(*
 * Copyright (C) 2002 Jun Kikuchi <kikuchi@bonnou.com>
 *
 * This file is part of ÍobN.
 *
 * ÍobN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ÍobN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *)
unit PacketIOUnit;

interface

uses
  Classes, SysUtils, Types, gcryptUnit, PacketUnit, KeyDataUnit, LockKeyDataUnit;

const
  EPACKET       = 'Ή̃t@C`ł';
  EPACKET_TAG   = 'pPbgwb_Ɉُ킪܂';
  EPACKET_TYPE  = 'Ή̃pPbg^Cvł';
  EPACKET_LEN   = 'Ή̃pPbg`ł';
  ESIGN         = 'TCpPbgɈُ킪܂';
  ESIGN_VER     = 'Ή̃TCo[Wł';
  ESIGN_LEN     = 'TCpPbgɈُ킪܂';
  ESIGN_TYPE    = 'Ή̃TC^Cvł';
  ESIGN_SUBTYPE = 'Ή̃TCTupPbg`ł';
  ES2K_SPEC     = 'ΉS2Kgp@ł';
  ES2K_TYPE     = 'ΉS2K^Cvł';
  EALGO_PUBKEY  = 'Ή̌JÍł';
  EALGO_CIPHER  = 'Ή̔閧Íł';
  EALGO_COMP    = 'Ή̈kł';
  EALGO_HASH    = 'Ή̃nbVł';
  EFEATURE      = 'ΉFeatureł';
  EPUBKEY_VER   = 'Ή̌JpPbgo[Wł';
  EPUBENC_VER   = 'Ή̈ÍZbVL[pPbgo[Wł';

type
  PACKET_TYPES = (
    PACKET_ESKEY         = 1,
    PACKET_SIG           = 2,
    PACKET_SECKEY        = 5,
    PACKET_PUBKEY        = 6,
    PACKET_SECSUBKEY     = 7,
    PACKET_COMPRESS      = 8,
    PACKET_LITERAL       = 11,
    PACKET_USERID        = 13,
    PACKET_PUBSUBKEY     = 14,
    PACKET_ENCMDC        = 18
  );

  SIGSUB_PACKET_TYPES = (
    SIGSUB_UNKNOWN         = 00,
    SIGSUB_SIG_TIME        = 02,
    SIGSUB_PREF_SYM_ALGOS  = 11,
    SIGSUB_KEYID           = 16,
    SIGSUB_PREF_HASH_ALGOS = 21,
    SIGSUB_PREF_COMP_ALGOS = 22,
    SIGSUB_FEATURES        = 30
  );

  EPacketIO    = class(Exception);
  EPacketRead  = class(EPacketIO);
  EPacketWrite = class(EPacketIO);

  TPacketIO = class
  protected
    procedure WriteHeader(S: TStream; PT: PACKET_TYPES; Len: Longword); overload;
    procedure WriteHeader(S: TStream; ST: SIGSUB_PACKET_TYPES; Len: Longword); overload;
    
    procedure ReadHeader(S: TStream; var PT: PACKET_TYPES; var Len: Longword); overload;
    procedure ReadHeader(S: TStream; var ST: SIGSUB_PACKET_TYPES; var Len: Longword); overload;

    procedure Write(S: TStream; P: TKeyData); overload;

    procedure WriteSigSub(S: TStream; P: PByteArray; Len: Integer); overload;
    procedure WriteSigSub(S: TStream; P: TSymKeyAlgoList); overload;
    procedure WriteSigSub(S: TStream; P: THashAlgoList); overload;
    procedure WriteSigSub(S: TStream; P: TCompAlgoList); overload;
    procedure WriteSigSub(S: TStream; P: TFeatureList); overload;
    procedure WriteSigSub(S: TStream; P: TKeyID); overload;

    procedure Write(S: TStream; P: TDEK); overload;
    procedure Write(S: TStream; P: TS2K); overload;

    procedure Read(S: TStream; P: TKeyData); overload;

    procedure ReadSigSub(S: TStream; var P: TSymKeyAlgoList; Len: Longword); overload;
    procedure ReadSigSub(S: TStream; var P: THashAlgoList; Len: Longword); overload;
    procedure ReadSigSub(S: TStream; var P: TCompAlgoList; Len: Longword); overload;
    procedure ReadSigSub(S: TStream; var P: TFeatureList; Len: Longword); overload;
    procedure ReadSigSub(S: TStream; P: TKeyID; Len: Longword); overload;
    
    procedure Read(S: TStream; P: TDEK); overload;
    procedure Read(S: TStream; P: TS2K); overload;
  public
    procedure WriteLongword(S: TStream; Value: Longword);
    procedure WriteWord(S: TStream; Value: Word);

    procedure ReadLongword(S: TStream; var Value: Longword);
    procedure ReadWord(S: TStream; var Value: Word);

    procedure WritePacketBody(S: TStream; P: TPubKeyPacket); overload;
    procedure WritePacketBody(S: TStream; P: TSecKeyPacket); overload;

    procedure ReadPacketBody(S: TStream; P: TPubKeyPacket); overload;
    procedure ReadPacketBody(S: TStream; P: TSecKeyPacket; Len: Longword); overload;

    function WriteSigHashedBody(S: TStream; P: TSignPacket): Cardinal; overload;
    function WriteSigHashedBody(S: TStream; P: TCertSignPacket): Cardinal; overload;
    procedure WriteSigUnhashedBody(S: TStream; P: TSignPacket);

    procedure Save(S: TStream; P: TPubKeyPacket); overload;
    procedure Save(S: TStream; P: TPubSubkeyPacket); overload;
    procedure Save(S: TStream; P: TSecKeyPacket); overload;
    procedure Save(S: TStream; P: TSecSubkeyPacket); overload;
    procedure Save(S: TStream; P: TUserIDPacket); overload;
    procedure Save(S: TStream; P: TSignPacket); overload;
    procedure Save(S: TStream; P: TCertSignPacket); overload;
    procedure Save(S: TStream; P: TPubKeyEncSessKeyPacket); overload;

    procedure Load(S: TStream; P: TPubKeyPacket); overload;
    procedure Load(S: TStream; P: TPubSubkeyPacket); overload;
    procedure Load(S: TStream; P: TSecKeyPacket); overload;
    procedure Load(S: TStream; P: TSecSubkeyPacket); overload;
    procedure Load(S: TStream; P: TUserIDPacket); overload;
    procedure Load(S: TStream; P: TSignPacket); overload;
    procedure Load(S: TStream; P: TCertSignPacket); overload;
    procedure Load(S: TStream; P: TPubKeyEncSessKeyPacket); overload;
  end;

function IsPacketType(Value: Byte): Boolean;
function IsSigSubPacketType(Value: Byte): Boolean;
function IsPubKeyAlgo(Value: Byte): Boolean;
function IsCipherAlgo(Value: Byte): Boolean;
function IsCompAlgo(Value: Byte): Boolean;
function IsHashAlgo(Value: Byte): Boolean;

implementation

uses
  UTCUnit;

function IsPacketType(Value: Byte): Boolean;
begin
  Result := (Value in [1,2,5,6,7,8,11,13,14,18]);
end;

function IsSigSubPacketType(Value: Byte): Boolean;
begin
  Result := (Value in [2,11,16,21,22,30]);
end;

function IsPubKeyAlgo(Value: Byte): Boolean;
begin
  Result := (Value in [1..3, 16..21]);
end;

function IsCipherAlgo(Value: Byte): Boolean;
begin
  Result := (Value in [0..9]);
end;

function IsCompAlgo(Value: Byte): Boolean;
begin
  Result := (Value in [0..2]);
end;

function IsHashAlgo(Value: Byte): Boolean;
begin
  Result := (Value in [1..7]);
end;

procedure TPacketIO.WriteHeader(S: TStream; PT: PACKET_TYPES; Len: Longword);
var
  B: Byte;
begin
  B := $C0 or Byte(PT);
  S.WriteBuffer(B, 1);

  case Len of
  0..191:
    begin
      B := Len;
      S.WriteBuffer(B, 1);
    end;
  192..8383:
    begin
      Len := Len - 192;
      B := (Len div 256) + 192;
      S.WriteBuffer(B, 1);
      B := Len mod 256;
      S.WriteBuffer(B, 1);
    end;
  8384..4294967295:
    begin
      B := $FF;
      S.WriteBuffer(B, 1);
      B := (Len shr 24) and $FF;
      S.WriteBuffer(B, 1);
      B := (Len shr 16) and $FF;
      S.WriteBuffer(B, 1);
      B := (Len shr 8) and $FF;
      S.WriteBuffer(B, 1);
      B := Len and $FF;
      S.WriteBuffer(B, 1);
    end;
  else
    Assert(False);
  end;
end;
{
procedure TPacketIO.WriteHeader(S: TStream; PT: PACKET_TYPES; Len: Longword);
var
  B: Byte;
begin
  B := $80 or ((Byte(PT) and $0F) shl 2);

  case Len of
  $0..$FF:
    begin
      S.WriteBuffer(B, 1);
      B := Byte(Len);
      S.WriteBuffer(B, 1);
    end;
  $100..$FFFF:
    begin
      B := B or $01;
      S.WriteBuffer(B, 1);
      WriteWord(S, Word(Len));
    end;
  $10000..$FFFFFFFF:
    begin
      B := B or $02;
      S.WriteBuffer(B, 1);
      WriteLongword(S, Len);
    end;
  else
    Assert(False);
  end;
end;
}
procedure TPacketIO.WriteHeader(S: TStream; ST: SIGSUB_PACKET_TYPES; Len: Longword);
var
  B: Byte;
begin
  Len := Len + 1;

  case Len of
  0..191:
    begin
      B := Len;
      S.WriteBuffer(B, 1);
    end;
  else
    Assert(False);
  end;
  
  B := Byte(ST);
  S.WriteBuffer(B, 1);
end;

procedure TPacketIO.ReadHeader(S: TStream; var PT: PACKET_TYPES; var Len: Longword);
var
  B: Byte;
  B1, B2, B3, B4, B5: Byte;
begin
  S.ReadBuffer(B, 1);
  if (B and $80) <> $80 then
    raise EPacketRead.Create(EPACKET_TAG);

  if (B and $40) = $40 then
  begin
    B := B and $3F;
    if not IsPacketType(B) then
      raise EPacketRead.Create(EPACKET_TYPE);
    PT := PACKET_TYPES(B);

    S.ReadBuffer(B1, 1);
    case B1 of
    0..191:
      Len := B1;
    192..254:
      begin
        S.ReadBuffer(B2, 1);
        Len := ((B1 - 192) shl 8) + B2 + 192;
      end;
    255:
      begin
        S.ReadBuffer(B2, 1);
        S.ReadBuffer(B3, 1);
        S.ReadBuffer(B4, 1);
        S.ReadBuffer(B5, 1);
        Len := (B2 shl 24) or (B3 shl 16) or (B4 shl 8) or B5;
      end;
    else
      raise EPacketRead.Create(EPACKET_LEN);
    end;
  end
  else
  begin
    if not IsPacketType((B and $3C) shr 2) then
      raise EPacketRead.Create(EPACKET_TYPE);
    PT := PACKET_TYPES((B and $3C) shr 2);

    case B and $03 of
    0:
      begin
        S.ReadBuffer(B1, 1);
        Len := B1;
      end;
    1:
      begin
        S.ReadBuffer(B1, 1);
        S.ReadBuffer(B2, 1);
        Len := (B1 shl 8) or B2;
      end;
    2:
      begin
        S.ReadBuffer(B1, 1);
        S.ReadBuffer(B2, 1);
        S.ReadBuffer(B3, 1);
        S.ReadBuffer(B4, 1);
        Len := (B1 shl 24) or (B2 shl 16) or (B3 shl 8) or B4;
      end;
    else
      raise EPacketRead.Create(EPACKET_LEN);
    end;
  end;
end;

procedure TPacketIO.ReadHeader(S: TStream; var ST: SIGSUB_PACKET_TYPES; var Len: Longword);
var
  B, B1, B2, B3, B4, B5: Byte;
begin
  S.ReadBuffer(B1, 1);
  case B1 of
  0..191:
    begin
      Len := B1;
    end;
  192..254:
    begin
      S.ReadBuffer(B2, 1);
      Len := ((B1 - 192) shl 8) + B2 + 192;
    end;
  255:
    begin
      S.ReadBuffer(B2, 1);
      S.ReadBuffer(B3, 1);
      S.ReadBuffer(B4, 1);
      S.ReadBuffer(B5, 1);
      Len := (B2 shl 24) or (B3 shl 16) or (B4 shl 8) or B5;
    end;
  else
    raise EPacketRead.Create(ESIGN_LEN);
  end;
  
  S.ReadBuffer(B, 1);
  if IsSigSubPacketType(B) then
    ST := SIGSUB_PACKET_TYPES(B)
  else
    ST := SIGSUB_UNKNOWN;

  Len := Len - 1;
end;

procedure TPacketIO.Write(S: TStream; P: TKeyData);
var
  I, J: Integer;
  B: Byte;
begin
  for I := 0 to P.NumKey -1 do
  begin
    for J := 0 to P.KeyLen[I] -1 do
    begin
      B := P[I, J];
      S.WriteBuffer(B, 1);
    end;
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: PByteArray; Len: Integer);
var
  B: Byte;
  I: Integer;
begin
  for I := 0 to Len -1 do
  begin
    B := Byte(P[I]);
    S.WriteBuffer(B, 1);
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: TSymKeyAlgoList);
begin
  if P <> nil then
  begin
    WriteHeader(S, SIGSUB_PREF_SYM_ALGOS, Length(P));
    WriteSigSub(S, PByteArray(P), Length(P));
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: THashAlgoList);
begin
  if P <> nil then
  begin
    WriteHeader(S, SIGSUB_PREF_HASH_ALGOS, Length(P));
    WriteSigSub(S, PByteArray(P), Length(P));
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: TCompAlgoList);
begin
  if P <> nil then
  begin
    WriteHeader(S, SIGSUB_PREF_COMP_ALGOS, Length(P));
    WriteSigSub(S, PByteArray(P), Length(P));
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: TFeatureList);
begin
  if P <> nil then
  begin
    WriteHeader(S, SIGSUB_FEATURES, Length(P));
    WriteSigSub(S, PByteArray(P), Length(P));
  end;
end;

procedure TPacketIO.WriteSigSub(S: TStream; P: TKeyID);
begin
  if P <> nil then
  begin
    WriteHeader(S, SIGSUB_KEYID, Length(P));
    WriteSigSub(S, PByteArray(P), Length(P));
  end;
end;

procedure TPacketIO.Write(S: TStream; P: TDEK);
var
  B: Byte;
  I: Integer;
begin
  B := 254;
  S.WriteBuffer(B, 1);

  B := Byte(P.CipherAlgo);
  S.WriteBuffer(B, 1);

  Write(S, P.S2K);

  for I := 0 to High(P.IV) do
  begin
    B := P.IV[I];
    S.WriteBuffer(B, 1);
  end;
end;

procedure TPacketIO.Write(S: TStream; P: TS2K);
var
  B: Byte;
  I: Integer;
begin
  B := $03;
  S.Write(B, 1);
  
  B := Byte(P.HashAlgo); 
  S.WriteBuffer(B, 1);
  
  for I := 0 to High(P.Salt) do
  begin
    B := P.Salt[I];
    S.WriteBuffer(B, 1);
  end;
  
  B := P.Count;
  S.WriteBuffer(B, 1);
end;

procedure TPacketIO.Read(S: TStream; P: TKeyData);
var
  I, J: Integer;
  B, B1, B2: Byte;
  Bit, Byte: Cardinal;
begin
  for I := 0 to P.NumKey -1 do
  begin
    S.ReadBuffer(B1, 1);
    S.ReadBuffer(B2, 1);
    Bit := (B1 shl 8) or B2;
    Byte := (Bit + 7) div 8;

    P.KeyLen[I] := Byte + 2;
    S.Position := S.Position - 2;
    for J := 0 to P.KeyLen[I] -1 do
    begin
      S.ReadBuffer(B, 1);
      P[I, J] := B;
    end;
  end;
end;

procedure TPacketIO.ReadSigSub(S: TStream; var P: TSymKeyAlgoList; Len: Longword);
var
  I: Integer;
  B: Byte;
begin
  SetLength(P, Len);
  for I := 0 to High(P) do
  begin
    S.ReadBuffer(B, 1);
    if not IsCipherAlgo(B) then
      raise EPacketRead.Create(EALGO_CIPHER);
    P[I] := GCRY_CIPHER_ALGOS(B)
  end;
end;

procedure TPacketIO.ReadSigSub(S: TStream; var P: THashAlgoList; Len: Longword);
var
  I: Integer;
  B: Byte;
begin
  SetLength(P, Len);
  for I := 0 to High(P) do
  begin
    S.ReadBuffer(B, 1);
    if not IsHashAlgo(B) then
      raise EPacketRead.Create(EALGO_HASH);
    P[I] := GCRY_MD_ALGOS(B)
  end;
end;

procedure TPacketIO.ReadSigSub(S: TStream; var P: TCompAlgoList; Len: Longword);
var
  I: Integer;
  B: Byte;
begin
  SetLength(P, Len);
  for I := 0 to High(P) do
  begin
    S.ReadBuffer(B, 1);
    if not IsCompAlgo(B) then
      raise EPacketRead.Create(EALGO_COMP);
    P[I] := COMPRESS_ALGOS(B)
  end;
end;

procedure TPacketIO.ReadSigSub(S: TStream; var P: TFeatureList; Len: Longword);
var
  I: Integer;
  B: Byte;
begin
  SetLength(P, Len);
  for I := 0 to High(P) do
  begin
    S.ReadBuffer(B, 1);
    if B <> $01 then
      raise EPacketRead.Create(EFEATURE);
    P[I] := FEATURES(B)
  end;
end;

procedure TPacketIO.ReadSigSub(S: TStream; P: TKeyID; Len: Longword);
var
  I: Integer;
  B: Byte;
begin
  for I := 0 to High(P) do
  begin
    S.ReadBuffer(B, 1);
    P[I] := B;
  end
end;

procedure TPacketIO.Read(S: TStream; P: TDEK);
var
  B: Byte;
  I: Integer;
begin
  S.ReadBuffer(B, 1);
  if B <> 254 then
    raise EPacketRead.Create(ES2K_SPEC);

  S.ReadBuffer(B, 1);
  if not IsCipherAlgo(B) then
    raise EPacketRead.Create(EALGO_CIPHER);
  P.CipherAlgo := GCRY_CIPHER_ALGOS(B);

  Read(S, P.S2K);

  for I := 0 to High(P.IV) do
  begin
    S.ReadBuffer(B, 1);
    P.IV[I] := B;
  end;
end;

procedure TPacketIO.Read(S: TStream; P: TS2K);
var
  B: Byte;
  I: Integer;
begin
  S.ReadBuffer(B, 1);
  if B <> $03 then
    raise EPacketRead.Create(ES2K_TYPE);

  S.ReadBuffer(B, 1);
  if not IsHashAlgo(B) then
    raise EPacketRead.Create(EALGO_HASH);
  P.HashAlgo := GCRY_MD_ALGOS(B);

  for I := 0 to High(P.Salt) do
  begin
    S.ReadBuffer(B, 1);
    P.Salt[I] := B;
  end;

  S.ReadBuffer(B, 1);
  P.Count := B;
end;

procedure TPacketIO.WriteLongword(S: TStream; Value: Longword);
var
  B: Byte;
begin
  B := Value shr 24;
  S.WriteBuffer(B, 1);
  B := (Value shr 16) and $FF;
  S.WriteBuffer(B, 1);
  B := (Value shr  8) and $FF;
  S.WriteBuffer(B, 1);
  B := Value and $FF;
  S.WriteBuffer(B, 1);
end;

procedure TPacketIO.WriteWord(S: TStream; Value: Word);
var
  B: Byte;
begin
  B := (Value shr 8) and $FF;
  S.WriteBuffer(B, 1);
  B := Value and $FF;
  S.WriteBuffer(B, 1);
end;

procedure TPacketIO.ReadLongword(S: TStream; var Value: Longword);
var
  B1, B2, B3, B4: Byte;
begin
  S.ReadBuffer(B1, 1);
  S.ReadBuffer(B2, 1);
  S.ReadBuffer(B3, 1);
  S.ReadBuffer(B4, 1);
  Value := (B1 shl 24) or (B2 shl 16) or (B3 shl 8) or B4;
end;

procedure TPacketIO.ReadWord(S: TStream; var Value: Word);
var
  B1, B2: Byte;
begin
  S.ReadBuffer(B1, 1);
  S.ReadBuffer(B2, 1);
  Value := (B1 shl 8) or B2;
end;

procedure TPacketIO.WritePacketBody(S: TStream; P: TPubKeyPacket);
var
  UTC: Longword;
  B: Byte;
begin
  Assert(P.Ver = 4);
  B := P.Ver;
  S.WriteBuffer(B, 1);

  UTC := DateTimeToUTC(P.Date);
  WriteLongword(S, UTC);

  B := Byte(P.PubKeyAlgo);
  S.WriteBuffer(B, 1);

  Write(S, P.KeyData);
end;

procedure TPacketIO.WritePacketBody(S: TStream; P: TSecKeyPacket);
begin
  WritePacketBody(S, TPubKeyPacket(P));
  Write(S, P.DEK);
  Write(S, P.SecKeyData);
end;

procedure TPacketIO.ReadPacketBody(S: TStream; P: TPubKeyPacket);
var
  B: Byte;
  UTC: Longword;
begin
  S.ReadBuffer(B, 1);
  if B <> 4 then
    raise EPacketRead.Create(EPUBKEY_VER);
  P.Ver := B;

  ReadLongword(S, UTC);
  P.Date := UTCToDateTime(UTC);

  S.ReadBuffer(B, 1);
  P.PubKeyAlgo := GCRY_PK_ALGOS(B);

  Read(S, P.KeyData);

  P.KeyID := P.CalcKeyID;
end;

procedure TPacketIO.ReadPacketBody(S: TStream; P: TSecKeyPacket; Len: Longword);
var
  B: Byte;
  I: Cardinal;
  SP: Int64;
begin
  SP := S.Position;
  
  ReadPacketBody(S, TPubKeyPacket(P));
  Read(S, P.DEK);

  P.SecKeyData.KeyLen[0] := (Len + SP) - S.Position;
  for I := 0 to P.SecKeyData.KeyLen[0] -1 do
  begin
    S.ReadBuffer(B, 1);
    P.SecKeyData[0, I] := B;
  end;
end;

function TPacketIO.WriteSigHashedBody(S: TStream; P: TSignPacket): Cardinal;
var
  B: Byte;
  UTC: Longword;
  MS, MS2: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    B := P.Ver;
    MS.WriteBuffer(B, 1);

    B := Byte(P.SignType);
    MS.WriteBuffer(B, 1);

    B := Byte(P.PubKeyAlgo);
    MS.WriteBuffer(B, 1);

    B := Byte(P.HashAlgo);
    MS.WriteBuffer(B, 1);

    MS2 := TMemoryStream.Create;
    try
      WriteHeader(MS2, SIGSUB_SIG_TIME, 4);
      UTC := DateTimeToUTC(P.Date);
      WriteLongword(MS2, UTC);
      
      WriteWord(MS, MS2.Size);
      MS2.SaveToStream(MS);
    finally
      MS2.Free;
    end;

    Result := MS.Size;
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

function TPacketIO.WriteSigHashedBody(S: TStream; P: TCertSignPacket): Cardinal;
var
  B: Byte;
  UTC: Longword;
  MS, MS2: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    B := P.Ver;
    MS.WriteBuffer(B, 1);

    B := Byte(P.SignType);
    MS.WriteBuffer(B, 1);

    B := Byte(P.PubKeyAlgo);
    MS.WriteBuffer(B, 1);

    B := Byte(P.HashAlgo);
    MS.WriteBuffer(B, 1);

    MS2 := TMemoryStream.Create;
    try
      WriteHeader(MS2, SIGSUB_SIG_TIME, 4);
      UTC := DateTimeToUTC(P.Date);
      WriteLongword(MS2, UTC);

      WriteSigSub(MS2, P.SymKeyAlgoList);
      WriteSigSub(MS2, P.HashAlgoList);
      WriteSigSub(MS2, P.CompAlgoList);
      WriteSigSub(MS2, P.FeatureList);
      
      WriteWord(MS, MS2.Size);
      MS2.SaveToStream(MS);
    finally
      MS2.Free;
    end;

    Result := MS.Size;
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.WriteSigUnhashedBody(S: TStream; P: TSignPacket);
var
  MS: TMemoryStream;
  B: Byte;
begin
  MS := TMemoryStream.Create;
  try
    WriteSigSub(MS, P.KeyID);
    WriteWord(S, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;

  B := P.HashLeft2[0];
  S.WriteBuffer(B, 1);
  B := P.HashLeft2[1];
  S.WriteBuffer(B, 1);

  Write(S, P.KeyData);
end;

procedure TPacketIO.Save(S: TStream; P: TPubKeyPacket);
var
  MS: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    WritePacketBody(MS, P);
    WriteHeader(S, PACKET_PUBKEY, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TPubSubkeyPacket);
var
  MS: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    WritePacketBody(MS, P);
    WriteHeader(S, PACKET_PUBSUBKEY, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TSecKeyPacket);
var
  MS: TMemoryStream;
begin
  Assert(P.IsLocked);
  MS := TMemoryStream.Create;
  try
    WritePacketBody(MS, P);
    WriteHeader(S, PACKET_SECKEY, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TSecSubkeyPacket);
var
  MS: TMemoryStream;
begin
  Assert(P.IsLocked);
  MS := TMemoryStream.Create;
  try
    WritePacketBody(MS, P);
    WriteHeader(S, PACKET_SECSUBKEY, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TUserIDPacket);
var
  UTF8: UTF8String;
  B: Char;
  I: Integer;
begin
  UTF8 := UTF8Encode(P.UID);

  WriteHeader(S, PACKET_USERID, Length(UTF8));
  
  for I := 1 to Length(UTF8) do
  begin
    B := UTF8[I];
    S.WriteBuffer(B, 1);
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TSignPacket);
var
  MS: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    WriteSigHashedBody(MS, P);
    WriteSigUnhashedBody(MS, P);

    WriteHeader(S, PACKET_SIG, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TCertSignPacket);
var
  MS: TMemoryStream;
begin
  MS := TMemoryStream.Create;
  try
    WriteSigHashedBody(MS, P);
    WriteSigUnhashedBody(MS, P);

    WriteHeader(S, PACKET_SIG, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Save(S: TStream; P: TPubKeyEncSessKeyPacket);
var
  MS: TMemoryStream;
  B: Byte;
  I: Integer;
begin
  MS := TMemoryStream.Create;
  try
    B := P.Ver;
    MS.WriteBuffer(B, 1);

    for I := 0 to High(P.KeyID) do
    begin
      B := P.KeyID[I];
      MS.WriteBuffer(B, 1);
    end;

    B := Byte(P.PubKeyAlgo);
    MS.WriteBuffer(B, 1);

    Write(MS, P.KeyData);

    WriteHeader(S, PACKET_ESKEY, MS.Size);
    MS.SaveToStream(S);
  finally
    MS.Free;
  end;
end;

procedure TPacketIO.Load(S: TStream; P: TPubKeyPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_PUBKEY then
    raise EPacketRead.Create(EPACKET);
  ReadPacketBody(S, P);
  
  if P.PubKeyAlgo <> GCRY_PK_DSA then
    raise EPacketRead.Create(EPACKET_TYPE);
end;

procedure TPacketIO.Load(S: TStream; P: TPubSubkeyPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_PUBSUBKEY then
    raise EPacketRead.Create(EPACKET);
  ReadPacketBody(S, P);

  if P.PubKeyAlgo <> GCRY_PK_ELG_E then
    raise EPacketRead.Create(EPACKET_TYPE);
end;

procedure TPacketIO.Load(S: TStream; P: TSecKeyPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_SECKEY then
    raise EPacketRead.Create(EPACKET);
  ReadPacketBody(S, P, Len);
end;

procedure TPacketIO.Load(S: TStream; P: TSecSubkeyPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_SECSUBKEY then
    raise EPacketRead.Create(EPACKET);
  ReadPacketBody(S, P, Len);
end;

procedure TPacketIO.Load(S: TStream; P: TUserIDPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
  I: Integer;
  B: Byte;
  UTF8: UTF8String;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_USERID then
    raise EPacketRead.Create(EPACKET);

  UTF8 := '';
  for I := 0 to Len -1 do
  begin
    S.ReadBuffer(B, 1);
    UTF8 := UTF8 + Char(B);
  end;

  P.UID := UTF8Decode(UTF8);
end;

procedure TPacketIO.Load(S: TStream; P: TSignPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
  SubLen: Word;
  B: Byte;
  ST: SIGSUB_PACKET_TYPES;
  SigSubLen: Longword;
  UTC: Longword;
  MS: TMemoryStream;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_SIG then
    raise EPacketRead.Create(EPACKET);

  S.ReadBuffer(B, 1);
  if B <> 4 then
    raise EPacketRead.Create(ESIGN_VER);
  P.Ver := B;

  S.ReadBuffer(B, 1);
  if not (B in [$13, $18]) then
    raise EPacketRead.Create(ESIGN_TYPE);
  P.SignType := SIGN_TYPES(B);

  S.ReadBuffer(B, 1);
  if not IsPubKeyAlgo(B) then
    raise EPacketRead.Create(EALGO_PUBKEY);
  P.PubKeyAlgo := GCRY_PK_ALGOS(B);

  S.ReadBuffer(B, 1);
  if not IsHashAlgo(B) then
    raise EPacketRead.Create(EALGO_HASH);
  P.HashAlgo := GCRY_MD_ALGOS(B);

  ReadWord(S, SubLen);
  if SubLen = 0 then
    raise EPacketRead.Create(ESIGN);
  MS := TMemoryStream.Create;
  try
    MS.CopyFrom(S, SubLen);
    MS.Position := 0;
    while MS.Position < MS.Size do
    begin
      ReadHeader(MS, ST, SigSubLen);
      case ST of
      SIGSUB_SIG_TIME:
        begin
          ReadLongword(MS, UTC);
          P.Date := UTCToDateTime(UTC);
        end;
      else
        MS.Position := MS.Position + SigSubLen;
      end;
    end;
  finally
    MS.Free;
  end;

  ReadWord(S, SubLen);
  if SubLen <> 0 then
  begin
    MS := TMemoryStream.Create;
    try
      MS.CopyFrom(S, SubLen);
      MS.Position := 0;
      while MS.Position < MS.Size do
      begin
        ReadHeader(MS, ST, SigSubLen);
        case ST of
        SIGSUB_KEYID:
          ReadSigSub(MS, P.KeyID, SigSubLen);
        else
          MS.Position := MS.Position + SigSubLen;
        end;
      end;
    finally
      MS.Free;
    end;
  end;

  S.ReadBuffer(B, 1);
  P.HashLeft2[0] := B;
  S.ReadBuffer(B, 1);
  P.HashLeft2[1] := B;

  Read(S, P.KeyData);
end;

procedure TPacketIO.Load(S: TStream; P: TCertSignPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
  SubLen: Word;
  B: Byte;
  ST: SIGSUB_PACKET_TYPES;
  SigSubLen: Longword;
  UTC: Longword;
  MS: TMemoryStream;
begin
  ReadHeader(S, PT, Len);
  if PT <> PACKET_SIG then
    raise EPacketRead.Create(EPACKET);

  S.ReadBuffer(B, 1);
  if B <> 4 then
    raise EPacketRead.Create(ESIGN_VER);
  P.Ver := B;

  S.ReadBuffer(B, 1);
  if not (B in [$13, $18]) then
    raise EPacketRead.Create(ESIGN_TYPE);
  P.SignType := SIGN_TYPES(B);

  S.ReadBuffer(B, 1);
  if not IsPubKeyAlgo(B) then
    raise EPacketRead.Create(EALGO_PUBKEY);
  P.PubKeyAlgo := GCRY_PK_ALGOS(B);

  S.ReadBuffer(B, 1);
  if not IsHashAlgo(B) then
    raise EPacketRead.Create(EALGO_HASH);
  P.HashAlgo := GCRY_MD_ALGOS(B);

  ReadWord(S, SubLen);
  if SubLen = 0 then
    raise EPacketRead.Create(ESIGN);
  MS := TMemoryStream.Create;
  try
    MS.CopyFrom(S, SubLen);
    MS.Position := 0;
    while MS.Position < MS.Size do
    begin
      ReadHeader(MS, ST, SigSubLen);
      case ST of
      SIGSUB_SIG_TIME:
        begin
          ReadLongword(MS, UTC);
          P.Date := UTCToDateTime(UTC);
        end;
      SIGSUB_PREF_SYM_ALGOS:
        ReadSigSub(MS, P.SymKeyAlgoList, SigSubLen);
      SIGSUB_PREF_HASH_ALGOS:
        ReadSigSub(MS, P.HashAlgoList, SigSubLen);
      SIGSUB_PREF_COMP_ALGOS:
        ReadSigSub(MS, P.CompAlgoList, SigSubLen);
      SIGSUB_FEATURES:
        ReadSigSub(MS, P.FeatureList, SigSubLen);
      else
        MS.Position := MS.Position + SigSubLen;
      end;
    end;
  finally
    MS.Free;
  end;

  ReadWord(S, SubLen);
  if SubLen <> 0 then
  begin
    MS := TMemoryStream.Create;
    try
      MS.CopyFrom(S, SubLen);
      MS.Position := 0;
      while MS.Position < MS.Size do
      begin
        ReadHeader(MS, ST, SigSubLen);
        case ST of
        SIGSUB_KEYID:
          ReadSigSub(MS, P.KeyID, SigSubLen);
        else
          MS.Position := MS.Position + SigSubLen;
        end;
      end;
    finally
      MS.Free;
    end;
  end;

  S.ReadBuffer(B, 1);
  P.HashLeft2[0] := B;
  S.ReadBuffer(B, 1);
  P.HashLeft2[1] := B;

  Read(S, P.KeyData);
end;

procedure TPacketIO.Load(S: TStream; P: TPubKeyEncSessKeyPacket);
var
  PT: PACKET_TYPES;
  Len: Longword;
  B: Byte;
  I: Integer;
begin
  ReadHeader(S, PT, Len);

  if PT <> PACKET_ESKEY then
    raise EPacketRead.Create(EPACKET);

  S.ReadBuffer(B, 1);
  if B <> 3 then
    raise EPacketRead.Create(EPUBENC_VER);
  P.Ver := B;

  for I := 0 to High(P.KeyID) do
  begin
    S.ReadBuffer(B, 1);
    P.KeyID[I] := B;
  end;

  S.ReadBuffer(B, 1);
  if not IsPubKeyAlgo(B) then
    raise EPacketRead.Create(EALGO_PUBKEY);
  P.PubKeyAlgo := GCRY_PK_ALGOS(B);

  Read(S, P.KeyData);
end;

end.
