/*$Id$*/
package nicobrowser.config;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.configuration.Configuration;
import org.apache.commons.configuration.ConfigurationException;
import org.apache.commons.configuration.PropertiesConfiguration;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public final class Config {

    private static Log log = LogFactory.getLog(Config.class);
    private static Config instance;
    private final PropertiesConfiguration properties;
    private static final String APPLICATION_NAME = "nicobrowser";
    private static final String CONFIG_NAME = APPLICATION_NAME + ".properties";
    private static final String FEEDURL_NAME = "feedurl.txt";
    private static final String MYLIST_CONFIG_NAME = "mylist.txt";
    private static final File APP_HOME = new File(System.getProperty("user.home", "."), "." + APPLICATION_NAME);
    private static final File CONFIG_FILE = new File(APP_HOME, CONFIG_NAME);
    private static final File FEEDURL_FILE = new File(APP_HOME, FEEDURL_NAME);
    private static final File MYLIST_CONFIG_FILE = new File(APP_HOME, MYLIST_CONFIG_NAME);
    public static final String P_PATH_DB = "path.db";
    public static final String P_PATH_SAVEFILE = "path.savefile";
    public static final String P_SAVEFILE_PATTERN = "savefilename.pattern";
    public static final String P_SAVEFILE_REPLACE_FROM = "savefilename.replace.from";
    public static final String P_SAVEFILE_REPLACE_TO = "savefilename.replace.to";
    public static final String P_FILE_ENCODING = "encoding";
    public static final String P_NICOVIDEO_MAIL = "nicovideo.mail";
    public static final String P_NICOVIDEO_PASSWORD = "nicovideo.password";
    public static final String P_DOWNLOAD_RETRY = "download.retry";
    public static final String P_DOWNLOAD_WAIT = "download.wait";
    public static final String P_DOWNLOAD_LOW = "download.low";
    public static final String P_DOWNLOAD_MYLIST = "download.mylist";
    private static final int DEFAULT_MAX_RETRY = 3;
    private static final int DEFAULT_WAIT_TIME = 15;
    private static final boolean DEFAULT_LOW_FILE = true;

    /**
     * プログラム実行に必要なコンフィグファイルを作成する.
     * @return 今回コンフィグを作成したのであればtrue. 既に存在していたため, ファイル作成を行わなかった場合にはfalse.
     * @throws java.io.IOException ファイル作成に失敗した.
     */
    public static boolean createNewConfigFiles() throws IOException {
        boolean result = false;
        if (!CONFIG_FILE.exists()) {
            createNewConfigFile(CONFIG_FILE);
            result = true;
            log.info("コンフィグファイルを作成しました: " + CONFIG_FILE.getCanonicalPath());
        }
        if (!FEEDURL_FILE.exists()) {
            InputStream resource = null;
            try {
                resource = ClassLoader.getSystemResourceAsStream("resources/" + FEEDURL_NAME);
                createNewFileFromResource(resource, FEEDURL_FILE);
                result = true;
                log.info("FEED URLファイルを作成しました: " + FEEDURL_FILE.getCanonicalPath());
            } finally {
                if (resource != null) {
                    resource.close();
                }
            }
        }
        if (!MYLIST_CONFIG_FILE.exists()) {
            InputStream resource = null;
            try {
                resource = ClassLoader.getSystemResourceAsStream("resources/" + MYLIST_CONFIG_NAME);
                createNewFileFromResource(resource, MYLIST_CONFIG_FILE);
                log.info("MYLISTファイルを作成しました: " + MYLIST_CONFIG_FILE.getCanonicalPath());
            } finally {
                if (resource != null) {
                    resource.close();
                }
            }
        }
        return result;
    }

    /**
     * 新しいプロパティを設定する. 新しいプロパティに値がない場合, 現在のままとなる.
     * @param p 新規プロパティ.
     * @throws IOException
     */
    public void updateConfigFile(Configuration p) throws ConfigurationException {
        updatePropertyValue(p, P_NICOVIDEO_MAIL);
        updatePropertyValue(p, P_NICOVIDEO_PASSWORD);
        updatePropertyValue(p, P_FILE_ENCODING);

        updatePropertyValue(p, P_PATH_DB);
        updatePropertyValue(p, P_PATH_SAVEFILE);
        updatePropertyValue(p, P_SAVEFILE_PATTERN);
        updatePropertyValue(p, P_SAVEFILE_REPLACE_FROM);
        updatePropertyValue(p, P_SAVEFILE_REPLACE_TO);

        updatePropertyValue(p, P_DOWNLOAD_RETRY);
        updatePropertyValue(p, P_DOWNLOAD_WAIT);
        updatePropertyValue(p, P_DOWNLOAD_LOW);

        updatePropertyValueArray(p, P_DOWNLOAD_MYLIST);

        properties.save();
    }

    /**
     * 新しいプロパティを設定する. 新しいプロパティに値がない場合, 現在のままとなる.
     * @param newProp 新規プロパティ.
     * @param key 設定するプロパティキー.
     */
    private void updatePropertyValue(Configuration newProp, String key) {
        String value = newProp.getString(key, properties.getString(key));
        properties.setProperty(key, value);
    }

    private void updatePropertyValueArray(Configuration newProp, String key) {
        String[] values = newProp.getStringArray(key);
        if (values.length < 1) {
            values = properties.getStringArray(key);
        }
        properties.setProperty(key, values);
    }

    private static void createNewConfigFile(File file) throws IOException {
        ArrayList<CharSequence> props = new ArrayList<CharSequence>();

        StringBuilder dbpath = new StringBuilder(P_PATH_DB + "=");
        File dbDir = new File(APP_HOME, "db");
        dbDir.mkdirs();
        //Windowsのパス区切りバックスペースをエスケープするための処理も入れている.
        dbpath.append(dbDir.getAbsolutePath().replaceAll("\\\\", "\\\\\\\\"));
        props.add(dbpath);

        StringBuilder savepath = new StringBuilder(P_PATH_SAVEFILE + "=");
        File saveDir = new File(APP_HOME, "flv");
        saveDir.mkdirs();
        savepath.append(saveDir.getAbsolutePath().replaceAll("\\\\", "\\\\\\\\"));
        props.add(savepath);

        props.add(P_SAVEFILE_PATTERN + "={title}");
        props.add(P_SAVEFILE_REPLACE_FROM + "=\\/\\\\:*?\\\"<>|.");
        props.add(P_SAVEFILE_REPLACE_TO + "=_");

        props.add(P_FILE_ENCODING + "=" + System.getProperty("file.encoding"));

        props.add(P_NICOVIDEO_MAIL + "=");
        props.add(P_NICOVIDEO_PASSWORD + "=");
        props.add(P_DOWNLOAD_RETRY + "=" + DEFAULT_MAX_RETRY);
        props.add(P_DOWNLOAD_WAIT + "=" + DEFAULT_WAIT_TIME);
        props.add(P_DOWNLOAD_LOW + "=" + DEFAULT_LOW_FILE);
        props.add(P_DOWNLOAD_MYLIST + "=");

        FileUtils.writeLines(file, props);
    }

    /**
     * リソースから新しいファイルを作成します. リソースファイルはUTF-8で記述する必要があります.
     * 出力されるファイルはOSのネイティブエンコーディングになります.
     * @param resource 入力リソース.
     * @param dest 出力ファイル.
     * @throws IOException 出力できなかった場合.
     */
    private static void createNewFileFromResource(InputStream resource, File dest) throws IOException {
        List<String> list = new ArrayList<String>();
        BufferedReader br = new BufferedReader(new InputStreamReader(resource, "UTF-8"));
        while (true) {
            String text = br.readLine();
            if (text == null) {
                break;
            }
            list.add(text);
        }
        FileUtils.writeLines(dest, list);
    }

    private Config() {
        try {
            properties = new PropertiesConfiguration(CONFIG_FILE);
        } catch (Exception ex) {
            log.fatal("コンフィグの読み込みに失敗: " + CONFIG_FILE);
            throw new RuntimeException(ex);
        }

        try {
            FileUtils.readLines(FEEDURL_FILE);
        } catch (IOException ex) {
            log.fatal("コンフィグの読み込みに失敗: " + CONFIG_FILE);
            throw new RuntimeException(ex);
        }
    }

    public static Config getInstance() {
        if (instance == null) {
            instance = new Config();
        }
        return instance;
    }

    /**
     * @return ニコニコ動画ログインID.
     */
    public String getNicoMail() {
        return properties.getString(P_NICOVIDEO_MAIL);
    }

    /**
     * @return ニコニコ動画ログインパスワード.
     */
    public String getNicoPassword() {
        return properties.getString(P_NICOVIDEO_PASSWORD);
    }

    /** @return DBパス. コンフィグで設定した値(保存ディレクトリ)でなく, DBファイルのパスが変えることに注意. */
    public String getDbFile() {
        return new File(properties.getString(P_PATH_DB), "nicodb").getAbsolutePath();
    }

    /** @return 保存先の指定. */
    public String getSrcSaveDir() {
        return properties.getString(P_PATH_SAVEFILE);
    }

    /**@return 保存ファイル名の命名規則. */
    public String getFileNamePattern() {
        return properties.getString(P_SAVEFILE_PATTERN, "{title}");
    }

    public String getFileNameReplaceFrom() {
        return properties.getString(P_SAVEFILE_REPLACE_FROM, "/\\:*?\"<>|.");
    }

    public String getFileNameReplaceTo() {
        return properties.getString(P_SAVEFILE_REPLACE_TO, "_");
    }

    /** @return feedurl.txtの文字エンコーディング. */
    public String getEncoding() {
        String res = properties.getString(P_FILE_ENCODING, System.getProperty("file.encoding"));
        return res;
    }

    /**
     * 失敗したダウンロードファイルの最大リトライ回数を取得する.
     * @return リトライ回数.
     */
    public int getMaxRetry() {
        return properties.getInt(P_DOWNLOAD_RETRY, DEFAULT_MAX_RETRY);
    }

    /**
     * 前回ダウンロード開始から最低何秒後から次回ダウンロードを開始するか.
     * @return 待ち時間(秒).
     */
    public int getWaitTime() {
        return properties.getInt(P_DOWNLOAD_WAIT, DEFAULT_WAIT_TIME);
    }

    /**
     * エコノミー動画をダウンロードする必要があるか.
     * @return エコノミー動画もダウンロード対称にする場合はtrue.
     */
    public boolean needLowFile() {
        return properties.getBoolean(P_DOWNLOAD_LOW, DEFAULT_LOW_FILE);
    }

    public List<String> getDownLoadMyList() {
        final List<String> mylists = new ArrayList<String>();
        try {
            final List<?> lines = FileUtils.readLines(MYLIST_CONFIG_FILE);
            final Pattern p = Pattern.compile("(^\\d+)");
            for (Object l : lines) {
                Matcher m = p.matcher(l.toString());
                if (m.find()) {
                    mylists.add(m.group(1));
                }
            }
        } catch (IOException ex) {
            Logger.getLogger(Config.class.getName()).log(Level.SEVERE, "読み込みに失敗: " + MYLIST_CONFIG_NAME, ex);
        }
        String[] res = properties.getStringArray(P_DOWNLOAD_MYLIST);
        mylists.addAll(Arrays.asList(res));
        return mylists;
    }

    public List<NicoFeed> getNicoFeeds() {
        List<NicoFeed> list = new ArrayList<NicoFeed>();
        try {
            List<?> lines = FileUtils.readLines(FEEDURL_FILE, getEncoding());
            for (Object line : lines) {
                final String str = line.toString();
                if (StringUtils.isBlank(str) || str.startsWith("#")) {
                    // 空行, コメント行はスキップ.
                    continue;
                }

                String[] values = str.split(",");
                NicoFeed feed = new NicoFeed(values[0].trim(), Integer.parseInt(values[1].trim()));
                list.add(feed);
            }
        } catch (IOException ex) {
            log.error("ファイルが見つかりません: " + FEEDURL_FILE);
        }
        return list;
    }

    public static File getConfigfile() {
        return CONFIG_FILE;
    }

    public static File getFeedUrlFile() {
        return FEEDURL_FILE;
    }

    public static File getMylistConfigFile() {
        return MYLIST_CONFIG_FILE;
    }

    public static File getAppHome() {
        return APP_HOME;
    }
}
