/* ------------------------------------------------------------------------- */
/*
 *  smtp.h
 *
 *  Copyright (c) 2004 - 2008, clown. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *    - Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *    - Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *    - No names of its contributors may be used to endorse or promote
 *      products derived from this software without specific prior written
 *      permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 *  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 *  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  Last-modified: Sun 26 Oct 2008 17:14:00 JST
 */
/* ------------------------------------------------------------------------- */
#ifndef CLX_SMTP_H
#define CLX_SMTP_H

#include <iostream>
#include <string>
#include <deque>
#include <stdexcept>
#include "tcp.h"
#include "lexical_cast.h"
#include "strip.h"

namespace clx {
	/* --------------------------------------------------------------------- */
	//  smtp_error
	/* --------------------------------------------------------------------- */
	class smtp_error : public std::runtime_error {
	public:
		typedef char char_type;
		typedef std::string string_type;
		
		explicit smtp_error(int code, const string_type& what_arg) :
			std::runtime_error(what_arg), code_(code) {}
		
		virtual ~smtp_error() throw() {}
		
		int code() const { return code_; }
		
	private:
		int code_;
	};
	
	/* --------------------------------------------------------------------- */
	//  basic_smtp
	/* --------------------------------------------------------------------- */
	template <class Socket>
	class basic_smtp : public Socket {
	public:
		typedef Socket socket_type;
		typedef char char_type;
		typedef std::basic_string<char_type> string_type;
		typedef std::pair<int, string_type> response_type;
		typedef std::deque<response_type> response_array;
		
		basic_smtp() : socket_type(), id_(), host_(), port_(0), res_() {}
		
		basic_smtp(const basic_smtp& cp) :
			socket_type(cp), id_(cp.id_), host_(cp.host_), port_(cp.port_), res_(cp.res_) {}
		
		explicit basic_smtp(const string_type& host, int port = 25) :
			socket_type(), id_(), host_(host), port_(port), res_() {
			try {
				this->start();
			}
			catch (socket_error& e) {
				(*this) = basic_smtp();
				throw e;
			}
			catch (std::runtime_error& e) {
				(*this) = basic_smtp();
				throw e;
			}
		}
		
		explicit basic_smtp(const char_type* host, int port = 25) :
			socket_type(), id_(), host_(host), port_(port), res_() {
			try {
				this->start();
			}
			catch (socket_error& e) {
				(*this) = basic_smtp();
				throw e;
			}
			catch (std::runtime_error& e) {
				(*this) = basic_smtp();
				throw e;
			}
		}
		
		virtual ~basic_smtp() throw() { this->finish(); }
		
		basic_smtp& start() {
			if (!this->is_open()) this->connect(host_, port_);
			basic_sockbuf<socket_type, char_type> sbuf(*this);
			std::basic_iostream<char_type> ss(&sbuf);
			this->xresponse(ss);
			return *this;
		}
		
		basic_smtp& start(const string_type& host, int port = 25) {
			host_ = host;
			port_ = port;
			return this->start();
		}
		
		basic_smtp& start(const char_type* host, int port = 25) {
			string_type tmp(host);
			return this->start(tmp, port);
		}
		
		void finish() {
			if (this->is_open()) {
				basic_sockbuf<socket_type, char_type> sbuf(*this);
				std::basic_iostream<char_type> ss(&sbuf);
				ss << "QUIT\r\n";
				this->xresponse(ss);
				this->close();
			}
		}
		
		basic_smtp& login(const string_type& id) {
			id_ = id;
			
			basic_sockbuf<socket_type, char_type> sbuf(*this);
			std::basic_iostream<char_type> ss(&sbuf);
			ss << ("EHLO " + id_ + " " + host_ + "\r\n");
			this->xresponse(ss);
			
			return *this;
		}
		
		basic_smtp& login(const char_type* id) {
			string_type tmp(id);
			return this->login(tmp);
		}
		
		basic_smtp& mail(const string_type& from, const string_type& to, const string_type& message) {
			basic_sockbuf<socket_type, char_type> sbuf(*this);
			std::basic_iostream<char_type> ss(&sbuf);
			string_type buf;
			
			// A mail transaction has 4 steps.
			buf = "MAIL FROM:<" + from + ">\r\n";
			ss << buf;
			this->xresponse(ss);
			
			buf = "RCPT TO:<" + to + ">\r\n";
			ss << buf;
			this->xresponse(ss);
			
			ss << "DATA\r\n";
			this->xresponse(ss);
			
			ss << message;
			this->xresponse(ss);
			
			return *this;
		}
		
		basic_smtp& mail(const char_type* from, const char_type* to, const char_type* message) {
			string_type tmp_from(from);
			string_type tmp_to(to);
			string_type tmp_msg(message);
			return this->mail(tmp_from, tmp_to, tmp_msg);
		}
		
		// get response messages from the server
		response_array& responses() { return res_; }
		const response_array& responses() const { return res_; }
		const response_type& response(size_t index) { return res_.at(index); }
		
	private:
		string_type id_;
		string_type host_;
		int port_;
		response_array res_;
		
		template <class InputStream>
		void xresponse(InputStream& sin) {
			std::string buf;
			while (std::getline(sin, buf)) {
				chomp(buf);
				response_type tmp;
				tmp.first = lexical_cast<int>(buf.substr(0, 3));
				tmp.second = buf.substr(4);
				res_.push_front(tmp);
				if (tmp.first >= 400) throw smtp_error(tmp.first, tmp.second);
				if (buf[3] == ' ') break;
			}
		}
	};
	
	typedef basic_smtp<tcp::socket> smtp;
}

#endif // CLX_SMTP_H
