package charactermanaj.model.io;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import charactermanaj.model.ColorGroup;
import charactermanaj.model.PartsCategory;
import charactermanaj.model.PartsIdentifier;
import charactermanaj.model.PartsSpec;

/**
 * パーツ名の末尾が、カラーグループの表記を括弧でくくったものと同じであれば、
 * そのパーツ固有のカラーグループとして設定するためのデコレータ.<br>
 * このクラス自身はパーツのロードは行わず、コンストラクタで指定したローダーによりロードを行い、
 * その結果に対してカラーグループの設定を行う.<br>
 * 
 * @author seraphy
 *
 */
public class PartsSpecDecorateLoader implements PartsDataLoader {

	private PartsDataLoader parent;
	
	private Collection<ColorGroup> colorGroups;

	/**
	 * パーツローダとカラーグループを指定して構築する.
	 * @param parent 元パーツローダー
	 * @param colorGroups カラーグループのコレクション、nullの場合は空とみなす.
	 */
	public PartsSpecDecorateLoader(PartsDataLoader parent, Collection<ColorGroup> colorGroups) {
		if (parent == null) {
			throw new IllegalArgumentException();
		}
		if (colorGroups == null) {
			colorGroups = Collections.emptyList();
		}
		this.parent = parent;
		this.colorGroups = colorGroups;
	}
	
	public Map<PartsIdentifier, PartsSpec> load(PartsCategory category) {
		Map<PartsIdentifier, PartsSpec> partsSpecs = parent.load(category);
		decolatePartsSpec(partsSpecs);
		return partsSpecs;
	}
	
	/**
	 * パーツ識別子の表示名の末尾がカラーグループの表示名を括弧でくくったものと等しい場合、
	 * パーツ設定のカラーグループを、そのカラーグループとして設定する.
	 * @param partsSpecs パーツマップ
	 */
	protected void decolatePartsSpec(Map<PartsIdentifier, PartsSpec> partsSpecs) {
		// パーツ名にカラーグループが含まれる場合、それを登録する.
		for (ColorGroup colorGroup : colorGroups) {
			String suffix = "(" + colorGroup.getLocalizedName() + ")";
			for (PartsSpec partsSpec : partsSpecs.values()) {
				if (partsSpec.getPartsIdentifier().getLocalizedPartsName().endsWith(suffix)) {
					partsSpec.setColorGroup(colorGroup);
				}
			}
		}
	}
	
}
