package charactermanaj.graphics.io;

import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.IOException;

import javax.imageio.ImageIO;

/**
 * 画像を読み取ります.<br>
 * @author seraphy
 */
public class ImageLoaderImpl implements ImageLoader {

	/**
	 * 画像リソースからBufferedImageを返します.<br>
	 * 返される形式はARGBに変換されています.<br>
	 * @param imageResource 画像リソース
	 * @throws IOException 読み取りに失敗した場合
	 */
	public BufferedImage load(ImageResource imageResource) throws IOException {
		if (imageResource == null) {
			throw new IllegalArgumentException();
		}

		BufferedImage img = ImageIO.read(imageResource.openStream());
		if (img == null) {
			throw new IOException("unsupported image");
		}
		
		// ARGB形式でなければ変換する.
		img = convertARGB(img);
		
		return img;
	}
	
	/**
	 * イメージがARGB形式でなければ、ARGB形式に変換して返す.<br>
	 * そうでなければ、そのまま返す. 
	 * @param image イメージ
	 * @return ARGB形式のイメージ
	 */
	protected BufferedImage convertARGB(BufferedImage image) {
		if (image == null) {
			throw new IllegalArgumentException();
		}
		int typ = image.getType();
		if (typ == BufferedImage.TYPE_INT_ARGB) {
			return image;
		}
		// ARGB形式でなければ変換する.
		BufferedImage img2 = new BufferedImage(image.getWidth(), image.getHeight(), BufferedImage.TYPE_INT_ARGB);
		Graphics g = img2.getGraphics();
		try {
			g.drawImage(image, 0, 0, null);
		} finally {
			g.dispose();
		}
		return img2;
	}
}
