package charactermanaj.model.io;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;

import charactermanaj.model.CharacterData;
import charactermanaj.model.PartsIdentifier;
import charactermanaj.model.PartsSpec;

public abstract class AbstractCharacterDataFileWriter implements CharacterDataWriter {

	protected File outFile;
	
	protected File tmpFile;
	
	protected Exception occureException;
	
	protected AbstractCharacterDataFileWriter(File outFile) throws IOException {
		if (outFile == null) {
			throw new IllegalArgumentException();
		}
		
		if (outFile.exists()) {
			if (!outFile.canWrite()) {
				throw new IOException("not writable: " + outFile);
			}
		}
		File tmpFile = new File(outFile.getPath() + ".tmp");
		
		this.tmpFile = tmpFile;
		this.outFile = outFile;
	}
	
	public void writeExportProp(Properties prop) throws IOException {
		if (prop == null) {
			throw new IllegalArgumentException();
		}
		
		try {
			internalWriteExportProp(prop);
			
		} catch (IOException ex) {
			occureException = ex;
			throw ex;

		} catch (Exception ex) {
			occureException = ex;
			IOException ex2 = new IOException("write characterdata failed.");
			ex2.initCause(ex);
			throw ex2;
		}
	}
	
	protected abstract void internalWriteExportProp(Properties prop) throws IOException;
	

	public void writeCharacterData(CharacterData characterData) throws IOException {
		if (characterData == null) {
			throw new IllegalArgumentException();
		}
		
		try {
			internalWriteCharacterData(characterData);
			
		} catch (IOException ex) {
			occureException = ex;
			throw ex;

		} catch (Exception ex) {
			occureException = ex;
			IOException ex2 = new IOException("write characterdata failed.");
			ex2.initCause(ex);
			throw ex2;
		}
	}
	
	protected abstract void internalWriteCharacterData(CharacterData characterData) throws IOException;
	
	public void writeTextUTF16LE(String name, String contents) throws IOException {
		if (name == null) {
			throw new IllegalArgumentException();
		}
		
		try {
			internalWriteTextUTF16LE(name, contents);
			
		} catch (IOException ex) {
			occureException = ex;
			throw ex;

		} catch (Exception ex) {
			occureException = ex;
			IOException ex2 = new IOException("internalWriteTextUTF16 failed.");
			ex2.initCause(ex);
			throw ex2;
		}
	}
	
	protected abstract void internalWriteTextUTF16LE(String name, String contents) throws IOException;
	
	public void writeSamplePicture(BufferedImage samplePicture) throws IOException {
		if (samplePicture == null) {
			throw new IllegalArgumentException();
		}

		try {
			internalWriteSamplePicture(samplePicture);

		} catch (IOException ex) {
			occureException = ex;
			throw ex;

		} catch (Exception ex) {
			occureException = ex;
			IOException ex2 = new IOException("write sample picture failed.");
			ex2.initCause(ex);
			throw ex2;
		}
	}

	protected abstract void internalWriteSamplePicture(BufferedImage samplePicture) throws IOException;

	public void writePartsImages(Map<PartsIdentifier, PartsSpec> partsImages) throws IOException {
		if (partsImages == null) {
			throw new IllegalArgumentException();
		}

		try {
			internalWritePartsImages(partsImages);

		} catch (IOException ex) {
			occureException = ex;
			throw ex;

		} catch (Exception ex) {
			occureException = ex;
			IOException ex2 = new IOException("write parts images failed.");
			ex2.initCause(ex);
			throw ex2;
		}
	}
	
	protected abstract void internalWritePartsImages(Map<PartsIdentifier, PartsSpec> partsImages) throws IOException;

	public void close() throws IOException {
		try {
			internalClose();
			
			outFile.delete();

		} catch (Exception ex) {
			if (occureException == null) {
				occureException = ex;
			}
		}
		if (occureException != null) {
			tmpFile.delete();
			return;
		}
		
		if (!tmpFile.renameTo(outFile)) {
			throw new IOException("rename failed. " + tmpFile);
		}
	}
	
	protected abstract void internalClose() throws IOException;
}
