/*
 * Crackerjack Project
 *
 * Copyright (C) 2007, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/time.h>
#include <getopt.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/syscall.h>

#include "../include/time.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"time"
#define INVALP		((void *)-1)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static time_t t;


/*
 * Data Structure
 */
struct test_case {
	time_t *time;
	time_t ret;
	time_t diff;
	int err;
};


/*
 * Test Data
 */
static struct test_case tcase[] = {
	{ // case00
		.time	= NULL,
		.ret	= 0,	// This value is filled in do_test() function
		.diff	= 0,	// This value is filled in do_test() function
		.err	= 0
	},{ // case01
		.time	= &t,
		.ret	= 0,	// This value is filled in do_test() function
		.diff	= 0,	// This value is filled in do_test() function
		.err	= 0
	},{ // case02
		.time	= INVALP,
		//.time	= (void*)0xc0000000,
		.ret	= -1,
		.diff	= 0,
		.err	= EFAULT
	}
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int
do_test(struct test_case *tc)
{
	struct timeval prep_tv, post_tv;
	time_t sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int ret;


	/*
	 * Save current time by gettimeofday syscall
	 */
	ret = gettimeofday(&prep_tv, NULL);
	if (ret != 0) {
		EPRINTF("gettimeofday failed: return: %d (errno=%d)\n",
			ret, errno);
		return RESULT_NG;
	}
	PRINTF("prep_time: %lu.%06lu\n", prep_tv.tv_sec, prep_tv.tv_usec);


	/*
	 * Execute system call
	 */
	errno = 0;
	//sys_ret = time(tc->time);
	sys_ret = syscall(__NR_time, tc->time);
	sys_errno = errno;
	PRINTF("RESULT: return value(arg)=%ld\n", sys_ret);

	/*
	 * Save current time again by gettimeofday syscall
	 */
	ret = gettimeofday(&post_tv, NULL);
	if (ret != 0) {
		EPRINTF("gettimeofday failed: return: %d (errno=%d)\n",
			ret, errno);
		return RESULT_NG;
	}
	PRINTF("post_time: %lu.%06lu\n", post_tv.tv_sec, post_tv.tv_usec);


	/*
	 * Check results
	 */
	if (tc->err == 0) {
		tc->ret  = prep_tv.tv_sec;
		tc->diff = post_tv.tv_sec - prep_tv.tv_sec;
	}

	if ((tc->time != NULL) && (tc->time != INVALP) && (sys_ret != -1)) {
		result |= (sys_ret != *tc->time);
		PRINTF("RESULT: return value(arg)=%ld\n", sys_ret);
	}

	result |= (sys_ret < tc->ret && sys_ret > (tc->ret + tc->diff));
	result |= (sys_errno != tc->err);

	if (tc->diff) {
		PRINTF("RESULT: return value(ret)=%10ld            errno=%d (%s)\n",
		       sys_ret, sys_errno, strerror(sys_errno));
		PRINTF("EXPECT: return value(ret)=%10ld-%10ld errno=%d (%s)\n",
		       tc->ret, tc->ret + tc->diff, tc->err, strerror(tc->err));
	} else {
		PRINTF("RESULT: return value(ret)=%ld errno=%d (%s)\n",
		       sys_ret, sys_errno, strerror(sys_errno));
		PRINTF("EXPECT: return value(ret)=%ld errno=%d (%s)\n",
		       tc->ret, tc->err, strerror(tc->err));
	}

	return result;
}


/*
 * usage()
 */
static void
usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n", SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int
main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

/*
 * gcc -W -Wall -O2 -o test <filename>.c
 */
