/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
////#include <linux/unistd.h>
////#include <linux/types.h>
#include <sys/syscall.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <linux/sysctl.h>

#include "../../common.j.h/include_j_h.h"


#ifndef SYSCTL_NOT_SUPPORTED

/*
 * Macros
 */
#define SYSCALL_NAME	"sysctl"
#define INVALP		((void *)-1)
#define BUF_SIZE	100


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char buf[BUF_SIZE];
static size_t oldlen = BUF_SIZE;


/*
 * Data Structure
 */
struct test_case {
	char *user;
	int *name;
	int nlen;
	void *oldval;
	size_t *oldlenp;
	void *newval;
	size_t newlen;
	int ret;
	int err;
};


/*
 * Test Data
 *   Error variation tests are impmlemented.
 */
static int ostype_name[] = { CTL_KERN, KERN_OSTYPE };
static int osrelease_name[] = { CTL_KERN, KERN_OSRELEASE };
static int version_name[] = { CTL_KERN, KERN_VERSION };
static int ng_name[]     = { 0 };	/* CTL_XXX is start from 1. */

static struct test_case tcase[] = {
	{ // case00
		.name	= ostype_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= 0,
		.err	= 0,
	},
	{ // case01
		.user	= "nobody",
		.name	= ostype_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= 0,
		.err	= 0,
	},
	{ // case02
		.name	= osrelease_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= 0,
		.err	= 0,
	},
	{ // case03
		.name	= version_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= 0,
		.err	= 0,
	},
	{ // case04
		.name	= ostype_name,
		.nlen	= 2,
		.oldval	= INVALP,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case05
		.name	= ostype_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= INVALP,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case06
		.name	= ng_name,	// invalid name
		.nlen	= 1,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= NULL,
		.newlen	= 0,
		.ret	= -1,
		.err	= ENOTDIR,
	},
	{ // case07
		.name	= ostype_name,
		.nlen	= 2,
		.oldval	= buf,
		.oldlenp= &oldlen,
		.newval	= "RTLinux",	// not writable
		.newlen	= 7,
		.ret	= -1,
		.err	= EPERM,
	},
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct __sysctl_args args;
	int rc;
	uid_t old_uid;

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0)
			return 1;
	}
	args.name = tc->name;
	args.nlen = tc->nlen;
	if (tc->newval) {
		args.newval = tc->newval;
		args.newlen = tc->newlen;
	} else {
		args.newval = NULL;
		args.newlen = 0;
	}
	args.oldval = tc->oldval;
	args.oldlenp = tc->oldlenp;

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR__sysctl, &args);
	sys_errno = errno;
	if (opt_debug && sys_ret >= 0)
		EPRINTF("value: %s\n", buf);

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);

	PRINTF("RESULT: return value(ret)=%d errno=%d (%s)\n",
	       sys_ret, sys_errno, strerror(sys_errno));
	PRINTF("EXPECT: return value(ret)=%d errno=%d (%s)\n",
	       tc->ret, tc->err, strerror(tc->err));

	if (tc->user != NULL)
		cleanup_euid(old_uid);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

#else

int main(int argc, char *argv[])
{
	RPRINTF("OK\n");
	return 0;
}

#endif


/*
 * gcc -W -Wall -O2 -o test <filename>.c
 */
