/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <unistd.h>
//#include <asm/page.h>
#include <sys/swap.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <limits.h>
#include <libgen.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"swapoff"
#define SWAPFILE_SIZE	(64 * 1024)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;
static char fpath[PATH_MAX];


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *user;
	char *path;
	int flags;
	size_t size;
	int ret;
	int err;
};


enum test_type {
	NORMAL,
	NOT_SWAP_FILE,
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EINVAL		v (not a regular file nor a block device)
 *   ENOENT		v (path does not exist)
 *   EPERM		v (permission denied)
 *
 *   test status of errors NOT on man page
 *
 *   EFAULT		v (invalid pointer)
 *   EISDIR		v (is a directory)
 */
static struct test_case tcase[] = {
	{ // case00
		.path		= fpath,
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.path		= fpath,
		.flags		= SWAP_FLAG_PREFER,
		.size		= SWAPFILE_SIZE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.path		= fpath,
		.flags		= 0,
		.size		= (SWAPFILE_SIZE * 1024),
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= NOT_SWAP_FILE,
		.path		= fpath,
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case04
		.path		= "not_exist_file",
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= -1,
		.err		= ENOENT,
	},
	{ // case05
		.user		= "nobody",
		.path		= fpath,
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= -1,
		.err		= EPERM,
	},
	{ // case06
		.path		= NULL,
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= -1,
		.err		= EFAULT,
	},
	{ // case07
		.path		= "/",
		.flags		= 0,
		.size		= SWAPFILE_SIZE,
		.ret		= -1,
		.err		= EISDIR,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc, fd = -1;
	uid_t old_uid;

	/*
	 * Setup swap file
	 */
	if (tc->ret >= 0) {
		fd = setup_swapfile(progdir, "test.file", fpath, tc->size);
		if (fd < 0)
			return 1;
		rc = swapon(tc->path, tc->flags);
		if (rc < 0) {
			EPRINTF("swapon failed.\n");
			result = 1;
			goto EXIT;
		}
	} else if (tc->ttype == NOT_SWAP_FILE) {
		fd = setup_file(progdir, "test.file", fpath);
		if (fd < 0)
			return 1;
	}

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0) {
			result = 1;
			goto EXIT;
		}
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	EPRINTF("path: %s\n", tc->path);
	sys_ret = swapoff(tc->path);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);
EXIT:
	if (tc->user != NULL)
		cleanup_euid(old_uid);
	if (fd >= 0) {
		close(fd);
		cleanup_file(fpath);
	}

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

