/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"socketpair"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	INVALID_POINTER,
	NO_FILE,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	int type;
	int proto;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EMFILE		v (process file table overflow)
 *   EAFNOSUPPORT	v (unsupported address family)
 *   EPROTONOSUPPORT	v (unsupported protocol)
 *   EOPNOTSUPP		v (incorrect socket type)
 *   EFAULT		v (points to not process address space)
 *
 *   test status of errors NOT on man page
 *
 *   EINVAL		v (invalid arguments)
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM + 1000, // this socket type must be unused
		.proto	= 0,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.proto	= 1000,	// this protocol must be unused
		.ret	= -1,
		.err	= EPROTONOSUPPORT,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= INVALID_POINTER,
		.type	= SOCK_STREAM,
		.proto	= 0,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= NO_FILE,
		.type	= SOCK_STREAM,
		.proto	= 0,
		.ret	= -1,
		.err	= EMFILE,
	},
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.proto	= 0,
		.ret	= 0,
		.err	= 0,
	},
	{ // case05
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_DGRAM,
		.proto	= 0,
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case06
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM + 1000, // this socket type must be unused
		.proto	= 0,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case07
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.proto	= 1000,	// this protocol must be unused
		.ret	= -1,
		.err	= EPROTONOSUPPORT,
	},
	{ // case08
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.proto	= 0,
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
	{ // case09
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_PACKET,
		.proto	= 0,
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
	// -------- PF_XXX --------
	{ // case10
		.domain	= PF_MAX,	// invalid domain
		.ttype	= NORMAL,
		.ret	= -1,
		.err	= EAFNOSUPPORT,
	},
	{ // case11
		.domain	= PF_IPX,	// not support domain
		.ttype	= NORMAL,
		.ret	= -1,
		.err	= EAFNOSUPPORT,
	},
};

#define ULIMIT_FNUM	0

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sv[2];
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	rlim_t oldlim;
	int rc;

	if (tc->ttype == NO_FILE) {
		rc = setup_ulimit_fnum(ULIMIT_FNUM, &oldlim);
		if (rc < 0)
			return 1;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	if (tc->ttype == INVALID_POINTER)
		sys_ret = socketpair(tc->domain, tc->type, tc->proto, NULL);
	else
		sys_ret = socketpair(tc->domain, tc->type, tc->proto, sv);
	sys_errno = errno;
	if (sys_ret >= 0) {
		close(sv[0]);
		close(sv[1]);
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);

	if (tc->ttype == NO_FILE)
		cleanup_ulimit_fnum(oldlim);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

/*
 * gcc -W -Wall -O2 -o test <filename>.c
 */
