/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"setsockopt"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	INVALID_POINTER,
	INVALID_OPTLEN,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	int type;
	int level;
	int oname;
	int val;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   ENOTSOCK		v (file descriptor)
 *   ENOPROTOOPT	v (unknown option)
 *   EFAULT		v (points to not process address space)
 *   EINVAL		v (invalid optlen)
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= INVALID_POINTER,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= INVALID_OPTLEN,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case05
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG + 1000,	// this option must be unused
		.val	= 1,
		.ret	= -1,
		.err	= ENOPROTOOPT,
	},
	{ // case06
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case07
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case08
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case09
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case10
		.domain	= PF_INET,
		.ttype	= INVALID_POINTER,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case11
		.domain	= PF_INET,
		.ttype	= INVALID_OPTLEN,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case12
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG + 1000,	// this option must be unused
		.val	= 1,
		.ret	= -1,
		.err	= ENOPROTOOPT,
	},
	{ // case13
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.type	= SOCK_STREAM,
		.level	= SOL_SOCKET,
		.oname	= SO_DEBUG,
		.val	= 1,
		.ret	= 0,
		.err	= 0,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, rc;
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int old_is_valid, oldval, val, chkval;
	socklen_t oldlen, len, chklen;
	int cmp_ok = 1;

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case INVALID_POINTER:
	case INVALID_OPTLEN:
	case NORMAL:
		fd = socket(tc->domain, tc->type, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			result = 1;
			goto EXIT;
		}
		break;
	}

	// save old value
	oldlen = sizeof(oldval);
	rc = getsockopt(fd, tc->level, tc->oname, &oldval, &oldlen);
	/* When error cases, getsockopt may be failed.
	 * So, mark that the error occures.
	 */
	old_is_valid = rc >= 0;

	// set new value
	val = tc->val;
	len = sizeof(val);
	if (tc->ttype == INVALID_OPTLEN)
		len -= 1;

	errno = 0;
	if (tc->ttype == INVALID_POINTER)
		sys_ret = setsockopt(fd, tc->level, tc->oname, NULL, len);
	else
		sys_ret = setsockopt(fd, tc->level, tc->oname, &val, len);
	sys_errno = errno;

	if (sys_ret == 0) {
		// check set value
		chklen = sizeof(chkval);
		rc = getsockopt(fd, tc->level, tc->oname, &chkval, &chklen);
		cmp_ok = rc == 0 && sizeof(val) == chklen && val == chkval;

		// restore old value
		if (old_is_valid)
			setsockopt(fd, tc->level, tc->oname, &oldval, oldlen);
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);
EXIT:
	if (fd >= 0)
		close(fd);
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

