/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"listen"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	INCORRECT_TYPE,
	ADDR_IN_USE,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	char *path;
	u_int16_t port;
	u_int32_t addr;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EADDRINUSE		can't check because 'bind' command is failed.
 *   EBADF		v (not a valid descriptor)
 *   ENOTSOCK		v (file descriptor)
 *   EOPNOTSUPP		v (incorrect socket type)
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.path	= "",
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= INCORRECT_TYPE,
		.path	= "",
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
#if 0
	{ // case0X
		.domain	= PF_UNIX,
		.ttype	= ADDR_IN_USE,
		.path	= "",
		.ret	= -1,
		.err	= EADDRINUSE,	// RHEL4U1 + 2.6.18 returns EBADF
	},
#endif
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case05
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case06
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case07
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case08
		.domain	= PF_INET,
		.ttype	= INCORRECT_TYPE,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
#if 0
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= ADDR_IN_USE,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= -1,
		.err	= EADDRINUSE,	// RHEL4U1 + 2.6.18 returns EBADF
	},
#endif
	{ // case09
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.port	= 11000,	// this port must be unused
		.ret	= 0,
		.err	= 0,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, other_fd = -1, rc;
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct sockaddr_un un_addr;
	struct sockaddr_in in_addr;
	struct sockaddr *addr;
	int len;
	int type = SOCK_STREAM;

	if (tc->domain == PF_UNIX) {
		un_addr.sun_family = AF_UNIX;
		strcpy(un_addr.sun_path, tc->path);
		len = sizeof(un_addr);
		addr = (struct sockaddr*)&un_addr;
	} else {
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = tc->port;
		in_addr.sin_addr.s_addr = tc->addr;
		len = sizeof(in_addr);
		addr = (struct sockaddr*)&in_addr;
	}

	if (tc->ttype == INCORRECT_TYPE)
		type = SOCK_DGRAM;

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case ADDR_IN_USE:
		other_fd = socket(tc->domain, type, 0);
		if (other_fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}
		rc = bind(other_fd, addr, len);
		if (rc < 0) {
			EPRINTF("bind failed1.\n");
			result = 1;
			goto EXIT;
		}
		/* fallthrough */
	case INCORRECT_TYPE:
	case NORMAL:
		fd = socket(tc->domain, type, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}
		rc = bind(fd, addr, len);
		if (rc < 0) {
			EPRINTF("bind failed.\n");
			result = 1;
			goto EXIT;
		}
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			result = 1;
			goto EXIT;
		}
		break;
	}
	errno = 0;
	sys_ret = listen(fd, 0);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);
EXIT:
	if (fd >= 0)
		close(fd);
	if (other_fd >= 0)
		close(other_fd);
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

