/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <asm/errno.h>
#include "../include/socketcall.h"

#define MBUF_LEN	64

static void
sighandler(int sig)
{
	if (sig == SIGUSR2)
		return;
	return;
}

int
do_server_proc(int type, int protocol, struct sockaddr *addr, socklen_t len,
	       int control, pid_t pid)
{
	int fd = -1, rc, sig_needed;
	int cfd = -1;
	struct sockaddr *caddr;
	struct sockaddr_in in_caddr;
	struct sockaddr_un un_caddr;
	socklen_t clen;
	struct msghdr msg;
	struct iovec miovec[1];
	char mbuf[MBUF_LEN];

	sig_needed = 1;
	//if (type != SOCK_STREAM)
	//	goto ERROR;

	/*
	 * Create server socket
	 */
	switch (addr->sa_family) {
	case AF_UNIX:
		fd = socket(PF_UNIX, type, protocol);
		break;
	case AF_INET:
		fd = socket(PF_INET, type, protocol);
		break;
	default:
		goto ERROR;	// not supported family
	}
	if (fd < 0)
		goto ERROR;

	/*
	 * do bind
	 */
	do {
		rc = bind(fd, addr, len);
		usleep(10000);	// relax
	} while (rc < 0 && errno == EADDRINUSE);
	if (rc < 0)
		goto ERROR;

	if (control == SV_NOT_RESPONSE) {
		// wake up client
		kill(pid, SIGUSR2);
		sig_needed = 0;
		for (;;)
			usleep(10000);	// 10msec
	}

	/*
	 * do listen
	 */
	if (type == SOCK_STREAM) {
		rc = listen(fd, 1);
		if (rc < 0)
			goto ERROR;

		/*
		 * wake up client
		 */
		kill(pid, SIGUSR2);
		sig_needed = 0;

		/*
		 * do accept
		 */
		switch (addr->sa_family) {
		case AF_UNIX:
			caddr = (struct sockaddr*)&un_caddr;
			clen = sizeof(un_caddr);
			break;
		case AF_INET:
			caddr = (struct sockaddr*)&in_caddr;
			clen = sizeof(in_caddr);
			break;
		default:
			goto ERROR;	// not supported family
		}
		cfd = accept(fd, caddr, &clen);
		if (cfd < 0)
			goto ERROR;
	} else {
		/*
		 * wake up client
		 */
		kill(pid, SIGUSR2);
		sig_needed = 0;

		cfd = fd;
		fd = -1;
	}

	/*
	 * control server
	 */
	switch (control) {
	case SV_SLOW_RESPONSE:
		usleep(1000000); // 1sec
		break;
	case SV_DISCONN_BY_ACCEPT:
		shutdown(cfd, SHUT_RDWR);
		for (;;)
			usleep(1000000); // 1sec
		//close(cfd);
		break;
	case SV_SEND_SIGINT_BY_ACCEPT:
		kill(pid, SIGINT);
		break;
	}

	/*
	 * do echo back loop
	 */
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = miovec;
	msg.msg_iovlen = 1;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags = 0;
	miovec[0].iov_base = mbuf;
	for (;;) {
		miovec[0].iov_len = MBUF_LEN;
		rc = recvmsg(cfd, &msg, 0);
		if (rc == 0)
			break;
		if (rc < 0)
			goto ERROR;

		switch (control) {
		case SV_DISCONN_BY_RECV:
			shutdown(cfd, SHUT_RDWR);
			for (;;)
				usleep(1000000); // 1sec
			break;
		case SV_SEND_SIGINT_BY_RECV:
			// make sure that client is receiving
			////////usleep(100000);
			kill(pid, SIGINT);
			break;
		}

		miovec[0].iov_len = rc;
		sendmsg(cfd, &msg, 0);
	}
	close(cfd);
	return 0;

ERROR:
	if (sig_needed)
		kill(pid, SIGUSR2);
	if (fd >= 0)
		close(fd);
	return -1;
}

int
do_client_proc(int type, int protocol, struct sockaddr *addr, socklen_t len,
	       int control, pid_t pid)
{
	int fd = -1, rc;

	/*
	 * Create server socket
	 */
	switch (addr->sa_family) {
	case AF_UNIX:
		fd = socket(PF_UNIX, type, protocol);
		break;
	case AF_INET:
		fd = socket(PF_INET, type, protocol);
		break;
	default:
		goto ERROR;	// not supported family
	}
	if (fd < 0)
		goto ERROR;

	if (control == CL_SEND_SIGINT) {
		kill(pid, SIGINT);
		for (;;)
			usleep(1000000); // 1sec
	}

	/*
	 * do connect
	 */
	rc = connect(fd, addr, len);
	if (rc < 0)
		goto ERROR;

	for (;;)
		usleep(1000000);
	close(fd);
	return 0;

ERROR:
	if (fd >= 0)
		close(fd);
	return -1;
}

pid_t
create_server_proc(int type, int protocol, struct sockaddr *addr, socklen_t len,
		   int control)
{
	pid_t pid, cpid;
	int rc;

	signal(SIGUSR2, sighandler);
	pid = getpid();
	cpid = fork();
	switch (cpid) {
	case 0:
		rc = do_server_proc(type, protocol, addr, len, control, pid);
		_exit(rc);
		break;
	case -1:
		EPRINTF("fork failed.\n");
		return cpid;
	default:
		pause();
		return cpid;
	}
}

pid_t
create_client_proc(int type, int protocol, struct sockaddr *addr, socklen_t len,
		   int control)
{
	pid_t pid, cpid;
	int rc;

	signal(SIGUSR2, sighandler);
	pid = getpid();
	cpid = fork();
	switch (cpid) {
	case 0:
		pause();
		rc = do_client_proc(type, protocol, addr, len, control, pid);
		_exit(rc);
		break;
	case -1:
		EPRINTF("fork failed.\n");
		return cpid;
	default:
		kill(cpid, SIGUSR2);
		return cpid;
	}
}

