/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <libgen.h>
#include <dirent.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"bind"

#define UNIX_PATH_MAX	108


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;

enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	FD_ALREADY_BOUND,
	FD_ALREADY_USED_ADDR,
	NAME_ZERO_LEN,
	NAME_MINUS_LEN,
	NAME_LONG_LEN,
	NAME_NONE_TERM_PATH,
	LOOP_LINK,
	NOT_DIR,
	INVALID_POINTER,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;		// PF_UNIX or PF_INET
	int ttype;
	char *path;		// for PF_UNIX
	u_int16_t port;		// for PF_INET
	u_int32_t addr;		// for PF_INET
	char *user;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINVAL		v (already bound)
 *   EACCES		v (permission is denied)
 *   ENOTSOCK		v (file descriptor)
 *   --- following errors are specific for UNIX domain sockets: ---
 *   EINVAL		v (wrong addrlen)
 *   EROFS		can't check because it's difficult to create read-only
 *   			file system
 *   EFAULT		v (points to not process address space)
 *   ENAMETOOLONG	can't check because UNIX_PATH_MAX is smaller than
 *                      PATH_MAX
 *   ENOENT		v (file does not exist)
 *   ENOMEM		can't check because it's difficult to create no-memory
 *   ENOTDIR		v (path prefix is not a directory)
 *   ELOOP		v (too many symbolic links)
 *
 *   test status of errors NOT on man page
 *
 *   EADDRNOTAVAIL	v (address not available)
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.path	= "",
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= FD_ALREADY_BOUND,
		.path	= "",
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= FD_ALREADY_USED_ADDR,
		.path	= "",
		.ret	= -1,
		.err	= EADDRINUSE,
	},
	{ // case05
		.domain	= PF_UNIX,
		.ttype	= NAME_ZERO_LEN,
		.path	= "",
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case06
		.domain	= PF_UNIX,
		.ttype	= NAME_MINUS_LEN,
		.path	= "",
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case07
		.domain	= PF_UNIX,
		.ttype	= NAME_LONG_LEN,
		.path	= "",
		.ret	= -1,
		.err	= EINVAL,
	},
#if 0
	{ // case0X
		.domain	= PF_UNIX,
		.ttype	= NAME_NONE_TERM_PATH,
		.path	= "",		// don't care
		.ret	= -1,
		.err	= ENAMETOOLONG,
	},
#endif
	{ // case08
		.domain	= PF_UNIX,
		.ttype	= LOOP_LINK,
		.path	= "",
		.ret	= -1,
		.err	= ELOOP,
	},
	{ // case09
		.domain	= PF_UNIX,
		.ttype	= NOT_DIR,
		.path	= "test_file",
		.ret	= -1,
		.err	= ENOTDIR,	// RHEL4U1 + 2.6.18 returns EADDRINUSE
	},
	{ // case10
		.domain	= PF_UNIX,
		.ttype	= INVALID_POINTER,
		.path	= "",
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case11
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "/crackerjack.test.bind/not.exist.dir",
		.ret	= -1,
		.err	= ENOENT,
	},
	{ // case12
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "/crackerjack.test.bind",
		.user	= "nobody",
		.ret	= -1,
		.err	= EACCES,
	},
	{ // case13
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "/crackerjack.test.bind",
		.ret	= 0,
		.err	= 0,
	},
	{ // case14
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case15
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case16
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case17
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case18
		.domain	= PF_INET,
		.ttype	= FD_ALREADY_BOUND,
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case19
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_LOOPBACK,
		.ret	= -1,
		.err	= EADDRNOTAVAIL,
	},
	{ // case20
		.domain	= PF_UNIX,
		.ttype	= INVALID_POINTER,
		.path	= "",
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case21
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.ret	= 0,
		.err	= 0,
	},
	{ // case22
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_BROADCAST,
		.ret	= 0,
		.err	= 0,
	},
};

static int setup_looped_link(char *testdir, char *link_top)
{
	char oldname[PATH_MAX], newname[PATH_MAX], *old;
	int rc, i, n = 9;

	sprintf(oldname, "%s/test_link.dir", testdir);
	rc = mkdir(oldname, 0777);
	if (rc < 0) {
		EPRINTF("mkdir failed.\n");
		return -1;
	}
	for (i = n; i > 0; i--) {
		sprintf(newname, "%s/test_link.lnk%d", testdir, i);
		old = basename(oldname);
		rc = symlink(old, newname);
		if (rc < 0) {
			EPRINTF("link failed.\n");
			return -1;
		}
		strcpy(oldname, newname);
	}
	strcpy(link_top, newname);
	return 0;
}

static int cleanup_looped_link(char *testdir)
{
	DIR *d;
	struct dirent *ent;
	char name[PATH_MAX];

	d = opendir(testdir);
	if (!d) {
		EPRINTF("opendir failed.\n");
		return -1;
	}
	while ((ent = readdir(d))) {
		if (strstr(ent->d_name, "test_link.")) {
			sprintf(name, "%s/%s", testdir, ent->d_name);
			if (ent->d_type == DT_DIR)
				rmdir(name);
			else
				unlink(name);
		}
	}
	closedir(d);
	return 0;
}

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, used_fd = -1;
	int sys_ret = -1;
	int sys_errno;
	int result = RESULT_OK;
	struct sockaddr_un un_addr;
	struct sockaddr_in in_addr;
	struct sockaddr *addr;
	int len;
	int rc;
	uid_t old_uid;

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0)
			return 1;
	}

	/*
	 * do setup
	 */
	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	case FD_ALREADY_USED_ADDR:
		used_fd = socket(tc->domain, SOCK_STREAM, 0);
		if (used_fd < 0) {
			EPRINTF("can't create socket.\n");
			return 1;
		}
		/* fallthrough */
	default:
		fd = socket(tc->domain, SOCK_STREAM, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			return 1;
		}
		break;
	}
	if (tc->domain == PF_UNIX) {
		un_addr.sun_family = AF_UNIX;
		switch (tc->ttype) {
		case NAME_NONE_TERM_PATH:
			memset(un_addr.sun_path, '@', UNIX_PATH_MAX);
			break;
		case LOOP_LINK:
			rc = setup_looped_link(progdir, un_addr.sun_path);
			if (rc < 0)
				return 1;
			strcat(un_addr.sun_path, "/path");
			break;
		case NOT_DIR:
			rc = setup_file(progdir, tc->path, un_addr.sun_path);
			if (rc < 0)
				return 1;
			break;
		default:
			strcpy(un_addr.sun_path, tc->path);
			break;
		}
		len = sizeof(un_addr);
		if (tc->ttype == NAME_ZERO_LEN)
			len = 0;
		else if (tc->ttype == NAME_MINUS_LEN)
			len = -1;
		else if (tc->ttype == NAME_LONG_LEN)
			len += 1;
		addr = (struct sockaddr*)&un_addr;
	} else {
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = tc->port;
		in_addr.sin_addr.s_addr = tc->addr;
		len = sizeof(in_addr);
		addr = (struct sockaddr*)&in_addr;
	}
	
	if (tc->ttype == FD_ALREADY_BOUND) {
		if (tc->domain == PF_INET)
			in_addr.sin_addr.s_addr = INADDR_BROADCAST;
		errno = 0;
		sys_ret = bind(fd, addr, len);
		sys_errno = errno;
		if (sys_ret < 0) {
			EPRINTF("can't bind.\n");
			result = 1;
			goto EXIT;
		}
		if (tc->domain == PF_INET)
			in_addr.sin_addr.s_addr = INADDR_ANY;
	} else if (tc->ttype == FD_ALREADY_USED_ADDR) {
		if (tc->domain == PF_INET) {
			if (inet_aton("192.168.0.1", &in_addr.sin_addr) == 0) {
				EPRINTF("inet_aton failed.\n");
				result = 1;
				goto EXIT;
			}
		}
		errno = 0;
		sys_ret = bind(used_fd, addr, len);
		sys_errno = errno;
		if (sys_ret < 0) {
			EPRINTF("can't bind.\n");
			result = 1;
			goto EXIT;
		}
	}

	/*
	 * do bind
	 */
	if (tc->ttype == INVALID_POINTER)
		addr = (struct sockaddr*)-1;
	errno = 0;
	sys_ret = bind(fd, addr, len);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);
EXIT:
	/*
	 * Remove UNIX domain socket file
	 */
	if (tc->domain == PF_UNIX && tc->path != NULL && strlen(tc->path) > 0 &&
	    sys_ret >= 0) {
		rc = unlink(tc->path);
		if (rc < 0) {
			EPRINTF("unlink failed.\n");
			return 1;
		}
	}
	if (tc->ttype == LOOP_LINK) {
		rc = cleanup_looped_link(progdir);
		if (rc < 0)
			return 1;
	} else if (tc->ttype == NOT_DIR) {
		rc = cleanup_file(un_addr.sun_path);
		if (rc < 0)
			return 1;
	}

	/*
	 * Restore effective user id
	 */
	if (tc->user != NULL) {
		rc = cleanup_euid(old_uid);
		if (rc < 0)
			return 1;
	}

	if (fd >= 0)
		close(fd);
	if (used_fd >= 0)
		close(used_fd);
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

