/*
 * Crackerjack Project
 *
 * Copyright (C) 2007, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>

#include "../include/pipe.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"pipe"
#define USERID		99


/*
 * Global variables
 */
static int opt_debug;
static char *progname;


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int
do_test(void)
{
	int result = RESULT_OK;
	int file_max, max_desc;
	int *fds;
	int i;


	/*
	 * Prepare file descriptors
	 */
	file_max = sysconf(_SC_OPEN_MAX);
	PRINTF("file_max=%d\n", file_max);

	max_desc = file_max * 2;
	fds = malloc(sizeof(int) * max_desc);
	if (fds == NULL) {
		EPRINTF("Memroy Allocation failed. (errno=%d)\n", errno);
		return RESULT_NG;
	}

	for (i = 0; i < max_desc; i++)
		fds[i] = -1;

	/*
	 * Execute syscall call
	 */
	for (i = 0; i < max_desc; i += 2) {
		int ret;

		ret = pipe(&fds[i]);
		if (ret < 0) {
			PRINTF("return value=%d errno=%d (%s)\n",
			       ret, errno, strerror(errno));
			if (errno != EMFILE) {
				EPRINTF("Unexpected error: ret=%d errno=%d (%s)\n",
					ret, errno, strerror(errno));
				result = RESULT_NG;
			}
			break;
		}
	}

	PRINTF("The number of opened pipes: %d\n", (i-2) / 2);

	for (i = 0; i < max_desc; i += 2) {
		if (fds[i] != -1)
			close(fds[i]);
		else
			break;
	}

	return result;
}


/*
 * usage()
 */
static void
usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n", SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int
main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Setup (Change User)
	 */
	result = setuid(USERID);
	if (result != 0) {
		EPRINTF("CHILD : setuid failed: return: %d (errno=%d (%s))\n",
			result, errno, strerror(errno));
		result = RESULT_NG;
		goto result;
	}
	PRINTF("Changed user to ID(%d)\n", getuid());


	/*
	 * Execute test
	 */
	PRINTF("(case00) START\n");
	result = do_test();
	PRINTF("(case00) END => %s\n", (result == RESULT_OK) ? "OK" : "NG");


	/*
	 * Check results
	 */
 result:
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

/*
 * gcc -W -Wall -O2 -o test <filename>.c
 */
