/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Yoshihide Sonoda <yoshihide.sonoda.ua@hitachi.com>
 *            Takahiro Yasui <takahiro.yasui.mp@hitachi.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/wait.h>

#include "../include/pipe.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"pipe"
#define USERID		99
#define INVALP		((void *)-1)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;


/*
 * Data Structure
 */
struct test_case {
	int *fildes;
	void (*setup)(struct test_case *tc);
	void (*cleanup)(struct test_case *tc);
	void *data;
	int ret;
	int err;

};


static void
do_open(void)
{
	int file_max;
	int *fds;
	int i;

	i = open("/dev/null", O_RDONLY);
	if (i < 0) {
		EPRINTF("Can't open file any more!\n");
		exit(1);
	}
	close(i);

	file_max = sysconf(_SC_OPEN_MAX);
	fds = malloc(sizeof(int) * file_max);
	for (i = 0; i < file_max; i++) {
		fds[i] = -1;
	}

	for (i = 0; i < file_max; i++) {
		fds[i] = open("/dev/null", O_RDONLY);
		if (fds[i] < 0)
			break;
	}

	pause();

	for (i = 0; i < file_max; i++) {
		if (fds[i] != -1)
			close(fds[i]);
		else
			break;
	}
}


struct post_func_data_02 {
	int file_max;
	int *fds;
};

struct post_func_data_03 {
	int proc_max;
	pid_t *pids;
};

static void
pre_func_00(struct test_case *tc)
{
	tc->fildes = malloc(sizeof(int)*2);
	tc->fildes[0] = 0;
	tc->fildes[1] = 0;
}

static void
pre_func_01(struct test_case *tc)
{
	tc->fildes = NULL;
}

static void
pre_func_02(struct test_case *tc)
{
	int i;
	struct post_func_data_02 *pfd;

	pfd = malloc(sizeof(struct post_func_data_02));
	pre_func_00(tc);

	pfd->file_max = sysconf(_SC_OPEN_MAX);
	pfd->fds = malloc(sizeof(int) * pfd->file_max);

	for (i = 0; i < pfd->file_max; i++) {
		pfd->fds[i] = open("/dev/null", O_RDONLY);
	}

	tc->data = (void *)pfd;
}

static volatile sig_atomic_t nochild = 0;


static void
sigchild_handler(int sig)
{
	int rt, status;

	rt = wait(&status);
	if (rt > 0 && WIFEXITED(status)) {
		if (WEXITSTATUS(status) != 0)
			nochild = 1;
	}
}


static void
pre_func_03(struct test_case *tc)
{
	int i;
	struct post_func_data_03 *pfd;

	pre_func_00(tc);

	pfd = malloc(sizeof(struct post_func_data_03));
	tc->data = pfd;
	pfd->proc_max = sysconf(_SC_CHILD_MAX);
	pfd->pids = malloc(sizeof(int) * pfd->proc_max);
	memset(pfd->pids, 0, sizeof(int) * pfd->proc_max);

	signal(SIGCHLD, sigchild_handler);

	for (i = 0; i < pfd->proc_max; i++) {
		if (nochild)
			break;

		pfd->pids[i] = fork();
		if (pfd->pids[i] == 0) {
			// child
			//execlp("sub/open", "open", NULL);
			do_open();
			DPRINTF("execlp failed!\n");
			exit(1);
		} else if (pfd->pids[i] > 0) {
			DPRINTF("fork = %d\n", pfd->pids[i]);
			// parent
		} else {
			// fork failed
			break;
		}
	}
}


static void
post_func(struct test_case *tc)
{
	if (tc->fildes != NULL && tc->fildes != INVALP) {
		close(tc->fildes[0]);
		close(tc->fildes[1]);
		free(tc->fildes);
	}
}


static void
post_func_02(struct test_case *tc)
{
	struct post_func_data_02 *pfd;
	int i;

	if (tc->data != NULL) {
		pfd = (struct post_func_data_02 *)tc->data;
		for (i = 0; i < pfd->file_max; i++) {
			if (pfd->fds[i] != -1)
				close(pfd->fds[i]);
		}

		free(pfd->fds);
		free(pfd);
	}

	post_func(tc);
}


static void
post_func_03(struct test_case *tc)
{
	struct post_func_data_03 *pfd;
	int i, status;

	if (tc->data != NULL) {
		pfd = (struct post_func_data_03 *)tc->data;

		for (i = 0; i < pfd->proc_max; i++) {
			if (pfd->pids[i] > 0) {
				status = kill(pfd->pids[i], SIGTERM);
				// if (status < 0) DPRINTF("kill failed\n");
			}
		}

		while (1) {
			i = wait(&status);
			if (i < 0 && errno == ECHILD)
				break;
		}

		free(pfd->pids);
		free(pfd);
	}

	signal(SIGCHLD, SIG_DFL);

	post_func(tc);
}

static void
pre_func_04(struct test_case *tc)
{
	tc->fildes = INVALP;
}


static struct test_case tcase[] = {
	{ // case00
		.setup	= pre_func_00,
		.cleanup= post_func,
		.data   = NULL,
		.ret    = 0,
		.err	= 0
	},{ // case01
		.setup	= pre_func_01,
		.cleanup= post_func,
		.data   = NULL,
		.ret    = -1,
		.err	= EFAULT
	},{ // case02
		.setup	= pre_func_02,
		.cleanup= post_func_02,
		.data   = NULL,
		.ret    = -1,
		.err    = EMFILE
	},{ // case03
		.setup	= pre_func_03,
		.cleanup= post_func_03,
		.data   = NULL,
		.ret    = -1,
		.err    = ENFILE
	},{ // case04
		.setup	= pre_func_04,
		.cleanup= post_func,
		.data   = NULL,
		.ret    = -1,
		.err	= EFAULT
	}
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
int
do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;


	/*
	 * Execute pre-test function
	 */
	if (tc->setup != NULL)
		tc->setup(tc);

	errno = 0;
	/*
	 * Execute system call
	 */
	sys_ret = pipe(tc->fildes);
	sys_errno = errno;
	PRINTF("RESULT: return value(arg)=%d\n", sys_ret);

	/*
	 * Check results
	 */
	PRINTF("RESULT: return value=%d errno=%d\n", sys_ret, sys_errno);
	PRINTF("EXPECT: return value=%d errno=%d\n", tc->ret, tc->err);

	result |= (tc->ret != sys_ret);
	result |= (tc->err != sys_errno);

	/*
	 * Execute post-test function
	 */
	if (tc->cleanup != NULL)
		tc->cleanup(tc);

	return result;
}


/*
 * usage()
 */
static void
usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n", SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int
main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Setup (Change User)
	 */
	result = setuid(USERID);
	if (result != 0) {
		EPRINTF("CHILD : setuid failed: return: %d (errno=%d (%s))\n",
			result, errno, strerror(errno));
		result = RESULT_NG;
		goto result;
	}
	PRINTF("Changed user to ID(%d)\n", getuid());


	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
 result:
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

/*
 * gcc -W -Wall -O2 -o test <filename>.c
 */
