/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <dirent.h>
#include <limits.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"openat"

#ifndef __NR_openat
#  define __NR_openat	295
#endif
#ifndef AT_FDCWD
#  define AT_FDCWD	-100
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progpath;
static char *progdir;

enum test_type {
	NORMAL,
	CURRENT_DIR,
	CREATE_FILE,
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *path;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   ENOTDIR		v (not a directory discriptor)
 *
 *   test status of errors on NOT man page
 *
 *   ENOENT		v (not exist file or directory)
 */
static struct test_case tcase[] = {
	{ // case00
		.path		= "test",
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= CURRENT_DIR,
		.path		= "", // don't care when 'CURRENT_DIR' test
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= CREATE_FILE,
		.path		= "not_exist_file",
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= FD_NONE,
		.path		= "/", // don't need dirfd when absolute path
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= FD_NONE,
		.path		= "test",
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case05
		.ttype		= FD_NOT_EXIST,
		.path		= "test",
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case06
		.ttype		= FD_FILE,
		.path		= "test",
		.ret		= -1,
		.err		= ENOTDIR,
	},
	{ // case07
		.path		= "not_exist_file",
		.ret		= -1,
		.err		= ENOENT,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int dirfd = -1, flags = 0;
	char *path = tc->path, fpath[PATH_MAX];
	DIR *d;
	struct dirent *dent;

	fpath[0] = '\0';
	switch (tc->ttype) {
	case CURRENT_DIR:
		d = opendir(".");
		if (!d) {
			EPRINTF("opendir failed.\n");
			return 1;
		}
		while ((dent = readdir(d))) {
			if ((dent->d_type == DT_REG || dent->d_type == DT_DIR)
			    && strcmp(dent->d_name, ".") != 0
			    && strcmp(dent->d_name, "..") != 0) {
				path = strdup(dent->d_name);
				if (opt_debug)
					EPRINTF("test '%s' (current dir)\n",
						path);
				dirfd = AT_FDCWD;
				break;
			}
		}
		closedir(d);
		break;
	case FD_NOT_EXIST:
		dirfd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		dirfd = open(progpath, O_RDONLY);
		if (dirfd < 0) {
			EPRINTF("open failed(%s).\n", progpath);
			return 1;
		}
		break;
	case CREATE_FILE:
		snprintf(fpath, PATH_MAX, "%s/%s", progdir, path);
		flags = O_CREAT | O_WRONLY;
		/* fallthrough */
	default:
		dirfd = open(progdir, O_RDONLY);
		if (dirfd < 0) {
			EPRINTF("open failed(%s).\n", progdir);
			return 1;
		}
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_openat, dirfd, path, flags, 0);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= sys_errno != tc->err;
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);

	if (sys_ret >= 0)
		close(sys_ret);
	if (dirfd >= 0)
		close(dirfd);
	if (fpath[0] != '\0')
		unlink(fpath);
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progpath = strdup(argv[0]);

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

