/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <limits.h>
#include <libgen.h>

#include "../../common.j.h/include_j_h.h"

#ifdef __i386__

/*
 * Macros
 */
#define SYSCALL_NAME	"old_mmap"
#define INVALP		((void *)-1)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;
static char fpath[PATH_MAX];


/*
 * Data Structure
 */
struct mmap_arg_struct {
	void *start;
	size_t length;
	int prot;
	int flags;
	int fd;
	off_t offset;
};

struct test_case {
	char *fname;
	int oflags;	/* open flags */
	struct mmap_arg_struct arg;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EACCES		v (not a regular file
 *                         or MAP_PRIVATE but fd is not for reading
 *                         or MAP_SHARED and PROT_WRITE but fd is not O_RDWR
 *                         or PROT_WRITE but file is append-only.)
 *   EINVAL		v (start, length or offset are too large or not aligned)
 *   ETXTBSY		can't check because MAP_DENYWRITE is ignored (from man
 *   			page).
 *   EAGAIN		can't check because we can't re-mount file system and
 *   			lock the test file by mandatory-lock.
 *   ENOMEM		can't check because it's difficult to create no-memory
 *   ENODEV		v (filesystem does not support memory mapping)
 */
static struct test_case tcase[] = {
	{ // case00
		.fname		= "test.file",
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_PRIVATE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.fname		= "test.file",
		.oflags		= O_RDWR,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_WRITE|PROT_WRITE,
		.arg.flags	= MAP_PRIVATE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.fname		= NULL,		// fd equals -1
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_PRIVATE,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case03
		.fname		= "test.file",
		.oflags		= O_WRONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_PRIVATE,
		.ret		= -1,
		.err		= EACCES,
	},
	{ // case04
		.fname		= "test.file",
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_WRITE,
		.arg.flags	= MAP_SHARED,
		.ret		= -1,
		.err		= EACCES,
	},
#if 0
	{ // caseXX
		.fname		= "test.file",
		.oflags		= O_APPEND,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_WRITE,
		.arg.flags	= MAP_PRIVATE,
		.ret		= -1,
		.err		= EACCES,
	},
#endif
	{ // case05
		.fname		= "test.file",
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 0,		// invalid
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_PRIVATE,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case06
		.fname		= "test.file",
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_FIXED,	//invalid
		.ret		= -1,
		.err		= EINVAL,
	},
#ifndef PROC_NOT_SUPPORTED
	{ // case07
		.fname		= "/proc/meminfo", // mmap not supported
		.oflags		= O_RDONLY,
		.arg.offset	= 0,
		.arg.length	= 4096,
		.arg.start	= NULL,
		.arg.prot	= PROT_READ,
		.arg.flags	= MAP_PRIVATE,
		.ret		= -1,
		.err		= ENODEV,
	},
#endif
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	void *sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int ret;
	int fd = -1;

	/*
	 * Prepare the file descriptor.
	 */
	if (tc->fname) {
		if (tc->fname[0] != '/') {
			int flags = O_CREAT|O_EXCL|tc->oflags;
			mode_t mode;
			switch (tc->oflags) {
			case O_RDONLY: mode = S_IRUSR; break;
			case O_WRONLY: mode = S_IWUSR; break;
			default:       mode = S_IRUSR|S_IWUSR; break;
			}
			fd = _setup_file(progdir, tc->fname, fpath, flags,mode);
			if (fd < 0) {
				result |= RESULT_NG;
				goto TEST_EXIT;
			}
		} else {
			fd = open(tc->fname, tc->oflags);
			if (fd < 0) {
				EPRINTF("open(\"%s\") failed. "
					"(ret)=%d errno=%d (%s)\n",
					tc->fname, fd, errno, strerror(errno));
				result |= RESULT_NG;
				goto TEST_EXIT;
			}
		}
	}

	/*
	 * Execute system call
	 */
	tc->arg.fd = fd;
	errno = 0;
	sys_ret = (void*)syscall(SYS_mmap, &tc->arg);
	sys_errno = errno;
	if (sys_ret != MAP_FAILED) {
		if ((ret = munmap(NULL, tc->arg.length)) < 0) {
			EPRINTF("munmap failed. (ret)=%d errno=%d (%s)\n",
				ret, errno, strerror(errno));
			result |= RESULT_NG;
		}
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);

	if (sys_ret != MAP_FAILED) {
		PRINTF("RESULT: return value(ret)=%p errno=%d (%s)\n",
		       sys_ret, sys_errno, strerror(sys_errno));
		PRINTF("EXPECT: return value(ret)=ANY        errno=%d (%s)\n",
		       tc->err, strerror(tc->err));
	} else {
		PRINTF("RESULT: return value(ret)=%d errno=%d (%s)\n",
		       (int)sys_ret, sys_errno, strerror(sys_errno));
		PRINTF("EXPECT: return value(ret)=%d errno=%d (%s)\n",
		       tc->ret, tc->err, strerror(tc->err));
	}
TEST_EXIT:
	if (fd >= 0) {
		close(fd);
		if (tc->fname != NULL && tc->fname[0] != '/')
			cleanup_file(fpath);
	}
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

#else

int main(int argc, char *argv[])
{
	RPRINTF("OK\n");
	return 0;
}

#endif

