/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <getopt.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <mqueue.h>
#include <signal.h>
#include <limits.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"mq_getsetattr"

#ifndef __NR_mq_getsetattr
#  define __NR_mq_getsetattr		282
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;



/*
 * Data Structure
 */
struct test_case {
	int ttype;
	long mq_flags;	// Message queue flags
	long mq_n_msgs;	// Number of messages
	int ret;
	int err;
};


enum test_type {
	NORMAL,
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
};
#define MAX_MSGSIZE	8192
#define MSG_SIZE	16


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINVAL		v (invalid mq_flags)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.mq_flags	= 0,
		.mq_n_msgs	= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.mq_flags	= O_NONBLOCK,
		.mq_n_msgs	= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.mq_flags	= 0,
		.mq_n_msgs	= 10,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= FD_NONE,
		.mq_flags	= 0,
		.mq_n_msgs	= 0,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case04
		.ttype		= FD_NOT_EXIST,
		.mq_flags	= 0,
		.mq_n_msgs	= 0,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case05
		.ttype		= FD_FILE,
		.mq_flags	= 0,
		.mq_n_msgs	= 0,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case06
		.ttype		= NORMAL,
		.mq_flags	= ~O_NONBLOCK,
		.mq_n_msgs	= 0,
		.ret		= -1,
		.err		= EINVAL,
	},
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc, i, fd = -1, cmp_ok = 1;
	struct mq_attr new, old;
	char smsg[MAX_MSGSIZE];

	/*
	 * When test ended with SIGTERM etc, mq discriptor is left remains.
	 * So we delete it first.
	 */
	mq_unlink(QUEUE_NAME);

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			result = 1;
			goto EXIT;
		}
		break;
	default:
		/*
		 * Open message queue
		 */
		fd = mq_open(QUEUE_NAME, O_CREAT|O_EXCL|O_RDWR, S_IRWXU, NULL);
		if (fd < 0) {
			EPRINTF("mq_open failed\n");
			result = 1;
			goto EXIT;
		}
	}

	new.mq_flags = tc->mq_flags;
	new.mq_curmsgs = tc->mq_n_msgs;

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_mq_getsetattr, fd, &new, NULL);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	/*
	 * Update mq_curmsgs in struct mq_attr
	 */
	for (i = 0; i < MSG_SIZE; i++)
		smsg[i] = i;
	for (i = 0; i < tc->mq_n_msgs; i++) {
		rc = mq_timedsend(fd, smsg, MSG_SIZE, 0, NULL);
		if (rc < 0) {
			EPRINTF("mq_timedsend failed.\n");
			result = 1;
			goto EXIT;
		}
	}

	errno = 0;
	sys_ret = syscall(__NR_mq_getsetattr, fd, NULL, &old);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	if (opt_debug) {
		EPRINTF("mq_flags E:0x%lx,\tR:0x%lx\n",
			new.mq_flags, old.mq_flags);
		EPRINTF("mq_curmsgs E:%ld,\tR:%ld\n",
			new.mq_curmsgs, old.mq_curmsgs);
	}
	cmp_ok = old.mq_flags == new.mq_flags &&
		 old.mq_curmsgs == new.mq_curmsgs;

TEST_END:
	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);

EXIT:
	if (fd >= 0) {
		close(fd);
		mq_unlink(QUEUE_NAME);
	}

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

