/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include "numaif.h"

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"move_pages"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	NORMAL,
	NO_SUCH_PROC,
	INVALID_NR_PAGES,
	INVALID_POINTER,
	TO_OTHER_NODE,
	TO_SELF_NODE,
	GET_INVALID_PAGE,
	GET_NOT_EXIST_PAGE,
	MOV_INVALID_PAGE,   // can't check in the 1-Node NUMA system
	MOV_NOT_EXIST_PAGE, // can't check in the 1-Node NUMA system
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *user;
	int flags;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   (NONE)		man page doesn't exist
 *
 *   test status of errors NOT on man page
 *
 *   EINVAL		v (invalid argument)
 *   ESRCH		v (not exist process)
 *   EPERM		v (permission denied)
 *   E2BIG		v (too big nr_pages)
 *   ENOMEM		can't check because it's difficult to create no-memory
 *   EFAULT		v (invalid pointer)
 *   ENODEV		v (not exist node)
 *   EACCESS		can't check because we don't have N-node NUMA system
 *   ENOENT		v (not exist page or no movable pages)
 */
static struct test_case tcase[] = {
	//------ get pages stat
	{ // case00
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.flags		= MPOL_MF_STRICT, // invalid flags
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case02
		.ttype		= NO_SUCH_PROC,
		.ret		= -1,
		.err		= ESRCH,
	},
	{ // case03
		.user		= "nobody",
		.ret		= -1,
		.err		= EPERM,
	},
	{ // case04
		.ttype		= INVALID_NR_PAGES,
		.ret		= -1,
		.err		= E2BIG,
	},
	{ // case05
		.ttype		= INVALID_POINTER,
		.ret		= -1,
		.err		= EFAULT,
	},
	{ // case06
		.ttype		= GET_INVALID_PAGE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case07
		.ttype		= GET_NOT_EXIST_PAGE,
		.ret		= 0,
		.err		= 0,
	},
	//------ move pages
	{ // case08
		.ttype		= TO_OTHER_NODE,
		.ret		= -1,
		.err		= ENODEV,
	},
	{ // case09
		.ttype		= TO_SELF_NODE,
		.ret		= -1,
		.err		= ENOENT,
	},
};

#define TEST_PAGES	16
#define PAGE_SIZE	4096
/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc;
	pid_t pid;
	unsigned long nr_pages = TEST_PAGES;
	void *p = NULL;
	void **pages, *page_array[TEST_PAGES];
	int status[TEST_PAGES], *nodes, node_array[TEST_PAGES];
	int i, cmp_ok = 1;

	/*
	 * mmap memory
	 */
	p = mmap(NULL, TEST_PAGES * PAGE_SIZE,
		 PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANONYMOUS, 0, 0);
	if (p == (void*)-1) {
		EPRINTF("malloc failed.\n");
		result = 1;
		goto EXIT;
	}
	for (i = 0; i < TEST_PAGES; i++) {
		int j;

		// set node array
		node_array[i] = (tc->ttype == TO_OTHER_NODE) ? 1 : 0;

		// set page array
		switch (tc->ttype) {
		case GET_INVALID_PAGE:
		case MOV_INVALID_PAGE:
			page_array[i] = (void*)0xc0000000;
			break;
		case GET_NOT_EXIST_PAGE:
		case MOV_NOT_EXIST_PAGE:
			page_array[i] = p;
			p += PAGE_SIZE;
			break;
		default:
			page_array[i] = p;
			for (j = 0; j < PAGE_SIZE; j++) {
				*(char*)p = 0;
				p++;
			}
			break;
		}
	}
	pages = page_array;
	// set nodes
	switch (tc->ttype) {
	case TO_SELF_NODE: case TO_OTHER_NODE:
	case MOV_INVALID_PAGE: case MOV_NOT_EXIST_PAGE:
		nodes = node_array;
		break;
	default:
		nodes = NULL;
	}

	pid = getpid();
	switch (tc->ttype) {
	case NO_SUCH_PROC:
		pid = get_unexist_pid();
		if (pid < 0) {
			result = 1;
			goto EXIT;
		}
		break;
	case INVALID_NR_PAGES:
		nr_pages = 0xffffffff;
		break;
	case INVALID_POINTER:
		// 'nr_pages' is not zero, but 'pages' is NULL
		pages = NULL;
		break;
	default:
		break;
	}

	if (tc->user != NULL) {
		// It's no use for changing the euid. We should change the uid.
		rc = setup_uid("nobody");
		if (rc < 0) {
			result = 1;
			goto EXIT;
		}
		pid = 1; // set the target process ('init'), so always fail.
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_move_pages, pid, nr_pages, pages,
			  nodes, status, tc->flags);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	if (opt_debug) {
		for (i = 0; i < TEST_PAGES; i++)
			EPRINTF("page status[%d]: %d\n", i, status[i]);
	}
	for (i = 0; i < TEST_PAGES; i++)
		switch (tc->ttype) {
		case GET_INVALID_PAGE:
		case MOV_INVALID_PAGE:
			if (status[i] != -EFAULT)
				cmp_ok = 0;
			break;
		case GET_NOT_EXIST_PAGE:
		case MOV_NOT_EXIST_PAGE:
			if (status[i] != -ENOENT)
				cmp_ok = 0;
			break;
		default:
			if (status[i] != 0)
				cmp_ok = 0;
		}

TEST_END:
	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(0, tc->ret, tc->err, sys_ret, sys_errno, cmp_ok);
EXIT:
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

