/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <dirent.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <libgen.h>
#include <pwd.h>
//#include <linux/ioprio.h>

#include "../../common.j.h/include_j_h.h"
#include "../../common.j.h/ioprio.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"ioprio_set"

#ifndef __NR_ioprio_set
#  define __NR_ioprio_set	289
#  define __NR_ioprio_get	290
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;


enum test_type {
	NORMAL,
	NO_PROCESS,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *user;
	int which;
	int class;
	int level;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EINVAL		v (invalid argument)
 *   EPERM		v (permission denied)
 *   ESRCH		v (no process found)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_RT,
		.level		= 7,	// low
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_RT,
		.level		= 0,	// high
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_BE,
		.level		= 7,	// low
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_BE,
		.level		= 0,	// high
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_IDLE,
		.level		= 7,	// low
		.ret		= 0,
		.err		= 0,
	},
	{ // case05
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_IDLE,
		.level		= 0,	// high
		.ret		= 0,
		.err		= 0,
	},
	{ // case06
		.ttype		= NORMAL,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_NONE,	// invalid
		.level		= 7,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case07
		.ttype		= NORMAL,
		.user		= "nobody", // not have permission to CLASS_RT
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_RT,
		.level		= 0,
		.ret		= -1,
		.err		= EPERM,
	},
	{ // case08
		.ttype		= NO_PROCESS,
		.which		= IOPRIO_WHO_PROCESS,
		.class		= IOPRIO_CLASS_BE,
		.level		= 7,
		.ret		= -1,
		.err		= ESRCH,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int rc, oldprio = 0, cmp_ok = 1;
	uid_t old_uid;
	struct passwd *pw;
	int who = 0, ioprio;

	switch (tc->which) {
	case IOPRIO_WHO_PROCESS:
		who = getpid();
		break;
	case IOPRIO_WHO_PGRP:
	case IOPRIO_WHO_USER:
		old_uid = geteuid();
		pw = getpwuid(old_uid);
		if (!pw) {
			EPRINTF("getpwuid failed.\n");
			return 1;
		}
		who = tc->which == IOPRIO_WHO_PGRP ? pw->pw_gid : pw->pw_uid;
		break;
	}
	ioprio = IOPRIO_PRIO_VALUE(tc->class, tc->level);

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0)
			return 1;
	}

	/*
	 * Save ioprio setting
	 */
	if (tc->ttype == NO_PROCESS)
		who = -1;
	else {
		oldprio = syscall(__NR_ioprio_get, tc->which, who);
		if (oldprio < 0 && errno != ENOSYS) {
			EPRINTF("ioprio_get failed.\n");
			result = 1;
			goto EXIT;
		}
		if (IOPRIO_PRIO_CLASS(oldprio) == IOPRIO_CLASS_NONE)
			oldprio = IOPRIO_PRIO_VALUE(IOPRIO_CLASS_BE,
						    IOPRIO_PRIO_DATA(oldprio));
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_ioprio_set, tc->which, who, ioprio);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	/*
	 * Check set proirity
	 */
	rc = syscall(__NR_ioprio_get, tc->which, who);
	cmp_ok = ioprio == rc;
	PRINTF("ioprio= E:%d <=> R:%d\n", ioprio, rc);

	/*
	 * Restore ioprio setting
	 */
	errno = 0;
	sys_ret = syscall(__NR_ioprio_set, tc->which, who, oldprio);
	sys_errno = errno;

	/*
	 * Check results
	 */
TEST_END:
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);

	/*
	 * Restore effective user id
	 */
EXIT:
	if (tc->user != NULL) {
		rc = cleanup_euid(old_uid);
		if (rc < 0)
			return 1;
	}
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

