/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <libgen.h>
#include <limits.h>
#include <linux/inotify.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"inotify_add_watch"

#ifndef __NR_inotify_init
#  define __NR_inotify_init		291
#  define __NR_inotify_add_watch	292
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;


enum test_type {
	NORMAL,
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	NULL_POINTER,
	INVALID_MASK,
	NO_FILE,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	char *user;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EACCES		v (permission is denied)
 *   EBADF		v (not a valid descriptor)
 *   EFAULT		v (points to not process address space)
 *   EINVAL		v (invalid event-mask or fd)
 *   ENOMEM		can't check because it's difficult to create no-mem
 *   ENOSPC		v (watching file table overflow)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.user		= "nobody",
		.ret		= -1,
		.err		= EACCES,
	},
	{ // case02
		.ttype		= FD_NONE,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case03
		.ttype		= FD_NOT_EXIST,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case04
		.ttype		= NULL_POINTER,
		.ret		= -1,
		.err		= EFAULT,
	},
	{ // case05
		.ttype		= FD_FILE,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case06
		.ttype		= INVALID_MASK,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case07
		.ttype		= NO_FILE,
		.ret		= -1,
		.err		= ENOSPC,
	},
};


static void print_inotify_event(char *p, int len)
{
	struct inotify_event *ev;
	int rec_len;

	if (len <= 0 || !opt_debug)
		return;
	PRINTF("================\n");
	while (len >= sizeof(struct inotify_event)) {
		ev = (struct inotify_event*)p;
		PRINTF("- - - - - - - - \n");
		PRINTF("wd:\t%d\n", ev->wd);
		PRINTF("mask:\t0x%08x\n", ev->mask);
		PRINTF("cookie:\t%d\n", ev->cookie);
		PRINTF("name:\t>%.*s<\n", ev->len, ev->name);
		rec_len = sizeof(struct inotify_event) + ev->len;
		p += rec_len;
		len -= rec_len;
	}
	PRINTF("\n");
}


#define ev_ok(ev, _wd, _mask, _cookie, _len, _name)		\
	(ev->wd == (_wd) && ev->mask == (_mask) &&		\
	 ev->cookie == (_cookie) && ev->len >= (_len) &&	\
	 strcmp(ev->name, (_name)) == 0)

#define TEST_FILE1	"test.file"
#define TEST_FILE2	"test.file2"

#define PROC_MAX_WATCHES	"/proc/sys/fs/inotify/max_user_watches"

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	char fp1[PATH_MAX], fp2[PATH_MAX], buf[4096];
	int fd = -1, fd1 = -1;
	int rc, cmp_ok = 1;
	struct inotify_event *ev;
	int rec_len, n;
	char *p;
	unsigned long saved_cookie = 0;
	uid_t old_uid;
	struct stat old_st;
	char *path;
	unsigned long mask;
	int oldval;

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	default:
		fd = syscall(__NR_inotify_init);
		if (fd < 0) {
			EPRINTF("inotify_init failed.\n");
			return 1;
		}
		if (tc->ttype == NO_FILE) {
			rc = setup_proc_fs(PROC_MAX_WATCHES, 0, &oldval);
			if (rc < 0)
				return 1;
		}
		break;
	}

	/*
	 * Change effective user id
	 */
	if (tc->user != NULL) {
		rc = stat(progdir, &old_st);
		if (rc < 0) {
			EPRINTF("stat failed.\n");
			result = 1;
			goto EXIT;
		}
		rc = chmod(progdir, old_st.st_mode & ~(S_IRGRP|S_IROTH));
		if (rc < 0) {
			EPRINTF("chmod failed.\n");
			result = 1;
			goto EXIT;
		}
		rc = setup_euid(tc->user, &old_uid);
		if (rc < 0) {
			result = 1;
			goto EXIT;
		}
	}

	path = progdir;
	mask = IN_CREATE | IN_OPEN | IN_MODIFY | IN_CLOSE_WRITE
		| IN_MOVED_FROM | IN_MOVED_TO | IN_DELETE;
	switch (tc->ttype) {
	case NULL_POINTER:
		path = NULL;
		break;
	case INVALID_MASK:
		mask = 0;
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_inotify_add_watch, fd, path, mask);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	/*
	 * Check inotify
	 */
	// create file
	sprintf(fp1, "%s/%s", progdir, TEST_FILE1);
	fd1 = open(fp1, O_CREAT|O_EXCL|O_RDWR, S_IRUSR|S_IWUSR);
	if (fd1 < 0) {
		EPRINTF("open failed.\n");
		result = 1;
		goto EXIT;
	}
	// write file
	rc = write(fd1, "hoge\n", 5);
	if (rc != 5) {
		EPRINTF("write failed.\n");
		result = 1;
		goto EXIT;
	}
	// close file
	rc = close(fd1);
	if (rc < 0) {
		EPRINTF("close failed.\n");
		result = 1;
		goto EXIT;
	}
	fd1 = -1;

	// move file
	sprintf(fp2, "%s/%s", progdir, TEST_FILE2);
	rc = rename(fp1, fp2);
	if (rc < 0) {
		EPRINTF("rename failed.\n");
		result = 1;
		goto EXIT;
	}

	// delete file
	rc = unlink(fp2);
	if (rc < 0) {
		EPRINTF("unlink failed.\n");
		result = 1;
		goto EXIT;
	}

	// get inotify events
	do {
		rc = read(fd, buf, 4096);
	} while (rc < 0 && errno == EAGAIN);
	if (rc < 0) {
		EPRINTF("read failed.\n");
		result = 1;
		goto EXIT;
	}

	// check inotify events
	print_inotify_event(buf, rc);
	p = buf;
	for (n = 0; rc >= sizeof(struct inotify_event); n++) {
		ev = (struct inotify_event*)p;
		switch (n) {
		case 0:
			if (!ev_ok(ev, sys_ret, IN_CREATE, 0,
				   strlen(TEST_FILE1), TEST_FILE1))
				cmp_ok = 0;
			break;
		case 1:
			if (!ev_ok(ev, sys_ret, IN_OPEN, 0,
				   strlen(TEST_FILE1), TEST_FILE1))
				cmp_ok = 0;
			break;
		case 2:
			if (!ev_ok(ev, sys_ret, IN_MODIFY, 0,
				   strlen(TEST_FILE1), TEST_FILE1))
				cmp_ok = 0;
			break;
		case 3:
			if (!ev_ok(ev, sys_ret, IN_CLOSE_WRITE, 0,
				   strlen(TEST_FILE1), TEST_FILE1))
				cmp_ok = 0;
			break;
		case 4:
			saved_cookie = ev->cookie;
			if (!ev_ok(ev, sys_ret, IN_MOVED_FROM, ev->cookie,
				   strlen(TEST_FILE1), TEST_FILE1))
				cmp_ok = 0;
			break;
		case 5:
			if (!ev_ok(ev, sys_ret, IN_MOVED_TO, saved_cookie,
				   strlen(TEST_FILE2), TEST_FILE2))
				cmp_ok = 0;
			break;
		case 6:
			if (!ev_ok(ev, sys_ret, IN_DELETE, 0,
				   strlen(TEST_FILE2), TEST_FILE2))
				cmp_ok = 0;
			break;
		default:
			cmp_ok = 0;
			break;
		}
		if (opt_debug)
			PRINTF("%d cmk_ok: %d\n", n, cmp_ok);
		rec_len = sizeof(struct inotify_event) + ev->len;
		p += rec_len;
		rc -= rec_len;
	}
	if (rc != 0)
		cmp_ok = 0;

	/*
	 * Check results
	 */
TEST_END:
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);
EXIT:
	/*
	 * Restore effective user id
	 */
	if (tc->user != NULL) {
		cleanup_euid(old_uid);
		chmod(progdir, old_st.st_mode);
	}
	if (tc->ttype == NO_FILE)
		cleanup_proc_fs(PROC_MAX_WATCHES, oldval);

	if (fd >= 0)
		close(fd);
	if (fd1 >= 0) {
		close(fd1);
		unlink(fp1);
	}

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

