/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#define _GNU_SOURCE
#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <getopt.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <fcntl.h>
#include <libgen.h>
#include <string.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"fadvise64"

#ifndef __NR_fadvise64
#  define __NR_fadvise64		250
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	int advice;
	int ret;
	int err;
};


enum test_type {
	NORMAL,
	FD_NONE,
	FD_NOT_EXIST,
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINVAL		v (invalid advise)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_SEQUENTIAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_RANDOM,
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_NOREUSE,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_WILLNEED,
		.ret		= 0,
		.err		= 0,
	},
	{ // case05
		.ttype		= NORMAL,
		.advice		= POSIX_FADV_DONTNEED,
		.ret		= 0,
		.err		= 0,
	},
	{ // case06
		.ttype		= FD_NONE,
		.advice		= POSIX_FADV_NORMAL,
		.ret		= EBADF,
		.err		= 0,
	},
	{ // case07
		.ttype		= FD_NOT_EXIST,
		.advice		= POSIX_FADV_NORMAL,
		.ret		= EBADF,
		.err		= 0,
	},
	{ // case08
		.ttype		= NORMAL,
		.advice		= -1,
		.ret		= EINVAL,
		.err		= 0,
	},
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int fd = -1;
	char fpath[PATH_MAX];

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	default:
		fd = setup_file(progdir, "test.file", fpath);
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = posix_fadvise(fd, 0, 0, tc->advice);
	//sys_ret = syscall(__NR_fadvise64, fd, 0, 0, tc->advice);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(0, tc->ret, tc->err, sys_ret, sys_errno);

	if (fd >= 0) {
		close(fd);
		cleanup_file(fpath);
	}

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

