/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bfd_if.c - BFD library interface                                         */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <elf.h>
#include <argp.h>
#include "bt.h"
#include "bfd_if.h"

static int last_error;

/*-----------------------------------------------------------------------------
 *  misc functions
 *-----------------------------------------------------------------------------
 */
#if 0
static void dump_symbols(const char *header, asymbol **symbols, long cnt)
{
	long i;
	asymbol *sym;
	asection *sect;

	printf("%s\n", header);
	for (i = 0; i < cnt; i++) {
		sym = symbols[i];
		sect = sym->section;
		printf("\tSYM 0x%08lx", (unsigned long)bfd_asymbol_value(sym));
		printf(" <%s>%08x\t<%s>%08x:%d\n",
		       sym->name, sym->flags, sect->name, sect->flags,
		       sect->index);
	}
}
static void dump_synsymbols(const char *header, asymbol *symbols, long cnt)
{
	long i;
	asymbol *sym;
	asection *sect;

	printf("%s\n", header);
	for (i = 0; i < cnt; i++) {
		sym = &symbols[i];
		sect = sym->section;
		printf("\tSYM 0x%08lx", (unsigned long)bfd_asymbol_value(sym));
		printf(" <%s>%08x\t<%s>%08x:%d\n",
		       sym->name, sym->flags, sect->name, sect->flags,
		       sect->index);
	}
}
static void dump_relocs(struct bfd_if *bi)
{
	long i;
	arelent *r;

	for (i = 0; i < bi->n_relocs; i++) {
		r = bi->p_relocs[i];
		if (r->howto->type != RELOC_TYPE_PC32)
			continue;
		printf("RELOC=> CODE-OFFSET:0x%08lx",
		       (unsigned long)r->address);
		printf(" SYM-ADDR:0x%08lx",
		       (unsigned long)(*r->sym_ptr_ptr)->value);
		printf(" <%s>", (*r->sym_ptr_ptr)->name);
		printf("\n");
	}
}
#endif

static int is_code_sect(struct bfd_if *bi, asection *sect)
{
	int *p, *p_max;

	for (p = bi->p_code_sects, p_max = p + bi->n_code_sects; p < p_max; p++)
		if (sect->index == *p)
			return 1;
	return 0;
}

static asection* get_sect_has_addr(struct bfd_if *bi, unsigned long addr)
{
	asection *s;

	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			continue;
		if (addr >= (unsigned long)s->vma &&
		    addr < (unsigned long)(s->vma + bfd_get_section_size(s)))
			return s;
	}
	return NULL;
}

#ifdef RELOC_TEST
int reloc_test;

void enable_reloc_test(void)
{
	reloc_test = 1;
}

static void test_print_reloc(struct bfd_if *bi, arelent *r)
{
	asection *s;

	if (!reloc_test)
		return;
	s = get_sect_has_addr(bi, (unsigned long)r->address);
	//printf("Resolved reloc: ");
	printf("%s 0x%08lx", s->name, (unsigned long)(r->address - s->vma));
	printf(" %s\n", (*r->sym_ptr_ptr)->name);
}
#else
#  define test_print_reloc(bi, r)
#endif

static int f_cmp_reloc_to_addr(const void *p1, const void *p2)
{
	bfd_vma addr;
	arelent *r;

	addr = (bfd_vma)(*(unsigned long*)p1);
	r = *(arelent**)p2;
	if (addr < r->address)
		return -1;
	if (addr > r->address)
		return 1;
	return 0;
}

static int is_reloc_branch(struct bfd_if *bi, unsigned long next)
{
	arelent **p;

	if (!bi->p_relocs)
		return 0;
	next -= sizeof(unsigned long);
	p = bsearch(&next, bi->p_relocs, bi->n_relocs, sizeof(arelent*),
		    f_cmp_reloc_to_addr);
	return (p != NULL);
}

static unsigned long get_relocated_baddr(struct bfd_if *bi,
					 unsigned long branch,
					 unsigned long next,
					 unsigned long offset)
{
	arelent **p;
	asymbol *sym;

	if (!bi->p_relocs)
		return branch + offset;
	next -= sizeof(unsigned long);
	p = bsearch(&next, bi->p_relocs, bi->n_relocs, sizeof(arelent*),
		    f_cmp_reloc_to_addr);
	if (!p)
		return branch + offset;
	test_print_reloc(bi, *p);
	sym = *(*p)->sym_ptr_ptr;
	branch = branch - next + (unsigned long)bfd_asymbol_value(sym);
	if (sym->flags & BSF_FUNCTION) {
		/* symbol defined same section of this module */
		return branch + offset;
	} else if (sym->flags & BSF_SECTION_SYM) {
		/* symbol defined other section of this module */
		offset += (unsigned long)sym->section->vma;
		return branch + offset;
	} else {
		/* symbol defined outside the module */
		return branch;
	}
}

static int prepare_print_insn(struct bfd_if *bi, unsigned long addr)
{
	asection *sect;
	struct disassemble_info *info = &bi->info;
	struct sect_cache *cache = &bi->cache;

	if (cache->data) {
		if (addr >= (unsigned long)cache->vma &&
		    addr < (unsigned long)(cache->vma + cache->size)) {
			goto CACHE_FINISH;
		} else {
			free(cache->data);
			cache->data = NULL;
		}
	}
	sect = get_sect_has_addr(bi, addr);
	if (!sect)
		return -1;
		//last_error |= CHK_BTYPE_E_NOT_FOUND;

	cache->sect = sect;
	cache->vma = sect->vma;
	cache->size = bfd_get_section_size(sect);
	if (cache->size == 0)
		return 0;
	cache->data = xmalloc(cache->size);
	bfd_get_section_contents(bi->abfd, sect, cache->data, 0, cache->size);

CACHE_FINISH:
	info->buffer = cache->data;
	info->buffer_vma = cache->vma;
	info->buffer_length = cache->size;
	info->section = cache->sect;
	//info->symbols = NULL;
	//info->num_symbols = 0;
	info->insn_info_valid = 0;
	info->bytes_per_line = 0;
	info->bytes_per_chunk = 0;
	info->flags = 0;
	return 0;
}

/*-----------------------------------------------------------------------------
 *  intialize
 *-----------------------------------------------------------------------------
 */
static int chk_elf_header(struct bfd_if *bi, const char *obj_name)
{
	int rc;
	unsigned int i;
	FILE *f;
	unsigned char e_ident[EI_NIDENT];
	Elf32_Ehdr ehdr;
	Elf32_Phdr phdr;

	rc = -1;
	f = fopen(obj_name, "rb");
	if (!f) {
		fprintf(stderr, "%s file can't open.(%s)\n",
			obj_name, strerror(errno));
		return rc;
	}
	if (fread(e_ident, EI_NIDENT, 1, f) != 1) {
		fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
		goto EXIT;
	}
	if (e_ident[EI_DATA] != ELFDATA2LSB) {
		fprintf(stderr, "not little endian object.\n");
		goto EXIT;
	}
	if (e_ident[EI_CLASS] != ELFCLASS32) {
		fprintf(stderr, "not 32-bit architecture object.\n");
		goto EXIT;
	}
	if (fread(&ehdr.e_type, sizeof(ehdr) - EI_NIDENT, 1, f) != 1) {
		fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
		goto EXIT;
	}
	for (i = 0; i < ehdr.e_phnum; i++) {
		if (fread(&phdr, sizeof(phdr), 1, f) != 1) {
			fprintf(stderr, "fread failed.(%s)\n", strerror(errno));
			goto EXIT;
		}
		if (phdr.p_type != PT_LOAD || !(phdr.p_flags & PF_X))
			continue;
		bi->load_vaddr = phdr.p_vaddr;
		break;
	}
	rc = 0;
EXIT:
	if (f)
		fclose(f);
	return rc;
}

/* for DEBUG
static asection* get_section_has_index(bfd *abfd, int index)
{
	asection *s;

	for (s = abfd->sections; s; s = s->next)
		if (s->index == index)
			return s;
	return NULL;
}
*/

static int is_valid_code_sect(bfd *abfd, asection *s, int is_module)
{
	if (!(s->flags & SEC_CODE) || !(s->flags & SEC_ALLOC))
		return 0;
	if (!is_module)
		return 1;
	return (strncmp(s->name, ".init", 5) != 0);
}

static int chk_valid_code_sects(struct bfd_if *bi, int is_module)
{
	int *top, *p;
	asection *s;

	top = xcalloc(bi->abfd->section_count, sizeof(int));
	p = top;
	for (s = bi->abfd->sections; s; s = s->next) {
		if (is_valid_code_sect(bi->abfd, s, is_module))
			*p++ = s->index;
	}
	bi->p_code_sects = top;
	bi->n_code_sects = p - top;
	/* for DEBUG
	printf("VALID CODE SECTS:");
	for (p = bi->p_code_sects, top = p + bi->n_code_sects; p < top; p++) {
		s = get_section_has_index(bi->abfd, *p);
		if (s)
			printf(" %d:%s,", *p, s->name);
	}
	printf("\n");
	*/
	return 0;
}

static void set_sect_offset(struct bfd_if *bi)
{
	bfd_vma sect_offset;
	asection *s;

	if (!(bfd_get_file_flags(bi->abfd) & HAS_RELOC))
		return;
	sect_offset = 0;
	for (s = bi->abfd->sections; s; s = s->next) {
		/* If the section address already set, then do not add the
		 * section offset (e.g. relocatable kernel).
		 */
		if (!is_code_sect(bi, s) || s->vma)
			continue;
		s->vma += sect_offset;
		sect_offset += bfd_get_section_size(s);
	}
	/* for DEBUG
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			continue;
		printf("code-sect:%s\t%08lx\n", s->name, s->vma);
	}
	*/
	return;
}

static int read_all_symbols(bfd *abfd, int is_dynamic,
			    asymbol ***symbols, long *nsyms)
{
	long size;

	if (!(bfd_get_file_flags(abfd) & HAS_SYMS))
		return 0;

	if (is_dynamic)
		size = bfd_get_dynamic_symtab_upper_bound(abfd);
	else
		size = bfd_get_symtab_upper_bound(abfd);
	if (size <= (long)sizeof(asymbol*))
		return 0;

	*symbols = xmalloc(size);
	if (is_dynamic)
		*nsyms = bfd_canonicalize_dynamic_symtab(abfd, *symbols);
	else
		*nsyms = bfd_canonicalize_symtab(abfd, *symbols);
	return 0;
}

static int __compare_symbols(const void *d1, const void *d2)
{
	asymbol *sym1 = *(asymbol**)d1, *sym2 = *(asymbol**)d2;
	bfd_vma a1, a2;
	flagword f1, f2;
	const char *n1, *n2;

	a1 = bfd_asymbol_value(sym1);
	a2 = bfd_asymbol_value(sym2);
	if (a1 > a2)
		return 1;
	if (a1 < a2)
		return -1;

	if (sym1->section > sym2->section)
		return 1;
	if (sym1->section < sym2->section)
		return -1;

	/* Try to sort global symbols before local symbols before function
	 * symbols before debugging symbols.
	 */
	f1 = sym1->flags;
	f2 = sym2->flags;
	if ((f1 & BSF_DEBUGGING) != (f2 & BSF_DEBUGGING)) {
		if ((f1 & BSF_DEBUGGING))
			return 1;
		else
			return -1;
	}
	if ((f1 & BSF_FUNCTION) != (f2 & BSF_FUNCTION)) {
		if ((f1 & BSF_FUNCTION))
			return -1;
		else
			return 1;
	}
	if ((f1 & BSF_LOCAL) != (f2 & BSF_LOCAL)) {
		if ((f1 & BSF_LOCAL))
			return 1;
		else
			return -1;
	}
	if ((f1 & BSF_GLOBAL) != (f2 & BSF_GLOBAL)) {
		if ((f1 & BSF_GLOBAL))
			return -1;
		else
			return 1;
	}

	/* Symbols that start with '.' might be section names, so sort them
	 * after symbols that don't start '.'.
	 */
	n1 = bfd_asymbol_name(sym1);
	n2 = bfd_asymbol_name(sym2);
	if (n1[0] == '.' && n2[0] != '.')
		return 1;
	if (n1[0] != '.' && n2[0] == '.')
		return -1;

	return strcmp(n1, n2);
}

static long remove_useless_symbols(struct bfd_if *bi, asymbol **syms,
				   long count)
{
	asymbol **p_in = syms, **p_out = syms, *sym_prev = NULL;
	asection *sect;

	while (--count >= 0) {
		asymbol *sym = *p_in++;
		if (sym->name == NULL || sym->name[0] == '\0')
			continue;
		if (sym->flags & (BSF_DEBUGGING))
			continue;
		sect = sym->section;
		/*
		if (bfd_is_und_section(sym->section)
		    || bfd_is_com_section(sym->section))
			continue;
			*/
		if (bfd_is_com_section(sect))
			continue;
		if (!is_code_sect(bi, sect) && !bfd_is_und_section(sect))
			continue;
		*p_out++ = sym_prev = sym;
	}
	return p_out - syms;
}

static int __resolve_undsym(struct bfd_if *bi, char *sname, unsigned long addr)
{
	long i;
	asymbol *sym;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (sym->name[0] != sname[0] || strcmp(sym->name, sname) != 0)
			continue;
		sym->value = addr;
		return 0;
	}
	//fprintf(stdout, "WARN: symbol(%s:0x%08lx) not found.\n", sname, addr);
	return 0;
}

static int resolve_undsyms(struct bfd_if *bi, const char *obj_name,
			   const char *kallsyms)
{
	int rc, len, num;
	FILE *f = NULL;
	char *mod_name = strdup(obj_name);
	char buf[MAX_LINE_LEN + 1], sym[MAX_LINE_LEN], mod[MAX_LINE_LEN], *p;
	unsigned long addr;

	if (!kallsyms)
		return 0;
	rc = -1;
	mod_name = basename(mod_name);
	len = strlen(mod_name);
	if (len <= 3 || strcmp(&mod_name[len - 3], ".ko") != 0)
		return 0;	/* target object is not kernel module */
	mod_name[len - 3] = '\0';
	f = fopen(kallsyms, "r");
	if (!f) {
		fprintf(stderr, "%s open failed.\n", kallsyms);
		goto EXIT;
	}
	buf[MAX_LINE_LEN] = '\0';
	while ((p = fgets(buf, MAX_LINE_LEN, f)) != NULL) {
		len = strlen(buf);
		if (buf[len - 2] != ']' || (buf[9] != 'U' && buf[9] != 'u'))
			continue;
		num = sscanf(buf, "%lx %*c %s\t[%[^]]]", &addr, sym, mod);
		if (num != 3 || strcmp(mod_name, mod) != 0)
			continue;
		__resolve_undsym(bi, sym, addr);
	}
	rc = 0;
EXIT:
	if (f)
		fclose(f);
	return rc;
}

static void chk_relocs_sect(struct bfd_if *bi, asection *sect,
			    bfd_vma sect_offset)
{
	long size, i, n;

	if (!(is_code_sect(bi, sect) && (sect->flags & SEC_RELOC)))
		return;
	size = bfd_get_reloc_upper_bound(bi->abfd, sect);
	if (size < 0 || size == sizeof(arelent*))
		return;
	n = size / sizeof(arelent*) - 1;
	if (bi->n_relocs)
		bi->p_relocs = xrealloc(bi->p_relocs,
					(bi->n_relocs * sizeof(arelent*))+size);
	else
		bi->p_relocs = xmalloc(size);
	memset(&bi->p_relocs[bi->n_relocs], 0, size);

	bfd_canonicalize_reloc(bi->abfd, sect,
			       &bi->p_relocs[bi->n_relocs], bi->p_syms);
	if (sect_offset)
		for (i = bi->n_relocs; i < bi->n_relocs + n; i++)
			bi->p_relocs[i]->address += sect_offset;
	bi->n_relocs += n;
}

static int f_cmp_relocs(const void *p1, const void *p2)
{
	arelent *r1, *r2;

	r1 = *(arelent**)p1;
	r2 = *(arelent**)p2;
	if (r1->address < r2->address)
		return -1;
	if (r1->address > r2->address)
		return 1;
	return 0;
}

static int chk_relocs(struct bfd_if *bi)
{
	bfd_vma sect_offset;
	asection *s;

	if (!(bfd_get_file_flags(bi->abfd) & HAS_RELOC))
		return 0;

	sect_offset = 0;
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			continue;
		chk_relocs_sect(bi, s, sect_offset);
		sect_offset += bfd_get_section_size(s);
	}
	qsort(bi->p_relocs, bi->n_relocs, sizeof(arelent*), f_cmp_relocs);
	//dump_relocs(bi);
	return 0;
}

static int build_sorted_func_symbols(struct bfd_if *bi, const char *obj_name,
				     const char *kallsyms)
{
	long i;

	bi->n_syms = bi->n_dynsyms = bi->n_synsyms = 0;
	bi->p_syms = bi->p_dynsyms = NULL;
	bi->p_synsyms = NULL;

	if (read_all_symbols(bi->abfd, 0, &bi->p_syms, &bi->n_syms) < 0)
		goto EXIT;
	//dump_symbols("syms", bi->p_syms, bi->n_syms);
	if (read_all_symbols(bi->abfd, 1, &bi->p_dynsyms, &bi->n_dynsyms) < 0)
		goto EXIT;
	//dump_symbols("dynsyms", bi->p_dynsyms, bi->n_dynsyms);

	chk_relocs(bi);
	bi->n_synsyms = bfd_get_synthetic_symtab(bi->abfd,
						 bi->n_syms, bi->p_syms,
						 bi->n_dynsyms, bi->p_dynsyms,
						 &bi->p_synsyms);
	if (bi->n_synsyms < 0)
		goto EXIT;
	//dump_synsymbols("synsyms", bi->p_synsyms, bi->n_synsyms);

	bi->n_fsyms = bi->n_syms ? bi->n_syms : bi->n_dynsyms;
	bi->p_fsyms = xmalloc((bi->n_fsyms + bi->n_synsyms) * sizeof(asymbol*));
	memcpy(bi->p_fsyms, bi->n_syms ? bi->p_syms : bi->p_dynsyms,
	       bi->n_fsyms * sizeof(asymbol*));
	bi->n_fsyms = remove_useless_symbols(bi, bi->p_fsyms, bi->n_fsyms);

	for (i = 0; i < bi->n_synsyms; i++) {
		bi->p_fsyms[bi->n_fsyms] = bi->p_synsyms + i;
		bi->n_fsyms++;
	}
	if (resolve_undsyms(bi, obj_name, kallsyms) < 0)
		goto EXIT;
	//dump_relocs(bi);
	qsort(bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*), __compare_symbols);
	//dump_symbols("sorted-fsyms", bi->p_fsyms, bi->n_fsyms);
	return 0;
EXIT:
	if (bi->p_syms)
		free(bi->p_syms);
	if (bi->p_dynsyms)
		free(bi->p_dynsyms);
	if (bi->p_synsyms)
		free(bi->p_synsyms);
	if (bi->p_fsyms)
		free(bi->p_fsyms);
	if (bi->p_relocs)
		free(bi->p_relocs);
	return -1;
}

/* We check all section because HAS_DEBUG never set in abfd->flags */
static void chk_has_debuginfo(struct bfd_if *bi)
{
	asection *s;

	for (s = bi->abfd->sections; s; s = s->next)
		if (s->flags & SEC_DEBUGGING) {
			bi->has_debuginfo = 1;
			break;
		}
}

#ifdef USE_ELFUTILS
int init_dwarf(struct bfd_if *bi, const char *obj_name)
{
	int argi, n = 3;
	char *s[] = { "", "-e", (char*)obj_name, NULL };

	argp_parse(dwfl_standard_argp(), n, s, 0, &argi, &bi->dwfl);
	if (!bi->dwfl)
		return -1;
	return 0;
}
#endif

int init_bfd_if(struct bfd_if *bi, const char *obj_name, const char *kallsyms)
{
	struct disassemble_info *info = &bi->info;
	int len;

	memset(bi, 0, sizeof(struct bfd_if));

	if (chk_elf_header(bi, obj_name) < 0)
		return -1;

	bi->abfd = bfd_openr(obj_name, NULL);
	if (!bi->abfd) {
		fprintf(stderr, "%s bfd_openr failed.\n", obj_name);
		return -1;
	}
	if (!bfd_check_format(bi->abfd, bfd_object)) {
		fprintf(stderr, "%s bfd_check_format failed.\n", obj_name);
		return -1;
	}
	len = strlen(obj_name);
	chk_valid_code_sects(bi, (strncmp(obj_name + len - 3, ".ko", 3) == 0));
	set_sect_offset(bi);
	if (build_sorted_func_symbols(bi, obj_name, kallsyms) < 0)
		return -1;

	init_disassemble_info(info, stdout, (fprintf_ftype)fprintf);
	info->flavour = bfd_get_flavour(bi->abfd);
	info->arch = bfd_get_arch(bi->abfd);
	info->mach = bfd_get_mach(bi->abfd);
	info->disassembler_options = NULL;
	info->octets_per_byte = bfd_octets_per_byte(bi->abfd);

	if (bfd_big_endian(bi->abfd))
		info->display_endian = info->endian = BFD_ENDIAN_BIG;
	else if (bfd_little_endian(bi->abfd))
		info->display_endian = info->endian = BFD_ENDIAN_LITTLE;
	else
		info->endian = BFD_ENDIAN_UNKNOWN;
	disassemble_init_for_target(info);
	chk_has_debuginfo(bi);
	return 0;
}

unsigned long get_offset_addr(struct bfd_if *bi, unsigned long begin)
{
	if (bfd_get_file_flags(bi->abfd) & HAS_RELOC)
		return begin;
	return begin - bi->load_vaddr;
}

/* This function is specified for kernel. It doesn't calculate section offset.
 */
void get_kernel_min_max_addr(struct bfd_if *bi,
			     unsigned long *min, unsigned long *max)
{
	bfd_vma sect_max;
	asection *s;

	*min = 0xffffffff;
	*max = 0;
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			continue;
		if ((unsigned long)s->vma < *min)
			*min = (unsigned long)s->vma;
		sect_max = s->vma + bfd_get_section_size(s) - 1;
		if ((unsigned long)sect_max > *max)
			*max = (unsigned long)sect_max;
	}
	return;
}

void remove_useless_fsyms(struct bfd_if *bi, unsigned long begin,
			  unsigned long end, unsigned long offset)
{
	long i, cnt;
	asymbol *sym, **symbols = bi->p_fsyms;
	bfd_vma val, prev_val = 0xffffffff;
	asection *sect;

	cnt = 0;
	for (i = 0; i < bi->n_fsyms; i++) {
		sym = symbols[i];
		val = bfd_asymbol_value(sym);
		sect = sym->section;
		if (!(sect->flags & SEC_ALLOC) || !(sect->flags & SEC_CODE))
			continue;
		if (val < sect->vma ||
		    val >= sect->vma + bfd_get_section_size(sect))
			continue;
		if ((unsigned long)val + offset < begin ||
		    (unsigned long)val + offset > end)
			continue;

		if (val != prev_val) {
			symbols[cnt++] = sym;
			prev_val = val;
		}
	}
	bi->n_fsyms = cnt;
}

/*-----------------------------------------------------------------------------
 *  branch check
 *-----------------------------------------------------------------------------
 */
static int btype;
static unsigned long baddr;

#define PRINT_BUF_MAX	128
static char print_buf[PRINT_BUF_MAX + 1];
static int seq_num;
static int offset;
static int baddr_offset;


static int chk_btype_dummy_fprintf(FILE *stream, const char *fmt, ...)
{
	va_list argp;
	int rc;
	char *p_tmp;

	va_start(argp, fmt);
	rc = vsnprintf(print_buf, PRINT_BUF_MAX, fmt, argp);
	if (rc >= PRINT_BUF_MAX)
		last_error |= CHK_BTYPE_E_OVERFLOW;
	va_end(argp);

	//printf("DBG>%.*s<\n", rc, print_buf);
	switch (seq_num) {
	case 0:
		if (strncmp(print_buf, "jmp", 3) == 0 ||
		    strncmp(print_buf, "ljmp", 4) == 0) {
			btype = BTYPE_JMP;
		} else if (print_buf[0] == 'j' ||
			   strncmp(print_buf, "loop", 4) == 0) {
			btype = BTYPE_BRANCH;
		} else if (strncmp(print_buf, "call", 4) == 0) {
			btype = BTYPE_CALL;
		} else if (strncmp(print_buf, "ret", 3) == 0) {
			btype = BTYPE_RET;
		} else if (strncmp(print_buf, "iret", 4) == 0 ||
			   strncmp(print_buf, "sysexit", 7) == 0) {
			btype = BTYPE_IRET;
		} else if (strncmp(print_buf, "int", 3) == 0 ||
			   strncmp(print_buf, "sysenter", 8) == 0) {
			btype = BTYPE_INT;
		} else if (strncmp(print_buf, "ud2", 3) == 0) {
			btype = BTYPE_BREAK;
		} else
			btype = BTYPE_OTHER;
		baddr = 0;
		break;
	case 1:
		switch (btype) {
		case BTYPE_JMP:
		case BTYPE_BRANCH:
		case BTYPE_CALL:
			baddr = strtoll(print_buf, &p_tmp, 0);
			if (p_tmp - print_buf != rc)
				baddr = UNKNOWN_BADDR;
				//last_error |= CHK_BTYPE_E_BADDR;
			break;
		}
		break;
	case 2:
		if (btype != BTYPE_OTHER && baddr != UNKNOWN_BADDR)
			last_error |= CHK_BTYPE_E_SEQNUM;
		break;
	}
	seq_num++;

	return rc;
}

static int __get_branch_info(struct bfd_if *bi, unsigned long addr, int *type,
			     unsigned long *addr_next,
			     unsigned long *addr_branch)
{
	int bytes;
	struct disassemble_info *info = &bi->info;
	fprintf_ftype func_save;

	if (prepare_print_insn(bi, addr) < 0)
		return -CHK_BTYPE_E_NOT_FOUND;

	seq_num = last_error = 0;
	baddr = 0;
	func_save = info->fprintf_func;
	info->fprintf_func = (fprintf_ftype)chk_btype_dummy_fprintf;
	//bytes = print_insn_i386_intel(addr, info);
	//bytes = print_insn_i386(addr, info);
	bytes = print_insn_i386_att((bfd_vma)addr, info);
	if (bytes < 0) {
		last_error |= CHK_BTYPE_E_INTERNAL;
		goto EXIT;
	}
	info->fprintf_func = func_save;
	*addr_next = addr + bytes;
	*addr_branch = 0;
	switch (btype) {
	case BTYPE_JMP:
	case BTYPE_CALL:
	case BTYPE_BRANCH:
		*addr_branch = baddr;
		break;
	}
	*type = btype;
EXIT:
	return (last_error ? -last_error : 0);
}

int get_branch_info(struct bfd_if *bi, unsigned long addr, int *type,
		    unsigned long *next, unsigned long *branch,
		    unsigned long offset)
{
	if (__get_branch_info(bi, addr - offset, type, next, branch) < 0)
		return -1;
	if (*branch != UNKNOWN_BADDR)
		*branch = get_relocated_baddr(bi, *branch, *next, offset);
	*next += offset;
	return 0;
}

/*-----------------------------------------------------------------------------
 *  path check
 *-----------------------------------------------------------------------------
 */
typedef struct {
	struct bfd_if		*bi;
	node			*pt;
	node			*node_addrs;
	int			skip_ud2_srcinfo;
} pack_bi_pt;

static int ____f_chk_range(struct path *p, unsigned long addr)
{
	if (p->cnt < 0) {
		if (addr < p->addr)
			return -1;
		if (addr > p->addr)
			return 1;
	} else {
		if (addr < p->addr)
			return -1;
		if (addr >= p->next)
			return 1;
	}
	return 0;
}

static int __f_chk_range(void *__target, void *__dt)
{
	unsigned long addr = *((unsigned long*)__target);
	struct path *p = (struct path*)__dt;

	return ____f_chk_range(p, addr);
}

static struct path* __find_path_from_addr(node *tree, unsigned long addr)
{
	return (struct path*)search_tree(&addr, tree, __f_chk_range);
}

static int f_chk_addr(void *__target, void *__dt)
{
	unsigned long a1 = *((unsigned long*)__target);
	unsigned long a2 = *((unsigned long*)__dt);

	if (a1 < a2)
		return -1;
	if (a1 > a2)
		return 1;
	return 0;
}

static int add_node_addrs(struct bfd_if *bi, node *pt, node **node_addrs,
			  unsigned long addr)
{
	unsigned long *p;

	if (addr == UNKNOWN_BADDR || __find_path_from_addr(pt,addr)
	    ||search_tree(&addr, *node_addrs, f_chk_addr))
		return 0;

#ifdef DBG_CHK_PATH
	printf("ADD:0x%08lx\n", addr);
#endif
	p = xmalloc(sizeof(unsigned long));
	*p = addr;
	*node_addrs = insert_tree(p, *node_addrs, f_chk_addr);
	return *node_addrs == NULL ? -1 : 0;
}

/*
 * Check ud2 instruction from 'begin' to 'end', and return the possibility
 * of having source information.
 * If no ud2 instruction found, it return -1.
 */
int __chk_ud2_src_info(struct bfd_if *bi, unsigned long begin,unsigned long end)
{
	unsigned long addr, next, branch;
	int rc, type;
	struct sect_cache *cache = &bi->cache;

	for (addr = begin; addr < end; addr = next) {
		rc = __get_branch_info(bi, addr, &type, &next, &branch);
		if (rc < 0) {
			fprintf(stderr, "__get_branch_info failed at 0x%08lx",
				addr);
			fprintf(stderr, ".(%d)\n", rc);
			return -1;
		}
		/* check ud2 instruction */
		if (type != BTYPE_BREAK ||
		    cache->data[addr - (unsigned long)cache->vma] != 0x0f ||
		    cache->data[addr - (unsigned long)cache->vma + 1] != 0x0b)
			continue;
		/* Caution:
		 * At this point, we have to finish checking this function
		 * because we don't know whether we should skip source
		 * information right now.
		 * So, we continue checking from next function's top address.
		 */
		/* ud2 instruction (2bytes) and src info (6bytes) */
		addr += 2 + 5;
		if (addr < (unsigned long)cache->vma ||
		    addr >= (unsigned long)(cache->vma + cache->size))
			return -1;
		return (cache->data[addr - (unsigned long)cache->vma]
				== (addr >> 24));
	}
	return -1;
}

#define MAX_CHK_UD2	10
#define MAX_CHK_FUNC	100
int chk_kernel_has_ud2_src_info(struct bfd_if *bi)
{
	int rc, i, cnt, func_cnt;
	asymbol *sym, *sym2;
	asection *s;
	unsigned long begin, end, sect_end;

	cnt = func_cnt = 0;
	for (i = 0; i + 1 < bi->n_fsyms && func_cnt < MAX_CHK_FUNC; i++) {
		sym = bi->p_fsyms[i];
		sym2 = bi->p_fsyms[i+1];
		if (!(sym->flags & BSF_FUNCTION))
			continue;
		if (!cnt)
			func_cnt++;
		/* get 'begin' and 'end' addresses */
		begin = (unsigned long)bfd_asymbol_value(sym);
		end = (unsigned long)bfd_asymbol_value(sym2);
		s = get_sect_has_addr(bi, begin);
		sect_end = (unsigned long)(s->vma + bfd_get_section_size(s));
		if (end > sect_end)
			end = sect_end;
		//printf("checking <%s>:%08lx-%08lx\t", sym->name, begin, end);
		rc = __chk_ud2_src_info(bi, begin, end);
		switch (rc) {
		case -1:
			//printf("---\n");
			continue;
		case 0:
			//printf("ud2 (NO src info)\n");
			return 0;
		case 1:
			//printf("ud2 (WITH src info)\n");
			if (++cnt >= MAX_CHK_UD2)
				return 1;
		}
	}
	return 0;
}

static int add_func_and_section_start_addrs(struct bfd_if *bi,
					    node **node_addrs)
{
	int i;
	asymbol *sym;
	asection *s;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (!(sym->flags & BSF_FUNCTION))
			continue;
		if (add_node_addrs(bi, NULL, node_addrs,
				   (unsigned long)bfd_asymbol_value(sym)) < 0)
			return -1;
	}
	for (s = bi->abfd->sections; s; s = s->next) {
		if (!is_code_sect(bi, s))
			continue;
		if (add_node_addrs(bi, NULL, node_addrs, (unsigned long)s->vma)
		    < 0)
			return -1;
	}
	return 0;
}

static int chk_one_node_addr(struct bfd_if *bi,
			     unsigned long begin, unsigned long max,
			     node **node_addrs, node **pt, int skip_ud2_srcinfo)
{
	unsigned long addr, next, branch;
	int rc, type;
	struct path *p;

#ifdef DBG_CHK_PATH
	printf("CHK:0x%08lx", begin);
	printf(" max(0x%08lx)\n", max);
#endif
	for (addr = begin; addr < max; addr = next) {
		rc = __get_branch_info(bi, addr, &type, &next, &branch);
		if (rc < 0) {
			fprintf(stderr, "__get_branch_info failed at 0x%08lx",
				addr);
			fprintf(stderr, ".(%d)\n", rc);
			return rc;
		}
		if (type == BTYPE_OTHER)
			continue;

		p = xcalloc(1, sizeof(struct path));
		p->addr = begin;
		p->cnt = -1;
		p->type = type;
		p->base = addr;
		p->next = next;
		p->jmp = branch;
		/* When switch case jump, set next_cnt value to -1. */
		p->next_cnt = type == BTYPE_JMP ? -1 : 0;
		*pt = insert_tree(p, *pt, f_chk_addr);
		if (skip_ud2_srcinfo && type == BTYPE_BREAK)
			break;
		if (!is_reloc_branch(bi, next)) {
			if ((rc = add_node_addrs(bi, *pt, node_addrs, branch))
			    < 0)
				return rc;
		}
		/* When switch case jmp, we need to check the next code */
		if ((rc = add_node_addrs(bi, *pt, node_addrs, next)) < 0)
			return rc;
		break;
	}
	return 0;
}

static int f_chk_each_node_addr(void *__dt, void *user_data)
{
	unsigned long addr = *((unsigned long*)__dt);
	pack_bi_pt *pack = (pack_bi_pt*)user_data;
	node **node_addrs = &pack->node_addrs;
	asection *s = get_sect_has_addr(pack->bi, addr);

	if (!s)
		return 0;
	if (chk_one_node_addr(pack->bi, addr,
			      (unsigned long)s->vma + bfd_get_section_size(s),
			      node_addrs, &pack->pt, pack->skip_ud2_srcinfo)
	    < 0)
		return -1;
	return 0;
}

static void f_free_each_node_addr(void *__dt)
{
	unsigned long *p = (unsigned long*)__dt;

	if (p)
		free(p);
}

static int chk_node_addrs(pack_bi_pt *pack)
{
	int rc;
	node *tmp;

	if (!pack->node_addrs)
		return 0;
	tmp = pack->node_addrs;
	pack->node_addrs = NULL;
	rc = for_each_node(tmp, f_chk_each_node_addr, pack);
	if (rc < 0)
		return -1;
	free_tree(tmp, f_free_each_node_addr);
	return 0;
}

//#define GIVEUP_MAX	8
static int loop_chk_req_addrs(pack_bi_pt *pack)
{
	long left_cnt, prev_left, giveup_cnt;

	prev_left = 0;
	//for (giveup_cnt = 0; pack->node_addrs && giveup_cnt < GIVEUP_MAX;) {
	for (;;) {
		left_cnt = get_node_cnt(pack->node_addrs);
		if (!left_cnt)
			break;
#ifdef DBG_CHK_PATH
		printf(">>>>>> loop chk (%ld) <<<<<<\n", left_cnt);
#endif
		if (prev_left == left_cnt)
			giveup_cnt++;
		else
			giveup_cnt = 0;
		prev_left = left_cnt;

		if (chk_node_addrs(pack) < 0)
			return -1;
	}
	/*
	if (giveup_cnt >= GIVEUP_MAX) {
		printf("give up file analysis...\n");
		//for (r = pack->req_addrs; r; r = r->next) {
		//	printf("    ");
		//	printf_bfd_vma(r->addr);
		//	printf("\n");
		//}
		return -1;
	}
	*/
	return 0;
}

static long conv_id;
static int f_conv_node_tree_to_array(void *__dt, void *user_data)
{
	struct path *p = (struct path*)__dt;
	struct path **array = (struct path**)user_data;

	array[conv_id] = p;
	conv_id++;
	return 0;
}
static int conv_path_tree_to_array(node *pt, struct path ***pt_array, long *cnt)
{
	struct path **__pt;

	*cnt = get_node_cnt(pt);
	__pt = xmalloc(sizeof(struct path*) * (*cnt));
	conv_id = 0;
	for_each_node(pt, f_conv_node_tree_to_array, __pt);
	free_tree(pt, NULL);

	*pt_array = __pt;
	return 0;
}

static int fix_all_path(struct bfd_if *bi, struct path **parray, long cnt)
{
	long i;
	struct path *p, *p_next;
	int rc, type;
	unsigned long next, branch;

	for (i = 0; i < cnt; i++) {
		p = parray[i];
		p_next = i + 1 < cnt ? parray[i + 1] : NULL;
		if (p_next && p_next->addr < p->next) {
			p->type = BTYPE_OTHER;
			p->next = p_next->addr;
			p->base = p->jmp = 0;
			continue;
		}
		if (p->type != BTYPE_BRANCH && p->type != BTYPE_JMP
		    && p->type != BTYPE_CALL)
			continue;
		rc = __get_branch_info(bi, p->base, &type, &next, &branch);
		if (rc < 0) {
			fprintf(stderr, "fix_all_path failed at 0x%08lx.(%d)\n",
				p->base, rc);
			return -1;
		}
		if (branch != 0 && branch != UNKNOWN_BADDR)
			p->jmp = branch;
		if (p->type != BTYPE_JMP)
			p->next = next;
	}
	for (i = 0; i < cnt; i++) {
		p = parray[i];
		p->cnt = 0;
	}
	return 0;
}

static void resolve_relocs_and_offset(struct bfd_if *bi, struct path **parray,
				      long cnt, unsigned long offset)
{
	long i;
	struct path *p;
	unsigned long rc_addr;

	for (i = 0; i < cnt; i++) {
		p = parray[i];
		p->addr += offset;
		rc_addr = p->next;
		p->next += offset;
		if (p->type == BTYPE_OTHER)
			continue;
		p->base += offset;
		if (p->jmp != UNKNOWN_BADDR) {
			if (p->type != BTYPE_CALL && p->type != BTYPE_JMP) {
				p->jmp += offset;
				continue;
			}
			p->jmp = get_relocated_baddr(bi, p->jmp, rc_addr,
						     offset);
		}
	}
}

/*
 * Build the path tree
 *   First, we check all path node addresses, and build the path tree.
 *   (At this point, base addr has the possibility of overlapping with the
 *    following path.)
 *   Second, fix the overlapping addresses.
 */
int chk_path_tree(struct bfd_if *bi, struct path ***parray, long *cnt,
		  unsigned long offset, int skip_ud2_srcinfo)
{
	pack_bi_pt pack;

	pack.bi = bi;
	pack.pt = NULL;
	pack.node_addrs = NULL;
	pack.skip_ud2_srcinfo = skip_ud2_srcinfo;

	/* 1st process */
	if (add_func_and_section_start_addrs(bi, &pack.node_addrs) < 0) {
		last_error = 1;
		goto EXIT;
	}
	if (last_error || loop_chk_req_addrs(&pack) < 0) {
		last_error = 1;
		goto EXIT;
	}
	/* 2nd process */
	if (last_error || conv_path_tree_to_array(pack.pt, parray, cnt) < 0) {
		last_error = 1;
		goto EXIT;
	}
	if (last_error || fix_all_path(bi, *parray, *cnt) < 0) {
		last_error = 1;
		goto EXIT;
	}
	if (!last_error) {
	    	resolve_relocs_and_offset(bi, *parray, *cnt, offset);
		goto EXIT;
	}
EXIT:
#ifdef DBG_CHK_PATH
	if (!last_error) {
		printf("------ after left code check ------\n");
		dump_path_tree(*parray, *cnt);
	}
#endif
	return (last_error ? -last_error : 0);
}

/*-----------------------------------------------------------------------------
 *  path utility
 *-----------------------------------------------------------------------------
 */
void free_path(struct path *p)
{
	struct unknown *uk, *next;

	if (!p)
		return;
	if (p->jmp == UNKNOWN_BADDR) {
		for (uk = (struct unknown*)p->jmp_cnt; uk; uk = next) {
			next = uk->next;
			free(uk);
		}
	}
	free(p);
}

void printf_path(struct path *p)
{
	struct unknown *uk;
	int type = p->type;

	if (type & IS_INVALID_PATH) {
		printf("(INV)\t ");
		type &= ~IS_INVALID_PATH;
	} else
		printf("(%ld)\t ", p->cnt);
	printf("0x%08lx", p->addr);
	switch (type) {
	case BTYPE_BRANCH:
		printf(" JXX  "); break;
	case BTYPE_JMP:
		printf(" JMP  "); break;
	case BTYPE_CALL:
		printf(" CALL "); break;
	case BTYPE_RET:
		printf(" RET  "); break;
	case BTYPE_IRET:
		printf(" IRET "); break;
	case BTYPE_INT:
		printf(" INT  "); break;
	case BTYPE_BREAK:
		printf(" BRK  "); break;
	default:
		printf(" ---  ---------- => ----------(-) : ");
		printf("0x%08lx(%ld)\n", p->next, p->next_cnt);
		return;
	}
	printf("0x%08lx => ", p->base);
	switch (type) {
	case BTYPE_BRANCH:
	case BTYPE_CALL:
	case BTYPE_JMP:
		if (p->jmp == UNKNOWN_BADDR) {
			uk = (struct unknown*)p->jmp_cnt;
			if (p->cnt < 0 || !uk)
				printf("XXXXXXXXXX(0) : ");
			else {
				for (; uk; uk = uk->next) {
					printf("0x%08lx(%ld), ",
					       uk->addr, uk->cnt);
				}
				printf(": ");
			}
		} else {
			printf("0x%08lx(%ld) : ", p->jmp, p->jmp_cnt);
		}
		if (type == BTYPE_JMP)
			printf("----------(-)");
		else {
			printf("0x%08lx(%ld)", p->next, p->next_cnt);
		}
		break;
	case BTYPE_RET:
	case BTYPE_IRET:
	case BTYPE_INT:
	case BTYPE_BREAK:
		printf("----------(-) : ----------(-)");
		break;
	}
	printf("\n");
}

void dump_path_tree(struct path **tree, long cnt)
{
	long i;

	printf("------ path tree ------\n");
	for (i = 0; i < cnt; i++)
		printf_path(tree[i]);
	/* for DEBUG
	{
		struct path *p_prev, *p;
		unsigned long addr;

		p_prev = tree[0];
		printf("start: %08lx\n", p_prev->addr);
		for (i = 1; i < cnt; i++, p_prev = p) {
			p = tree[i];
			addr = p_prev->next;
			if (p->addr - addr > 30)
				printf("brank: %08lx:%08lx\n", addr, p->addr);
		}
		printf("end: %08lx\n", p->next);
	}
	*/
}

static int f_chk_range(const void *p1, const void *p2)
{
	unsigned long addr = *((unsigned long*)p1);
	struct path *p = *(struct path**)p2;

#ifdef DBG_CHK_PATH
	printf("DBG:0x%08lx:0x%08lx\n", addr, p->addr);
#endif
	return ____f_chk_range(p, addr);
}

struct path* find_path_from_addr(struct path **tree, long *cnt,
				 unsigned long addr)
{
	struct path **p;

	p = (struct path**)bsearch(&addr, tree, *cnt, sizeof(struct path*),
				   f_chk_range);
	if (p) {
		*cnt = p - tree;
		return *p;
	} else
		return NULL;
}

/*-----------------------------------------------------------------------------
 *  other utility
 *-----------------------------------------------------------------------------
 */
int get_source_info_binutils(struct bfd_if *bi, unsigned long addr,
			     const char **src_name, const char **func_name,
			     int *lno);
#ifdef USE_ELFUTILS
int get_source_info_elfutils(struct bfd_if *bi, unsigned long addr,
			     const char **src_name, const char **func_name,
			     int *lno)
{
	Dwfl_Module *mod = NULL;
	Dwfl_Line *line = NULL;
	Dwarf_Addr a = addr;

	*src_name = *func_name = NULL; /* func_name is not used */
	*lno = 0;

	mod = dwfl_addrmodule(bi->dwfl, a);
	line = dwfl_module_getsrc(mod, a);
	if (!line ||
	    !(*src_name = dwfl_lineinfo(line, &a, lno, NULL, NULL, NULL)))
		return -1;
	return 0;
}

inline int get_source_info(struct bfd_if *bi, unsigned long addr,
			   const char **src_name, const char **func_name,
			   int *lno)
{
	if (bi->dwfl)
		return get_source_info_elfutils(bi, addr, src_name, func_name,
						lno);
	else
		return get_source_info_binutils(bi, addr, src_name, func_name,
						lno);
}
#else
inline int get_source_info(struct bfd_if *bi, unsigned long addr,
			   const char **src_name, const char **func_name,
			   int *lno)
{
	return get_source_info_binutils(bi, addr, src_name, func_name, lno);
}
#endif
int get_source_info_binutils(struct bfd_if *bi, unsigned long addr,
			     const char **src_name, const char **func_name,
			     int *lno)
{
	int rc;
	asection *sect;

	*src_name = *func_name = NULL;
	*lno = 0;
#ifdef TEST
	return -1;
#endif
	if (!bi->has_debuginfo)
		return -1;
	sect = get_sect_has_addr(bi, addr);
	if (!sect)
		return -1;
	rc = bfd_find_nearest_line(bi->abfd, sect,
				   bi->n_syms ? bi->p_syms : bi->p_dynsyms,
				   (bfd_vma)addr - sect->vma,
				   src_name, func_name, (unsigned int*)lno);
	return (rc ? 0 : -1);
}

/* This function nearly equal bsearch(3).
 * Except, this function also returns small index if not match found.
 */
static void* __bsearch(const void *key, const void *base, size_t nmemb,
		       size_t size, int (*comp)(const void*, const void*),
		       size_t *small_index)
{
	size_t min, max, mid = 0;
	const void *p;
	int rc = 0;

	min = 0;
	max = nmemb;
	while (min < max) {
		mid = (min + max) / 2;
		p = (void *)(((const char *)base) + (mid * size));
		rc = (*comp)(key, p);
		if (rc < 0)
			max = mid;
		else if (rc > 0)
			min = mid + 1;
		else {
			if (small_index)
				*small_index = mid;
			return (void*)p;
		}
	}
	if (small_index)
		*small_index = rc > 0 ? mid : mid - 1;
	return NULL;
}

static int __comp_fsym_addr(const void *key, const void *elem)
{
	unsigned long addr = *(unsigned long*)key;
	asymbol *sym = *(asymbol**)elem;
	unsigned long sym_addr = (unsigned long)bfd_asymbol_value(sym);

	if (addr > sym_addr)
		return 1;
	else if (addr < sym_addr)
		return -1;
	return 0;
}

/* addr is relative */
int addr_to_func_name_and_offset(struct bfd_if *bi, unsigned long addr,
				 const char **func_name, size_t *foffset)
{
	asymbol *sym, **p_sym;
	size_t sym_index;

	if (addr == UNKNOWN_BADDR)
		goto NOT_FOUND;
	p_sym = __bsearch(&addr, bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*),
			  __comp_fsym_addr, &sym_index);
	if (!p_sym) {
		if (sym_index < 0)
			goto NOT_FOUND;
		sym = bi->p_fsyms[sym_index];
	} else
		sym = *p_sym;
	*func_name = sym->name;
	*foffset = addr - (unsigned long)bfd_asymbol_value(sym);
	return 0;

NOT_FOUND:
	*func_name = "";
	*foffset = 0;
	return -1;
}

static int dummy_sprintf(FILE *stream, const char *fmt, ...)
{
	va_list argp;
	int rc;

	va_start(argp, fmt);
	if (seq_num == 1)
		baddr_offset = offset;
	rc = vsnprintf(&print_buf[offset], PRINT_BUF_MAX - offset, fmt, argp);
	if (rc < 0)
		return -1;
	offset += rc;
	seq_num++;
	va_end(argp);
	return 0;
}

void printf_func_name(const char *func_name, size_t offset)
{
	if (offset)
		printf("<%s+0x%x>", func_name, offset);
	else
		printf("<%s>", func_name);
}

int printf_mnemonic(struct bfd_if *bi, unsigned long addr,
		    unsigned long *p_baddr)
{
	int rc;
	char *p_tmp;
	struct disassemble_info *info = &bi->info;
	fprintf_ftype func_save;

	if (prepare_print_insn(bi, addr) < 0)
		return -1;

	seq_num = 0;
	offset = baddr_offset = 0;
	func_save = info->fprintf_func;
	info->fprintf_func = (fprintf_ftype)dummy_sprintf;
	rc = print_insn_i386_att(addr, &bi->info);
	info->fprintf_func = func_save;
	
	if (p_baddr) {
		*p_baddr = UNKNOWN_BADDR;
		if (baddr_offset > 0) {
			*p_baddr = strtoll(print_buf + baddr_offset, &p_tmp, 0);
			if (*p_tmp != '\0')
				*p_baddr = UNKNOWN_BADDR;
		}
	}
	if (rc >= 0)
		printf("%s", print_buf);
	return rc;
}

static bfd_vma get_sect_offset(struct bfd_if *bi, asection *sect)
{
	bfd_vma sect_offset;
	asection *s;

	if (!(bfd_get_file_flags(bi->abfd) & HAS_RELOC))
		return 0;
	sect_offset = 0;
	for (s = bi->abfd->sections; s; s = s->next) {
		/* If the section address already set, then do not add the
		 * section offset (e.g. relocatable kernel).
		 */
		if (!is_code_sect(bi, s) || s->vma)
			continue;
		if (s == sect)
			return sect_offset;
		sect_offset += bfd_get_section_size(s);
	}
	return sect_offset;
}

static int __for_each_fsyms(struct bfd_if *bi, unsigned long offset,
			    int (*f)(unsigned long,unsigned long,unsigned long,
				     const char*,void*),
			    void *data)
{
	int rc;
	long i;
	asymbol *sym, **symbols = bi->p_fsyms;
	bfd_vma val, sect_offset;
	asection *sect;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = symbols[i];
		val = bfd_asymbol_value(sym);
		sect = sym->section;
		sect_offset = get_sect_offset(bi, sect);
		rc = f((unsigned long)val, (unsigned long)sect_offset, offset,
		       sym->name, data);
		if (rc != 0)
			return rc;
	}
	return 0;
}

int for_each_fsymbols(struct bfd_if *bi, unsigned long offset,
		      t_func_for_fsym f, void *data)
{
	int rc;
	long i;
	asymbol *sym, **symbols = bi->p_fsyms;
	bfd_vma val, sect_offset;
	asection *sect;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = symbols[i];
		val = bfd_asymbol_value(sym);
		sect = sym->section;
		sect_offset = get_sect_offset(bi, sect);
		rc = f((unsigned long)val + offset, sym->name, data);
		if (rc != 0)
			return rc;
	}
	return 0;
}

static int f_print_fsym(unsigned long addr, unsigned long sect_offset,
			unsigned long offset, const char *name, void *data)
{
	printf("0x%08lx %s\n", addr - sect_offset, name);
	return 0;
}

void dump_bfd_symbols(struct bfd_if *bi, unsigned long begin, unsigned long end,
		      unsigned long offset)
{
	printf("------ bfd symbols ------\n");
	__for_each_fsyms(bi, offset, f_print_fsym, NULL);
}

struct find_fsym_data {
	const char *name;
	unsigned long addr;
	int cnt;
};

static int f_find_fsym(unsigned long addr, const char *name, void *data)
{
	struct find_fsym_data *dt = (struct find_fsym_data*)data;

	if (strcmp(name, dt->name) == 0) {
		if (dt->addr == UNKNOWN_BADDR)
			dt->addr = addr;
		dt->cnt++;
	}
	return 0;
}

/* This function returns the number that a function name was found in.
 * In the 'addr' variable, an address of a function found first is set.
 */
int get_addr_of_symbol(struct bfd_if *bi, unsigned long offset,
		       const char *symstr, unsigned long *addr)
{
	struct find_fsym_data dt;

	dt.name = symstr;
	dt.addr = UNKNOWN_BADDR;	/* initialize */
	dt.cnt = 0;
	for_each_fsymbols(bi, offset, f_find_fsym, &dt);
	if (dt.cnt)
		*addr = dt.addr;
	return dt.cnt;
}

struct find_fsym_all_data {
	const char *name;
	unsigned long *addrs;
	int cnt;
};

static int f_find_fsym_all(unsigned long addr, const char *name, void *data)
{
	struct find_fsym_all_data *dt = (struct find_fsym_all_data*)data;
	int step = 16;

	if (strcmp(name, dt->name) == 0) {
		if (!(dt->cnt % step)) {
			if (dt->cnt) {
				dt->addrs = xrealloc(dt->addrs,
						     (dt->cnt + step) *
						    	sizeof(unsigned long));
			} else {
				dt->addrs = xmalloc(step *
						    sizeof(unsigned long));
			}
		}
		dt->addrs[dt->cnt++] = addr;
	}
	return 0;
}

int get_addr_of_symbol_all(struct bfd_if *bi, unsigned long offset,
			   const char *symstr, unsigned long **addrs)
{
	struct find_fsym_all_data dt;

	dt.name = symstr;
	dt.addrs = NULL;
	dt.cnt = 0;
	for_each_fsymbols(bi, offset, f_find_fsym_all, &dt);
	if (dt.cnt)
		*addrs = dt.addrs;
	return dt.cnt;
}

int get_begin_of_func(struct bfd_if *bi, unsigned long addr,
		      unsigned long *begin)
{
	asymbol *sym, **p_sym;
	size_t sym_index;

	p_sym = __bsearch(&addr, bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*),
			  __comp_fsym_addr, &sym_index);
	if (!p_sym) {
		if (sym_index < 0)
			return 0;
		sym = bi->p_fsyms[sym_index];
	} else
		sym = *p_sym;
	*begin = (unsigned long)bfd_asymbol_value(sym);
	return 1;
}

int get_end_of_func(struct bfd_if *bi, unsigned long begin, unsigned long *end)
{
	asymbol *sym, **p_sym;
	long i;

	p_sym = bsearch(&begin, bi->p_fsyms, bi->n_fsyms, sizeof(asymbol*),
			__comp_fsym_addr);
	if (!p_sym)
		return 0;
	sym = *p_sym;
	i = (p_sym - bi->p_fsyms);
	if (i < bi->n_fsyms - 1) {
		sym = bi->p_fsyms[i + 1];
		*end = (unsigned long)bfd_asymbol_value(sym);
	}
	return 1;
}

