/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_execpath.c - execution path display program                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <getopt.h>
#include "chk_repeat.h"
#include "bt_ar_parse.h"
#include "bt_utils.h"

#define BT_EXECPATH_VER	VERSION
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-" RELEASE_YEAR

struct proc_each_rec_data {
	struct addr_range	*range;
	int			nest_chk;
};

struct proc_each_range_data {
	int	fd;
	off_t	i_rec_max;
	off_t	i_start;
	off_t	i_stop;
	int	match_found;
	off_t	match_i_rec;
};

int output_summary = 0;
int verbose = 0;
struct r2n_info r2n_info;

/*----------------------------------------------------------------------------
 *  get function name and mnemonic
 *----------------------------------------------------------------------------
 */
int print_exec_line(unsigned long addr, struct range_to_name *r2n)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;
	struct bfd_if *bi = &r2n->bi;
	unsigned long baddr;

	addr -= r2n->offset;
	printf("%s\t0x%08lx\t", r2n->basename, addr);
	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name && lno) {
		addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
		printf_func_name(func_name, offset);
		if (src_name)
			printf("\t%s:%d", src_name, lno);
		printf("\n");
	} else {
		if (!addr_to_func_name_and_offset(bi, addr, &func_name,
						  &offset)) {
			printf_func_name(func_name, offset);
			printf("\t");
		}
		rc = printf_mnemonic(&r2n->bi, addr, &baddr);
		//printf("\nBADDR: 0x%08lx\n", baddr);
		if (rc > 0 &&
		    !addr_to_func_name_and_offset(bi, baddr, &func_name,
						  &offset)) {
			printf(" ");
			printf_func_name(func_name, offset);
		}
		printf("\n");
	}
	return 0;
}

/*----------------------------------------------------------------------------
 *  normal output support
 *----------------------------------------------------------------------------
 */
int __comp_N(const void *p1, const void *p2)
{
	return ((unsigned long)p1 == (unsigned long)p2);
}

void __print_nest_space_N(int nest)
{
	int i;

	for (i = 0; i < nest; i++)
		printf("  ");
}

void __print_data_N(int nest, const void *p)
{
	struct range_to_name *r2n;
	unsigned long addr = (unsigned long)p;

	__print_nest_space_N(nest);
	r2n = addr_to_r2n(&r2n_info, addr);
	if (!r2n) {
		if (addr != 0)
			printf("--------\t0x%08lx\n", addr);
	} else
		print_exec_line(addr, r2n);
}

void __print_start_repeat_N(int nest, unsigned long cnt)
{
	__print_nest_space_N(nest);
	printf("===> repeat %ld times\n", cnt);
}

struct chk_repeat_funcs chk_repeat_funcs_N =
{
	__comp_N,
	NULL,
	__print_data_N,
	__print_start_repeat_N,
	NULL,
};

int proc_each_addr(unsigned long addr)
{
	static unsigned long last_addr = 0;

	if (addr == last_addr)
		return 0;
	last_addr = addr;

	return chk_repeat_each((const void*)addr);
}

int proc_each_record_for_normal(struct bt_record *__p, off_t i_rec, void *tmp)
{
	struct proc_each_rec_data *data = (struct proc_each_rec_data*)tmp;
	struct bt_record p = *__p;
	int rc;

	if (is_warn_record(&p)) {
		struct warn_record *warn = (struct warn_record*)&p;
		printf("WARN: bts left only: %ld\n", warn->left);
		return 0;
	}
	if (is_pid_record(&p) || is_comm_record(&p))
		return 0;
	rc = chk_fix_from_cache(&r2n_info, &p.from, p.to, NULL, NULL, NULL);
	if (rc < 0)
		return rc;
	if (!is_addr_range_match(p.from, p.to, data->range))
		return 0;

	if (proc_each_addr(p.from) < 0)
		return -1;
	if (proc_each_addr(p.to) < 0)
		return -1;
	return 0;
}

/*----------------------------------------------------------------------------
 *  summary output support
 *----------------------------------------------------------------------------
 */
#define PRINTABLE_NEST_MAX	20
#define PRINTABLE_NEST_STEP	20

struct summary {
	int type;
	int nest;
	unsigned long from;
	unsigned long to;
};

int __comp_S(const void *p1, const void *p2)
{
	return (memcmp(p1, p2, sizeof(struct summary)) == 0);
}

void __free_S(const void *p)
{
	free((struct summary*)p);
}

void __print_nest_space_S(int nest)
{
	int i;

	for (i = 0; i < nest; i++)
		printf("  ");
}

void __print_each_addr_S(unsigned long addr)
{
	size_t offset;
	const char *func_name;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(&r2n_info, addr);
	if (!r2n) {
		//printf("--------:0x%08lx", addr);
		printf("0x%08lx", addr);
		return;
	}
	//printf("%s:", r2n->basename);
	addr -= r2n->offset;
	if (addr_to_func_name_and_offset(&r2n->bi, addr, &func_name,
					 &offset) < 0) {
		printf("0x%08lx", addr);
		return;
	}
	printf_func_name(func_name, offset);
	return;
}

void __print_data_S(int nest, const void *p)
{
	struct summary *s = (struct summary*)p;
	int i;
	char c;

	if (s->nest > PRINTABLE_NEST_MAX) {
		int n;
		n = s->nest % PRINTABLE_NEST_STEP;
		if (n == 0)
			n += PRINTABLE_NEST_STEP;
		printf("(N:%4d)", s->nest - n);
		for (i = 0; i < n; i++)
			printf("+-");
	} else {
		for (i = 0; i < s->nest; i++)
			printf("+-");
	}
	switch (s->type) {
	case BTYPE_JMP:  c = 'J'; break;
	case BTYPE_CALL: c = 'C'; break;
	case BTYPE_INT:
	case BTYPE_BREAK:
			 c = 'I'; break;
	default:         c = '?'; break;
	}
	printf("%c ", c);
	__print_each_addr_S(s->to);
	printf(" (");
	//__print_each_addr_S(s->from);
	printf("0x%08lx", s->from);
	printf(")\n");
}

void __print_start_repeat_S(int nest, unsigned long cnt)
{
	__print_nest_space_S(nest);
	printf("===> repeat %ld times\n", cnt);
}

void __print_end_repeat_S(int nest)
{
	__print_nest_space_S(nest);
	printf("<===\n");
}

struct chk_repeat_funcs chk_repeat_funcs_S =
{
	__comp_S,
	__free_S,
	__print_data_S,
	__print_start_repeat_S,
	__print_end_repeat_S,
};

int proc_each_record_for_summary(struct bt_record *__p, off_t i_rec, void *tmp)
{
	struct bt_record p = *__p;
	struct proc_each_rec_data *data = (struct proc_each_rec_data*)tmp;
	int nest_chk = data->nest_chk, rc, inst_type, type, nest;
	struct summary *s;

	if (!nest_chk && is_warn_record(&p)) {
		struct warn_record *warn = (struct warn_record*)&p;
		printf("WARN: bts left only: %ld\n", warn->left);
		return 0;
	}
	if (is_pid_record(&p) || is_comm_record(&p))
		return 0;
	rc = chk_enter_leave(&r2n_info, &p, nest_chk, &inst_type, &type, &nest,
			     NULL, NULL, NULL, NULL);
	if (rc < 0)
		return -1;
	if (!(rc == 1 || rc == 2))
		return 0;
	/* normal jump or enter */
	switch (type) {
	case BTYPE_JMP:
	case BTYPE_CALL:
	case BTYPE_INT:
	case BTYPE_BREAK:
		if (!nest_chk) {
			s = xcalloc(1, sizeof(struct summary));
			s->type = type;
			s->nest = nest;
			s->from = p.from;
			s->to = p.to;
			if (chk_repeat_each((const void*)s) < 0)
				return -1;
		}
	}
	return 0;
}

int chk_nest(int fd, off_t from, off_t to, struct proc_each_rec_data *data)
{
	chk_nest_initialize();
	if (for_each_block_record(fd, from, to,
				  proc_each_record_for_summary, data) < 0)
		return -1;
	chk_nest_initialize();
	return 0;
}

/* search address option */
int search;
char *search_sym;
unsigned long search_addr;
off_t before;
off_t after;
off_t display_from;
off_t display_to;
/* for debug (hidden options: -W) */
int search_warn;

/*----------------------------------------------------------------------------
 *  program core
 *----------------------------------------------------------------------------
 */
/* Don't check p->from cause Pentium-M's p->from is not correct. */
#define __is_search_record(p)	\
	(is_bt_record(p) && (p)->to == search_addr)

static int is_search_record(struct bt_record *p, off_t i_rec)
{
	if (search_warn)
		return is_warn_record(p);
	else if (display_from)
		return (i_rec == display_from);
	else
		return __is_search_record(p);
}

int __get_block_range(struct bt_record *p, off_t i_rec, void *tmp)
{
	struct proc_each_range_data *data = (struct proc_each_range_data*)tmp;

	if (!is_search_record(p, i_rec))
		return 0;
	data->match_found = 1;
	data->match_i_rec = i_rec + 1;
	//printf("FOUND  :%lld:%lld\n", i_rec, data->match_i_rec);
	return 1;
}

int get_block_range(struct bt_record *p, off_t i_rec, void *tmp)
{
	struct proc_each_range_data *data = (struct proc_each_range_data*)tmp;
	off_t i_tmp;

	data->match_found = 0;
	data->match_i_rec = -1;

	if (!is_search_record(p, i_rec))
		return 0;
	if (data->i_start < 0) {
		data->i_start = i_rec - before;
		if (data->i_start < 0)
			data->i_start = 0;
	}
	data->match_found = 1;
	data->match_i_rec = i_rec + 1;
	//printf("FOUND>>:%lld:%lld\n", i_rec, data->match_i_rec);

	if (display_to) {
		data->i_stop = display_to;
		if (data->i_stop > data->i_rec_max || data->i_stop < 0)
			data->i_stop = data->i_rec_max;
		return 1;
	}

	/* Check whether 'before' or 'after' range contains the search
	 * address.
	 */
	while (data->match_found) {
		i_tmp = data->match_i_rec + (after > before ? after : before);
		if (i_tmp > data->i_rec_max) {
			i_tmp = data->i_rec_max;
			break;
		}
		data->match_found = 0;
		for_each_block_record(data->fd, data->match_i_rec, i_tmp,
				      __get_block_range, data);
	}
	//printf("B:%lld, A:%lld, M:%lld\n", before, after, data->match_i_rec);
	data->i_stop = data->match_i_rec + after;
	if (data->i_stop > data->i_rec_max)
		data->i_stop = data->i_rec_max;
	return 1;
}

int proc_block_records(int fd, off_t from, off_t to)
{
	struct proc_each_rec_data data = { NULL, 0 };

	if (to < from) {
		fprintf(stderr, "from-recnum is bigger than to-recnum\n");
		return -1;
	}
	data.range = get_pid_addr_range(ALL_PID);
	if (output_summary) {
		data.nest_chk = 1;
		/*
		if (chk_nest(fd, from, to, &data) < 0)
			return -1;
			*/
		data.nest_chk = 0;
		chk_repeat_start(&chk_repeat_funcs_S);
		if (for_each_block_record(fd, from, to,
					  proc_each_record_for_summary,
					  &data) < 0)
			return -1;
		chk_repeat_end();
	} else {
		chk_repeat_start(&chk_repeat_funcs_N);
		if (for_each_block_record(fd, from, to,
					  proc_each_record_for_normal,
					  &data) < 0)
			return -1;
		chk_repeat_end();
	}
	return 0;
}

/* Return value:  1 = Find Next (Continue)
 *                0 = Not Found
 *               -1 = Error
 */
int proc_search_range(off_t i_rec, void *tmp)
{
	struct proc_each_range_data *data = (struct proc_each_range_data*)tmp;

	data->i_start = data->i_stop = -1;
	if (for_each_block_record(data->fd, i_rec, data->i_rec_max,
				  get_block_range, data) < 0)
		return -1;
	if (data->i_start < 0 || data->i_stop < 0)
		return 0;
	printf("======== records from %lld to %lld ========\n",
	       data->i_start, data->i_stop);
	if (proc_block_records(data->fd, data->i_start, data->i_stop) < 0)
		return -1;
	printf("\n");
	if (display_to)
		return 0;
	return 1;
}

int proc_logfile(char *logfile)
{
	int fd, rc = -1, tmp;
	off_t size;
	char *dir, *func_name;
	struct proc_each_range_data data;

	if ((fd = u_open(logfile, &size)) < 0)
		goto EXIT;

	func_name = basename(logfile);
	dir = dirname(logfile);
	if (is_exists_maps(dir, func_name)) {
		if (parse_maps(&r2n_info, dir, func_name,
			       PM_WITH_PT, PM_VERBOSE) <0)
			goto EXIT;
	}
	if (parse_modules(&r2n_info, dir, PM_WITH_PT, PM_VERBOSE) < 0)
		goto EXIT;
	r2n_info.from_is_next = chk_from_is_next(dir);

	if (search_sym) {
		if (!get_symbol_addr(&r2n_info, search_sym, &search_addr)) {
			fprintf(stderr, "symbol not found(%s).\n", search_sym);
			goto EXIT;
		}
	}
	printf("start\n");
	if (verbose)
		dump_r2n(&r2n_info);
	if (search || display_from || search_warn) {
		if (search) {
			if (search_sym)
				printf("search: %s(0x%08lx)",
				       search_sym, search_addr);
			else
				printf("search: 0x%08lx", search_addr);
			printf(" -B %lld -A %lld\n", before, after);
		}
		data.fd = fd;
		data.i_rec_max = size / sizeof(struct bt_record);
		data.i_start = -1;
		data.i_stop = 0;
		data.match_found = 0;
		data.match_i_rec = -1;
		do {
			tmp = proc_search_range(data.i_stop, &data);
			if (tmp < 0)
				goto EXIT;
		} while (tmp);
	} else {
		if (proc_block_records(fd, 0, size / sizeof(struct bt_record))
		    < 0)
			goto EXIT;
	}
	rc = 0;
EXIT:
	u_close(fd);
	return rc;
}

void err_exit(void)
{
	free_ranges();
	free_r2n(&r2n_info);
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_execpath %s\n", BT_EXECPATH_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
#if 0
	fprintf(stderr, "bt_execpath [-s] [-a top:end] [-d top:end] [--usr|--ker|--all]\n");
	fprintf(stderr, "            [-S aors|-F recnum] [-B n] [-A n|-T recnum] -f logfile\n");
#else
	fprintf(stderr, "bt_execpath [-s] [-a top:end] [-d top:end] [--usr|--ker|--all] -f logfile\n");
#endif
	fprintf(stderr, "  -s: output execution path summary\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  --usr: alias of '-a 0:0xbfffffff'\n");
	fprintf(stderr, "  --ker: alias of '-a 0xc0000000:0xffffffff'\n");
	fprintf(stderr, "  --all: alias of '-a 0:0xffffffff'\n");
#if 0
	fprintf(stderr, "  -S: search address or symbol\n");
	fprintf(stderr, "  -B: print n records before the record of -S or -F option\n");
	fprintf(stderr, "  -A: print n records after the record of -S or -F option\n");
	fprintf(stderr, "  -F: specify the start record number\n");
	fprintf(stderr, "  -T: specify the end record number\n");
#endif
	fprintf(stderr, "  -f: logfile\n");
}

int main(int argc, char *argv[])
{
	int opt_index;
	char c, *logfile = NULL, *p_end;
	unsigned long begin, end;
	long tmp;
	off_t tmp_index;
	struct option long_options[] = {
		{"usr", no_argument, NULL, 0},
		{"ker", no_argument, NULL, 0},
		{"all", no_argument, NULL, 0},
	};

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	while ((c = getopt_long(argc, argv, "sa:d:S:F:B:A:T:f:WvD:",
				long_options, &opt_index)) != -1) {
		switch (c) {
		case 0:
			switch (opt_index) {
			case 0:
				add_range(ALL_PID, 0, 0xbfffffff);
				break;
			case 1:
				add_range(ALL_PID, 0xc0000000, 0xffffffff);
				break;
			case 2:
				add_range(ALL_PID, 0, 0xffffffff);
				break;
			}
			break;
		case 's':
			output_summary = 1;
			break;
		case 'a':
		case 'd':
			if (!range2ulongs(optarg, &begin, &end))
				err_exit();
			if (c == 'a')
				add_range(ALL_PID, begin, end);
			else
				del_range(ALL_PID, begin, end);
			break;
		case 'S':
			tmp = strtoul(optarg, &p_end, 0);
			if (*p_end != '\0' && *p_end != ' ')
				search_sym = optarg;
			search = 1;
			search_addr = tmp;
			break;
		case 'B':
		case 'A':
		case 'F':
		case 'T':
			tmp_index = strtoull(optarg, &p_end, 0);
			if (tmp_index < 0
			    || (*p_end != '\0' && *p_end != ' ')) {
				fprintf(stderr, "invalid value(%s).\n", optarg);
				err_exit();
			}
			switch (c) {
			case 'B':
				before = tmp_index;
				break;
			case 'A':
				after = tmp_index;
				break;
			case 'F':
				display_from = tmp_index;
				break;
			case 'T':
				display_to = tmp_index;
				break;
			}
			break;
		case 'f':
			logfile = optarg;
			break;
		case 'W':
			search_warn = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'D':
			set_elf_path_prefix(optarg);
			break;
		default:
			usage();
			err_exit();
		}
	}
	//dump_ranges();	/* for debug */
	if (optind < argc ||
	    !logfile || (search && display_from) || (after && display_to) ||
	    (!search && !display_from && (after || display_to))) {
		usage();
		err_exit();
	}
	set_uname(&r2n_info, logfile);
	if (search_warn)
		after = 0;
	r2n_info.r2n_max = 16;
	r2n_info.all_r2n = xcalloc(r2n_info.r2n_max,
				   sizeof(struct range_to_name*));
	if (proc_logfile(logfile) < 0)
		err_exit();
	free_ranges();
	free_r2n(&r2n_info);
	exit(0);
}
