/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_coverage.c - coverage information display program                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2008                         */
/*             Authors: Yumiko Sugita (yumiko.sugita.yf@hitachi.com),        */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <getopt.h>
#include "bt_ar_parse.h"
#include "bt_utils.h"
#include "bt_hconv.h"

#define BT_COVERAGE_VER	VERSION
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-" RELEASE_YEAR

int output_summary = 0;
int verbose = 0;
int eliminate_out_of_context = 0;
int print_enter_leave = 0;
int chk_type;
char *outdir;
char *includes;
char *excludes;

struct r2i_pack r2p1, r2p2;

#define HTML_OUTPUT	(outdir != NULL)
#define LIMIT_BY_FUNCS	(r2p1.include_funcs != NULL)

#define INC_CNT(p_cnt)						\
	do {							\
		if (*(p_cnt) != 0x7fffffff) *(p_cnt) += 1;	\
	} while(0)

#define ADD_CNT(p_cnt, cnt)					\
	do {							\
		*(p_cnt) += (cnt);				\
		if (*(p_cnt) < (cnt))				\
			*(p_cnt) = 0x7fffffff;			\
	} while(0)

int inc_tracking_cnt_with_addr(struct path *p, unsigned long addr)
{
	struct unknown *uk;

	for (uk = (struct unknown *)p->jmp_cnt; uk; uk = uk->next) {
		if (uk->addr == addr) {
			INC_CNT(&uk->cnt);
			return 0;
		}
	}
	uk = xcalloc(1, sizeof(struct unknown));
	uk->addr = addr;
	uk->cnt = 1;
	uk->next = (struct unknown*)p->jmp_cnt;
	p->jmp_cnt = (long)uk;
	return 0;
}

/*-----------------------------------------------------------------------------
 *  filter by function name support
 *-----------------------------------------------------------------------------
 */
#define is_addr_range_in(r2n, fc) \
	((fc) == NULL || \
	 ((fc)->addr >= (r2n)->begin && (fc)->addr <= (r2n)->end))

static int f_cmp_faddr(void *target, void *each)
{
	unsigned long a1 = (unsigned long)target;
	unsigned long a2 = (unsigned long)each;

	if (a1 < a2)
		return -1;
	if (a1 > a2)
		return 1;
	return 0;
}

int create_filter_funcs(struct r2i_pack *r2p, char *funcs, int type)
{
	node **list;
	char *p;
	unsigned long *addrs;
	int i, n;

	if (!funcs)
		return 0;
	list = type == 0 ? &r2p->include_funcs : &r2p->exclude_funcs;
	p = strtok(strdup(funcs), ",");
	while (p) {
		n = get_symbol_addr_all(&r2p->r2i, p, &addrs);
		if (n < 0)
			return -1;
		for (i = 0; i < n; i++) {
			if (!search_tree((void*)addrs[i], *list, f_cmp_faddr))
				*list = insert_tree((void*)addrs[i], *list,
						    f_cmp_faddr);
		}
		if (n)
			free(addrs);
		p = strtok(NULL, ",");
	}
	if (!*list) {
		printf("WARN: include functions were not found.\n");
		return -1;
	}
	return 0;
}

int create_both_filter_funcs(struct r2i_pack *r2p)
{
	if (create_filter_funcs(r2p, includes, 0) < 0)
		return -1;
	if (create_filter_funcs(r2p, excludes, 1) < 0)
		return -1;
	return 0;
}

int is_include_faddr(struct r2i_pack *r2p, unsigned long addr)
{
	return (search_tree((void*)addr, r2p->include_funcs, f_cmp_faddr)
		&& !search_tree((void*)addr, r2p->exclude_funcs, f_cmp_faddr));
}

int is_exclude_faddr(struct r2i_pack *r2p, unsigned long addr)
{
	return (search_tree((void*)addr, r2p->exclude_funcs, f_cmp_faddr)
		!= NULL);
}

static struct range_to_name* get_kernel_r2n(struct r2n_info *r2i) {
	int i;
	struct range_to_name *r2n;

	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (!r2n || !is_kernel(r2n))
			continue;
		return r2n;
	}
	fprintf(stderr, "kernel not found.\n");
	return NULL;
}

/*-----------------------------------------------------------------------------
 *  process each record
 *-----------------------------------------------------------------------------
 */
struct enter_leave_dt {
	int nest;
	unsigned long addr;
};

struct proc_each_rec_data {
	struct r2i_pack *r2p;
	struct addr_range *range;
	unsigned long last;

	/* checking out of context execution */
	int context_status;	/* 0: out of context
				 * 1: into the context
				 * 2: interrupt from context
				 * 3: unknown call from context
				 * 4: excluded function call from context
				 */
	struct enter_leave_dt leave;	/* valid when context_status == 1 */
	struct enter_leave_dt enter;	/* valid when context_status == 2,3,4 */
};

void debug_print_symbol_name(struct range_to_name *r2n, unsigned long addr)
{
	const char *fname;
	size_t offset;

	if (!r2n)
		goto PRINT_HEX;
	addr_to_func_name_and_offset(&r2n->bi, addr - r2n->offset,
				     &fname, &offset);
	if (!fname)
		goto PRINT_HEX;
	printf_func_name(fname, offset);
	return;
PRINT_HEX:
	printf("%08lx", addr);
}

inline void debug_print_eltype(int rc, int type, int inst_type)
{
	switch (type) {
	case BTYPE_CALL:
		switch (inst_type) {
		case BTYPE_CALL:
			printf("(C)");
			break;
		case BTYPE_JMP:
			printf("(C')");
			break;
		case BTYPE_RET:
			printf("(R)");
			break;
		default:
			goto PRINT_RAW;
		}
		break;
	case BTYPE_INT:
		switch (inst_type) {
		case BTYPE_IRET:
			printf("(Q)");
			break;
		default:
			if (rc == 2)
				printf("(I)");
			else
				goto PRINT_RAW;
			break;
		}
		break;
	default:
		goto PRINT_RAW;
	}
	printf(" ");
	return;
PRINT_RAW:
	printf("(%d:%d) ", type, inst_type);
}

void debug_print_enter_leave(int rc, int context_status,
			     int type, int inst_type, int nest,
			     struct range_to_name *r2n_from,
			     struct range_to_name *r2n_to, struct bt_record *p)
{
	int i;

	if (!(rc == 2 || rc == 3 || verbose))
		return;
	if (context_status == 1)
		printf("ooooo ");
	else
		printf("__(%d) ", context_status);
	printf(r2n_from ? "F" : "_");
	printf(r2n_to ?   "T" : "_");
	if (rc == 2 || rc == 3) { /* enter or leave occured ? */
		for (i = 0; i < nest; i++)
			printf("+-");
		printf("%s ", (rc == 2 ? ">" : "<"));
		debug_print_eltype(rc, type, inst_type);
	} else { /* verbose */
		for (i = 0; i < nest; i++)
			printf("+-");
		printf(". (-,%d)", inst_type);
	}
	debug_print_symbol_name(r2n_from, p->from);
	printf(" ");
	debug_print_symbol_name(r2n_to, p->to);
	printf(" (%08lx => %08lx)\n", p->from, p->to);
}

void update_context_status(struct bt_record *rec, struct path *p_from,
			   struct proc_each_rec_data *dt,
			   int inst_type, int type, int is_enter, int nest)
{
	/* When return from unfound address (i.e. in case of iret),
	 * 'p_from' variable is NULL.
	 */
	switch (dt->context_status) {
	case 0: /* not into the context yet */
		if (p_from && inst_type == BTYPE_CALL &&
		    is_include_faddr(dt->r2p, rec->to)) {
			dt->leave.addr = p_from->next;
			dt->leave.nest = nest;
			dt->context_status = 1;
		}
		break;
	case 1: /* into the context */
		if (inst_type == BTYPE_RET && dt->leave.addr == rec->to)
			dt->context_status = 0;
		else if (inst_type != BTYPE_INT && type == BTYPE_INT) {
			/* out of context interrupt */
			dt->enter.addr = rec->from;
			dt->enter.nest = nest;
			dt->context_status = 2;
		} else if (p_from && inst_type == BTYPE_CALL &&
			   p_from->jmp == UNKNOWN_BADDR) {
			/* out of context unknown call */
			dt->enter.addr = p_from->next;
			dt->enter.nest = nest;
			dt->context_status = 3;
		} else if (p_from && inst_type == BTYPE_CALL &&
			   is_exclude_faddr(dt->r2p, rec->to)) {
			/* out of context unknown call */
			dt->enter.addr = p_from->next;
			dt->enter.nest = nest;
			dt->context_status = 4;
		}
		break;
	case 2: /* interrupt from context */
		/* Checking for 'fixup_exception'.
		 * For detail, refer to the src/common/bt_ar_parse.c.
		 */
		if ((inst_type == BTYPE_IRET || inst_type == BTYPE_OTHER)
		    && type == BTYPE_INT && !is_enter && nest == dt->enter.nest)
			dt->context_status = 1;
		////else if (!is_enter && dt->enter.addr == rec->to)
		////	dt->context_status = 1;
		break;
	case 3: /* unknown call from context */
	case 4: /* excluded function call from context */
		if (dt->enter.addr == rec->to)
			dt->context_status = 1;
		break;
	}
}

#define IN_CONTEXT(context_status)	((context_status) == 1)

int proc_each_record(struct bt_record *__rec, off_t i_rec, void *tmp)
{
	struct bt_record rec = *__rec;
	struct proc_each_rec_data *data = (struct proc_each_rec_data*)tmp;
	long i, idx_last, idx_from, chk_fallthrough;
	struct r2n_info *r2i = &data->r2p->r2i;
	struct range_to_name *r2n, *r2n_to;
	struct path *p, *p_last, *p_from, *p_to = NULL;
	int rc, inst_type, type, nest, prev_st;

	if (is_warn_record(&rec)) {
		struct warn_record *warn = (struct warn_record*)&rec;
		printf("WARN: bts left only: %ld\n", warn->left);
		data->context_status = 0;
		goto EXIT;
	}
	if (is_pid_record(&rec)) {
		if (print_enter_leave && verbose)
			printf("CX CHG\n");
		goto EXIT;
	}
	if (is_comm_record(&rec))
		goto EXIT;
	prev_st = data->context_status;
	if (LIMIT_BY_FUNCS && eliminate_out_of_context) {
		rc = chk_enter_leave(r2i, &rec, 0, &inst_type, &type, &nest,
				     &r2n, &r2n_to, &p_from, &idx_from);
		if (print_enter_leave)
			debug_print_enter_leave(rc, data->context_status,
						type, inst_type, nest,
						r2n, r2n_to, &rec);
		if (rc <= 0)
			goto EXIT;
		if (rc != 1) {	/* enter or leave occured ? */
			update_context_status(&rec, p_from, data, inst_type,
					      type, (rc == 2), nest);
			if (print_enter_leave && verbose &&
			    prev_st != data->context_status)
				printf("ST CHG(%d) %d->%d (%08lx, %08lx)\n",
				       nest, prev_st, data->context_status,
				       rec.from, rec.to);
		}
	} else {
		r2n_to = addr_to_r2n(r2i, rec.to);
		rc = chk_fix_from_cache(r2i, &rec.from, rec.to,
					&r2n, &p_from, &idx_from);
	}
	if (rc <= 0 || r2n == NULL || p_from == NULL)
		goto EXIT;
	/*
	if (!is_addr_range_match(rec.from, rec.to, data->range))
		goto EXIT;
		*/

	idx_last = r2n->cnt;
	p_last = find_path_from_addr(r2n->pt, &idx_last, data->last);

	/*
	printf("DBG:0x%08lx", data->last);
	printf("->0x%08lx", rec.from);
	printf("->0x%08lx", rec.to);
	printf(" ");
	printf_path(p_from);
	if (p_to)
		printf_path(p_to);
		*/
	if (p_last) {
		/* check fallthrough */
		chk_fallthrough = 1;
		for (i = idx_last; i < idx_from; i++) {
			p = r2n->pt[i];
			/* The address becomes an execution address when
			 * interrupt enters the branch ahead when returning
			 * from interrupt.
			 * Therefore, after it returns from the int instruction,
			 * it becomes fallthrough check from passing the int
			 * instruction.
			 */
			if (i == idx_last && p->type == BTYPE_INT)
				continue;
			if (p->type != BTYPE_BRANCH && p->type != BTYPE_OTHER) {
				chk_fallthrough = 0;
				/* The error occurs because the address between
				 * from turning off the bts facility to turning
				 * on seems to drop off.
				 * When the error occurs in bt_mod.ko, the
				 * fallthrough check is not done, and the
				 * following processing is done.
				 */
				if (strcmp(r2n->basename, MOD_NAME ".ko") != 0){
					printf("WARN: detect lack of log");
					printf(" (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
					       data->last, rec.from, rec.to);
					break;
					/*
					fprintf(stdout, "!!! NG path tree\n");
					dump_r2n();
					return -1;
					*/
				}
			}
		}
		if (chk_fallthrough && IN_CONTEXT(prev_st))
			for (i = idx_last; i < idx_from; i++) {
				p = r2n->pt[i];
				if (i != idx_last)
					INC_CNT(&p->cnt);
				if (p->type == BTYPE_BRANCH)
					INC_CNT(&p->next_cnt);
			}
	}
	if (p_last != p_from && IN_CONTEXT(prev_st))
		INC_CNT(&p_from->cnt);

	/* check branch */
	if (p_from->base == rec.from) {
		if (p_from->jmp == UNKNOWN_BADDR &&
		    (IN_CONTEXT(prev_st) || IN_CONTEXT(data->context_status))) {
			if (inc_tracking_cnt_with_addr(p_from, rec.to) < 0) {
				printf("!!! Error: (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
				       data->last, rec.from, rec.to);
				return -1;
			}
		} else if (p_from->jmp == rec.to &&
			   IN_CONTEXT(data->context_status))
			INC_CNT(&p_from->jmp_cnt);
	}
	/* check to */
	if (r2n_to && IN_CONTEXT(data->context_status)) {
		i = r2n_to->cnt;
		p_to = find_path_from_addr(r2n_to->pt, &i, rec.to);
		if (p_to)
			INC_CNT(&p_to->cnt);
	}
EXIT:
	data->last = rec.to;
	return 0;
}

void printf_funcname(struct bfd_if *bi, unsigned long addr)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;

	rc = addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
	if (rc >= 0) {
		printf_func_name(func_name, offset);
		rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
		if (rc >= 0 && src_name && lno)
			printf(":%s,%d", src_name, lno);
	} else
		printf("0x%08lx", addr);
}

void printf_srcname_and_lno(struct bfd_if *bi, unsigned long addr)
{
	int rc, lno;
	const char *src_name, *func_name;

	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name && lno)
		printf("%s,%d", src_name, lno);
	else
		printf("0x%08lx", addr);
		//printf_funcname(bi, addr);
}

#define for_each_path(r2n, i, p) \
	for ((i) = 0; (i) < (r2n)->cnt && ((p) = (r2n)->pt[(i)]); (i)++)

/*-----------------------------------------------------------------------------
 *  display function coverage
 *-----------------------------------------------------------------------------
 */
static int f_cmp_addr2fc(void *__target, void *__dt)
{
	unsigned long addr = (unsigned long)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (addr < fc->addr)
		return -1;
	if (addr > fc->addr)
		return 1;
	return 0;
}

static int f_cmp_fc(void *__target, void *__dt)
{
	struct func_chk *t_fc = (struct func_chk*)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (t_fc->addr < fc->addr)
		return -1;
	if (t_fc->addr > fc->addr)
		return 1;
	return 0;
}

#define CNT_AS_UK(r2p, addr, from_uk) \
	(LIMIT_BY_FUNCS && (from_uk) && !is_include_faddr(r2p, addr))

/* Meaning of return value is -1:error, 0:already checked, 1: checked. */
int __check_func(struct r2i_pack *r2p, node **fc_list, unsigned long addr,
		 unsigned long cnt, int from_uk)
{
	struct func_chk *fc;

	fc = search_tree((void*)addr, *fc_list, f_cmp_addr2fc);
	if (fc) {
		if (CNT_AS_UK(r2p, addr, from_uk))
			ADD_CNT(&fc->uk_cnt, cnt);
		else
			ADD_CNT(&fc->cnt, cnt);
		return 0;
	}
	fc = xmalloc(sizeof(struct func_chk));
	fc->addr = addr;
	fc->end = 0;
	fc->cnt = 0;
	fc->uk_cnt = 0;
	if (CNT_AS_UK(r2p, addr, from_uk))
		fc->uk_cnt = cnt;
	else
		fc->cnt = cnt;
	SET_FC_VALID(fc);
	fc->childs = NULL;
	fc->excluded_tree_cnt = 0;

	*fc_list = insert_tree(fc, *fc_list, f_cmp_fc);
	if (!(*fc_list))
		return -1;
	return 1;
}

/* Meaning of return value is -1:error, 0:normal-end. */
int __check_func_by_jaddr(struct r2i_pack *r2p, node **fc_list,
			  unsigned long addr, unsigned long cnt, int from_uk)
{
	struct func_chk *fc;

	if (!cnt)
		return 0;
	fc = search_tree((void*)addr, *fc_list, f_cmp_addr2fc);
	if (!fc)
		return 0;
	if (CNT_AS_UK(r2p, addr, from_uk))
		ADD_CNT(&fc->uk_cnt, cnt);
	else
		ADD_CNT(&fc->cnt, cnt);
	return 0;
}

int check_func(struct r2i_pack *r2p, node **fc_list, struct path *p)
{
	struct unknown *uk;
	int (*func)(struct r2i_pack*, node**, unsigned long, unsigned long,int);

	func = LIMIT_BY_FUNCS ? __check_func_by_jaddr : __check_func;
	if (p->jmp == UNKNOWN_BADDR) {
		for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next) {
			if (func(r2p, fc_list, uk->addr, uk->cnt, 1) < 0)
				return -1;
		}
	} else if (func(r2p, fc_list, p->jmp, p->jmp_cnt, 0) < 0)
		return -1;
	return 0;
}

struct get_valid_funcs_pack {
	node		*valid_funcs;
	func_compare	f_cmp;
};

static void f_free_unnecessary_fc(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	if (!IS_CHECKED_FC(fc))
		free(fc);
}

static int f_slim_down_flist(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct get_valid_funcs_pack *pack =
		(struct get_valid_funcs_pack*)user_data;

	if (!IS_CHECKED_FC(fc))
		return 0;
	pack->valid_funcs = insert_tree(fc, pack->valid_funcs, pack->f_cmp);
	return 0;
}

static void slim_down_flist(node **fc_list, func_compare f_compare)
{
	struct get_valid_funcs_pack pack;

	pack.valid_funcs = NULL;
	pack.f_cmp = f_compare;
	for_each_node(*fc_list, f_slim_down_flist, &pack);
	free_tree(*fc_list, f_free_unnecessary_fc);
	*fc_list = pack.valid_funcs;
}

static int f_cmp_fc_excluded_tcnt(void *__target, void *__dt)
{
	struct func_chk *t_fc = (struct func_chk*)__target;
	struct func_chk *fc = (struct func_chk*)__dt;

	if (t_fc->excluded_tree_cnt > fc->excluded_tree_cnt)
		return -1;
	if (t_fc->excluded_tree_cnt < fc->excluded_tree_cnt)
		return 1;
	if (t_fc->addr < fc->addr)
		return -1;
	if (t_fc->addr > fc->addr)
		return 1;
	return 0;
}

static inline void get_excluded_tcnt_func_list(node **fc_list)
{
	slim_down_flist(fc_list, f_cmp_fc_excluded_tcnt);
}

int mark_valid_path(struct r2i_pack *r2p, struct range_to_name *r2n,
		    unsigned long root, struct func_chk *fc,
		    unsigned long end, node *fc_list);

int __mark_valid_path(struct r2i_pack *r2p, struct range_to_name *r2n,
		      unsigned long root, struct func_chk *fc,
		      unsigned long faddr, node *fc_list, int from_uk)
{
	struct range_to_name *c_r2n;
	unsigned long c_end;
	struct func_chk *c_fc;

	/* If 'faddr' is the other includes function, 'faddr' would be validate
	 * in other includes function's validate process.
	 */
	if (root != faddr && is_include_faddr(r2p, faddr))
		return 0;

	if (!is_exclude_faddr(r2p, faddr) &&
	    get_func_info(&r2p->r2i, faddr, &c_end, &c_r2n)) {
		c_fc = search_tree((void*)faddr, fc_list, f_cmp_addr2fc);
		if (c_fc) {
			if (from_uk) {
				if (c_fc->uk_cnt)
					SET_FC_UT(c_fc);
				return 0;
			}
			fc->childs = insert_tree(c_fc, fc->childs, f_cmp_fc);
			if (!(fc->childs))
				return -1;
			if (IS_CHECKED_FC(c_fc))
				return 0;
			SET_FC_VALID(c_fc);
			if (mark_valid_path(r2p, c_r2n, root, c_fc, c_end,
					    fc_list) < 0)
				return -1;
		}
	}
	return 0;
}

int mark_valid_path(struct r2i_pack *r2p, struct range_to_name *r2n,
		    unsigned long root, struct func_chk *fc,
		    unsigned long end, node *fc_list)
{
	struct path *p;
	long i, top;
	struct unknown *uk;

	top = r2n->cnt;
	p = find_path_from_addr(r2n->pt, &top, fc->addr);
	if (!p) {
		fprintf(stderr,
			"function is not in path-tree.(%s:0x%08lx:0x%08lx)\n",
			r2n->basename, fc->addr, end);
		return -1;
	}
	fc->end = end;
	for (i = top; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->next > end)
			break;
		p->type &= ~IS_INVALID_PATH;
		switch (p->type) {
		case BTYPE_CALL:
		case BTYPE_JMP:
		case BTYPE_BRANCH:
			/* Function calls by the indirect addressing are not
			 * included in the function call tree.
			 */
			if (p->jmp == UNKNOWN_BADDR) {
				if (chk_type != CHK_SINGLE)
					break;
				for (uk = (struct unknown*)p->jmp_cnt; uk;
				     uk = uk->next) {
					if (__mark_valid_path(r2p, r2n, root,
							      fc, uk->addr,
							      fc_list, 1) < 0)
						return -1;
				}
			} else {
				if (__mark_valid_path(r2p, r2n, root, fc,
						      p->jmp, fc_list, 0) < 0)
					return -1;
			}
			break;
		}
	}
	return 0;
}

struct f_chk_func_data {
	struct r2i_pack *r2p;
	struct range_to_name *r2n;
	node **fc_list;
};

static int f_check_func(unsigned long addr, const char *name, void *data)
{
	struct f_chk_func_data *dt = (struct f_chk_func_data*)data;

	if (__check_func(dt->r2p, dt->fc_list, addr, 0, 0) >= 0)
		return 0;
	return -1;
}

int chk_all_func_syms(node **fc_list, struct r2i_pack *r2p,
		      struct range_to_name *r2n)
{
	struct f_chk_func_data dt;
	int rc;

	dt.r2p = r2p;
	dt.r2n = r2n;
	dt.fc_list = fc_list;
	rc = for_each_fsymbols(&r2n->bi, r2n->offset, f_check_func, &dt);
	if (rc < 0)
		return rc;
	return 0;
}

static void f_fc_free(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	free(fc);
}

static int f_mark_include_funcs(void *__dt, void *user_data)
{
	unsigned long addr = (unsigned long)__dt;
	struct r2i_pack *r2p = (struct r2i_pack*)user_data;
	struct range_to_name *r2n;
	unsigned long end;
	struct func_chk *fc;
	
	if (is_exclude_faddr(r2p, addr) ||
	    !get_func_info(&r2p->r2i, addr, &end, &r2n))
		return 0;
	fc = search_tree((void*)addr, r2p->fc_list, f_cmp_addr2fc);
	if (!fc)
		return 0;
	r2p->include_fcs = insert_tree(fc, r2p->include_fcs, f_cmp_fc);
	if (IS_CHECKED_FC(fc))
		return 0;
	SET_FC_VALID(fc);
	if (mark_valid_path(r2p, r2n, addr, fc, end, r2p->fc_list) < 0)
		return -1;
	return 0;
}

static int f_all_fc_mark_invalid(void *__dt, void *user_data)
{
	struct func_chk *cc = (struct func_chk*)__dt;

	SET_FC_NOT_CHECKED(cc);
	return 0;
}

static int f_get_fname(void *__dt, void *user_data)
{
	struct r2i_pack *r2p = (struct r2i_pack*)user_data;
	struct func_chk *fc = (struct func_chk*)__dt;

	fc->fname = get_fname(r2p->r2n, fc->addr);
	if (fc->fname == NULL)
		return -1;
	return 0;
}

static int f_cmp_fc_fname(void *__target, void *__dt)
{
	struct func_chk *t_fc = (struct func_chk*)__target;
	struct func_chk *fc = (struct func_chk*)__dt;
	int rc;

	rc = strcmp(t_fc->fname, fc->fname);
	if (rc != 0)
		return rc;
	SET_FC_DOUBLE_NAME(t_fc);
	SET_FC_DOUBLE_NAME(fc);
	if (t_fc->addr < fc->addr)
		return -1;
	if (t_fc->addr > fc->addr)
		return 1;
	return 0;
}

static inline void get_fname_sorted_func_list(struct r2i_pack *r2p)
{
	slim_down_flist(&r2p->fc_list, f_cmp_fc);
	for_each_node(r2p->fc_list, f_get_fname, r2p);
	slim_down_flist(&r2p->fc_list, f_cmp_fc_fname);
}

int chk_func_coverage(struct r2i_pack *r2p)
{
	unsigned long i, j;
	struct r2n_info *r2i = &r2p->r2i;
	struct range_to_name *r2n;
	struct path *p;
	node **fc_list = &r2p->fc_list;
	int type;
	long cnt;

	*fc_list = NULL;
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			if (chk_all_func_syms(fc_list, r2p, r2n) < 0)
				goto ERR_EXIT;
		}
	}
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			for (j = 0; j < r2n->cnt; j++) {
				p = r2n->pt[j];
				type = p->type & ~IS_INVALID_PATH;
				if (type == BTYPE_CALL) {
					if (check_func(r2p, fc_list, p) < 0)
						goto ERR_EXIT;
				} else if (type == BTYPE_JMP ||
					   type == BTYPE_BRANCH) {
					if (__check_func_by_jaddr(r2p, fc_list,
								  p->jmp,
								  p->jmp_cnt,
								  0) < 0)
						goto ERR_EXIT;
				}
			}
		}
	}
	if (LIMIT_BY_FUNCS) {
		/* When limiting the coverage output by function-list, mark all
		 * paths as invalid. Then, mark the valid function paths as
		 * valid.
		 */
		for_each_node(*fc_list, f_all_fc_mark_invalid, NULL);
		for (i = 0; i < r2i->r2n_max; i++) {
			r2n = r2i->all_r2n[i];
			if (r2n) {
				for (j = 0; j < r2n->cnt; j++) {
					p = r2n->pt[j];
					p->type |= IS_INVALID_PATH;
				}
			}
		}
		for_each_node(r2p->include_funcs, f_mark_include_funcs, r2p);
		slim_down_flist(fc_list, f_cmp_fc);
		for (i = 0; i < r2i->r2n_max; i++) {
			r2n = r2i->all_r2n[i];
			if (r2n) {
				for (cnt = 0, j = 0; j < r2n->cnt; j++) {
					p = r2n->pt[j];
					if ((p->type & IS_INVALID_PATH)) {
						free_path(p);
						continue;
					}
					r2n->pt[cnt++] = p;
				}
				r2n->cnt = cnt;
			}
		}
	}
	if (chk_type == CHK_DIFF_KERNEL) {
		r2p->r2n = get_kernel_r2n(r2i);
		get_fname_sorted_func_list(r2p);
	}
	return 0;

ERR_EXIT:
	free_tree(*fc_list, f_fc_free);
	return -1;
}

struct fc_cnt {
	long	called;
	long	all;
};

struct call_cnt_pack {
	struct fc_cnt		*fc_cnt;
	struct range_to_name	*r2n;
};

static int f_cnt_each_fc(void *__dt, void *user_data)
{
	struct call_cnt_pack *pack = (struct call_cnt_pack*)user_data;
	struct fc_cnt *cnt = pack->fc_cnt;
	struct range_to_name *r2n = pack->r2n;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (is_addr_range_in(r2n, cc)) {
		if (IS_VALID_FC(cc) || IS_DOUBLE_NAME_FC(cc)) {
			cnt->all++;
			if (cc->cnt)
				cnt->called++;
		}
	}
	return 0;
}

static void get_all_fc_cnt(struct r2i_pack *r2p, struct fc_cnt *cnt)
{
	struct call_cnt_pack pack;

	cnt->all = 0;
	cnt->called = 0;
	pack.fc_cnt = cnt;
	pack.r2n = r2p->r2n;
	for_each_node(r2p->fc_list, f_cnt_each_fc, &pack);
}

#define __print_one_func_cov(r2n, fc, cnt) \
do { \
	printf_funcname(&(r2n)->bi, (fc)->addr - (r2n)->offset); \
	if ((fc)->excluded_tree_cnt > 0) { \
		if (cnt) \
			printf("\t(%ld, F:%ld)\n", cnt, \
			       (fc)->excluded_tree_cnt); \
		else \
			printf("\t(0, F:%ld)\n", (fc)->excluded_tree_cnt); \
	} else { \
		if (cnt) \
			printf("\t(%ld)\n", cnt); \
		else \
			printf("\n"); \
	} \
} while(0)

static int f_dump_each_fc(void *__dt, void *user_data)
{
	struct cov_out_data *dt = (struct cov_out_data*)user_data;
	struct func_chk *cc = (struct func_chk*)__dt;
	struct range_to_name *r2n = dt->r2p[0]->r2n;

	if (is_addr_range_in(r2n, cc)) {
		if (!(IS_VALID_FC(cc) || IS_DOUBLE_NAME_FC(cc)))
			return 0;
		if (HTML_OUTPUT)
			out_func_html_each(dt, cc);
		else {
			printf(cc->cnt ? "(OK) " : "(NT) ");
			__print_one_func_cov(r2n, cc, cc->cnt);
		}
	}
	return 0;
}

struct print_ftree_data {
	struct r2i_pack		*r2p;
	node			*fc_list;
	node			**printed;
	int			nest;
	struct cov_out_data	*cov_out_dt;
};

struct excluded_chk_pack {
	struct func_chk	*exclude_from;
	node		*excludes;
	node		*checked;
	node		*fc_list;
};

static int f_cnt_tree_leaf(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct excluded_chk_pack *pack = (struct excluded_chk_pack*)user_data;

	if (search_tree(fc, pack->excludes, f_cmp_fc))
		return 0;
	pack->excludes = insert_tree(fc, pack->excludes, f_cmp_fc);
	return for_each_node(fc->childs, f_cnt_tree_leaf, pack);
}

static int f_chk_not_exclude_fc(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct excluded_chk_pack *pack = (struct excluded_chk_pack*)user_data;

	if (fc == pack->exclude_from)
		return 0;

	if (search_tree(fc, pack->checked, f_cmp_fc))
		return 0;
	pack->checked = insert_tree(fc, pack->checked, f_cmp_fc);

	if (search_tree(fc, pack->excludes, f_cmp_fc))
		pack->excludes = delete_tree(fc, pack->excludes, f_cmp_fc,
					     NULL);
	return for_each_node(fc->childs, f_chk_not_exclude_fc, pack);
}

static void get_excluded_tree_cnt(struct print_ftree_data *dt,
				  node *fc_list, struct func_chk *fc)
{
	struct excluded_chk_pack pack;

	if (fc->excluded_tree_cnt > 0)
		return;
	pack.exclude_from = fc;
	pack.excludes = NULL;
	pack.checked = NULL;
	pack.fc_list = fc_list;

	/* 1. Check all exclude function's f-tree. */
	f_cnt_tree_leaf(fc, &pack);

	/* 2. Check the include functions' f-tree but exclude function's,
	 *    and if these valid functions are found in exclude functions,
	 *    then delete it from the exclude function list.
	 */
	for_each_node(dt->r2p->include_fcs, f_chk_not_exclude_fc, &pack);

	/* 3. Count exclude functions, and subtract from total function count.
	 */
	fc->excluded_tree_cnt = get_node_cnt(pack.excludes);
	free_tree(pack.excludes, NULL);
	return;
}

static void __f_print_func_tree(struct func_chk *fc,
				struct print_ftree_data *p_ftree_dt,
				int type, int has_child)
{
	int i, nest = p_ftree_dt->nest, kern;
	struct cov_out_data *dt = p_ftree_dt->cov_out_dt;
	node *fc_list = p_ftree_dt->fc_list;
	struct range_to_name *r2n;

	kern = 0;
	r2n = addr_to_r2n(&p_ftree_dt->r2p->r2i, fc->addr);
	if (!r2n)
		return;
	dt->r2p[kern]->r2n = r2n;
	get_excluded_tree_cnt(p_ftree_dt, fc_list, fc);
	if (HTML_OUTPUT)
		out_func_tree_html_each_enter(dt, kern, fc, nest, type,
					      has_child);
	else {
		switch (type) {
		case BCOV_TYPE_OK:
			printf("(OK) ");
			break;
		case BCOV_TYPE_NT:
			printf("(NT) ");
			break;
		default:
			printf("(--) ");
			break;
		}
		for (i = 0; i < nest; i++)
			printf("+-");
		__print_one_func_cov(r2n, fc, fc->cnt);
	}
}

static int f_print_invalid_func(void *__dt, void *user_data)
{
	struct print_ftree_data *dt = (struct print_ftree_data*)user_data;
	struct func_chk *fc = (struct func_chk*)__dt;
	struct range_to_name *r2n;

	r2n = addr_to_r2n(&dt->r2p->r2i, fc->addr);
	if (!r2n)
		return 0;
	dt->cov_out_dt->r2p[0]->r2n = r2n;
	if (IS_UT_FC(fc)) {
		if (HTML_OUTPUT)
			out_func_tree_html_each_invalid(dt->cov_out_dt, 0, fc);
		else {
			printf("(UT) ");
			__print_one_func_cov(r2n, fc, fc->uk_cnt);
		}
	}
	return 0;
}

static int f_print_func_tree(void *__dt, void *user_data)
{
	struct print_ftree_data dt = *(struct print_ftree_data*)user_data;
	struct func_chk *fc = (struct func_chk*)__dt;
	int printed, type, has_child;

	printed = search_tree((void*)fc->addr, *dt.printed, f_cmp_addr2fc)
									!= NULL;
	if (printed) {
		if (dt.nest) {
			has_child = 0;
			__f_print_func_tree(fc, &dt, BCOV_TYPE_HT, has_child);
			goto EXIT;
		}
		return 0;
	}
	*dt.printed = insert_tree(fc, *dt.printed, f_cmp_fc);
	if (!(*dt.printed))
		return -1;
	type = fc->cnt ? BCOV_TYPE_OK : BCOV_TYPE_NT;
	has_child = fc->childs != NULL;
	__f_print_func_tree(fc, &dt, type, has_child);
	dt.nest++;
	if (for_each_node(fc->childs, f_print_func_tree, &dt) < 0)
		return -1;
	dt.nest--;
EXIT:
	if (HTML_OUTPUT)
		out_func_tree_html_each_exit(dt.cov_out_dt, dt.nest,has_child);
	return 0;
}

void print_func_tree(struct r2i_pack *r2p, struct cov_out_data *cov_out_dt)
{
	int i;
	struct r2n_info *r2i = &r2p->r2i;
	struct range_to_name *r2n;
	struct fc_cnt cnt = { 0, 0 };
	struct call_cnt_pack pack = { &cnt, NULL };
	node *fc_list = r2p->fc_list;
	node *printed = NULL;
	struct print_ftree_data dt;
	char *s_inc, *s_exc;

	s_inc = includes ? includes : "(--)";
	s_exc = excludes ? excludes : "(--)";
	if (HTML_OUTPUT)
		out_func_tree_html_start(cov_out_dt, s_inc, s_exc);
	for (i = 0; i < r2i->r2n_max; i++) {
		r2n = r2i->all_r2n[i];
		if (r2n) {
			pack.r2n = r2n;
			for_each_node(fc_list, f_cnt_each_fc, &pack);
		}
	}
	dt.r2p = r2p;
	dt.fc_list = fc_list;
	dt.printed = &printed;
	dt.nest = 0;
	dt.cov_out_dt = cov_out_dt;
	if (!HTML_OUTPUT) {
		printf("====== includes: %s ======\n", s_inc);
		printf("====== excludes: %s ======\n", s_exc);
		printf("====== function tree (%ld/%ld=%.2f%%) ======\n",
		       cnt.called, cnt.all, get_percent(cnt.called, cnt.all));
	}
	for_each_node(r2p->include_fcs, f_print_func_tree, &dt);
	for_each_node(fc_list, f_print_invalid_func, &dt);
	if (HTML_OUTPUT)
		out_func_tree_html_end(cov_out_dt);
}

struct func_diff_chk {
	const char	*fname;
	struct func_chk	*fc1;
	struct func_chk	*fc2;
};

static int f_cmp_fdc(const void *p1, const void *p2)
{
	struct func_diff_chk *fdc1 = (struct func_diff_chk*)p1;
	struct func_diff_chk *fdc2 = (struct func_diff_chk*)p2;

	return strcmp(fdc1->fname, fdc2->fname);
}

struct pack_fdc {
	int			type;
	int			side;	/* r2p1 or r2p2 */
	struct range_to_name	*r2n;
	long			i;
	long			n;
	struct func_diff_chk	*fdc;
};

static int f_cp_fc2fdc(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct pack_fdc *pack = (struct pack_fdc*)user_data;
	struct func_diff_chk *fdc, tmp;

	if (IS_DOUBLE_NAME_FC(fc))
		return 0;
	if (!is_addr_range_in(pack->r2n, fc))
		return 0;
	if (pack->side == 0) {	/* Are we processing 'r2p1'? */
		/* processing 'r2p1' */
		fdc = &pack->fdc[pack->i++];
		fdc->fname = fc->fname;
		fdc->fc1 = fc;
		return 0;
	}
	/* processing 'r2p2' */
	if (pack->n == 0) {
		pack->n = pack->i;
		if (pack->type == CHK_SAME_KERNEL)
			pack->i = 0;
	}
	if (pack->type == CHK_SAME_KERNEL) {
		fdc = &pack->fdc[pack->i++];
		fdc->fc2 = fc;
	} else {
		tmp.fname = fc->fname;
		fdc = bsearch(&tmp, pack->fdc, pack->n,
			      sizeof(struct func_diff_chk), f_cmp_fdc);
		if (fdc) {
			fdc->fc2 = fc;
		} else {
			fdc = &pack->fdc[pack->i++];
			fdc->fname = fc->fname;
			fdc->fc2 = fc;
		}
	}
	return 0;
}

static int f_cp_double_name_fc2fdc(void *__dt, void *user_data)
{
	struct func_chk *fc = (struct func_chk*)__dt;
	struct pack_fdc *pack = (struct pack_fdc*)user_data;
	struct func_diff_chk *fdc;

	if (!IS_DOUBLE_NAME_FC(fc))
		return 0;
	if (!is_addr_range_in(pack->r2n, fc))
		return 0;
	fdc = &pack->fdc[pack->i++];
	fdc->fname = fc->fname;
	if (pack->side == 0)
		fdc->fc1 = fc;
	else
		fdc->fc2 = fc;
	return 0;
}

void dump_func_coverage(struct cov_out_data *dt,
			struct fc_cnt *cnt, struct fc_cnt *cnt2)
{
	long i, n, same, diff;
	struct pack_fdc pack = { chk_type, 0, NULL, 0, 0, NULL };
	struct func_diff_chk *fdc;
	struct range_to_name *r2n = dt->r2p[0]->r2n, *r2n2 = NULL;

	if (chk_type != CHK_SINGLE) {
		r2n2 = dt->r2p[1]->r2n;
		n = chk_type == CHK_DIFF_KERNEL ?
					cnt->all + cnt2->all : cnt->all;
		pack.fdc = xcalloc(n, sizeof(struct func_diff_chk));
		for (i = 0; i < 2; i++) {
			pack.side = i;
			pack.r2n = dt->r2p[i]->r2n;
			for_each_node(dt->r2p[i]->fc_list, f_cp_fc2fdc, &pack);
		}
		if (chk_type == CHK_DIFF_KERNEL) {
			qsort(pack.fdc, pack.i, sizeof(struct func_diff_chk),
			      f_cmp_fdc);
			for (i = 0; i < 2; i++) {
				pack.side = i;
				pack.r2n = dt->r2p[i]->r2n;
				for_each_node(dt->r2p[i]->fc_list,
					      f_cp_double_name_fc2fdc, &pack);
			}
		} else {
			same = diff = 0;
			for (i = 0; i < pack.i; i++) {
				fdc = &pack.fdc[i];
				if (!is_addr_range_in(r2n, fdc->fc1) ||
				    !is_addr_range_in(r2n2, fdc->fc2))
					continue;
				out_func_html_each2(dt, fdc->fc1, fdc->fc2,
						    &same, &diff);
			}
		}
	}
	if (HTML_OUTPUT) {
		out_summary_html_func(dt, 0, cnt->called, cnt->all);
		if (chk_type != CHK_SINGLE)
			out_summary_html_func(dt, 1, cnt2->called, cnt2->all);
		out_func_html_start(dt, same, diff);
	} else
		printf("------ function coverage (%ld/%ld=%.2f%%) ------\n",
		       cnt->called, cnt->all,
		       get_percent(cnt->called, cnt->all));
	if (output_summary)
		goto EXIT;
	if (chk_type == CHK_SINGLE) {
		if (LIMIT_BY_FUNCS)
			get_excluded_tcnt_func_list(&r2p1.fc_list);
		for_each_node(dt->r2p[0]->fc_list, f_dump_each_fc, dt);
		goto EXIT;
	}
	for (i = 0; i < pack.i; i++) {
		fdc = &pack.fdc[i];
		if (!is_addr_range_in(r2n, fdc->fc1) ||
		    !is_addr_range_in(r2n2, fdc->fc2))
			continue;
		out_func_html_each2(dt, fdc->fc1, fdc->fc2, NULL, NULL);
	}
	free(pack.fdc);
EXIT:
	if (HTML_OUTPUT)
		out_func_html_end(dt);
}

/*-----------------------------------------------------------------------------
 *  display branch coverage
 *-----------------------------------------------------------------------------
 */
struct unknown_diff {
	unsigned long	addr;
	unsigned long	cnt1;
	unsigned long	cnt2;
};

static int f_cmp_ud(const void *p1, const void *p2)
{
	struct unknown_diff *ud1 = (struct unknown_diff*)p1;
	struct unknown_diff *ud2 = (struct unknown_diff*)p2;

	if (ud1->addr < ud2->addr)
		return -1;
	if (ud1->addr > ud2->addr)
		return 1;
	return 0;
}

static struct unknown_diff* get_unknown_diff(struct path *p1, long n1,
					     struct path *p2, long n2)
{
	struct unknown_diff *p_ud;
	struct unknown *uk;
	long i = 0, j;

	p_ud = xcalloc(n1 + n2, sizeof(struct unknown_diff));
	for (uk = (struct unknown*)p1->jmp_cnt; uk; uk = uk->next, i++) {
		p_ud[i].addr = uk->addr;
		p_ud[i].cnt1 = uk->cnt;
	}
	for (uk = (struct unknown*)p2->jmp_cnt; uk; uk = uk->next) {
		for (j = 0; j < n1; j++) {
			if (p_ud[j].addr == uk->addr) {
				p_ud[j].cnt2 = uk->cnt;
				continue;
			}
		}
		p_ud[i].addr = uk->addr;
		p_ud[i].cnt2 = uk->cnt;
		i++;
	}
	qsort(p_ud, i, sizeof(struct unknown_diff), f_cmp_ud);
	return p_ud;
}

long get_unknown_bcnt(struct path *p)
{
	long i = 0;
	struct unknown *uk;

	if (IS_SWITCH_JMP(p) ||
	    (p->type == BTYPE_BRANCH && p->jmp == UNKNOWN_BADDR)) {
		for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next, i++);
		return i;
	}
	return 0;
}

void __dump_one_branch_coverage(struct cov_out_data *dt, int kern,
				int bcov_type, struct branch_info *bi)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;

	if (HTML_OUTPUT)
		out_branch_html_each(dt, kern, bi);
	else {
		switch (bcov_type) {
		case BCOV_TYPE_OK: printf("(OK) "); break;
		case BCOV_TYPE_HT: printf("(HT) "); break;
		case BCOV_TYPE_NT: printf("(NT) "); break;
		case BCOV_TYPE_UT: printf("(UT) "); break;
		case BCOV_TYPE_UN: printf("(UN) "); break;
		}
		printf_srcname_and_lno(&r2n->bi, bi->base);
		printf(" [%ld/", bi->b_cnt);
		if (bi->fall == UNKNOWN_BADDR)
			printf("x] ");
		else
			printf("%ld] ", bi->f_cnt);
		if (bi->branch == UNKNOWN_BADDR)
			printf("----------:");
		else {
			printf_srcname_and_lno(&r2n->bi, bi->branch);
			printf(":");
		}
		if (bi->fall == UNKNOWN_BADDR)
			printf("xxxxxxxxxx\n");
		else {
			printf_srcname_and_lno(&r2n->bi, bi->fall);
			printf("\n");
		}
	}
}

inline void do_one_branch_coverage(struct cov_out_data *dt, int kern,
				   struct path *p, func_do_one_bcov func)
{
	struct range_to_name *r2n = dt->r2p[kern]->r2n;
	struct unknown *uk;
	int bcov_type;
	struct branch_info bi;

	bi.base = p->base - r2n->offset;
	bi.uk_id = 0;
	if (IS_SWITCH_JMP(p) ||		/* switch case */
	    p->jmp == UNKNOWN_BADDR) {	/* indirect addressing branch */
		bi.fall = UNKNOWN_BADDR;
		bi.f_cnt = 0;
		if (get_unknown_bcnt(p)) {
			for (uk = (struct unknown*)p->jmp_cnt; uk;
			     uk = uk->next) {
				bcov_type = BCOV_TYPE_UT;
				bi.branch = uk->addr - r2n->offset;
				bi.b_cnt = uk->cnt;
				func(dt, kern, bcov_type, &bi);
				bi.uk_id++;
			}
		} else {
			bcov_type = BCOV_TYPE_UN;
			bi.branch = UNKNOWN_BADDR;
			bi.b_cnt = 0;
			func(dt, kern, bcov_type, &bi);
		}
	} else {
		if (p->jmp_cnt != 0 && p->next_cnt != 0) {
			bcov_type = BCOV_TYPE_OK;
		} else if (p->jmp_cnt == 0 && p->next_cnt == 0) {
			bcov_type = BCOV_TYPE_NT;
		} else {
			bcov_type = BCOV_TYPE_HT;
		}
		bi.branch = p->jmp - r2n->offset;
		bi.b_cnt = p->jmp_cnt;
		bi.fall = p->next - r2n->offset;
		bi.f_cnt = p->next_cnt;
		func(dt, kern, bcov_type, &bi);
	}
}

static inline void dump_one_branch_coverage2(struct cov_out_data *dt,
					     struct path *p, struct path *p2,
					     long *same, long *diff)
{
	struct range_to_name *r2n, *r2n2;
	struct unknown_diff *p_ud, *ud;
	long i, n_uk, n_uk2;
	struct branch_info bi, bi2;

	r2n = dt->r2p[0]->r2n;
	r2n2 = dt->r2p[1]->r2n;

	bi.base = bi2.base = p->base - r2n->offset;
	if (IS_SWITCH_JMP(p) ||		/* switch case */
	    p->jmp == UNKNOWN_BADDR) {	/* indirect addressing branch */
		bi.fall = bi2.fall = UNKNOWN_BADDR;
		bi.f_cnt = bi2.fall = 0;
		n_uk = get_unknown_bcnt(p);
		n_uk2 = get_unknown_bcnt(p2);
		if (n_uk == n_uk2 && n_uk == 0) {
			bi.branch = bi2.branch = UNKNOWN_BADDR;
			bi.b_cnt = bi2.b_cnt = 0;
			out_branch_html_each2(dt, &bi, &bi2, same, diff);
		} else {
			p_ud = get_unknown_diff(p, n_uk, p2, n_uk2);
			for (i = 0; i < n_uk + n_uk2; i++) {
				ud = &p_ud[i];
				if (!ud->addr)
					break;
				bi.branch = bi2.branch = ud->addr - r2n->offset;
				bi.b_cnt = ud->cnt1;
				bi2.b_cnt = ud->cnt2;
				out_branch_html_each2(dt, &bi, &bi2, same,diff);
			}
			free(p_ud);
		}
	} else {
		bi.branch = bi2.branch = p->jmp - r2n->offset;
		bi.b_cnt = p->jmp_cnt;
		bi2.b_cnt = p2->jmp_cnt;
		bi.fall = bi2.fall = p->next - r2n->offset;
		bi.f_cnt = p->next_cnt;
		bi2.f_cnt = p2->next_cnt;
		out_branch_html_each2(dt, &bi, &bi2, same, diff);
	}
}

void dump_one_branch_coverage(struct cov_out_data *dt, int kern,
			      struct path *p)
{
	do_one_branch_coverage(dt, kern, p, __dump_one_branch_coverage);
}

struct branch_cnts {
	unsigned long ok;
	unsigned long nt;
	unsigned long ht;
	unsigned long uk_through;
	unsigned long uk_not_through;
};

void dump_branch_coverage(struct cov_out_data *dt)
{
	long i, uk_bcnt, same, diff;
	struct range_to_name *r2n, *r2n2;
	struct branch_cnts bcs = {0,0,0,0,0};
	struct path *p, *p2;
	unsigned long n_all_branch, n_all_uk;
	int k, kmax;

	kmax = chk_type == CHK_SINGLE ? 1 : 2;
	for (k = 0; k < kmax; k++) {
		r2n = dt->r2p[k]->r2n;
		bcs.ok = bcs.nt = bcs.ht = 0;
		bcs.uk_through = bcs.uk_not_through = 0;
		for_each_path(r2n, i, p) {
			if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
				continue;
			if (p->jmp == UNKNOWN_BADDR) {
				uk_bcnt = get_unknown_bcnt(p);
				if (!uk_bcnt)
					bcs.uk_not_through++;
				else
					bcs.uk_through++;
			} else if (p->jmp_cnt != 0 && p->next_cnt != 0)
				bcs.ok++;
			else if (p->jmp_cnt == 0 && p->next_cnt == 0)
				bcs.nt++;
			else
				bcs.ht++;
		}
		n_all_branch = (bcs.ok + bcs.ht + bcs.nt) * 2;
		n_all_uk = bcs.uk_through + bcs.uk_not_through;
		if (HTML_OUTPUT)
			out_summary_html_branch(dt, k, bcs.ok, bcs.uk_through,
						bcs.ht, bcs.nt,
						n_all_branch, n_all_uk);
		else {
			printf("------ branch coverage (OK:%ld,HT:%ld,NT:%ld/%ld=%.2f%%",
			       bcs.ok, bcs.ht, bcs.nt, n_all_branch,
			       get_percent(bcs.ok * 2 + bcs.ht, n_all_branch));
			printf(" UK:%ld/%ld=%.2f%%) ------\n",
			       bcs.uk_through, n_all_uk,
			       get_percent(bcs.uk_through, n_all_uk));
		}
		if (output_summary || chk_type == CHK_SAME_KERNEL)
			continue;
		if (HTML_OUTPUT)
			out_branch_html_start(dt, 0, 0);
		for_each_path(r2n, i, p) {
			if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
				continue;
			dump_one_branch_coverage(dt, k, p);
		}
		if (HTML_OUTPUT)
			out_branch_html_end(dt);
	}
	if (chk_type != CHK_SAME_KERNEL)
		return;
	r2n = dt->r2p[0]->r2n;
	r2n2 = dt->r2p[1]->r2n;
	same = diff = 0;
	for_each_path(r2n, i, p) {
		if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
			continue;
		p2 = r2n2->pt[i];
		dump_one_branch_coverage2(dt, p, p2, &same, &diff);
	}
	out_branch_html_start(dt, same, diff);
	for_each_path(r2n, i, p) {
		if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
			continue;
		p2 = r2n2->pt[i];
		dump_one_branch_coverage2(dt, p, p2, NULL, NULL);
	}
	out_branch_html_end(dt);
}

struct state_cnts {
	unsigned long ok;
	unsigned long nt;
};

/*-----------------------------------------------------------------------------
 *  display state coverage
 *-----------------------------------------------------------------------------
 */
void dump_state_coverage(struct cov_out_data *dt)
{
	long i, diff, same;
	struct range_to_name *r2n, *r2n2;
	struct state_cnts scs;
	struct path *p, *p2;
	int is_exec, is_exec2, k, kmax;

	kmax = chk_type == CHK_SINGLE ? 1 : 2;
	for (k = 0; k < kmax; k++) {
		r2n = dt->r2p[k]->r2n;
		scs.ok = scs.nt = 0;
		for_each_path(r2n, i, p) {
			if (p->cnt > 0)
				scs.ok += 1;
			else
				scs.nt += 1;
		}
		if (HTML_OUTPUT)
			out_summary_html_state(dt, k, scs.ok, scs.ok + scs.nt);
		else
			printf("------ state coverage (%ld/%ld=%.2f%%) ------\n",
			       scs.ok, scs.ok + scs.nt,
			       get_percent(scs.ok, scs.ok + scs.nt));

		if (output_summary || chk_type == CHK_SAME_KERNEL)
			continue;
		if (HTML_OUTPUT)
			out_state_html_start(dt, 0, 0);
		for_each_path(r2n, i, p) {
			is_exec = p->cnt > 0;
			if (HTML_OUTPUT)
				out_state_html_each(dt, k, is_exec, p,
						    r2n->offset);
			else {
				if (is_exec)
					printf("(OK) ");
				else
					printf("(NT) ");
				printf_srcname_and_lno(&r2n->bi,
						       p->addr - r2n->offset);
				printf("\n");
			}
		}
		if (HTML_OUTPUT)
			out_state_html_end(dt);
	}
	if (chk_type != CHK_SAME_KERNEL)
		return;
	r2n = dt->r2p[0]->r2n;
	r2n2 = dt->r2p[1]->r2n;
	same = diff = 0;
	for_each_path(r2n, i, p) {
		p2 = r2n2->pt[i];
		is_exec = p->cnt > 0;
		is_exec2 = p2->cnt > 0;
		if (is_exec == is_exec2)
			same++;
		else
			diff++;
	}
	out_state_html_start(dt, same, diff);
	for_each_path(r2n, i, p) {
		p2 = r2n2->pt[i];
		is_exec = p->cnt > 0;
		is_exec2 = p2->cnt > 0;
		out_state_html_each2(dt, is_exec, is_exec2, p, r2n->offset);
	}
	out_state_html_end(dt);
}

/*-----------------------------------------------------------------------------
 *  display coverage
 *-----------------------------------------------------------------------------
 */
static int f_chk_ifuncs_are_executed(void *__dt, void *user_data)
{
	unsigned long addr = (unsigned long)__dt;
	struct r2i_pack *r2p = (struct r2i_pack*)user_data;
	struct range_to_name *r2n;
	unsigned long end;
	struct func_chk *fc;

	if (is_exclude_faddr(r2p, addr) ||
	    !get_func_info(&r2p->r2i, addr, &end, &r2n))
		return 0;
	fc = search_tree((void*)addr, r2p->include_fcs, f_cmp_addr2fc);
	if (!fc || !fc->cnt) {
		printf("WARN: ");
		printf_funcname(&r2n->bi, addr - r2n->offset);
		printf(" was not executed.\n");
	}
	return 0;
}

int dump_coverage(void)
{
	int i;
	struct r2n_info *r2i = &r2p1.r2i;
	struct r2n_info *r2i2 = &r2p2.r2i;
	struct range_to_name *r2n, *r2n2;
	struct cov_out_data dt;
	struct fc_cnt cnt, cnt2;

	if (!r2i->all_r2n)
		return 0;

	memset(&dt, 0, sizeof(struct cov_out_data));
	dt.outdir = outdir;
	dt.limit_by_funcs = LIMIT_BY_FUNCS;
	dt.chk_type = chk_type;
	dt.r2p[0] = &r2p1;
	if (chk_type != CHK_SINGLE)
		dt.r2p[1] = &r2p2;
	if (HTML_OUTPUT) {
		if (init_html_output(&dt) < 0)
			return -1;
	}
	if (chk_func_coverage(&r2p1) < 0)
		return -1;
	if (chk_type != CHK_SINGLE) {
		if (chk_func_coverage(&r2p2) < 0)
			return -1;
	}
	if (chk_type == CHK_SINGLE) {
		if (LIMIT_BY_FUNCS) {
			/* Check if the inculde functions are executed. */
			for_each_node(r2p1.include_funcs,
				      f_chk_ifuncs_are_executed, &r2p1);
			if (!output_summary)
				print_func_tree(&r2p1, &dt);
		}
		for (i = 0; i < r2i->r2n_max; i++) {
			r2n = r2i->all_r2n[i];
			if (!r2n)
				continue;
			r2p1.r2n = r2n;
			get_all_fc_cnt(&r2p1, &cnt);
			if (!cnt.all)
				continue;
			if (HTML_OUTPUT) {
				dt.name = r2n->basename;
				out_summary_html_name(&dt);
			} else
				printf("====== %s coverage ======\n",
				       r2n->basename);
			dump_func_coverage(&dt, &cnt, NULL);
			dump_branch_coverage(&dt);
			dump_state_coverage(&dt);
		}
	} else {
		if (!(r2n = get_kernel_r2n(r2i)) ||
		    !(r2n2 = get_kernel_r2n(r2i2)))
			return -1;
		if (chk_type == CHK_SAME_KERNEL && r2n->cnt != r2n2->cnt) {
			fprintf(stderr,
				"It's seemed like a BUG... (%ld <-> %ld)\n",
				r2n->cnt, r2n2->cnt);
			return -1;
		}
		r2p1.r2n = r2n;
		get_all_fc_cnt(&r2p1, &cnt);
		r2p2.r2n = r2n2;
		get_all_fc_cnt(&r2p2, &cnt2);
		if (cnt.all == cnt2.all && cnt.all == 0) {
			fprintf(stderr,
				"Both kernels do not contains any functions.\n");
			return -1;
		}
		if (chk_type == CHK_SAME_KERNEL)
			r2p2.srcdir = r2p1.srcdir;
		dt.name = r2n->basename;
		out_summary_html_name(&dt);
		dump_func_coverage(&dt, &cnt, &cnt2);
		dump_branch_coverage(&dt);
		dump_state_coverage(&dt);
	}
	if (HTML_OUTPUT) {
		if (exit_html_output(&dt, LIMIT_BY_FUNCS) < 0)
			return -1;
	}
	return 0;
}

int check_elf_files(struct r2i_pack *r2p, char *logfile)
{
	int mode;
	char *tmp = strdup(logfile);
	char *dir, *fname;
	struct r2n_info *r2i = &r2p->r2i;

	mode = output_summary || HTML_OUTPUT ? PM_SILENT : PM_VERBOSE;
	fname = basename(tmp);
	dir = dirname(tmp);
	if (is_exists_maps(dir, fname)) {
		if (parse_maps(r2i, dir, fname, PM_WITH_PT, mode) < 0)
			return -1;
	}
	if (parse_modules(r2i, dir, PM_WITH_PT, mode) <0)
		return -1;
	r2i->from_is_next = chk_from_is_next(dir);

	if (mode == PM_VERBOSE)
		printf("start\n");

	if (create_both_filter_funcs(r2p) < 0)
		return -1;
	return 0;
}

int proc_pid_block_log(void *__dt, void *data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;
	struct log_file_info *log_info = get_log_file_info(__p);

	if (print_enter_leave && verbose)
		printf("======== %s\t%lld:%lld ========\n",
		       log_info->fpath, __p->i_rec, __p->n_rec);
	if (for_each_block_record(log_info->fd,
				  __p->i_rec, __p->i_rec + __p->n_rec,
				  proc_each_record, data) < 0)
		return -1;
	return 0;
}

int proc_pid_logfiles(struct pid_log_info *p, void *data)
{
	struct proc_each_rec_data d;

	if (print_enter_leave)
		printf("PID:%d >>>>>>>>\n", p->pid);
	d.r2p = (struct r2i_pack*)data;
	d.range = get_pid_addr_range(ALL_PID);
	d.last = UNKNOWN_BADDR;
	d.context_status = LIMIT_BY_FUNCS && eliminate_out_of_context ? 0 : 1;
	if (for_each_node(p->info, proc_pid_block_log, &d) < 0)
		return -1;
	return 0;
}

int proc_logfiles_by_one_dir(char *files[], void *data)
{
	struct r2i_pack *r2p = (struct r2i_pack*)data;

	if (initialize_log_info(files) < 0)
		return -1;
	if (chk_pid_pos(files, 0, 0) < 0)
		return -1;
	free_r2n_but_kernel(&r2p->r2i);
	if (check_elf_files((struct r2i_pack*)data, files[0]) < 0)
		return -1;
	chk_nest_initialize();
	if (for_each_pid_log_info(proc_pid_logfiles, data) < 0)
		return -1;
	finalize_log_info();
	return 0;
}

int proc_logfiles_by_each_dir(struct r2i_pack *r2p, char *log, int n_logfs)
{
	int i, j, flags[n_logfs];
	char *files[MAX_CPU_FILES + 1];
	char *f, *target_dir = NULL, *tmp, *dir;

	for (i = 0; i < n_logfs; i++)
		flags[i] = 0;
	j = 0;
	for (;;) {
		for (f = log, i = 0; i < n_logfs; i++, f += strlen(f) + 1) {
			if (flags[i])
				continue;
			tmp = strdup(f);
			dir = dirname(tmp);
			if (target_dir) {
				if (strcmp(target_dir, dir) != 0)
					goto FREE_AND_NEXT;
			} else
				target_dir = strdup(dir);
			files[j++] = f;
			flags[i] = 1;
FREE_AND_NEXT:
			free(tmp);
		}
		if (j == 0)
			break;
		files[j] = NULL;
		j = 0;
		free(target_dir);
		target_dir = NULL;

		/* do process with 'files' */
		if (proc_logfiles_by_one_dir(files, r2p) < 0)
			return -1;
	}
	return 0;
}

int replace_comma_with_null(char *top, int *n)
{
	char *prev, *cur;

	*n = 0;
	prev = NULL;
	cur = strtok(top, ",");
	while (cur) {
		(*n)++;
		if ((!prev && cur != top) ||
		    (prev && cur != prev + strlen(prev) + 1))
			return -1;
		prev = cur;
		cur = strtok(NULL, ",");
	}
	return 0;
}

void err_exit(void)
{
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_coverage %s\n", BT_COVERAGE_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_coverage [-se] [-a top:end] [-d top:end] [--usr|--ker|--all]\n");
	fprintf(stderr, "            [-I func[,...]] [-E func[,...]] [[-S src_dir] -o html_out_dir]\n");
	fprintf(stderr, "            [-u kver] -f logfile[,...]\n");
	fprintf(stderr, "            [[[--u2 kver] --S2 src_dir] --f2 logfile[,...]]\n");
	fprintf(stderr, "  -s: output coverage summary\n");
	fprintf(stderr, "  -e: exclude out-of-context path (experimental)\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  --usr: alias of '-a 0:0xbfffffff'\n");
	fprintf(stderr, "  --ker: alias of '-a 0xc0000000:0xffffffff'\n");
	fprintf(stderr, "  --all: alias of '-a 0:0xffffffff'\n");
	fprintf(stderr, "  -I: include function name(s)\n");
	fprintf(stderr, "  -E: exclude function name(s)\n");
	fprintf(stderr, "      This option excludes only included function by -I option.\n");
	fprintf(stderr, "  -S: source directory\n");
	fprintf(stderr, "  -o: html output directory\n");
	fprintf(stderr, "      -o and -S options cannot be used with -s option.\n");
	fprintf(stderr, "  -u: kernel version ('uname -r')\n");
	fprintf(stderr, "  -f: logfile(s)\n");
	fprintf(stderr, "  --u2,--S2,--f2: display differences between two trace-log(s).\n");
	fprintf(stderr, "         These options should be used with -o option.\n");
}

int main(int argc, char *argv[])
{
	int opt_index, n_logfs = 0, n_logfs2 = 0;
	char c, *logfs_top = NULL, *logfs_top2 = NULL;
	unsigned long begin, end;
	struct option long_options[] = {
		{"usr", no_argument, NULL, 0},
		{"ker", no_argument, NULL, 0},
		{"all", no_argument, NULL, 0},
		{"u2", required_argument, NULL, 0},
		{"S2", required_argument, NULL, 0},
		{"f2", required_argument, NULL, 0},
	};

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	while ((c = getopt_long(argc, argv, "sa:d:f:I:E:S:o:u:evPD:",
				long_options, &opt_index)) != -1) {
		switch (c) {
		case 0:
			switch (opt_index) {
			case 0:
				add_range(ALL_PID, 0, 0xbfffffff);
				break;
			case 1:
				add_range(ALL_PID, 0xc0000000, 0xffffffff);
				break;
			case 2:
				add_range(ALL_PID, 0, 0xffffffff);
				break;
			case 3:
				r2p2.r2i.uname_r = optarg;
				break;
			case 4:
				r2p2.srcdir = optarg;
				break;
			case 5:
				logfs_top2 = optarg;
				if (replace_comma_with_null(logfs_top2,
							    &n_logfs2) < 0)
					goto ERR_EXIT;
				break;
			}
			break;
		case 's':
			output_summary = 1;
			break;
		case 'e':
			eliminate_out_of_context = 1;
			break;
		case 'a':
		case 'd':
			if (!range2ulongs(optarg, &begin, &end))
				err_exit();
			if (c == 'a')
				add_range(ALL_PID, begin, end);
			else
				del_range(ALL_PID, begin, end);
			break;
		case 'f':
			logfs_top = optarg;
			if (replace_comma_with_null(logfs_top, &n_logfs) < 0)
				goto ERR_EXIT;
			break;
		case 'I':
			includes = optarg;
			break;
		case 'E':
			excludes = optarg;
			break;
		case 'S':
		case 'o':
			if (c == 'S')
				r2p1.srcdir = optarg;
			else
				outdir = optarg;
			break;
		case 'u':
			r2p1.r2i.uname_r = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'P':
			print_enter_leave = 1;
			break;
		case 'D':
			set_elf_path_prefix(optarg);
			break;
		default:
			goto ERR_EXIT;
		}
	}
	if (optind < argc || !n_logfs || (output_summary && outdir) ||
	    (!includes && excludes))
		goto ERR_EXIT;
	set_uname(&r2p1.r2i, logfs_top);
	set_uname(&r2p2.r2i, logfs_top2);
	chk_type = CHK_SINGLE;
	if (logfs_top2) {
		if (!outdir)
			goto ERR_EXIT;
		if (strcmp(r2p1.r2i.uname_r, r2p2.r2i.uname_r) != 0)
			chk_type = CHK_DIFF_KERNEL;
		else if (!r2p2.srcdir || strcmp(r2p1.srcdir, r2p2.srcdir) == 0)
			chk_type = CHK_SAME_KERNEL;
		else {
ERR_EXIT:
			usage();
			err_exit();
		}
	}
	if (outdir) {
		if (dir_chk_and_create(outdir, 1) < 0)
			err_exit();
	}
	r2p1.r2i.r2n_max = 16;
	r2p1.r2i.all_r2n = xcalloc(r2p1.r2i.r2n_max,
				   sizeof(struct range_to_name*));
	if (proc_logfiles_by_each_dir(&r2p1, logfs_top, n_logfs) < 0)
		err_exit();
	if (logfs_top2) {
		r2p2.r2i.r2n_max = 16;
		r2p2.r2i.all_r2n = xcalloc(r2p2.r2i.r2n_max,
					   sizeof(struct range_to_name*));
		if (proc_logfiles_by_each_dir(&r2p2, logfs_top2, n_logfs2) < 0)
			err_exit();
	}
	if (print_enter_leave)
		exit(0);
	if (verbose) {	/* hidden option for debug */
		dump_r2n(&r2p1.r2i);
		if (logfs_top2) {
			printf("======== other kernel ========\n");
			dump_r2n(&r2p2.r2i);
		}
	}
	if (dump_coverage() < 0)
		err_exit();
	exit(0);
}

