/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "breeze.h"
#include "user_commands.h"
#include "config.h"

#define DEFAULT_WIDTH 400
#define DEFAULT_HEIGHT 360

static GtkWidget *dialog;
static GtkWidget *treeview;
static gboolean is_edited = FALSE;

enum {
  COLUMN_KEYWORD, COLUMN_COMMAND, COLUMN_EDITABLE, NUM_COLUMNS
};

static void list_store_append_func(gpointer keyword, gpointer command,
				   gpointer user_data)
{
  GtkListStore *store = user_data;
  GtkTreeIter iter;

  gtk_list_store_append(store, &iter);
  gtk_list_store_set(store, &iter, COLUMN_KEYWORD, keyword,
		     COLUMN_COMMAND, command, COLUMN_EDITABLE, TRUE, -1);
}

static GtkTreeModel *create_model(void)
{
  GtkListStore *store = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING,
					   G_TYPE_BOOLEAN);

  user_commands_table_foreach(list_store_append_func, store);

  return GTK_TREE_MODEL(store);
}

static void keyword_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data)
{
  GtkTreeModel *model = user_data;
  GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
  GtkTreeIter iter;

  gtk_tree_model_get_iter(model, &iter, path);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_KEYWORD,
		     new_text, -1);
  is_edited = TRUE;
}

static void command_renderer_edited_callback(GtkWidget *w,
					     gchar *path_string,
					     gchar *new_text,
					     gpointer user_data)
{
  GtkTreeModel *model = user_data;
  GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
  GtkTreeIter iter;

  gtk_tree_model_get_iter(model, &iter, path);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_COMMAND,
		     new_text, -1);
  is_edited = TRUE;
}

static void add_columns(void)
{
  GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
  GtkCellRenderer *keyword_renderer = gtk_cell_renderer_text_new();
  GtkCellRenderer *command_renderer = gtk_cell_renderer_text_new();

  gtk_tree_view_insert_column_with_attributes
    (GTK_TREE_VIEW(treeview), -1, "Keyword", keyword_renderer,
     "text", COLUMN_KEYWORD, "editable", COLUMN_EDITABLE, NULL);
  gtk_tree_view_insert_column_with_attributes
    (GTK_TREE_VIEW(treeview), -1, "Command", command_renderer,
     "text", COLUMN_COMMAND, "editable", COLUMN_EDITABLE, NULL);

  g_signal_connect(keyword_renderer, "edited",
		   G_CALLBACK(keyword_renderer_edited_callback), model);
  g_signal_connect(command_renderer, "edited",
		   G_CALLBACK(command_renderer_edited_callback), model);
}

static void delete_event_callback(GtkWidget *w, gpointer user_data)
{
  gtk_widget_destroy(dialog);
  gtk_main_quit();
}

static void addButton_clicked_callback(GtkWidget *w, gpointer user_data)
{
  GtkTreeIter iter;
  GtkTreeModel *model =
    gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
  GtkTreePath *path;

  gtk_list_store_append(GTK_LIST_STORE(model), &iter);
  gtk_list_store_set(GTK_LIST_STORE(model), &iter, COLUMN_KEYWORD, "New item",
		     COLUMN_COMMAND, "Edit this line before save!",
		     COLUMN_EDITABLE, TRUE, -1);

  path = gtk_tree_model_get_path(model, &iter);
  gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, NULL, FALSE);

  gtk_tree_path_free(path);
}

static void removeButton_clicked_callback(GtkWidget *w, gpointer user_data)
{
  GtkTreeIter iter;
  GtkTreeModel *model =
    gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
  GtkTreeSelection *selection =
    gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));

  if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
    gint i;
    GtkTreePath *path;

    path = gtk_tree_model_get_path(model, &iter);
    i = gtk_tree_path_get_indices(path)[0];
    gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
    gtk_tree_path_free(path);
    is_edited = TRUE;
  }
}

static void okButton_clicked_callback(GtkWidget *w, gpointer user_data)
{
  if (is_edited) {
    GHashTable *new_user_commands =
      g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
    GtkTreeIter iter;
    GtkTreeModel *model =
      gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
    gboolean has_element =
      gtk_tree_model_get_iter_first(model, &iter);

    while (has_element) {
      gchar *keyword, *command;

      gtk_tree_model_get(model, &iter, COLUMN_KEYWORD, &keyword,
			 COLUMN_COMMAND, &command, -1);

      g_hash_table_insert(new_user_commands,
			  g_strdup(keyword), g_strdup(command));

      has_element = gtk_tree_model_iter_next(model, &iter);
    }

    free_user_commands();

    user_commands = new_user_commands;
    user_commands_changed = TRUE;
  }

  delete_event_callback(dialog, NULL);
}

static void cancelButton_clicked_callback(GtkWidget *w, gpointer user_data)
{
  delete_event_callback(dialog, NULL);
}

static int editorwin_new(void)
{
  GtkWidget *hbox;
  GtkWidget *scrolled;
  GtkWidget *addButton;
  GtkWidget *removeButton;
  GtkWidget *okButton;
  GtkWidget *cancelButton;
  GtkTreeModel *model;
  GtkAccelGroup *accel;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), "Edit commands");
  gtk_container_set_border_width(GTK_CONTAINER(dialog), 4);
  gtk_window_set_default_size(GTK_WINDOW(dialog),
			      DEFAULT_WIDTH, DEFAULT_HEIGHT);

  scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  model = create_model();
  treeview = gtk_tree_view_new_with_model(model);
  gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
  gtk_tree_selection_set_mode
    (gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)),
     GTK_SELECTION_SINGLE);
  g_object_unref(model);
  add_columns();
  gtk_container_add(GTK_CONTAINER(scrolled), treeview);

  hbox = gtk_hbox_new(FALSE, 0);
  addButton = gtk_button_new_with_label("Add new");
  removeButton = gtk_button_new_with_label("Remove");
  gtk_box_pack_end(GTK_BOX(hbox), removeButton, FALSE, FALSE, 4);
  gtk_box_pack_end(GTK_BOX(hbox), addButton, FALSE, FALSE, 4);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     scrolled, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		     hbox, FALSE, FALSE, 4);

  okButton = gtk_button_new_with_label("OK");
  cancelButton = gtk_button_new_with_label("Cancel");

  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		   cancelButton, FALSE, FALSE, 4);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(dialog)->action_area),
		   okButton, FALSE, FALSE, 4);

  g_signal_connect(dialog, "delete_event",
		   G_CALLBACK(delete_event_callback), NULL);
  g_signal_connect(addButton, "clicked",
		   G_CALLBACK(addButton_clicked_callback), NULL);
  g_signal_connect(removeButton, "clicked",
		   G_CALLBACK(removeButton_clicked_callback), NULL);
  g_signal_connect(okButton, "clicked",
		   G_CALLBACK(okButton_clicked_callback), NULL);
  g_signal_connect(cancelButton, "clicked",
		   G_CALLBACK(cancelButton_clicked_callback), NULL);

  accel = gtk_accel_group_new();
  gtk_window_add_accel_group(GTK_WINDOW(dialog), accel);
  gtk_widget_add_accelerator(cancelButton, "clicked", accel, GDK_Escape, 0, 0);

  gtk_widget_show_all(dialog);

  is_edited = FALSE;

  return 0;
}

int popup_editorwin(void)
{
  editorwin_new();

  return 0;
}
