/**
 * Copyright (c) 2003-2005, David A. Czarnecki
 * All rights reserved.
 *
 * Portions Copyright (c) 2003-2005 by Mark Lussier
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * Neither the name of the "David A. Czarnecki" and "blojsom" nor the names of
 * its contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Products derived from this software may not be called "blojsom",
 * nor may "blojsom" appear in their name, without prior written permission of
 * David A. Czarnecki.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.blojsom.blog;

import org.blojsom.BlojsomDataBase;
import org.blojsom.BlojsomException;
import org.blojsom.util.BlojsomUtils;
import org.blojsom.util.BlojsomProperties;
import org.blojsom.util.BlojsomConstants;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.xmlrpc.XmlRpcException;

import java.io.*;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

/**
 * FileBackedPingback
 *
 * @author David Czarnecki
 * @since blojsom 2.26
 * @version $Id: FileBackedPingback.java,v 1.3 2005/09/18 20:58:18 czarneckid Exp $
 */
public class FileBackedPingback extends Pingback {

    private transient Log _logger = LogFactory.getLog(FileBackedPingback.class);

    protected static final int PINGBACK_GENERIC_FAULT_CODE = 0;
    protected static final int PINGBACK_SOURCE_URI_NON_EXISTENT_CODE = 16;
    protected static final int PINGBACK_NO_LINK_TO_TARGET_URI_CODE = 17;
    protected static final int PINGBACK_TARGET_URI_NON_EXISTENT_CODE = 32;
    protected static final int PINGBACK_TARGET_URI_NOT_ENABLED_CODE = 33;
    protected static final int PINGBACK_ALREADY_REGISTERED_CODE = 48;
    protected static final int PINGBACK_ACCESS_DENIED_CODE = 49;
    protected static final int PINGBACK_UPSTREAM_SERVER_ERROR_CODE = 50;

    protected transient File _source;

    /**
     * Default constructor
     */
    public FileBackedPingback() {
        super();
    }

    /**
     * Pingback constructor to take a title, excerpt, url, and blog name
     *
     * @param title    Title of the pingback
     * @param excerpt  Excerpt from the pingback
     * @param url      Url for the pingback
     * @param blogName Blog name of the pingback
     */
    public FileBackedPingback(String title, String excerpt, String url, String blogName) {
        super(title, excerpt, url, blogName);
    }

    /**
     * Retrieve the source file for this pingback
     *
     * @return {@link File} backing this pingback
     */
    public File getSource() {
        return _source;
    }

    /**
     * Set the source file for this pingback
     *
     * @param source {@link File} backing this pingback
     */
    public void setSource(File source) {
        _source = source;
    }

    /**
     * Load the pingback
     *
     * @param blogUser {@link BlogUser}
     * @since blojsom 2.26
     */
    public void load(BlogUser blogUser) throws BlojsomException {
        if (_source == null) {
			throw new BlojsomException("No source file set for this pingback");
		}
		Blog blog = blogUser.getBlog();
		File category_path = new File(_source.toString());
		String blog_home = BlojsomUtils.replace(category_path.getParent(),
				"\\", "/");
		String category_name = blog_home.replace(blog.getBlogHome(), "");

		BlojsomDataBase blojsomdb = new BlojsomDataBase();
		blojsomdb.startDB();
        String category_condition = "";
        if(category_name.equals("/")){
        	category_condition = "and category_name is null";
        }else{
        	category_condition = "and category_name='"+BlojsomUtils.removeSlashes(category_name)+"'";
        }
		ResultSet rs = blojsomdb.myExecuteQuery("SELECT * FROM pingback_pb WHERE weblog_id='"+blogUser._id+"' and pingback_pb_name='" + _source.getName()
				+ "' "+category_condition);
		try {
			while (rs.next()) {
				setTitle(rs.getString("title"));
				setUrl(rs.getString("url"));
				setBlogName(rs.getString("blogname"));
				setExcerpt(rs.getString("excerpt"));
				setTrackbackDateLong(rs.getDate("blojsom_date").getTime());
				setId(_source.getName());
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		blojsomdb.stopDB();

		// Load pingback meta-data if available
		File pingbackMetaData = new File(BlojsomUtils.getFilename(_source
				.toString())
				+ ".meta");

		_logger.debug("Loading pingback meta-data: "
				+ pingbackMetaData.toString());
		BlojsomProperties pingbackMetaDataProperties = new BlojsomProperties();

		blojsomdb.startDB();
		int categoryPropertyFiles = 0;
		ResultSet rs1 = null;
		Vector columnName = null;
		try {
	    	if(category_name.equals("/")){
				rs1 = blojsomdb.myExecuteQuery("SELECT * FROM pingback_meta where weblog_id='"+blogUser._id+"' and pingback_meta_name='"
						+ pingbackMetaData.getName() + "' AND category_name= is null");
			}else{
				rs1 = blojsomdb.myExecuteQuery("SELECT * FROM pingback_meta where weblog_id='"+blogUser._id+"' and pingback_meta_name='"
						+ pingbackMetaData.getName() + "' AND category_name='"
						+ category_name + "'");
	    	}

			rs1.last();
			categoryPropertyFiles = rs1.getRow();
			ResultSetMetaData rsmd;
			rsmd = rs1.getMetaData();
			int iCols = rsmd.getColumnCount();
			int column = 6;
			columnName = new Vector();
			while (column <= iCols) {
				columnName.add(rsmd.getColumnName(column));
				column++;
			}
		} catch (SQLException e1) {
			// TODO ꂽ catch ubN
			e1.printStackTrace();
		}
		blojsomdb.stopDB();
		blojsomdb.startDB();
		// ResultSet rs = null;
		try {
	    	if(category_name.equals("/")){
				rs1 = blojsomdb.myExecuteQuery("SELECT * FROM pingback_meta where weblog_id='"+blogUser._id+"' and pingback_meta_name='"
						+ pingbackMetaData.getName() + "' AND category_name= is null");
			}else{
				rs1 = blojsomdb.myExecuteQuery("SELECT * FROM pingback_meta where weblog_id='"+blogUser._id+"' and pingback_meta_name='"
						+ pingbackMetaData.getName() + "' AND category_name='"
						+ category_name + "'");
	    	}
		} catch (Exception e) {
			e.printStackTrace();
		}

		// Load properties file for category (if present)
		// File[] categoryPropertyFiles =
		// blog.listFiles(BlojsomUtils.getExtensionsFilter(propertiesExtensions));
		if ((categoryPropertyFiles != 0) && (categoryPropertyFiles > 0)) {
			for (int i = 0; i < categoryPropertyFiles; i++) {
				try {
					// ꂽs[v
					while (rs.next()) {
						for (int j = 0; j < columnName.size(); j++) {
							String line = columnName.get(j).toString();
							// line = BlojsomUtils.replace(line, "_", ".");
							line = line
									+ "="
									+ rs
											.getString(columnName.get(j)
													.toString());
							if (line != null) {
								if (line.length() > 0) {
									if (line.charAt(0) != '#') {
										int len = line.length();
										int keyIndex;
										for (keyIndex = 0; keyIndex < len; keyIndex++) {
											if (line.charAt(keyIndex) == '\\')
												keyIndex += 2;
											else if (BlojsomProperties.whiteSpaceChars
													.indexOf(line
															.charAt(keyIndex)) == -1)
												break;
										}

										int separatorIndex;
										for (separatorIndex = keyIndex; separatorIndex < len; separatorIndex++) {
											char currentChar = line
													.charAt(separatorIndex);
											if (currentChar == '\\')
												separatorIndex++;
											else if (BlojsomProperties.keyValueSeparators
													.indexOf(currentChar) != -1)
												break;
										}

										int valueIndex;
										for (valueIndex = separatorIndex; valueIndex < len; valueIndex++)
											if (BlojsomProperties.whiteSpaceChars
													.indexOf(line
															.charAt(valueIndex)) == -1)
												break;

										if (valueIndex < len)
											if (BlojsomProperties.strictKeyValueSeparators
													.indexOf(line
															.charAt(valueIndex)) != -1)
												valueIndex++;

										while (valueIndex < len) {
											if (BlojsomProperties.whiteSpaceChars
													.indexOf(line
															.charAt(valueIndex)) == -1)
												break;
											valueIndex++;
										}

										String key;
										String value;

										key = line.substring(keyIndex,
												separatorIndex);
										key = BlojsomUtils.replace(key, "\\",
												"");// key="BLOJSOM_pingback_PLUGIN_METADATA_IP"
										value = (separatorIndex < len) ? line
												.substring(valueIndex, len)
												: "";// value="127.0.0.1"

										List values;
										if (pingbackMetaDataProperties
												.containsKey(key)
												&& pingbackMetaDataProperties.allowMultipleValues) {
											Object previousValue = pingbackMetaDataProperties
													.get(key);

											if (previousValue instanceof List) {
												values = (List) previousValue;
												values.add(value);
											} else {
												values = new ArrayList(1);
												values.add(previousValue);
												values.add(value);
											}

											pingbackMetaDataProperties.put(key,
													values);
										} else {
											pingbackMetaDataProperties.put(key,
													value);
										}
									}
								}
							}
						}
					}
				} catch (Exception e) {
					e.printStackTrace();
					_logger.warn("Failed loading properties from: " + e);
					continue;
				}
			}
			setMetaData(pingbackMetaDataProperties);
		}
		blojsomdb.stopDB();
    }

    /**
	 * Save the pingback
	 * 
	 * @param blogUser
	 *            {@link BlogUser}
	 * @since blojsom 2.26
	 */
    public void save(BlogUser blogUser) throws BlojsomException {
        if (_blogEntry == null) {
            throw new BlojsomException("Blog entry for this pingback not available");
        }
        
        Blog blog = blogUser.getBlog();
        long originalTimestamp = -1;
        originalTimestamp = _trackbackDateLong;

        File pingbackEntry;
        String pingbackDir = "";
        if (_source == null) {
            StringBuffer pingbackDirectory = new StringBuffer();
            String permalinkFilename = _blogEntry.getPermalink();
            permalinkFilename = BlojsomUtils.urlDecode(permalinkFilename);
            if (permalinkFilename == null) {
                _logger.debug("Invalid permalink pingback for: " + _blogEntry.getPermalink());

                throw new BlojsomException("Invalid permalink pingback for: " + _blogEntry.getPermalink());
            }

            pingbackDirectory.append(blog.getBlogHome());
            pingbackDirectory.append(BlojsomUtils.removeInitialSlash(_blogEntry.getCategory()));
            pingbackDir = BlojsomUtils.removeInitialSlash(_blogEntry.getCategory());
            // File blogEntry = new File(pingbackDirectory.toString() +
			// File.separator + permalinkFilename);
            /*
			 * if (!blogEntry.exists()) { _logger.error("Trying to create
			 * pingback for invalid blog entry: " + _blogEntry.getPermalink());
			 * 
			 * throw new BlojsomException("Trying to create pingback for invalid
			 * blog entry: " + _blogEntry.getPermalink()); }
			 */

            pingbackDirectory.append(blog.getBlogPingbacksDirectory());
            pingbackDirectory.append(File.separator);
            pingbackDirectory.append(permalinkFilename);
            pingbackDirectory.append(File.separator);
            pingbackDir += blog.getBlogPingbacksDirectory() +"/"+ permalinkFilename;
            String pingbackFilename = pingbackDirectory.toString() + _id + BlojsomConstants.PINGBACK_EXTENSION;
            // setId(hashedpingback + BlojsomConstants.pingback_EXTENSION);

            pingbackEntry = new File(pingbackFilename);
        } else {
            pingbackEntry = _source;
            originalTimestamp = _trackbackDateLong;
        }
        String category_condition = "";
        if(pingbackDir.equals("/")){
        	category_condition = "and category_name is null";
        }else{
        	category_condition = "and category_name='"+BlojsomUtils.removeSlashes(pingbackDir)+"'";
        }
        try {
        	BlojsomDataBase blojsomdb = new BlojsomDataBase();
        	blojsomdb.startDB();
        	int i = blojsomdb
					.myExecuteUpdate("UPDATE pingback_pb SET title='"
							+ BlojsomUtils.nullToBlank(getTitle()).trim()
							+ "', url='" + BlojsomUtils.nullToBlank(getUrl()).trim()
							+ "', blogname='" + BlojsomUtils.nullToBlank(getBlogName()).trim()
							+ "', excerpt='" + BlojsomUtils.nullToBlank(getExcerpt()).trim()
							+ "' WHERE weblog_id='"+blogUser._id+"' and pingback_pb_name='" + pingbackEntry.getName()
							+ "' "+category_condition);		
        	if(i <= 0){
        		i = blojsomdb.myExecuteUpdate("INSERT INTO pingback_pb VALUES ('"+blogUser._id+"','"
            			+ pingbackEntry.getName() +"','"
    					+ BlojsomUtils.removeSlashes(pingbackDir) + "','"
    					+ BlojsomUtils.nullToBlank(getTitle()).trim() + "','"
    					+ BlojsomUtils.nullToBlank(getUrl()).trim() + "','"
    					+ BlojsomUtils.nullToBlank(getBlogName()).trim() + "','"
    					+ BlojsomUtils.nullToBlank(getExcerpt()).trim() + "',CURRENT_TIMESTAMP)");
        	}
        	blojsomdb.stopDB();

            _logger.debug("Added blog pinkback: " + _id);

            BlojsomProperties pingbackMetaDataProperties = (BlojsomProperties)BlojsomUtils.mapToProperties(_metaData, BlojsomConstants.UTF8);
            String pingbackMetaDataFilename = BlojsomUtils.getFilename(pingbackEntry.toString()) + BlojsomConstants.DEFAULT_METADATA_EXTENSION;
            //FileOutputStream fos = new FileOutputStream(new File(pingbackMetaDataFilename));
            //Rgmetat@C̕ۑ
            //pingbackMetaDataProperties.getProperty(pingbackMetaDataFilename,"pingback_META");
            //pingbackMetaDataProperties.store(fos, null);
            //fos.close();
            
            blojsomdb.startDB();
    		i = blojsomdb.myExecuteUpdate("UPDATE pingback_meta set blojsom_date=CURRENT_TIMESTAMP WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='" + pingbackEntry.getName().replaceAll("pb","meta")+ "' "+category_condition);
    		if(i <= 0){
    			blojsomdb.myExecuteUpdate("INSERT INTO pingback_meta (weblog_id,pingback_meta_name,category_name,blojsom_date) VALUES ('"+blogUser._id+"','" + pingbackEntry.getName().replaceAll("pb","meta")+ "','"+BlojsomUtils.removeSlashes(pingbackDir)+"',CURRENT_TIMESTAMP)");
    		}
    		for (Enumeration e = pingbackMetaDataProperties.keys(); e.hasMoreElements();) {
    			String key = e.nextElement().toString();
    			Object value = pingbackMetaDataProperties.get(key);
    			key = BlojsomUtils.replace(key, " ", "\\ ");
    			key = BlojsomUtils.replace(key,"-","_");
    			key = BlojsomUtils.replace(key,".","_");
    			key = key.toLowerCase();
    			if (value != null && value instanceof List
    					&& pingbackMetaDataProperties.allowMultipleValues) {
    				List values = (List) value;
    				for ( i = 0; i < values.size(); i++) {
    					value = values.get(i);
    					// writer.write(key + "=" + value);writer.newLine();
    					int j = blojsomdb.myExecuteUpdate("UPDATE pingback_meta SET " + key + "='" + value + "' WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='" + pingbackEntry.getName().replaceAll("pb","meta")+ "' "+category_condition);
    	        		if(j <= 0){
    	        			blojsomdb.myExecuteUpdate("ALTER TABLE pingback_meta ADD " + key + " varchar2(2000)");
    	        			blojsomdb.myExecuteUpdate("UPDATE pingback_meta SET " + key + "='" + value + "' WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='" + pingbackEntry.getName().replaceAll("pb","meta")+ "' "+category_condition);
    	        		}
    				}
    			} else {
    				value = (value != null) ? value.toString() : "";
    				// writer.write(key + "=" + value);writer.newLine();
    				i = blojsomdb.myExecuteUpdate("UPDATE pingback_meta SET " + key + "='" + value + "' WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='" + pingbackEntry.getName().replaceAll("pb","meta")+ "' "+category_condition);
            		if(i <= 0){
            			blojsomdb.myExecuteUpdate("ALTER TABLE pingback_meta ADD " + key + " varchar2(2000)");
            			blojsomdb.myExecuteUpdate("UPDATE pingback_meta SET " + key + "='" + value + "' WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='" + pingbackEntry.getName().replaceAll("pb","meta")+ "' "+category_condition);
            		}
    			}
    		}
            blojsomdb.stopDB();
            
            _logger.debug("Wrote pingback meta-data: " + pingbackMetaDataFilename);

            _source = pingbackEntry;

            if (originalTimestamp != -1) {
                _source.setLastModified(originalTimestamp);
                
                blojsomdb.startDB();
                blojsomdb.myExecuteUpdate("UPDATE pingback_pb SET blojsom_date=CURRENT_TIMESTAMP WHERE weblog_id='"+blogUser._id+"' and pingback_pb_name='"
						+ pingbackEntry.getName().replaceAll("pb", "meta")
						+ "' "+category_condition);	
                blojsomdb.stopDB();
            }
        } catch (Exception e) {
            _logger.error(e);

            throw new BlojsomException(e.getMessage(), new XmlRpcException(PINGBACK_GENERIC_FAULT_CODE, "Unknown exception occurred"));
        }
    }

    /**
     * Delete the pingback
     *
     * @param blogUser {@link BlogUser}
     * @since blojsom 2.26
     */
    public void delete(BlogUser blogUser) throws BlojsomException {
        Blog blog = blogUser.getBlog();
        BlojsomDataBase blojsomdb = new BlojsomDataBase();
        blojsomdb.startDB();
        if (_source == null || _blogEntry == null) {
            throw new BlojsomException("No source file found to delete pingback");
        }

        _logger.debug("Deleting pingback " + _source.getAbsolutePath());
        String _category = _blogEntry.getCategory() + blog.getBlogPingbacksDirectory() + "/" + _blogEntry.getPermalink();
        String category_condition = "";
        if(_category.equals("/")){
        	category_condition = "and category_name is null";
        }else{
        	category_condition = "and category_name='"+BlojsomUtils.removeSlashes(_category)+"'";
        }
        int i = blojsomdb.myExecuteUpdate("DELETE FROM pingback_pb WHERE weblog_id='"+blogUser._id+"' and pingback_pb_name='"+_source.getName()+"' "+category_condition);
		if(i <= 0){
			throw new BlojsomException("Unable to delete pingback: " + getId());
        }

        // Delete meta-data
        File metaFile = new File(blog.getBlogHome() + _blogEntry.getCategory() + blog.getBlogPingbacksDirectory()
                + File.separatorChar + _blogEntry.getPermalink() + File.separatorChar + BlojsomUtils.getFilename(_source.getName())
                + blog.getBlogEntryMetaDataExtension());
        
        i = blojsomdb.myExecuteUpdate("DELETE FROM pingback_meta WHERE weblog_id='"+blogUser._id+"' and pingback_meta_name='"+metaFile.getName()+"' "+category_condition);
		if(i <= 0){
			throw new BlojsomException("Unable to delete pingback meta-data: " + metaFile);
		}
		blojsomdb.stopDB();
    }
}