/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2012 NTT DATA BUSINESS BRAINS CORPORATION
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 NTT DATA BUSINESS BRAINS CORPORATION and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      NTT DATA BUSINESS BRAINS CORPORATION - initial API and implementation
 *******************************************************************************/
package blanco.db.tableaccessor;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import blanco.xml.bind.BlancoXmlBindingUtil;
import blanco.xml.bind.BlancoXmlUnmarshaller;
import blanco.xml.bind.valueobject.BlancoXmlDocument;
import blanco.xml.bind.valueobject.BlancoXmlElement;

/**
 * 「メッセージ定義書」Excel様式から情報を抽出します。
 * 
 * このクラスは、中間XMLファイルから情報抽出する機能を担います。
 * 
 * @author Toshiki Iga
 */
public class BlancoDbTableAccessorXmlParser {
    /**
     * 登録済みの SQL 定義 ID を記憶します。
     */
    private final Map<String, BlancoDbTableAccessorTargetInfo> sqlDefIdMap = new HashMap<String, BlancoDbTableAccessorTargetInfo>();

    /**
     * 中間XMLファイルのXMLドキュメントをパースして、情報の配列を取得します。
     * 
     * @param argMetaXmlSourceFile
     *            中間XMLファイル。
     * @return パースの結果得られた情報の配列。
     */
    public BlancoDbTableAccessorTargetInfo[] parse(final File argMetaXmlSourceFile) {
        final BlancoXmlDocument documentMeta = new BlancoXmlUnmarshaller().unmarshal(argMetaXmlSourceFile);
        if (documentMeta == null) {
            return null;
        }

        return parse(documentMeta);
    }

    /**
     * 中間XMLファイル形式のXMLドキュメントをパースして、バリューオブジェクト情報の配列を取得します。
     * 
     * @param argXmlDocument
     *            中間XMLファイルのXMLドキュメント。
     * @return パースの結果得られたバリューオブジェクト情報の配列。
     */
    public BlancoDbTableAccessorTargetInfo[] parse(final BlancoXmlDocument argXmlDocument) {
        final List<BlancoDbTableAccessorTargetInfo> listStructure = new ArrayList<BlancoDbTableAccessorTargetInfo>();
        // ルートエレメントを取得します。
        final BlancoXmlElement elementRoot = BlancoXmlBindingUtil.getDocumentElement(argXmlDocument);
        if (elementRoot == null) {
            // ルートエレメントが無い場合には処理中断します。
            return null;
        }

        // sheet(Excelシート)のリストを取得します。
        final List<BlancoXmlElement> listSheet = BlancoXmlBindingUtil.getElementsByTagName(elementRoot, "sheet");
        final int sizeListSheet = listSheet.size();
        for (int index = 0; index < sizeListSheet; index++) {
            final BlancoXmlElement elementSheet = listSheet.get(index);

            final List<BlancoDbTableAccessorTargetInfo> structures = parseElementSheet(elementSheet);
            // 得られた情報を記憶します。
            if (structures != null)
                listStructure.addAll(structures);
        }

        final BlancoDbTableAccessorTargetInfo[] result = new BlancoDbTableAccessorTargetInfo[listStructure.size()];
        listStructure.toArray(result);
        return result;
    }

    /**
     * 中間XMLファイル形式の「sheet」XMLエレメントをパースして、バリューオブジェクト情報を取得します。
     * 
     * @param argElementSheet
     *            中間XMLファイルの「sheet」XMLエレメント。
     * @return パースの結果得られたバリューオブジェクト情報。「name」が見つからなかった場合には nullを戻します。
     */
    public List<BlancoDbTableAccessorTargetInfo> parseElementSheet(final BlancoXmlElement argElementSheet) {
        final List<BlancoDbTableAccessorTargetInfo> result = new ArrayList<BlancoDbTableAccessorTargetInfo>();

        // 入力パラメータ情報を取得します。

        final BlancoXmlElement elementSimpleAccessorList = BlancoXmlBindingUtil.getElement(argElementSheet,
                "blancodbtablesimpleaccessor-list");
        if (elementSimpleAccessorList == null) {
            return null;
        }

        // 一覧の内容を取得します。
        final List<BlancoXmlElement> listField = BlancoXmlBindingUtil.getElementsByTagName(elementSimpleAccessorList,
                "field");
        for (int indexField = 0; indexField < listField.size(); indexField++) {
            final BlancoDbTableAccessorTargetInfo structure = new BlancoDbTableAccessorTargetInfo();

            final BlancoXmlElement elementField = listField.get(indexField);

            structure.setSqldefid(BlancoXmlBindingUtil.getTextContent(elementField, "sqldefid"));
            if (structure.getSqldefid() == null || structure.getSqldefid().length() == 0) {
                return null;
            }

            structure.setPackageName(BlancoXmlBindingUtil.getTextContent(elementField, "packageName"));
            if (structure.getPackageName() == null || structure.getPackageName().length() == 0) {
                return null;
            }

            structure.setDescription(BlancoXmlBindingUtil.getTextContent(elementField, "description"));

            structure.setTableName(BlancoXmlBindingUtil.getTextContent(elementField, "tableName"));
            if (structure.getTableName() == null || structure.getTableName().length() == 0) {
                return null;
            }

            structure.setSqlType(BlancoXmlBindingUtil.getTextContent(elementField, "sqlType"));
            if (structure.getSqlType() == null || structure.getSqlType().length() == 0) {
                return null;
            }

            final String excludeColumn = BlancoXmlBindingUtil.getTextContent(elementField, "excludeColumn");
            if (excludeColumn != null) {
                // カンマ区切りをリストに加工して記憶します。
                structure.getExcludeColumnList().addAll(splitByComma(excludeColumn));

                if ("DELETE".equals(structure.getSqlType())) {
                    System.err.println("SQL定義ID [" + structure.getSqldefid() + "] において '" + structure.getSqlType()
                            + "' に除外列 '" + excludeColumn + "' の指定があります。この値を同時に利用することはできません。");
                }
            }

            structure.setFieldNameForOptimisticLock(BlancoXmlBindingUtil.getTextContent(elementField,
                    "fieldNameForOptimisticLock"));
            if (structure.getFieldNameForOptimisticLock() != null
                    && structure.getFieldNameForOptimisticLock().length() > 0) {
                if ("INSERT".equals(structure.getSqlType()) || "SELECT".equals(structure.getSqlType())) {
                    System.err.println("SQL定義ID [" + structure.getSqldefid() + "] において '" + structure.getSqlType()
                            + "' に楽観的排他列 '" + structure.getFieldNameForOptimisticLock()
                            + "' の指定があります。この値を同時に利用することはできません。");
                }
            }

            if (sqlDefIdMap.get(structure.getSqldefid()) != null) {
                System.err.println("問題: SQL定義ID [" + structure.getSqldefid() + "] が重複しています。");
            } else {
                sqlDefIdMap.put(structure.getSqldefid(), structure);
            }

            result.add(structure);
        }

        return result;
    }

    /**
     * カンマで分割
     * 
     * @param value
     * @return
     */
    static List<String> splitByComma(final String value) {
        final List<String> result = new ArrayList<String>();
        final StringTokenizer tokenizer = new StringTokenizer(value, ",");
        for (; tokenizer.hasMoreTokens();) {
            result.add(tokenizer.nextToken().trim());
        }
        return result;
    }
}
