/*******************************************************************************
 * blanco Framework
 * Copyright (C) 2012 NTT DATA BUSINESS BRAINS CORPORATION
 * 
 * This library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *******************************************************************************/
/*******************************************************************************
 * Copyright (c) 2012 NTT DATA BUSINESS BRAINS CORPORATION and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *      NTT DATA BUSINESS BRAINS CORPORATION - initial API and implementation
 *******************************************************************************/
package blanco.db.tableaccessor;

import java.sql.Types;
import java.util.Map;

import blanco.commons.sql.format.BlancoSqlFormatter;
import blanco.commons.sql.format.BlancoSqlFormatterException;
import blanco.commons.sql.format.BlancoSqlRule;
import blanco.commons.util.BlancoNameAdjuster;
import blanco.db.common.stringgroup.BlancoDbSqlInfoScrollStringGroup;
import blanco.db.common.stringgroup.BlancoDbSqlInfoTypeStringGroup;
import blanco.db.common.valueobject.BlancoDbSqlInfoStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataColumnStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataKeyStructure;
import blanco.dbmetadata.valueobject.BlancoDbMetaDataTableStructure;

/**
 * DB テーブルアクセサが内部的に利用する SQL 生成のためのクラス。
 * 
 * @author Toshiki Iga
 */
public class BlancoDbTableAccessorSqlGenerator {
	/**
	 * INSERT 文の生成。
	 * 
	 * @param tableStructure
	 * @return
	 */
	public BlancoDbSqlInfoStructure generateInsert(
			final BlancoDbTableAccessorTargetInfo targetInfo,
			final BlancoDbMetaDataTableStructure tableStructure,
			final Map<String, BlancoDbMetaDataColumnStructure> usedExcludeColumnMap) {
		final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbSqlInfoStructure();
		sqlInfo.setName(BlancoNameAdjuster.toClassName(targetInfo.getSqldefid()));
		sqlInfo.setType(BlancoDbSqlInfoTypeStringGroup.INVOKER);
		sqlInfo.setDescription(targetInfo.getDescription());
		sqlInfo.setSingle(true);

		final StringBuilder sql = new StringBuilder();
		sql.append("INSERT INTO " + tableStructure.getName() + " (");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isExcludeColumn(column, targetInfo)) {
					// 使用済み除外列としてマークします。
					usedExcludeColumnMap.put(column.getName().toLowerCase(),
							column);
					continue;
				}
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(", ");
				}
				sql.append(column.getName());
			}
		}
		sql.append(") VALUES (");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isExcludeColumn(column, targetInfo)) {
					continue;
				}
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(", ");
				}

				// 入力パラメータ追加
				final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
				copy(column, newColumn);
				newColumn.setName(BlancoNameAdjuster.toParameterName(newColumn
						.getName()));
				sqlInfo.getInParameterList().add(newColumn);

				sql.append("#" + newColumn.getName());
			}
		}

		sql.append(")");

		try {
			sqlInfo.setQuery(new BlancoSqlFormatter(new BlancoSqlRule())
					.format(sql.toString()));
		} catch (BlancoSqlFormatterException e) {
			throw new IllegalArgumentException(e.getMessage() + e);
		}

		return sqlInfo;
	}

	/**
	 * SELECT 文の生成。
	 * 
	 * @param tableStructure
	 * @return
	 */
	public BlancoDbSqlInfoStructure generateSelect(
			final BlancoDbTableAccessorTargetInfo targetInfo,
			final BlancoDbMetaDataTableStructure tableStructure,
			final Map<String, BlancoDbMetaDataColumnStructure> usedExcludeColumnMap) {
		final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbSqlInfoStructure();
		sqlInfo.setName(BlancoNameAdjuster.toClassName(targetInfo.getSqldefid()));
		sqlInfo.setType(BlancoDbSqlInfoTypeStringGroup.ITERATOR);
		sqlInfo.setDescription(targetInfo.getDescription());
		sqlInfo.setScroll(BlancoDbSqlInfoScrollStringGroup.TYPE_FORWARD_ONLY);
		sqlInfo.setSingle(true);

		final StringBuilder sql = new StringBuilder();
		sql.append("SELECT ");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isExcludeColumn(column, targetInfo)) {
					// 使用済み除外列としてマークします。
					usedExcludeColumnMap.put(column.getName().toLowerCase(),
							column);
					continue;
				}
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}
				if (isPrimaryKey(column.getName(), tableStructure)) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(", ");
				}
				sql.append(column.getName());
			}
		}
		sql.append(" FROM " + tableStructure.getName() + " WHERE ");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}
				if (isPrimaryKey(column.getName(), tableStructure) == false) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(" AND ");
				}

				// 入力パラメータ追加
				final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
				copy(column, newColumn);
				newColumn.setName(BlancoNameAdjuster.toParameterName(newColumn
						.getName()));
				sqlInfo.getInParameterList().add(newColumn);

				sql.append(column.getName() + " = #" + newColumn.getName());
			}
		}

		try {
			sqlInfo.setQuery(new BlancoSqlFormatter(new BlancoSqlRule())
					.format(sql.toString()));
		} catch (BlancoSqlFormatterException e) {
			throw new IllegalArgumentException(e.getMessage() + e);
		}

		return sqlInfo;
	}

	/**
	 * UPDATE 文の生成。
	 * 
	 * @param tableStructure
	 * @param fieldNameForOptimisticLock
	 * @return
	 */
	public BlancoDbSqlInfoStructure generateUpdate(
			final BlancoDbTableAccessorTargetInfo targetInfo,
			final BlancoDbMetaDataTableStructure tableStructure,
			final Map<String, BlancoDbMetaDataColumnStructure> usedExcludeColumnMap,
			final Map<String, BlancoDbMetaDataColumnStructure> usedFieldNameForOptimisticLockMap) {
		final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbSqlInfoStructure();
		sqlInfo.setName(BlancoNameAdjuster.toClassName(targetInfo.getSqldefid()));
		sqlInfo.setType(BlancoDbSqlInfoTypeStringGroup.INVOKER);
		sqlInfo.setDescription(targetInfo.getDescription());
		sqlInfo.setSingle(true);

		final StringBuilder sql = new StringBuilder();
		sql.append("UPDATE " + tableStructure.getName() + " SET ");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isExcludeColumn(column, targetInfo)) {
					// 使用済み除外列としてマークします。
					usedExcludeColumnMap.put(column.getName().toLowerCase(),
							column);
					continue;
				}
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}
				if (isPrimaryKey(column.getName(), tableStructure)) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(", ");
				}

				// 入力パラメータ追加
				sqlInfo.getInParameterList().add(column);

				sql.append(column.getName() + " = #"
						+ BlancoNameAdjuster.toParameterName(column.getName()));
			}
		}
		sql.append(" WHERE ");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}
				if (isPrimaryKey(column.getName(), tableStructure) == false) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(" AND ");
				}

				// 入力パラメータ追加
				final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
				copy(column, newColumn);
				newColumn.setName(BlancoNameAdjuster.toParameterName(newColumn
						.getName()));
				sqlInfo.getInParameterList().add(newColumn);

				sql.append(column.getName() + " = #" + newColumn.getName());
			}
		}

		// UPD_DATE 関連処理。
		if (targetInfo.getFieldNameForOptimisticLock() != null) {
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (targetInfo.getFieldNameForOptimisticLock().toLowerCase()
						.equals(column.getName().toLowerCase())) {
					// 使用済み楽観排他列としてマークします。
					usedFieldNameForOptimisticLockMap.put(
							targetInfo.getFieldNameForOptimisticLock(), column);

					sql.append(" AND ");

					// 入力パラメータの特殊な追加
					final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
					copy(column, newColumn);
					sqlInfo.getInParameterList().add(newColumn);

					// 名称を上書き。
					newColumn.setName("originalValueForOptimisticLock");

					sql.append(column.getName()
							+ " = #originalValueForOptimisticLock");
				}
			}
		}

		try {
			sqlInfo.setQuery(new BlancoSqlFormatter(new BlancoSqlRule())
					.format(sql.toString()));
		} catch (BlancoSqlFormatterException e) {
			throw new IllegalArgumentException(e.getMessage() + e);
		}

		return sqlInfo;
	}

	/**
	 * DELETE 文の生成。
	 * 
	 * @param tableStructure
	 * @param fieldNameForOptimisticLock
	 * @return
	 */
	public BlancoDbSqlInfoStructure generateDelete(
			final BlancoDbTableAccessorTargetInfo targetInfo,
			final BlancoDbMetaDataTableStructure tableStructure,
			final Map<String, BlancoDbMetaDataColumnStructure> usedFieldNameForOptimisticLockMap) {
		final BlancoDbSqlInfoStructure sqlInfo = new BlancoDbSqlInfoStructure();
		sqlInfo.setName(BlancoNameAdjuster.toClassName(targetInfo.getSqldefid()));
		sqlInfo.setType(BlancoDbSqlInfoTypeStringGroup.INVOKER);
		sqlInfo.setDescription(targetInfo.getDescription());
		sqlInfo.setSingle(true);

		final StringBuilder sql = new StringBuilder();
		sql.append("DELETE FROM " + tableStructure.getName());
		sql.append(" WHERE ");
		{
			boolean isFirstColumn = true;
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (isNonSupportTypeForSimpleTable(column)) {
					continue;
				}
				if (isPrimaryKey(column.getName(), tableStructure) == false) {
					continue;
				}

				if (isFirstColumn) {
					isFirstColumn = false;
				} else {
					sql.append(" AND ");
				}

				// 入力パラメータ追加
				final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
				copy(column, newColumn);
				newColumn.setName(BlancoNameAdjuster.toParameterName(newColumn
						.getName()));
				sqlInfo.getInParameterList().add(newColumn);

				sql.append(column.getName() + " = #" + newColumn.getName());
			}
		}

		// UPD_DATE 関連処理。
		if (targetInfo.getFieldNameForOptimisticLock() != null) {
			for (BlancoDbMetaDataColumnStructure column : tableStructure
					.getColumns()) {
				if (targetInfo.getFieldNameForOptimisticLock().toLowerCase()
						.equals(column.getName().toLowerCase())) {
					// 使用済み楽観排他列としてマークします。
					usedFieldNameForOptimisticLockMap.put(
							targetInfo.getFieldNameForOptimisticLock(), column);

					sql.append(" AND ");

					// 入力パラメータの特殊な追加
					final BlancoDbMetaDataColumnStructure newColumn = new BlancoDbMetaDataColumnStructure();
					copy(column, newColumn);
					sqlInfo.getInParameterList().add(newColumn);

					// 名称を上書き。
					newColumn.setName("originalValueForOptimisticLock");

					sql.append(column.getName()
							+ " = #originalValueForOptimisticLock");
				}
			}
		}

		try {
			sqlInfo.setQuery(new BlancoSqlFormatter(new BlancoSqlRule())
					.format(sql.toString()));
		} catch (BlancoSqlFormatterException e) {
			throw new IllegalArgumentException(e.getMessage() + e);
		}

		return sqlInfo;
	}

	static boolean isPrimaryKey(final String columnName,
			BlancoDbMetaDataTableStructure tableStructure) {
		boolean isKey = false;
		for (BlancoDbMetaDataKeyStructure keyColumn : tableStructure
				.getPrimaryKeys()) {
			if (keyColumn.getPkcolumnName().toLowerCase()
					.equals(columnName.toLowerCase())) {
				isKey = true;
			}
		}
		return isKey;
	}

	/**
	 * バリューオブジェクトのコピー。
	 * 
	 * @param source
	 * @param target
	 */
	static void copy(final BlancoDbMetaDataColumnStructure source,
			final BlancoDbMetaDataColumnStructure target) {
		target.setName(source.getName());
		target.setDataType(source.getDataType());
		target.setDataTypeDisplayName(source.getDataTypeDisplayName());
		target.setTypeName(source.getTypeName());
		target.setColumnSize(source.getColumnSize());
		target.setBufferLength(source.getBufferLength());
		target.setDecimalDigits(source.getDecimalDigits());
		target.setNumPrecRadix(source.getNumPrecRadix());
		target.setNullable(source.getNullable());
		target.setNullableDisplayName(source.getNullableDisplayName());
		target.setRemarks(source.getRemarks());
		target.setColumnDef(source.getColumnDef());
		target.setSqlDataType(source.getSqlDataType());
		target.setSqlDatetimeSub(source.getSqlDatetimeSub());
		target.setCharOctetLength(source.getCharOctetLength());
		target.setOrdinalPosition(source.getOrdinalPosition());
		target.setIsNullable(source.getIsNullable());
		target.setScopeCatlog(source.getScopeCatlog());
		target.setScopeSchema(source.getScopeSchema());
		target.setScopeTable(source.getScopeTable());
		target.setSourceDataType(source.getSourceDataType());
		target.setWritable(source.getWritable());
	}

	/**
	 * 除外列かどうかを調べます。
	 * 
	 * @param column
	 * @param targetInfo
	 * @return
	 */
	static boolean isExcludeColumn(
			final BlancoDbMetaDataColumnStructure column,
			final BlancoDbTableAccessorTargetInfo targetInfo) {
		for (String excludeColumn : targetInfo.getExcludeColumnList()) {
			if (excludeColumn.toLowerCase().equals(
					column.getName().toLowerCase()))
				return true;
		}
		return false;
	}

	static boolean isNonSupportTypeForSimpleTable(
			final BlancoDbMetaDataColumnStructure columnStructure) {
		switch (columnStructure.getDataType()) {
		case Types.JAVA_OBJECT:
		case Types.DISTINCT:
		case Types.STRUCT:
		case Types.ARRAY:
		case Types.NULL:
		case Types.OTHER:
		case Types.REF:
		case Types.DATALINK:
		case Types.ROWID:// さしあたりサポート範囲外にマップします。
			return true;
		default:
			return false;
		}
	}
}
