package jp.co.powerbeans.powerql.vendor;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;

import jp.co.powerbeans.powerql.ColumnProperty;
import jp.co.powerbeans.powerql.Log;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.exceptions.POQLException;

import org.apache.oro.text.perl.MalformedPerl5PatternException;
import org.apache.oro.text.perl.Perl5Util;

/**
 * <p>タイトル: DBDependQLStatementPGSQL</p>
 * <p>説明: DBDependQLStatement の PostgreSQL実装。</p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * 
 * <p>Created on 2003/10/14</p>
 * @author 門田明彦
 * @version $Revision: 1.4 $
 */
public class DBDependQLStatementPGSQL extends DBDependQLStatementSupport {

	private static final String DB_TYPE = "PostgreSQL";
	
	/** regexp lib */
	private static Perl5Util perl = new Perl5Util();

	/** type mapping */
	private static Hashtable db2javaTypeMap = new Hashtable();

	/** type mapping */
	private static Hashtable java2dbTypeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] db2javaMap = { { "TINYINT", Short.class }, {
			"TINYINT\\([0-9]+\\)", Short.class }, {
			"SMALLINT", Integer.class }, {
			"SMALLINT\\([0-9]+\\)", Integer.class }, {
			"MEDIUMINT", Integer.class }, {
			"MEDIUMINT\\([0-9]+\\)", Integer.class }, {
			"INT", Integer.class }, {
			"INT\\([0-9]+\\)", Integer.class }, {
			"INTEGER", Integer.class }, {
			"INTEGER\\([0-9]+\\)", Integer.class }, {
			"BIGINT", Long.class }, {
			"BIGINT\\([0-9]+\\)", Long.class }, {

			"NUMERIC\\([0-9]+,[0-9]+\\)", Double.class }, {
			"DECIMAL\\([0-9]+,[0-9]+\\)", Double.class }, {
			"REAL\\([0-9]+,[0-9]+\\)", Float.class }, {
			"DOUBLE PRECISION\\([0-9]+,[0-9]+\\)", Double.class }, {

			"DATE", Timestamp.class }, {
			"DATETIME", Timestamp.class }, {
			"TIMESTAMP", Timestamp.class }, {
			"TIMESTAMP\\([0-9]+\\)", Timestamp.class }, {
			"TIME", Timestamp.class }, {

			"TINYBLOB", String.class }, {
			"TINYTEXT", String.class }, {
			"BLOB", String.class }, {
			"TEXT", String.class }, {
			"MEDIUMBLOB", String.class }, {
			"MEDIUMTEXT", String.class }, {
			"LONGBLOB", String.class }, {
			"LONGTEXT", String.class }, {

			"CHAR", String.class }, {
			"VARCHAR", String.class }, {
			"CHAR\\([0-9]+\\)", String.class }, {
			"VARCHAR\\([0-9]+\\)", String.class }, };

	/** type mapping data key */
	private static Object[][] java2dbMap = { { Byte.class, "TINYINT" }, {
			Short.class, "TINYINT" }, {
			Character.class, "TINYINT" }, {
			Integer.class, "INT" }, {
			Long.class, "BIGINT" }, {
			Double.class, "DOUBLE PRECISION" }, {

			String.class, "VARCHAR" }, {
			Timestamp.class, "DATE" }
	};
	
	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < db2javaMap.length; i++) {
			db2javaTypeMap.put(db2javaMap[i][0], db2javaMap[i][1]);
		}

		// typeMap.put(Java type, MySQL type)
		for (int i = 0; i < java2dbMap.length; i++) {
			java2dbTypeMap.put(java2dbMap[i][0], java2dbMap[i][1]);
		}
	}
	/** PostgreSQL > JDBC type mapping */
	private static Hashtable typeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] typeArray = {
            { "NUMBER\\([0-9]+\\)", Integer.class },
            { "NUMBER\\([0-9]+,[0-9]+\\)", Double.class },
            { "CHAR\\([0-9]+\\)", String.class },
            { "VARCHAR2\\([0-9]+\\)", String.class },
            { "NUMBER", Integer.class }, { "DATE", Timestamp.class },
            { "CHAR", String.class }, { "VARCHAR2", String.class },
            { "VARCHAR", String.class },
            { "VARCHAR2\\([0-9]+\\)", String.class },
            { "CHARACTER", String.class}, // DB2 Original
            { "TIMESTAMP", Timestamp.class}, 
            { "SMALLINT", Short.class}, 
            { "INTEGER", Integer.class}, // DB2 Original
            { "BPCHAR", String.class}, // PostgreSQL Original
            { "INT2", Short.class}, // PostgreSQL Original
            { "INT4", Integer.class}, // PostgreSQL Original
            { "INT8", Long.class}, // PostgreSQL Original
            { "NUMERIC", BigDecimal.class},
            { "FLOAT4", Float.class}, // PostgreSQL Original
            { "FLOAT8", Float.class}, // PostgreSQL Original
            { "TEXT", String.class}, // PostgreSQL Original
            { "BYTEA", byte[].class}, // PostgreSQL Original
            { "TIMESTAMPTZ", Timestamp.class}, // PostgreSQL Original
            { "INTERVAL", Integer.class}, // PostgreSQL Original
            { "TIME", Time.class}, // PostgreSQL Original 
            { "TIMETZ", Time.class}, // PostgreSQL Original 
            { "BOOL", Byte.class}, // PostgreSQL Original 
            { "POINT", String.class}, // PostgreSQL Original
            { "LINE", String.class}, // PostgreSQL Original
            { "LSEG", String.class}, // PostgreSQL Original
            { "BOX", String.class}, // PostgreSQL Original
            { "PATH", String.class}, // PostgreSQL Original
            { "POLYGON", String.class}, // PostgreSQL Original
            { "CIRCLE", String.class}, // PostgreSQL Original
            { "CIDR", String.class}, // PostgreSQL Original
            { "INET", String.class}, // PostgreSQL Original
            { "MACADDR", String.class}, // PostgreSQL Original
	};

	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < typeArray.length; i++) {
			typeMap.put(typeArray[i][0], typeArray[i][1]);
		}
	}

	
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getColumnPropertyList(java.lang.String)
     */
    public ArrayList getColumnPropertyList(String table, String schema) throws SQLException,
            POQLException {
        ArrayList list = new ArrayList(10);
        String table_up = table.toUpperCase();
        HashMap colMap = new HashMap(10);
        
        // 1 テーブルのカラム情報取得
        String sql =
         "select " + 
		 " c.relname as TABNAME, " +
		 " a.attname AS column_name, " +
		 " t.typname as data_type, " +
		 " a.attlen as data_length, " +
		 " a.attnotnull as notnull, " +
		 " a.atthasdef as default_exist, " +
		 " pk.conkey as pk," +
		 " seq.relname as seq " +
		 " from pg_attribute as a " +
		 "      left join  ( " + //  -- pk outer join
		 " 		select conrelid,contype,conkey " +
		 " 		from pg_constraint pk, pg_class c " +
		 " 		where  " +
		 " 		upper(c.relname)=upper('" + table + "') " + 
		 " 		and pk.contype='p' " +
		 " 		and pk.conrelid = c.oid " +
		 " ) as pk on a.attnum = ANY (pk.conkey) " +
		 " left join ( " + //  -- pk serial outer join
		 " select  " +
		 "      relname " +
//		 "      substring(s.relname, '^[^_]+') as tabname, " +
//		 "      substring(s.relname, '[^_]+_([^_]+)') as colname " +
		 "      from pg_class s " +
		 "where s.relkind = 'S' " +
		 " ) as seq on upper(seq.relname)=upper('" + table + "') || '_' || upper(a.attname) || '_SEQ' ," +
		 " pg_class as c, pg_type as t " +
		 " where a.attrelid = c.oid " +
		 " and a.atttypid = t.oid " +
		 " and upper(c.relname)=upper('" + table + "') " +
		 " and a.attnum > 0 " +
		 " order by a.attnum";
        
        ResultSet result = statement.executeQuery(sql.toString());
        
        while (result.next()) {
          // 各カラム情報を取得
          // ここではまだプライマリキー情報は取得しない
          ColumnProperty cp = new ColumnProperty();
          cp.setName(result.getString("column_name"));
          cp.setTypeAsDB(result.getString("data_type"));
          cp.setNullable(!result.getBoolean("notnull"));
          //cp.setDefaultValue(result.getObject("data_default")); // 2. デフォルト値の取得 を参照
          cp.setType(getJavaTypeFromDBType(cp.getTypeAsDB()));
          cp.setForceAutoValue(cp.getTypeAsDB().startsWith("serial"));
          cp.setPrimaryKey(result.getString("pk") != null);
          cp.setForceAutoValue(result.getString("seq") != null); // serial型
          colMap.put(cp.getName().toUpperCase(), cp);
          list.add(cp);
          Log.debug("add CPList " + cp.getName() + " type:" + cp.getTypeAsDB());
        }
        result.close();
        
        // 2. デフォルト値の取得
        // 以下のSQLで内部形式のデフォルト値は取得できるが
        // 再度PostgreSQLでコンパイルする必要があり、取得が複雑なので
        // 現段階では行わない
        
//        select c.oid, d.adnum, d.adbin from pg_attrdef d,pg_class c
//        where
//         d.adrelid = c.oid
//         AND upper(c.relname)=upper('mstcity') 

        return list;
    }

	/**
	 * getJavaTypeFromDBType<BR>
	 * DB(Oracle)のカラムの型からJAVAの型をClassで取得
	 * @param type_as_DB DB(Oracle)の型
	 * @return JAVAの型
	 */
	private Class getJavaTypeFromDBType(String type_as_DB)
		throws POQLException {

		String upper = type_as_DB.toUpperCase();

		if (typeMap.containsKey(upper)) {
			// 完全一致
			return (Class) typeMap.get(upper);

		} else {
			// 全パターンと正規表現比較
			for (int i = 0; i < typeArray.length; i++) {
				try {
                    if (perl.match(upper, (String) typeArray[i][0])) {
                    	return (Class) typeArray[i][1];
                    }
                } catch (MalformedPerl5PatternException e) {
                    throw new POQLException(
                        "not correspond column type(" + upper + ") ");
                }
			}

		}

		// 一致する型無し
		throw new POQLException(
			"not recoganize db TYPE(" + DB_TYPE + ") " + type_as_DB);
	}
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#isExistTable(java.lang.String)
     */
    public boolean isExistTable(String table_name) throws SQLException {
		String sql = "select count(*) as num from pg_tables where schemaname='public' AND " +
				" upper(tablename)=upper('" + table_name + "')";
		ResultSet result = statement.executeQuery(sql.toString());
		boolean is_exist = false;
		

		if (result.next()) {
		    // 1だったら存在
		    is_exist = result.getInt("num") > 0;
		}

		result.close();
		return is_exist;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#createTable(java.lang.String, jp.co.powerbeans.powerql.ColumnProperty[])
     */
    public void createTable(String table_name, ColumnProperty[] cp)
            throws POQLException, SQLException {

		if (cp == null || cp.length == 0) {
			throw new POQLException("ColumnProperty value is invalid:" + cp);
		}

		StringBuffer sql = new StringBuffer("CREATE TABLE " + table_name + "(");

		// column
		for (int i = 0; i < cp.length; i++) {
			sql.append(getJava2DBType(cp[i]) + ", ");
		}

		// pk
		sql.append(
				" primary key ("
				+ cp[0].getName()
				+ ") )");

		statement.executeUpdate(sql.toString());
    }

	/**
	 * getJava2DBType<BR>
	 * ColumnProperty から CREATE SQL 用カラム定義を作成
	 * @param property
	 * @return CREATE SQL 用カラム定義を作成
	 */
	private String getJava2DBType(ColumnProperty cp) throws POQLException {

		StringBuffer sql = new StringBuffer(cp.getName());
		if (!java2dbTypeMap.containsKey(cp.getType())) {
			// JAVAの型が設定されていない
			throw new POQLException(
				"cannot get type:"
					+ cp.getType()
					+ "(java 2 db) ["
					+ DB_TYPE
					+ "]");
		}

		sql.append(" " + (String) java2dbTypeMap.get(cp.getType()));

		if (cp.getSize() > 0) {
			sql.append("(" + cp.getSize() + ")");
		}

		if (!cp.isNullable()) {
			sql.append(" not null");
		}

		if (cp.getDefaultValue() != null
			&& cp.getDefaultValue().toString().length() > 0) {
			sql.append(" DEFAULT " + cp.getDefaultValue());
		}

		sql.append(" ");
		return sql.toString();
	}

	  /**
	   * getSelectSQL<BR>
	   * パラメータの条件からSQLを作成。
	   * PostgreSQLはgroup by がある場合はgroup by で指定したフィールドだけ SELECT 句に指定して取得。
	   * @param tableName テーブル名
	   * @param bpList BeanPropertyList
	   * @param where where句
	   * @param groupby groupby句
	   * @param orderby orderby句
	   * @return select SQL
	   */
	  public String getSelectSQL(String tableName, ArrayList bpList, String where, String groupby, String orderby) {

	    StringBuffer sql = null;
	    
	    // group by がある場合はSELECT句で利用
	    if (groupby != null && groupby.trim().length() > 0) {
	      sql = new StringBuffer("SELECT " + groupby + " FROM " + tableName);
	      
	    } else {
	      sql = new StringBuffer("SELECT * FROM " + tableName);
	    }
	    
	    if (where != null && where.trim().length() > 0) {
	      sql.append(" WHERE " + where.trim());
	    }
	    
	    if (groupby != null && groupby.trim().length() > 0) {
	      sql.append(" GROUP BY " + groupby.trim());
	    }
	    
	    if (orderby != null && orderby.trim().length() > 0) {
	      sql.append(" ORDER BY " + orderby.trim());
	    }
	    return sql.toString();
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW()
     */
    public String getSQL_NOW() {
        return " NOW()";
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW(java.lang.String)
     */
    public String getSQL_NOW(String column_type) {
        return getSQL_NOW();
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#formatDateVal(java.util.Date, java.lang.String)
     */
    public String formatDateVal(Date date, String column_type) {
        return formatDateVal_std(date, column_type);
    }

//    /* (non-Javadoc)
//     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getBlobClass()
//     */
//    public Class getBlobClass() {
//        return super.getBlobClass("org.postgresql.jdbc3.Jdbc3Blob");
//    }
    
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableAllSql()
	 */
	public String getSysSelectTableAllSql() {
		throw new RuntimeException("not support getSysSelectTableAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectSchemaAllSql()
	 */
	public String getSysSelectSchemaAllSql() {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}
	

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableBySchemaSql(java.lang.String)
	 */
	public String getSysSelectTableBySchemaSql(String schema) {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getGeneratedKey(jp.co.powerbeans.powerql.POQLPreparedStatement)
	 */
	public Object getGeneratedKey(POQLPreparedStatement st) throws SQLException {
		// ダミー実装.
		ResultSet key_rset = st.getGeneratedKeys();
		if (key_rset == null) {
			return Integer.valueOf(0);
		}
		return key_rset.getInt(1);
	}

}
