package jp.co.powerbeans.powerql.vendor;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;

import jp.co.powerbeans.powerql.ColumnProperty;
import jp.co.powerbeans.powerql.Log;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.exceptions.POQLException;

import org.apache.oro.text.perl.Perl5Util;

/**
 * <p>タイトル: DBDependQLStatementDB2</p>
 * <p>説明: DBDependQLStatement の DB2実装。</p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * 
 * <p>Created on 2003/10/14</p>
 * @author 門田明彦
 * @version $Revision: 1.4 $
 */
public class DBDependQLStatementDB2 extends DBDependQLStatementSupport {


	private static final String DB_TYPE = "DB2";
	
	/** regexp lib */
	private static Perl5Util perl = new Perl5Util();

	/** type mapping */
	private static Hashtable db2javaTypeMap = new Hashtable();

	/** type mapping */
	private static Hashtable java2dbTypeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] db2javaMap = { { "TINYINT", Short.class }, {
			"TINYINT\\([0-9]+\\)", Short.class }, {
			"SMALLINT", Short.class }, {
			"SMALLINT\\([0-9]+\\)", Short.class }, {
			"MEDIUMINT", Integer.class }, {
			"MEDIUMINT\\([0-9]+\\)", Integer.class }, {
			"INT", Integer.class }, {
			"INT\\([0-9]+\\)", Integer.class }, {
			"INTEGER", Integer.class }, {
			"INTEGER\\([0-9]+\\)", Integer.class }, {
			"BIGINT", Long.class }, {
			"BIGINT\\([0-9]+\\)", Long.class }, {

			"NUMERIC\\([0-9]+,[0-9]+\\)", Double.class }, {
			"DECIMAL\\([0-9]+,[0-9]+\\)", Double.class }, {
			"REAL\\([0-9]+,[0-9]+\\)", Double.class }, {
			"DOUBLE PRECISION\\([0-9]+,[0-9]+\\)", Double.class }, {

			"DATE", Timestamp.class }, {
			"DATETIME", Timestamp.class }, {
			"TIMESTAMP", Timestamp.class }, {
			"TIMESTAMP\\([0-9]+\\)", Timestamp.class }, {
			"TIME", Timestamp.class }, {

			"TINYBLOB", String.class }, {
			"TINYTEXT", String.class }, {
			"BLOB", String.class }, {
			"TEXT", String.class }, {
			"MEDIUMBLOB", String.class }, {
			"MEDIUMTEXT", String.class }, {
			"LONGBLOB", String.class }, {
			"LONGTEXT", String.class }, {

			"CHAR", String.class }, {
			"VARCHAR", String.class }, {
			"CHAR\\([0-9]+\\)", String.class }, {
			"VARCHAR\\([0-9]+\\)", String.class }, };

	/** type mapping data key */
	private static Object[][] java2dbMap = { { Byte.class, "TINYINT" }, {
			Short.class, "TINYINT" }, {
			Character.class, "TINYINT" }, {
			Integer.class, "INT" }, {
			Long.class, "BIGINT" }, {
			Double.class, "DOUBLE PRECISION" }, {

			String.class, "VARCHAR" }, {
			Timestamp.class, "DATE" }
	};
	
	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < db2javaMap.length; i++) {
			db2javaTypeMap.put(db2javaMap[i][0], db2javaMap[i][1]);
		}

		// typeMap.put(Java type, MySQL type)
		for (int i = 0; i < java2dbMap.length; i++) {
			java2dbTypeMap.put(java2dbMap[i][0], java2dbMap[i][1]);
		}
	}
	/** Oracld > JDBC type mapping */
	private static Hashtable typeMap = new Hashtable();

	/** type mapping data key */
	private static Object[][] typeArray = {
            { "NUMBER\\([0-9]+\\)", Integer.class },
            { "NUMBER\\([0-9]+,[0-9]+\\)", Double.class },
            { "CHAR\\([0-9]+\\)", String.class },
            { "VARCHAR2\\([0-9]+\\)", String.class },
            { "NUMBER", Integer.class }, { "DATE", Timestamp.class },
            { "CHAR", String.class }, { "VARCHAR2", String.class },
            { "VARCHAR", String.class },
            { "VARCHAR2\\([0-9]+\\)", String.class },
            { "CHARACTER", String.class}, // DB2 Original
            { "TIME", Timestamp.class}, 
            { "TIMESTAMP", Timestamp.class}, 
            { "SMALLINT", Short.class}, 
            { "INTEGER", Integer.class}, // DB2 Original
            { "BIGINT", Long.class}, // DB2 Original
            { "DOUBLE", Double.class }, // DB2 Original
            { "REAL", Float.class },
            { "DECIMAL", BigDecimal.class },
            { "LONG VARCHAR", String.class },
            { "BLOB", String.class },
            { "CLOB", String.class },
            { "DBCLOB", String.class },
            { "GRAPHIC", String.class },
            { "VARGRAPHIC", String.class },
            { "LONG VARGRAPHIC", String.class },
            { "DATALINK", String.class },
	};
	
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getColumnPropertyList(java.lang.String)
     */
    public ArrayList getColumnPropertyList(String table, String schema) throws SQLException,
            POQLException {
        ArrayList list = new ArrayList(10);
        HashMap colMap = new HashMap(10);
        
        // 1 テーブルのカラム情報取得
        String sql =
         "SELECT " +
           " TABNAME, " +
           " COLNAME column_name, " +
           " TYPENAME data_type, " +
           " LENGTH data_length, " +
           " NULLS nullable, " +
           " DEFAULT data_default, " +
           " GENERATED" +
           " FROM " +
           " SYSCAT.COLUMNS " +
           " WHERE " +
           " TABSCHEMA = UPPER('" + schema + "') AND " + 
           " TABNAME = UPPER('" + table + "') " +
           " ORDER BY COLNO FOR READ ONLY";
        
        ResultSet result = statement.executeQuery(sql.toString());
        
        while (result.next()) {
          // 各カラム情報を取得
          // ここではまだプライマリキー情報は取得しない
          ColumnProperty cp = new ColumnProperty();
          cp.setName(result.getString("column_name"));
          cp.setTypeAsDB(result.getString("data_type"));
          cp.setNullable(result.getString("nullable").equals("Y"));
          cp.setDefaultValue(result.getObject("data_default"));
          cp.setType(getJavaTypeFromDBType(cp.getTypeAsDB()));
          cp.setForceAutoValue(result.getString("GENERATED").equals("A"));
          colMap.put(cp.getName().toUpperCase(), cp);
          list.add(cp);
          Log.debug("add CPList " + cp.getName());
        }
        result.close();

        // 2. テーブルのプライマリキーカラム取得
        result = statement.executeQuery(
                "select constname, tabname, colname, colseq from syscat.keycoluse " +
                " WHERE TABNAME = UPPER('" + table + "')");
        
        while(result.next()) {
            // プライマリキーの数だけﾙｰﾌﾟ
            String colname = result.getString("colname").toUpperCase();
            if (colMap.containsKey(colname)) {
                // カラム名が一致したのでプライマリキー設定
                ColumnProperty cp = (ColumnProperty)colMap.get(colname);
                cp.setPrimaryKey(true);
            }
        }
        // POINT DB2 SQL
//        insert into MstPref VALUES('aa',DATE('2001-01-01'));
//
//        select COLNAMES from SYSCAT.INDEXES 
//        where INDSCHEMA = 'SYSIBM' AND TABNAME = 'MSTCITY';
//
//        select constname, tabname, colname, colseq from syscat.keycoluse
//         WHERE TABNAME = 'MSTPREF'
//
//        select * from syscat.tabconst WHERE TABNAME Like 'MSTCITY%'
//        select COLNAME from SYSCAT.INDEXCOLUSE WHERE TABNAME Like 'MSTPREF%'


        return list;
    }

	/**
	 * getJavaTypeFromDBType<BR>
	 * DB(Oracle)のカラムの型からJAVAの型をClassで取得
	 * @param type_as_DB DB(Oracle)の型
	 * @return JAVAの型
	 */
	private Class getJavaTypeFromDBType(String type_as_DB)
		throws POQLException {

		String upper = type_as_DB.toUpperCase();

		if (typeMap.containsKey(upper)) {
			// 完全一致
			return (Class) typeMap.get(upper);

		} else {
			// 全パターンと正規表現比較
			for (int i = 0; i < typeArray.length; i++) {
				if (perl.match(upper, (String) typeArray[i][0])) {
					return (Class) typeArray[i][1];
				}
			}

		}

		// 一致する型無し
		throw new POQLException(
			"not recoganize db TYPE(" + DB_TYPE + ") " + type_as_DB);
	}
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#isExistTable(java.lang.String)
     */
    public boolean isExistTable(String table_name) throws SQLException {
        String sql = "select count(*) num from SYSCAT.TABLES where UPPER(TABNAME) = UPPER('" + table_name + "') " +
                " FOR READ ONLY";
        ResultSet result = null;
        try {
            result = statement.executeQuery(sql.toString());
            // 検索が成功したら存在する
            return true;
        } catch (SQLException e) {
            return false;
        } finally {
            if (result != null) {
                result.close();
            }
        }
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#createTable(java.lang.String, jp.co.powerbeans.powerql.ColumnProperty[])
     */
    public void createTable(String table_name, ColumnProperty[] cp)
            throws POQLException, SQLException {

		if (cp == null || cp.length == 0) {
			throw new POQLException("ColumnProperty value is invalid:" + cp);
		}

		StringBuffer sql = new StringBuffer("CREATE TABLE " + table_name + "(");

		// column
		for (int i = 0; i < cp.length; i++) {
			sql.append(getJava2DBType(cp[i]) + ", ");
		}

		// pk
		sql.append(
				" primary key ("
				+ cp[0].getName()
				+ ") )");

		statement.executeUpdate(sql.toString());
    }

	/**
	 * getJava2DBType<BR>
	 * ColumnProperty から CREATE SQL 用カラム定義を作成
	 * @param property
	 * @return CREATE SQL 用カラム定義を作成
	 */
	private String getJava2DBType(ColumnProperty cp) throws POQLException {

		StringBuffer sql = new StringBuffer(cp.getName());
		if (!java2dbTypeMap.containsKey(cp.getType())) {
			// JAVAの型が設定されていない
			throw new POQLException(
				"cannot get type:"
					+ cp.getType()
					+ "(java 2 db) ["
					+ DB_TYPE
					+ "]");
		}

		sql.append(" " + (String) java2dbTypeMap.get(cp.getType()));

		if (cp.getSize() > 0) {
			sql.append("(" + cp.getSize() + ")");
		}

		if (!cp.isNullable()) {
			sql.append(" not null");
		}

		if (cp.getDefaultValue() != null
			&& cp.getDefaultValue().toString().length() > 0) {
			sql.append(" DEFAULT " + cp.getDefaultValue());
		}

		sql.append(" ");
		return sql.toString();
	}

	  /**
	   * getSelectSQL<BR>
	   * パラメータの条件からSQLを作成。
	   * DB2はgroup by がある場合はgroup by で指定したフィールドだけ SELECT 句に指定して取得。
	   * @param tableName テーブル名
	   * @param bpList BeanPropertyList
	   * @param where where句
	   * @param groupby groupby句
	   * @param orderby orderby句
	   * @return select SQL
	   */
	  public String getSelectSQL(String tableName, ArrayList bpList, String where, String groupby, String orderby) {

	    StringBuffer sql = null;
	    
	    // group by がある場合はSELECT句で利用
	    if (groupby != null && groupby.trim().length() > 0) {
	      sql = new StringBuffer("SELECT " + groupby + " FROM " + tableName);
	      
	    } else {
	      sql = new StringBuffer("SELECT " + getSelectFields(bpList) + " FROM " + tableName);
	    }
	    
	    if (where != null && where.trim().length() > 0) {
	      sql.append(" WHERE " + where.trim());
	    }
	    
	    if (groupby != null && groupby.trim().length() > 0) {
	      sql.append(" GROUP BY " + groupby.trim());
	    }
	    
	    if (orderby != null && orderby.trim().length() > 0) {
	      sql.append(" ORDER BY " + orderby.trim());
	    }
	    return sql.toString();
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW()
     */
    public String getSQL_NOW() {
        return " CURRENT TIMESTAMP";
        // 実際はフィールドの型によって異なるため NotSupportMethodException
        // をスローしてもいいかも
        // DATE型 < CURRNET DATE
        // TIME型 < CURRNET TIME
        // TIMESTAMP型 < CURRNET TIMESTAMP
        
        // ※DB2では getSQL_NOW() ではなく getSQL_NOW(String) を利用すること
    }


	
	static {
		// typeMap.put(Oracle Type (regex), Java type)
		for (int i = 0; i < typeArray.length; i++) {
			typeMap.put(typeArray[i][0], typeArray[i][1]);
		}
	}

	// getSQL_NOW で返す値
    private static HashMap nowColtype = new HashMap();
    static {
        nowColtype.put("DATE", " CURRENT DATE ");
        nowColtype.put("date", " CURRENT DATE ");
        nowColtype.put("TIME", " CURRENT TIME ");
        nowColtype.put("time", " CURRENT TIME ");
        nowColtype.put("TIMESTAMP", " CURRENT TIMESTAMP ");
        nowColtype.put("timestamp", " CURRENT TIMESTAMP ");
    }
    
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSQL_NOW(java.lang.String)
     */
    public String getSQL_NOW(String column_type) {
        if (nowColtype.containsKey(column_type)) {
            return (String)nowColtype.get(column_type);
        }
        throw new IllegalArgumentException("time column type:" + column_type + " not support.");

        // DATE型 < CURRNET DATE
        // TIME型 < CURRNET TIME
        // TIMESTAMP型 < CURRNET TIMESTAMP
        
    }

    /* (non-Javadoc)
     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#formatDateVal(java.util.Date, java.lang.String)
     */
    public String formatDateVal(Date date, String column_type) {
        return formatDateVal_std(date, column_type);
    }

//    /* (non-Javadoc)
//     * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getBlobClass()
//     */
//    public Class getBlobClass()  {
//        return super.getBlobClass("COM.ibm.db2.jdbc.app.DB2Blob");
//    }
    
	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableAllSql()
	 */
	public String getSysSelectTableAllSql() {
		throw new RuntimeException("not support getSysSelectTableAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectSchemaAllSql()
	 */
	public String getSysSelectSchemaAllSql() {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getSysSelectTableBySchemaSql(java.lang.String)
	 */
	public String getSysSelectTableBySchemaSql(String schema) {
		throw new RuntimeException("not support getSysSelectSchemaAllSql()");
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.vendor.DBDependQLStatement#getGeneratedKey(jp.co.powerbeans.powerql.POQLPreparedStatement)
	 */
	public Object getGeneratedKey(POQLPreparedStatement st) throws SQLException {
		// ダミー実装.
		ResultSet key_rset = st.getGeneratedKeys();
		if (key_rset == null) {
			return Integer.valueOf(0);
		}
		return key_rset.getInt(1);
	}

}
