package jp.co.powerbeans.powerql.dao;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import jp.co.powerbeans.powerql.BeanProperty;
import jp.co.powerbeans.powerql.CPCache;
import jp.co.powerbeans.powerql.ColumnProperty;
import jp.co.powerbeans.powerql.Count;
import jp.co.powerbeans.powerql.POQLManager;
import jp.co.powerbeans.powerql.POQLTransaction;
import jp.co.powerbeans.powerql.POQLUtil;
import jp.co.powerbeans.powerql.POQLViewPreparedStatement;
import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;

/**
 * POQLDynaViewDAO基底クラス。 <BR>
 * 独自のPOQLDynaViewDAOを実装する場合はこのクラスを継承する。
 * 
 * <p>
 * 著作権: 株式会社パワービーンズ
 * </p>
 * <p>
 * 会社名: 株式会社パワービーンズ
 * </p>
 * 
 * @author A.Monden
 */
public abstract class POQLDynaViewBaseDAO  extends CoreDAO implements POQLDynaViewDAO {

    private Class[] tableBeans;

    private String joinSql;

    private String dviewSql;
    
    private String countSql;
    
    /**
     * 動的VIEW用コンストラクタ
     * 
     * @param manager
     *            マネージャ
     * @param c
     *            検索結果格納JavaBean
     * @param table_beans
     *            結合テーブル型
     * @param join_sql
     *            結合SQL
     */
    public POQLDynaViewBaseDAO(POQLManager manager, Class c,
            Class[] table_beans, String join_sql) {

        super(manager, c);
        
        this.tableBeans = table_beans;
        this.joinSql = join_sql;
        if (joinSql == null) {
            joinSql = "";
        }

        this.dviewSql = "";

        // 結合SQLを作成
        StringBuffer buf = new StringBuffer(); // 検索用
        StringBuffer nbuf = new StringBuffer(); // 件数取得用

        if (c == null || table_beans == null || table_beans.length == 0) {
            throw new IllegalArgumentException("invalid param: manager="
                    + manager + ", bean_class=" + c + ", table_beans="
                    + table_beans);
        }

        POQLTransaction bqlTrn = null;
        try {
            // 1. table_beans からテーブルの定義を取得
            bqlTrn = manager.getPowerQLTransaction();
            bqlTrn.setRtrim(isRtrim());
//          // ベンダー依存Statementを取得
            DBDependQLStatement dbst = DBDependQLStatementFactory.create(bqlTrn
                    .getConnection(), manager);

            HashMap cp_map = new HashMap(20);
            HashMap cp_table_map = new HashMap(20);
            
            // テーブル名配列 
            // AMS TODO 同名テーブルを別名で使用するSQLに対応していない
            // 対応する場合は、FROM MstPref MstPref1,MstPref MstPref2 ... とする
            String[] table_names = new String[table_beans.length];

            for (int i = 0; i < table_beans.length; i++) {
                // テーブル名配列作成
                table_names[i] = POQLUtil.className2TableName(table_beans[i]);
                
                // テーブル定義取得(DB or Cache)
                ArrayList cp_list = CPCache.getColumnPropertyList(
                        bqlTrn.getConnection(),
                        manager, table_names[i]);
                        
//                ArrayList cp_list = dbst.getColumnPropertyList(table_names[i]);

                // 全テーブル共通のカラム名マップに格納
                // もし既に同名のカラムが存在する場合は格納しない
                for (Iterator it = cp_list.iterator(); it.hasNext();) {
                    ColumnProperty cp = (ColumnProperty) it.next();
                    if (!cp_map.containsKey(cp.getName())) {
                        cp_map.put(cp.getName().toUpperCase(), cp);
                        cp_table_map
                                .put(cp.getName().toUpperCase(), table_names[i]);
                    }
                }
            }

            // 2. SELECT 句の元となる BeanProperty リストを作成
            ArrayList bp_list = new ArrayList(20);
            BeanInfo info = Introspector.getBeanInfo(c);
            PropertyDescriptor[] properties = info.getPropertyDescriptors();

            for (int i = 0; i < properties.length; i++) {
                // プロパティ名とカラム名から一致するカラムを探し
                // 一致したらBeanPropertyを生成してbp_listに追加
                PropertyDescriptor p = properties[i];
                if (!cp_map.containsKey(p.getName().toUpperCase())) {
                    // 存在しない
                    continue;
                }

                ColumnProperty cp = (ColumnProperty) cp_map.get(p.getName().toUpperCase());

                // 存在したので生成
                Class type = p.getPropertyType();
                String name = p.getName();
                Method m_get = p.getReadMethod();
                Method m_set = p.getWriteMethod();
                // setter,getter,名前チェック
                if (m_set == null
                        || m_get == null
                        || !name.toUpperCase().equals(
                                cp.getName().toUpperCase())) {
                    continue;
                }

                // 一致したので追加
                BeanProperty bp = new BeanProperty(type, name, m_get, m_set);
                bp.setColumnProp(cp);
                bp.setColumnPropEscapedName(dbst.escape(cp.getName()));
                bp_list.add(bp);
            }

            // 3. select句作成
            buf.append("SELECT ");
            nbuf.append(" SELECT COUNT(*) AS NUM ");
            int fc = 0;
            for (Iterator it = bp_list.iterator(); it.hasNext();) {
                BeanProperty bp = (BeanProperty) it.next();
                if (fc++ > 0) {
                    buf.append(",");
                }
                buf.append(cp_table_map.get(bp.getColumnProp().getName().toUpperCase()) + "."
                        + bp.getColumnProp().getName());
            }

            // 4. from句作成
            buf.append(" FROM ");
            nbuf.append(" FROM ");
            fc = 0;
            for (int i = 0; i < table_beans.length; i++) {
                if (fc++ > 0) {
                    buf.append(",");
                    nbuf.append(",");
                }
                buf.append(table_names[i]);
                nbuf.append(table_names[i]);
            }

            // 5. 結合部分のWHEREは
            if (joinSql.trim().length() > 0 ) {
                buf.append(" WHERE " + join_sql);
                nbuf.append(" WHERE " + join_sql);
            }

            dviewSql = buf.toString();
            countSql = nbuf.toString();

        } catch (Exception e) {
            // onException(e);
           // コンストラクタは必ずエラーログ出力
            e.printStackTrace();
        } finally {
            if (bqlTrn != null) {
                try {
                    bqlTrn.close();
                } catch (POQLException e1) {
                    onException(e1);
                }
            }
        }
    }

    /**
     * SQL指定動的VIEW用コンストラクタ
     * @param manager マネージャ
     * @param c 検索結果格納JavaBean
     * @param sql 実行SQL
     */
    public POQLDynaViewBaseDAO(POQLManager manager, Class c, String sql) {
        
        super(manager, c);
        this.joinSql = "";
        this.dviewSql = sql;
    }
    
    /**
     * 静的VIEW用コンストラクタ
     * @param manager
     *            マネージャ
     * @param c
     *            検索結果格納JavaBean
     */ 
    public POQLDynaViewBaseDAO(POQLManager manager, Class c) { 
        
        super(manager, c);
        this.joinSql = "";
        this.dviewSql = "SELECT * FROM " + POQLUtil.className2TableName(c) + " ";
    }


    /**
     * @return @throws
     *         POQLException
     */
    protected POQLTransaction getPowerQLTransaction() throws POQLException {

        return getManager().getPowerQLTransaction();
    }

    /**
     * @return dviewSql
     */
    public String getDviewSql() {
        return dviewSql;
    }

    /**
     * @return joinSql
     */
    public String getJoinSql() {
        return joinSql;
    }

    /**
     * @return tableBeans
     */
    public Class[] getTableBeans() {
        return tableBeans;
    }

    public int countByAll() {
        return countBy(null);
    }

    public int countBy(String where) {
        POQLTransaction bqlTrn = null;
        int count = 0;
        StringBuffer sql = new StringBuffer(countSql);
        
        if (where != null && where.trim().length() > 0) {
            if (joinSql != null && joinSql.trim().length() > 0) {
                sql.append(" AND " + where);
            } else {
                sql.append(" WHERE " + where);
            }
        }
    
        try {
            bqlTrn = getPowerQLTransaction();
            POQLViewPreparedStatement st = bqlTrn.createViewPreparedStatement(
                    Count.class, sql.toString());
            Count ct = (Count) st.selectOne();
            count = ct.getNum();
    
        } catch (Exception e) {
            onException(e);
        } finally {
            close(bqlTrn);
        }
        
        return count;
    }
}