package jp.co.powerbeans.powerql.dao;

import java.sql.SQLException;
import java.util.HashSet;
import java.util.Set;

import jp.co.powerbeans.powerql.POQLManager;
import jp.co.powerbeans.powerql.POQLPreparedStatement;
import jp.co.powerbeans.powerql.POQLStatementIF;
import jp.co.powerbeans.powerql.POQLTransaction;
import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * POQLDAO実装クラスの抽象クラス。<BR>
 * 独自のPOQLDAOインターフェースを実装する場合は
 * このクラスを継承して実装する。<BR>
 *
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * @author A.Monden
 */
public abstract class POQLBaseDAO extends CoreDAO implements POQLDAO {

    /** 動作中のトランザクション */
    private POQLTransaction bqlTrn;
    
    /** 排他チェックフィールド名 */
    private String exclusiveCheckField;

    /** 1メソッドでトランザクションを完結するかどうか, true:完結する, false 完結しない */
    private boolean singleCallMethod;
    
    /** DAOで最後に実行したSQL */
    protected String lastSQL;

    /** DAOで最後に実行したSQLのbinds値 */
    protected Object[] lastBindValues;

    /** 自動で現在日付を格納するタイムスタンプフィールド名 */
    protected String[] autoTimestampColName;

    /** JTA等の外部トランザクション管理モードフラグ, true JTA利用, false JTA非利用(default) */
    private boolean useJTATransaction;

    /**
     * コンストラクタ
     * @param manager
     */
    public POQLBaseDAO(POQLManager manager, Class bean_class) {
        super(manager, bean_class);
        
        singleCallMethod = true;
        setReturnNull(true);
        lastSQL = "";
        lastBindValues = new Object[]{};
        useJTATransaction = false;
    }
    
    /**
     * PowerQLTransactionを生成して返す。
     * 複数メソッド呼び出しモードの場合は既に作成済みの
     * トランザクションを返す。
     * @return POQLTransaction
     * @throws POQLException
     */
    protected POQLTransaction getPowerQLTransaction() throws POQLException {
        
        // 1. JTA等の外部トランザクション管理モードの場合
        if (useJTATransaction) {
//            if (bqlTrn == null) {
            // 基本的に毎回生成する（実際のコネクション管理はJTAに任せる)
            bqlTrn = getManager().getPowerQLTransaction();
            // オートコミットはOFF
            bqlTrn.setAutoCommit(false);
//            }
        } else {
        
            // 2. JTA以外(PowerQL単体)の場合,,
            
            if (singleCallMethod) {
                // 単一メソッドモードの場合
                // もし既に存在したらクローズ
                close(bqlTrn);
                // 必ず新規作成する
                bqlTrn = getManager().getPowerQLTransaction();
                // オートコミットはON
                bqlTrn.setAutoCommit(true);

            } else {
                // 複数呼び出しモードの場合は存在していない場合だけ生成する
                if (bqlTrn == null) {
                    bqlTrn = getManager().getPowerQLTransaction();
                    // オートコミットはOFF
                    bqlTrn.setAutoCommit(false);
                }
            }
        }
        bqlTrn.setRtrim(isRtrim());
        bqlTrn.setAutoTimestampColName(this.autoTimestampColName);
        return bqlTrn;
    }
    
    /* (Javadoc なし)
     * @see java.lang.Object#finalize()
     */
    protected void finalize() throws Throwable {
        close(bqlTrn);
        super.finalize();
    }

    /**
     * 排他チェックフィールド名を取得
     * @return 排他チェックフィールド名
     */
    public String getExclusiveCheckField() {
        return exclusiveCheckField;
    }

    /**
     * トランザクションをコミットする。
     * 通常ビジネスロジックから複数メソッド呼び出しモードで実行される。
     * コネクションは保持しつづける。
     */
    public void commit() {
        if (bqlTrn != null) {
	        try {
	            if (bqlTrn.getConnection() != null &&
	                !bqlTrn.getConnection().isClosed()) {
			            bqlTrn.commit();
	            }
	        } catch (Exception e) {
	            onException(e);
//            } finally {
//                // トランザクションを利用しないので参照を解放
//                bqlTrn = null;
            }
        }
    }
    /**
     * 単一メソッド呼び出しモードの場合はコミットする。
     * それ以外の場合(JTAトランザクション有効の場合)はコミットしない。
     */
    protected void commitSafe() {
        
        if (useJTATransaction) {
            return;
        }
        if (singleCallMethod && bqlTrn != null) {
	        try {
	            if (bqlTrn.getConnection() != null &&
	                !bqlTrn.getConnection().isClosed()) {
			            bqlTrn.commit();
	            }
	        } catch (Exception e) {
	            onException(e);
//            } finally {
//                // トランザクションを利用しないので参照を解放
//                bqlTrn = null;
            }
        }
    }
    
    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#setSingleCallMethod(boolean)
     */
    public void setSingleCallMethod(boolean singleCallMethod) {
        if (bqlTrn != null) {
            throw new IllegalStateException("PowerQL can't change singleCallMethod while Transaction exist.");
        }
        this.singleCallMethod = singleCallMethod;
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#close()
     */
    public void close() {
        close(bqlTrn);
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#commitAndClose()
     */
    public void commitAndClose() {
        commit();
        close();
    }

    /* (Javadoc なし)
     * @see jp.co.powerbeans.powerql.dao.POQLDAO#rollback()
     */
    public void rollback() {
        if (bqlTrn != null) {
	        try {
	            if (bqlTrn.getConnection() != null &&
	                !bqlTrn.getConnection().isClosed()) {
			            bqlTrn.rollback();
	            }
	        } catch (Exception e) {
	            onException(e);
            } finally {
                // トランザクションを利用しないので参照を解放
                bqlTrn = null;
            }
        }
   }

//    /**
//     * 指定したPowerQLトランザクションをロールバックする
//     * @param bqlTrn2 POQLTransaction
//     */
//    protected void rollback(POQLTransaction bqlTrn2) {
//        if (singleCallMethod) {
//	        try {
//	            if (bqlTrn2 != null && bqlTrn2.getConnection() != null &&
//	                    !bqlTrn2.getConnection().isClosed()) {
//	                bqlTrn2.rollback();
//	            }
//	        } catch (Exception e) {
//	            onException(e);
//	        } finally {
//	            // トランザクションを利用しないので参照を解放
//	            // bql
//	        }
//        }
//    }
    /**
     * @param exclusiveCheckField exclusiveCheckField を設定します。
     */
    public void setExclusiveCheckField(String exclusiveCheckField) {
        this.exclusiveCheckField = exclusiveCheckField;
    }

    /**
     * @return singleCallMethod
     */
    public boolean isSingleCallMethod() {
        return singleCallMethod;
    }

    /**
     * 複数メソッド呼び出しモード(複数メソッドで1トランザクションの場合)はコネクションをクローズしない。
     * JTAを利用している場合、その他の場合はコネクションをクローズする。
     */
    protected void closeSafe() {
        if (useJTATransaction) {
//        	return;
             close(); // コネクション管理はJTAManagerに任せるので、アプリ側は何も気にせずクローズする
        }
        else if (singleCallMethod) {
            // 1メソッドでトランザクションが完結する場合はクローズ
            close();
        }
    }
    /**
     * @return lastSQL を戻します。
     */
    public String getLastSQL() {
        return lastSQL;
    }
    /**
     * @return lastBindValues を戻します。
     */
    public Object[] getLastBindValues() {
        return lastBindValues;
    }
    /**
     * @return useJTATransaction を戻します。
     */
    public boolean isUseJTATransaction() {
        return useJTATransaction;
    }
    /**
     * @param useJTATransaction useJTATransaction を設定。
     */
    public void setUseJTATransaction(boolean useJTATransaction) {
        this.useJTATransaction = useJTATransaction;
    }

	/**
	 * POQLStatementが存在する場合、最後に実行したSQLを返す。
	 * 
	 * @param st SQLを実行したPOQLStatement
	 */
	protected void setLastSqlSafe(POQLStatementIF st) {
	    if (st != null) {
	        lastSQL = st.getLastSQL();
	        lastBindValues = st.getLastBindValues();
	    }
	}

	/**
	 * JTAモードであればロールバックしない.
	 * それ以外のモードであればロールバックする
	 */
	protected void onRollback() {
	    if (!isUseJTATransaction()) {
	        rollback();
	    }
	}
	

	/**
	 * @param bqlTrn
	 * @param st
	 * @return
	 * @throws POQLException
	 * @throws SQLException
	 */
	protected Object getGeneratedKey(POQLTransaction bqlTrn,
			POQLPreparedStatement st) throws POQLException, SQLException {
		DBDependQLStatement dbst = DBDependQLStatementFactory.create(bqlTrn
		        .getConnection(), this.getManager());
		Object key = dbst.getGeneratedKey(st);
		return key;
	}

	public void setAutoTimestampColName(String[] auto_timestamp_colname) {
	    this.autoTimestampColName = auto_timestamp_colname;
	}

	/**
	 * カラム名配列からSetを作成
	 * @param columns
	 * @return カラム名Set
	 */
	protected Set createColSet(String[] columns) {
	    if (columns == null) {
	        return new HashSet();
	    }
	    Set set = new HashSet(columns.length);
	    for (int i = 0; i < columns.length; i++) {
	        if (columns[i] != null && columns[i].length() > 0) {
	            set.add(columns[i].toUpperCase());
	        }
	    }
	    return set;
	}

}
