package jp.co.powerbeans.powerql;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.util.StringUtl;


/**
 * <p>プロジェクト名: PowerQL</p>
 * <p>タイトル: POQLViewStatementSupport</p>
 * <p>説明:
 * POQLViewStatement 共通処理実装抽象クラス。<BR>
 *  </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * @version $Revision: 1.3 $
 */
public abstract class POQLViewStatementSupport implements POQLStatementIF {

  /** ResultSet から値取得用メソッド取得用パラメータ */
  protected static final Class[] RS_METHOD_PARAM = { String.class };

  /** BeanClass */
  protected Class bean;

  /** このBeanが利用するMapping(BeanProperty の ArrayList) */
  protected ArrayList bpList;

  /** All Common bean mapping map (BeanProperty の ArrayListを格納) */
  protected static Map beanMap = new HashMap();

  /** rtrim フラグ */
  private boolean rtrim;
  
  /**
  	 * createBeanFieldListMap<BR>
  	 * 渡されたViewBeanのフィールドLマッピングを作成<BR>
  	 * ※DBにはアクセスしない
  	 * 
  	 */
  protected void createBeanFieldListMap() throws POQLException {
  	if (beanMap.containsKey(bean.getName())) {
  		// マッピング作成済み
  		bpList = (ArrayList) beanMap.get(bean.getName());
  		return;
  	}
  
  	try {
  		// カラム情報リストの順で BeanPropertyのリストを作成していく
  		ArrayList bp_list = new ArrayList();
  		BeanInfo info = Introspector.getBeanInfo(bean);
  		PropertyDescriptor[] p = info.getPropertyDescriptors();
  		// 簡易Map作成
  		for (int i = 0; i < p.length; i++) {
  
  			Class type = p[i].getPropertyType();
  			String name = p[i].getName();
  			Method m_get = p[i].getReadMethod();
  			Method m_set = p[i].getWriteMethod();
  			if (m_set == null
  				|| m_get == null) {
  					
  				continue;
  			}
  			
  			BeanProperty bp = new BeanProperty(type, name, m_get, m_set);
  			// bp.setColumnProp(null); DBからは情報を取得しないので設定なし
//  			bp.setColumnPropEscapedName(column_property_escaped_name);
  
  			bp_list.add(bp);
  		}
  
  		beanMap.put(bean.getName(), bp_list);
  		bpList = bp_list;
  	} catch (IntrospectionException e) {
  		throw new POQLException(e);
  	}
  	
  }

  /**
  	 * getBeanByResultSet<BR>
  	 * 検索結果の現在のカーソルの行について
  	 * Mappingを利用して
  	 * @param result
  	 * @return 取得したbean
  	 */
  protected Object getBeanByResultSet(ResultSet result) throws POQLException {
  
  	try {
        // metadata から取得可能なカラム名Mapを作成
        ResultSetMetaData metadata =  result.getMetaData();
        Set metacol_map = new HashSet();
        int num = metadata.getColumnCount();
        for (int i = 1; i <= num; i++) {
            metacol_map.add(metadata.getColumnName(i).toUpperCase());
        }
        
  		// 引数無しのpublicコンストラクタでBeanを生成
  		Object o = bean.newInstance(); // ClassLoaderの関係で生成できない場合があるので直接newする
//  		Object o = Class.forName(bean.getName()).newInstance();
  
  		// 各Property名を利用してresultから値を取得
  		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
  			// Property
  			BeanProperty bp = (BeanProperty) bpit.next();
            
  			// BeanPropertyと一致すればカラム値取得
            if (metacol_map.contains(bp.getName().toUpperCase())) {
      			Object val = getValueByType(result, bp);
      
      			// call setter
      			bp.getM_set().invoke(o, new Object[] { val });
            }
  		}
  
  		return o;
  	} catch (Exception e) {
  	    e.printStackTrace();
  		throw new POQLException(e);
  	}
  }

  /**
  	 * getValueByType<BR>
  	 * Propertyの型を判別してResultSetから値を取得する。
  	 * @param result 検索結果
  	 * @param bp BeanProperty
  	 * @return
  	 */
  protected Object getValueByType(ResultSet result, BeanProperty bp) throws POQLException {
  
  	String get_method = (String) TypeMappings.java2ResultsetGetter.get(bp.getType().getName());
  	if (get_method == null) {
  		// not implement type
  	  Log.println("POQLSimpleViewStatement not implement type [" + bp.getType().getName() + "]");
  		return null;
  	} else {
  		try {
  			Method method =
  				ResultSet.class.getDeclaredMethod(
  					get_method,
  					RS_METHOD_PARAM);
  			Object val = method.invoke(result, new String[] { bp.getName()});
            if (rtrim && bp.getType() == String.class && val != null) {
              val = StringUtl.rtrim3(val.toString());
            }
            return val;
  		} catch (Exception e) {
  		    e.printStackTrace();
  			throw new POQLException(e);
  		}
  	}
  }
  
  /**
   * @return rtrim を戻します。
   */
  public boolean isRtrim() {
      return rtrim;
  }
  /**
   * @param rtrim rtrim を設定。
   */
  public void setRtrim(boolean rtrim) {
      this.rtrim = rtrim;
  }
}
