package jp.co.powerbeans.powerql;

import java.io.InputStream;
import java.io.Reader;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLExclusiveException;
import jp.co.powerbeans.powerql.exceptions.POQLResultEmptyException;
import jp.co.powerbeans.powerql.exceptions.POQLResultMultiException;
import jp.co.powerbeans.powerql.exceptions.POQLTableNotFoundException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * <p>タイトル: POQLPreparedStatement</p>
 * <p>説明:
 * SQL実行のためのPreparedStatement保持クラス<BR>
 * １つのテーブルに対してPreparedStetementを利用して検索する場合に利用する。
 *  </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/07</p>
 * @author 門田明彦
 * @version $Revision: 1.4 $
 */
public class POQLPreparedStatement extends POQLStatementSupport {

	/** PreparedStatement */
	private PreparedStatement preparedStatement;
    private String whereSql;
    /** 最終実行 SQL  */
    private String lastSQL;
    /** 最終実行時 Bind Values  */
    private Object[] lastBindValues;
    /** 自動で現在日付を格納するタイムスタンプフィールド名 */
    private String[] autoTimestampColName;
    /** autoTimestampColName のSet */
    private Set autoTimestampColNameSet;
    
	/**
	 * コンストラクタ
	 * @param bean_class
	 * @param ps
	 * @param super_st
	 */
	POQLPreparedStatement(
		Class bean_class,
		PreparedStatement ps,
		POQLTransaction super_tr)
		throws POQLException, POQLTableNotFoundException {
	    
	    this(bean_class, ps, null, super_tr);
	}

	/**
	 * コンストラクタ
	 * @param bean_class
	 * @param ps
     * @param where_sql
	 * @param super_tr
	 */
	POQLPreparedStatement(
		Class bean_class,
    PreparedStatement ps,
		String where_sql, POQLTransaction super_tr)
		throws POQLException, POQLTableNotFoundException {
		bean = bean_class;
		this.preparedStatement = ps;
		this.super_tr = super_tr;
		this.whereSql = where_sql;
        this.lastSQL = super_tr.getLastSQL();
        this.lastBindValues = new Object[]{};
        this.autoTimestampColName = super_tr.getAutoTimestampColName();
        autoTimestampColNameSet = new HashSet();
        if (autoTimestampColName != null) {
            for (int i = 0; i < autoTimestampColName.length; i++) {
                if (autoTimestampColName[i] != null) {
                    autoTimestampColNameSet.add(autoTimestampColName[i].toUpperCase());
                }
            }
        }

		// テーブル名作成
		tableName = POQLUtil.className2TableName(bean);

		// 渡されたBeanのSQLマッピングを作成
		try {
			createSQLMapping();
		} catch (SQLException e) {
			throw new POQLException("fail to create bean mapping [" + bean.getName() + "] (" + e.getMessage() + ")", e);
		}
	}


//	/**
//	 * getValueByType<BR>
//	 * Propertyの型を判別してResultSetから値を取得する。
//	 * @param o 値を取得するObject
//	 * @param bp 型情報
//	 * @return 取得したプロパティの値
//	 */
//	private Object getValueByType(Object o, BeanProperty bp)
//		throws POQLException {
//		Method method = bp.getM_get();
//		if (method == null) {
//			// not implement type
//			// log_?
//			//			return null;
//			throw new POQLException("not implement type:" + bp.getName());
//		} else {
//			try {
//				return method.invoke(o, null);
//			} catch (Exception e) {
//				throw new POQLException(e);
//			}
//		}
//	}
//

	/**
	 * @param prepareStatement
	 * @param transaction
	 */
	public POQLPreparedStatement(PreparedStatement ps,
			POQLTransaction super_tr) {
//		bean = bean_class;
		this.preparedStatement = ps;
		this.super_tr = super_tr;
//		this.whereSql = where_sql;
        this.lastSQL = super_tr.getLastSQL();
        this.lastBindValues = new Object[]{};
        this.autoTimestampColName = super_tr.getAutoTimestampColName();
        autoTimestampColNameSet = new HashSet();
        if (autoTimestampColName != null) {
            for (int i = 0; i < autoTimestampColName.length; i++) {
                if (autoTimestampColName[i] != null) {
                    autoTimestampColNameSet.add(autoTimestampColName[i].toUpperCase());
                }
            }
        }

		// テーブル名作成
//		tableName = POQLUtil.className2TableName(bean);

//		// 渡されたBeanのSQLマッピングを作成
//		try {
//			createSQLMapping();
//		} catch (SQLException e) {
//			throw new POQLException("fail to create bean mapping [" + bean.getName() + "] (" + e.getMessage() + ")", e);
//		}
   }

	/**
   * select<BR>
   * 検索した結果をBeanのListで返す
   * @return 検索結果
   */
  public Collection select()
    throws SQLException, POQLException {

    ArrayList result_list = new ArrayList();
    ResultSet result = preparedStatement.executeQuery();

    while (result.next()) {
      // 各検索結果をMappingに基づいてBeanに変換
      result_list.add(getBeanByResultSet(result));
    }

    result.close();
    return result_list;
  }
  
  /**
   * selectToMap<br>
   * 検索した結果をMapのListで返す
   * @return　検索結果
   * @throws SQLException
   * @throws POQLException
   */
  public Collection selectToMap() throws SQLException, POQLException {
	    ArrayList result_list = new ArrayList();
	    ResultSet result = preparedStatement.executeQuery();

	    while (result.next()) {
	      // 各検索結果をMappingに基づいてBeanに変換
		      result_list.add(getMapByResultSet(result));
//		      result_list.add(getBeanByResultSet(result));
	    }

	    result.close();
	    return result_list;
  }
  
  /**
   * selectOneToMap<br>
   * 検索した結果1件をMapで返す
   * @return　検索結果
 * @throws POQLResultEmptyException 
 * @throws POQLResultMultiException 
   */
  public Map selectOneToMap() throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException {
	  Collection col = selectToMap();
	if (col.isEmpty()) {
		throw new POQLResultEmptyException("検索結果 0件");
	} else if (col.size() > 1) {
		throw new POQLResultMultiException("検索結果 " + col.size() + "件");
	}
	
	Object o = null;
	for(Iterator it = col.iterator(); it.hasNext();) {
		o = it.next(); 
	}
	return (Map) o;
  }
  
	/**
	 * selectOne<BR>
	 * 単一オブジェクトだけ取得するものとしてSELECT
	 * @return 検索結果。Beanクラスインスタンス。
	 */
	public Object selectOne() throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException {
		Collection col = select();
		if (col.isEmpty()) {
			throw new POQLResultEmptyException("検索結果 0件");
		} else if (col.size() > 1) {
			throw new POQLResultMultiException("検索結果 " + col.size() + "件");
		}
		
		Object o = null;
		for(Iterator it = col.iterator(); it.hasNext();) {
			o = it.next(); 
		}
		return o;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.POQLStatementIF#close()
	 */
	public void close() throws POQLException {
		if (preparedStatement != null) {
			try {
				preparedStatement.close();
			} catch (SQLException e) {
				throw new POQLException(e);
			}
		}
	}

	/**
	 * StatementがNULL以外の場合はclose, NULLの場合は何もしない
	 * @param st Statement
	 */
	public static void closeSafe(POQLStatementIF st) {
		try {
			if (st != null) {
				st.close();
			}
		} catch (POQLException e) {
			e.printStackTrace();
		}
	}

  /**
   * setArray<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setArray(int i, Array x) throws SQLException {
    preparedStatement.setArray(i, x);
  }

  /**
   * setAsciiStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setAsciiStream(parameterIndex, x, length);
  }

  /**
   * setBigDecimal<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
    preparedStatement.setBigDecimal(parameterIndex, x);
  }

  /**
   * setBinaryStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setBinaryStream(parameterIndex, x, length);
  }

  /**
   * setBlob<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBlob(int i, Blob x) throws SQLException {
    preparedStatement.setBlob(i, x);
  }

  /**
   * setBoolean<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBoolean(int parameterIndex, boolean x) throws SQLException {
    preparedStatement.setBoolean(parameterIndex, x);
  }

  /**
   * setByte<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setByte(int parameterIndex, byte x) throws SQLException {
    preparedStatement.setByte(parameterIndex, x);
  }

  /**
   * setBytes<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBytes(int parameterIndex, byte[] x) throws SQLException {
    preparedStatement.setBytes(parameterIndex, x);
  }

  /**
   * setCharacterStream<BR>
   * @param parameterIndex
   * @param reader
   * @param length
   * @throws java.sql.SQLException
   */
  public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
    preparedStatement.setCharacterStream(parameterIndex, reader, length);
  }

  /**
   * setClob<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setClob(int i, Clob x) throws SQLException {
    preparedStatement.setClob(i, x);
  }

  /**
   * setCursorName<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setCursorName(String arg0) throws SQLException {
    preparedStatement.setCursorName(arg0);
  }

  /**
   * setDate<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setDate(int parameterIndex, Date x) throws SQLException {
    preparedStatement.setDate(parameterIndex, x);
  }

  /**
   * setDate<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
    preparedStatement.setDate(parameterIndex, x, cal);
  }

  /**
   * setDouble<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setDouble(int parameterIndex, double x) throws SQLException {
    preparedStatement.setDouble(parameterIndex, x);
  }

  /**
   * setEscapeProcessing<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setEscapeProcessing(boolean arg0) throws SQLException {
    preparedStatement.setEscapeProcessing(arg0);
  }

  /**
   * setFetchDirection<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setFetchDirection(int arg0) throws SQLException {
    preparedStatement.setFetchDirection(arg0);
  }

  /**
   * setFloat<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setFloat(int parameterIndex, float x) throws SQLException {
    preparedStatement.setFloat(parameterIndex, x);
  }

  /**
   * setInt<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setInt(int parameterIndex, int x) throws SQLException {
    preparedStatement.setInt(parameterIndex, x);
  }

  /**
   * setLong<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setLong(int parameterIndex, long x) throws SQLException {
    preparedStatement.setLong(parameterIndex, x);
  }

  /**
   * setMaxFieldSize<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setMaxFieldSize(int arg0) throws SQLException {
    preparedStatement.setMaxFieldSize(arg0);
  }

  /**
   * setMaxRows<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setMaxRows(int arg0) throws SQLException {
    preparedStatement.setMaxRows(arg0);
  }

  /**
   * setNull<BR>
   * @param parameterIndex
   * @param sqlType
   * @throws java.sql.SQLException
   */
  public void setNull(int parameterIndex, int sqlType) throws SQLException {
    preparedStatement.setNull(parameterIndex, sqlType);
  }

  /**
   * setNull<BR>
   * @param paramIndex
   * @param sqlType
   * @param typeName
   * @throws java.sql.SQLException
   */
  public void setNull(int paramIndex, int sqlType, String typeName) throws SQLException {
    preparedStatement.setNull(paramIndex, sqlType, typeName);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x) throws SQLException {
    preparedStatement.setObject(parameterIndex, x);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @param targetSqlType
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
    preparedStatement.setObject(parameterIndex, x, targetSqlType);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @param targetSqlType
   * @param scale
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x, int targetSqlType, int scale) throws SQLException {
    preparedStatement.setObject(parameterIndex, x, targetSqlType, scale);
  }

  /**
   * setQueryTimeout<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setQueryTimeout(int arg0) throws SQLException {
    preparedStatement.setQueryTimeout(arg0);
  }

  /**
   * setRef<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setRef(int i, Ref x) throws SQLException {
    preparedStatement.setRef(i, x);
  }

  /**
   * setShort<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setShort(int parameterIndex, short x) throws SQLException {
    preparedStatement.setShort(parameterIndex, x);
  }

  /**
   * setString<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setString(int parameterIndex, String x) throws SQLException {
    preparedStatement.setString(parameterIndex, x);
  }

  /**
   * setTime<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setTime(int parameterIndex, Time x) throws SQLException {
    preparedStatement.setTime(parameterIndex, x);
  }

  /**
   * setTime<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
    preparedStatement.setTime(parameterIndex, x, cal);
  }

  /**
   * setTimestamp<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
    preparedStatement.setTimestamp(parameterIndex, x);
  }

  /**
   * setTimestamp<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
    preparedStatement.setTimestamp(parameterIndex, x, cal);
  }

  /**
   * setUnicodeStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setUnicodeStream(parameterIndex, x, length);
  }

  /**
   * setURL<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setURL(int parameterIndex, URL x) throws SQLException {
    preparedStatement.setURL(parameterIndex, x);
  }

  /**
   * update<BR>
   * 生成済みSQLでexecuteUpdateを実行
   * @return 更新数
   */
  public int update() throws POQLException, SQLException {

    return preparedStatement.executeUpdate(); 
  }


/**
 * insert実行。BLOB型にも対応。
 * @param o insertする値をもつJavaBean
 * @return insertした件数
 * @throws POQLException
 * @throws SQLException
 */
public int insert(Object o) throws POQLException, SQLException { 
    validateBean(o);
    
    if (o instanceof LinkedHashMap) {
    	// Map の場合は専用メソッドへ
    	return insertByMap((LinkedHashMap)o);
	}
    
    // DBMS依存処理クラス取得
    DBDependQLStatement dep = DBDependQLStatementFactory.create(
            super_tr.getConnection(), super_tr.mgr);

	// 各Property名を利用して値を取得し PreparedStatement#setXxx で格納
    int c = 1;
    Timestamp timestamp = new Timestamp(System.currentTimeMillis());
    // 値格納用
    ArrayList list = new ArrayList(bpList.size());
	for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
		// Property
		BeanProperty bp = (BeanProperty) bpit.next();
		
        // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
        if (bp.getColumnProp().isNotUseInsertSql()) {
            continue;
        }
		
		// カラム値取得
		Object val = getValueByType(o, bp);

		// 値を格納
		setPreparedValue(c++, bp, val, dep, timestamp);
        list.add(val);
	}
    
    this.lastBindValues = list.toArray(new Object[0]);

	return preparedStatement.executeUpdate();
}


	/**
	 * map 値 を利用して insert実行。BLOB型にも対応。
	 * @param o insertする値をもつJavaBean
	 * @return insertした件数
	 * @throws POQLException
	 * @throws SQLException
	 */
	private int insertByMap(LinkedHashMap map) throws POQLException, SQLException {
		// DBMS依存処理クラス取得
		DBDependQLStatement dep = DBDependQLStatementFactory.create(super_tr
				.getConnection(), super_tr.mgr);

		// 各Property名を利用して値を取得し PreparedStatement#setXxx で格納
		int c = 1;
//		Timestamp timestamp = new Timestamp(System.currentTimeMillis());
		
		// 値を格納
		List list = new ArrayList(map.size());
		for (Iterator it = map.keySet().iterator(); it.hasNext();) {
			String name = (String) it.next();
			Object val = map.get(name);
			preparedStatement.setObject(c++, val);
//			setPreparedValue(c++, name, map, dep, timestamp);
			list.add(val);
		}
		
//		ArrayList list = new ArrayList(bpList.size());
//		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
//			// Property
//			BeanProperty bp = (BeanProperty) bpit.next();
//
//			// DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
//			if (bp.getColumnProp().isNotUseInsertSql()) {
//				continue;
//			}
//
//			// カラム値取得
//			Object val = getValueByType(o, bp);
//
//			// 値を格納
//			setPreparedValue(c++, bp, val, dep, timestamp);
//			list.add(val);
//		}

		this.lastBindValues = list.toArray(new Object[0]);

		return preparedStatement.executeUpdate();
	}

	/**
	 * PreparedStatementに値を格納。 BLOB型以外は setObject, BLOB型は PoqlBlob
	 * 型としてInputStreamで格納
	 * 
	 * @param i
	 *            インデックス
	 * @param val
	 *            値
	 * @param dep
	 * @param timestamp
	 * @throws SQLException
	 * @throws POQLException
	 */
	private void setPreparedValue(
	        int i, BeanProperty bp, Object val, DBDependQLStatement dep,
	        Timestamp timestamp) throws SQLException, POQLException {
	    
	    String type_as_db = bp.columnProp.getTypeAsDB().toUpperCase();
        if (val != null && val instanceof PoqlBlob) {
	        // PoqlBlob型として格納
	        PoqlBlob bl = (PoqlBlob) val;
            preparedStatement.setBlob(i, bl);
            Log.debug("preparedStatement setBlob(" + i + "," + val + ") DB:" + 
            bp.columnProp.getName() + "," + type_as_db + " JAVA:" + bp.getType());
            return;
            
        } else if (val != null && val instanceof Blob) {
            // PoqlBlob に変換して設定
            Blob ob = (Blob) val;
            PoqlBlob bl = new PoqlBlob(ob.getBytes(0, (int) ob.length()));
            preparedStatement.setBlob(i, bl);
            Log.debug("preparedStatement setBlob(" + i + "," + val + ") DB:" + 
            bp.columnProp.getName() + "," + type_as_db + " JAVA:" + bp.getType());
            return;
            
            
            // DB依存Blob型の場合はとりあえず更新しない
//            Blob bl = (Blob) val;
//            preparedStatement.setBlob(i, bl);
//            Log.debug("preparedStatement setBlob(" + i + "," + val + ") DB:" + 
//            bp.columnProp.getName() + "," + type_as_db + " JAVA:" + bp.getType());
            
	    } else if (bp.getType().isEnum()) {
	    	// Enum 型はtoString()を格納
	    	Enum en = (Enum) val;
	    	preparedStatement.setString(i, en.toString());
	    	return;
	    }
	    
	    // setHogeで格納
	    if (!TypeMappings.java2ResultsetSetter.containsKey(bp.getType().getName())) {
	        // 未対応のプロパティ型
	        throw new POQLException("setPreparedValue unsupport type:" + bp.getType().getName());
	    }
	    
        Method m_set = (Method) TypeMappings.java2ResultsetSetter.get(bp.getType().getName());
        
        if (type_as_db.equals("TIMESTAMP") &&
                val == null &&
                autoTimestampColName != null &&
                autoTimestampColNameSet.contains(
                        bp.getColumnProp().getName().toUpperCase())) {
            // タイムスタンプ型で値にnullが格納されていた場合は 現在のTimestamp を格納
            // DBサーバーのTimestampを利用できればいいが、"CURRENT TIMESTAMP" 
            // を設定できないのでアプリケーションで設定する
            preparedStatement.setTimestamp(i, timestamp);
//	            preparedStatement.setObject(i, dep.getSQL_NOW("TIMESTAMP"));
            Log.debug("preparedStatement " + m_set.getName() + "(" + i + "," + val + ") DB:" +
                    bp.columnProp.getName() + "," + type_as_db + " JAVA:" + bp.getType());
            return;
        }
        
        try {
            if (val instanceof java.util.Date) {
                if (type_as_db.equals("DATE")) {
                    if (super_tr.mgr.getVendorType() == POQLManager.VENDOR_ORACLE) {
                        // Oracle日付型の場合は java.sql.Date は getTimestamp で取得
                        m_set = (Method) TypeMappings.java2ResultsetSetter.get("java.sql.Timestamp");
                        val = new java.sql.Timestamp(((java.util.Date)val).getTime());
                    } else {
                        // Oracle以外日付型の場合は java.sql.Date に変換して格納
                        m_set = (Method) TypeMappings.java2ResultsetSetter.get("java.sql.Date");
                        val = new java.sql.Date(((java.util.Date)val).getTime());
                    }
	                
                } else if (type_as_db.equals("TIME") || type_as_db.equals("TIMETZ")) {
	                // 時間型の場合は java.sql.Time に変換して格納
	                val = new java.sql.Time(((java.util.Date)val).getTime());
	                m_set = (Method) TypeMappings.java2ResultsetSetter.get("java.sql.Time");

                } else if (type_as_db.startsWith("TIMESTAMP") ||
                        type_as_db.equals("DATETIME")) {
	                // 時間型の場合は java.sql.Timestamp に変換して格納
	                val = new java.sql.Timestamp(((java.util.Date)val).getTime());
	                m_set = (Method) TypeMappings.java2ResultsetSetter.get("java.sql.Timestamp");
                }
            }
            
	        // setter実行
            m_set.invoke(preparedStatement,
                    new Object[]{POQLUtil.getInteger(i), val});
            
            Log.debug("preparedStatement " + m_set.getName() + "(" + i + "," + val + ") DB:" +
                    bp.columnProp.getName() + "," + type_as_db + " JAVA:" + bp.getType());
        } catch (Exception e) {
            e.printStackTrace(); // for DEBUG
            throw new POQLException(e);
        }
	    
	}


    /**
     * 自動で現在日付を格納するタイムスタンプフィールド名を取得
     * @return 自動で現在日付を格納するタイムスタンプフィールド名 
     */
    private String[] getAutoTimestampColName() {
        
        return autoTimestampColName;
    }


    /**
     * 自動で現在日付を格納するタイムスタンプフィールド名を取得
     * @return 自動で現在日付を格納するタイムスタンプフィールド名 
     */
    private void setAutoTimestampColName(String[] autoTimestampColName) {
        
        this.autoTimestampColName = autoTimestampColName;
    }

    /**
     * exclusive_check_field フィールドで排他チェックを行い、レコード取得時と同じ値なら更新する。<BR>
     * レコード取得時と異なる場合は排他エラーとし、POQLExclusiveException をスローする。
     * 
	 * @param o 追加するObject(テーブルに対応しているBean)
     * @param exclusive_check_field 排他チェックフィールド名。null or "" の場合は排他チェックを行わない
     * @return 更新数
     * @throws POQLException
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     * @throws POQLExclusiveException
     * @throws SQLException
     */
    public int updateExclusive(Object o, String exclusive_check_field) 
    throws SQLException, POQLException, 
    	POQLResultEmptyException, POQLResultMultiException, POQLExclusiveException {
        validateBean(o);
        
	    DBDependQLStatement dep = DBDependQLStatementFactory.create(super_tr.getConnection(),
	            super_tr.mgr);
		Timestamp timestamp = new Timestamp(System.currentTimeMillis());
        
        // 1. SELECT FOR UPDATE 排他チェック
        if (exclusive_check_field != null && exclusive_check_field.length() > 0 &&
                whereSql != null && whereSql.trim().length() > 0) {
            // 排他チェックフィールドが有効 かつ where 条件有り の場合だけロックする
            // その場合は新にPreparedStatementを作成
            POQLPreparedStatement lockps = super_tr.createPreparedStatement(
                    bean, "SELECT * FROM " +  getTableName() + " WHERE " + this.whereSql + " FOR UPDATE ");

            // WHERE条件に値を格納
            int c = 1;
    		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
    			// Property
    			BeanProperty bp = (BeanProperty) bpit.next();
    			
    			if (bp.columnProp.isPrimaryKey()) {
        			// カラム値取得
        			Object val = getValueByType(o, bp);
    			    lockps.setPreparedValue(c++, bp, val, dep, timestamp);
    			}
    		}
            
            // 検索
            Object db_bean = lockps.selectOne();
            
            if (equalFieldVal(db_bean, o, exclusive_check_field)) {
                // 値が一致したので更新する
            } else {
                // 既に更新されていたので排他エラーとする
                throw new POQLExclusiveException(getTableName() + "." + exclusive_check_field + " is not match in bean.");
            }
//            lockps.close();
        }
        
        // 2. UPDATE

        // 値を格納
        int c = 1;
        ArrayList list = new ArrayList(bpList.size() + 3);
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();

			if (bp.getColumnProp().isNotUseInsertSql()) {
				continue;
			} 
			// カラム値取得
			Object val = null;
			String type_as_db = bp.getColumnProp().getTypeAsDB();
            if (bp.getColumnProp().getName().equalsIgnoreCase(exclusive_check_field) &&
                    ( type_as_db.equalsIgnoreCase("TIMESTAMP") ||
                      type_as_db.equalsIgnoreCase("DATETIME") ||
                      type_as_db.equalsIgnoreCase("DATE"))
                ) {
			    // (TIMESTAMP または DATE または DATETIME) かつ 排他チェックフィールドの場合は現在時刻を格納し更新時間格納
			    val = timestamp;
			} else {
			    // その他通常の値
				val = getValueByType(o, bp);
			}
		    setPreparedValue(c++, bp, val, dep, timestamp);
            list.add(val);
		}

        // WHERE条件に値を格納
		for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
			// Property
			BeanProperty bp = (BeanProperty) bpit.next();
			
			if (bp.columnProp.isPrimaryKey()) {
				Object val = getValueByType(o, bp);
			    setPreparedValue(c++, bp, val, dep, timestamp);
                list.add(val);
			}
		}
        this.lastBindValues = list.toArray(new Object[0]);
		return preparedStatement.executeUpdate();
    }

    /**
     * @return lastSQL を戻します。
     */
    public String getLastSQL() {
        return lastSQL;
    }
    /**
     * @return lastBindValues を戻します。
     */
    public Object[] getLastBindValues() {
        return lastBindValues;
    }

    /**
     * insert実行。BLOB型にも対応。
     * 
     * @param o
     *            insertする値をもつJavaBean
     * @param 値指定カラム名Set
     * @return insertした件数
     * @throws POQLException
     * @throws POQLException
     * @throws SQLException
     * @throws SQLException
     */
    public int insert(Object o, Set colset) throws SQLException, POQLException {
        validateBean(o);
        
        // DBMS依存処理クラス取得
        DBDependQLStatement dep = DBDependQLStatementFactory.create(
                super_tr.getConnection(), super_tr.mgr);

        // 各Property名を利用して値を取得し PreparedStatement#setXxx で格納
        int c = 1;
        Timestamp timestamp = new Timestamp(System.currentTimeMillis());
        // 値格納用
        ArrayList list = new ArrayList(bpList.size());
        for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
            // Property
            BeanProperty bp = (BeanProperty) bpit.next();
            
            // カラム名Setに存在しない または
            // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
            if (!colset.contains(bp.getColumnProp().getName().toUpperCase()) || bp.getColumnProp().isNotUseInsertSql()) {
                continue;
            }
            
            // カラム値取得
            Object val = getValueByType(o, bp);

            // 値を格納
            setPreparedValue(c++, bp, val, dep, timestamp);
            list.add(val);
        }
        
        this.lastBindValues = list.toArray(new Object[0]);

        return preparedStatement.executeUpdate();
    }

    /**
     * exclusive_check_field フィールドで排他チェックを行い、レコード取得時と同じ値なら更新する。<BR>
     * レコード取得時と異なる場合は排他エラーとし、POQLExclusiveException をスローする。
     * 
     * @param o 追加するObject(テーブルに対応しているBean)
     * @param exclusive_check_field 排他チェックフィールド名。null or "" の場合は排他チェックを行わない
     * @param colset 値設定カラム名
     * @return 更新数
     * @throws POQLException
     * @throws POQLResultMultiException
     * @throws POQLResultEmptyException
     * @throws POQLExclusiveException
     * @throws SQLException
     */
    public int updateExclusive(Object o, String exclusive_check_field, Set colset) throws SQLException, POQLException, POQLResultEmptyException, POQLResultMultiException, POQLExclusiveException {
        validateBean(o);
        
        DBDependQLStatement dep = DBDependQLStatementFactory.create(super_tr.getConnection(),
                super_tr.mgr);
        Timestamp timestamp = new Timestamp(System.currentTimeMillis());
        
        // 1. SELECT FOR UPDATE 排他チェック
        if (exclusive_check_field != null && exclusive_check_field.length() > 0 &&
                whereSql != null && whereSql.trim().length() > 0) {
            // 排他チェックフィールドが有効 かつ where 条件有り の場合だけロックする
            // その場合は新にPreparedStatementを作成
            POQLPreparedStatement lockps = super_tr.createPreparedStatement(
                    bean, "SELECT * FROM " +  getTableName() + " WHERE " + this.whereSql + " FOR UPDATE ");

            // WHERE条件に値を格納
            int c = 1;
            for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                // Property
                BeanProperty bp = (BeanProperty) bpit.next();
                
                if (bp.columnProp.isPrimaryKey()) {
                    // カラム値取得
                    Object val = getValueByType(o, bp);
                    lockps.setPreparedValue(c++, bp, val, dep, timestamp);
                }
            }
            
            // 検索
            Object db_bean = lockps.selectOne();
            
            if (equalFieldVal(db_bean, o, exclusive_check_field)) {
                // 値が一致したので更新する
            } else {
                // 既に更新されていたので排他エラーとする
                throw new POQLExclusiveException(getTableName() + "." + exclusive_check_field + " is not match in bean.");
            }
//            lockps.close();
        }
        
        // 2. UPDATE

        // 値を格納
        int c = 1;
        ArrayList list = new ArrayList(bpList.size() + 3);
        for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
            // Property
            BeanProperty bp = (BeanProperty) bpit.next();

            if (bp.getColumnProp().isNotUseInsertSql()) {
                continue;
            } 
            // カラム値取得
            Object val = null;
            String type_as_db = bp.getColumnProp().getTypeAsDB();
            
            if (!colset.contains(bp.getColumnProp().getName().toUpperCase())) {
                // カラム名に存在しない場合は追加しない
                continue;
            }
            else if (bp.getColumnProp().getName().equalsIgnoreCase(exclusive_check_field) &&
                    ( type_as_db.equalsIgnoreCase("TIMESTAMP") ||
                      type_as_db.equalsIgnoreCase("DATETIME") ||
                      type_as_db.equalsIgnoreCase("DATE"))
                ) {
                // 
                // (TIMESTAMP または DATE または DATETIME) かつ 排他チェックフィールドの場合は現在時刻を格納し更新時間格納
                val = timestamp;
            } else {
                // その他通常の値
                val = getValueByType(o, bp);
            }
            setPreparedValue(c++, bp, val, dep, timestamp);
            list.add(val);
        }

        // WHERE条件に値を格納
        for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
            // Property
            BeanProperty bp = (BeanProperty) bpit.next();
            
            if (bp.columnProp.isPrimaryKey()) {
                Object val = getValueByType(o, bp);
                setPreparedValue(c++, bp, val, dep, timestamp);
                list.add(val);
            }
        }
        this.lastBindValues = list.toArray(new Object[0]);
        return preparedStatement.executeUpdate();
    }

	/**
	 * @return
	 * @throws SQLException
	 * @see java.sql.Statement#getGeneratedKeys()
	 */
	public ResultSet getGeneratedKeys() throws SQLException {
		return preparedStatement.getGeneratedKeys();
	}

	/**
	 * 内部PreparedStatmenetに値を格納
	 * @param vals
	 * @throws SQLException 
	 */
	public void setPreparedValues(Object[] vals) throws SQLException {
		if (vals == null) {
			return;
		}
		for (int i = 0; i < vals.length; i++) {
			this.preparedStatement.setObject(i + 1, vals[i]);
		}
	}

//	/**
//	 * 内部PreparedStatementにMap,カラムセットをで値を格納
//	 * @param bean_map
//	 * @param colset
//	 */
//	public void setPreparedValuesByMap(LinkedHashMap bean_map, Set colset) {
//		// TODO Auto-generated method stub
//		
//	}
}
