package jp.co.powerbeans.powerql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Iterator;

import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLResultEmptyException;
import jp.co.powerbeans.powerql.exceptions.POQLResultMultiException;


/**
 * <p>タイトル: POQLViewPreparedStatement</p>
 * <p>説明:
 * VIEW SQL実行のためのPreparedStatement保持クラス<BR>
 * 複数のテーブルに対してPreparedStetementを利用して結合SQLで検索する場合に利用する。
 *  </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/07</p>
 * @author 門田明彦
 * @version $Revision: 1.2 $
 */
public class POQLViewPreparedStatement extends POQLViewStatementSupport {

	/** PreparedStatement */
	private PreparedStatement preparedStatement;
    /** 最終実行SQL */
    private String lastSQL;
    /** 来週実行SQL格納値 */
    private Object[] lastBindValues;

	/**
	 * コンストラクタ
	 * @param bean_class Beanクラス名
	 * @param ps 生成済みPreparedStatement
	 * @param sql SQL
	 */
	public POQLViewPreparedStatement(Class bean_class, PreparedStatement ps, String sql) throws POQLException {
		
		bean = bean_class;
		this.preparedStatement = ps;
        this.lastSQL = sql;
        this.lastBindValues = new Object[0];
		//this.super_tr = super_tr;

		// 渡されたViewBeanのフィールドマッピングを作成
		createBeanFieldListMap();
	}

//	/**
//	 * 結合外部キーを指定して生成
//	 * @param bean_class
//	 * @param statement
//	 * @param fkeys
//	 * @param transaction
//	 */
//	public POQLSimpleViewStatement(Class bean_class, Statement statement, String[] fkeys, POQLTransaction super_tr) throws POQLException {
//		
//		bean = bean_class;
//		this.statement = statement;
//		this.fkeys = fkeys;
//		this.super_tr = super_tr;
//
//
//		// 渡されたViewBeanのフィールドマッピングを作成
//		createBeanFieldListMap();
//	}

	/**
	 * select<BR>
	 * 全件検索し結果をBeanのCollectionで返す。
	 * @return 検索結果。BeanクラスインスタンスのCollection
	 * @throws SQLException SQL例外
	 * @throws POQLException PowerQL例外
	 */
	public Collection select() throws POQLException, SQLException {

		ArrayList result_list = new ArrayList(10);
		ResultSet result = preparedStatement.executeQuery();

		while (result.next()) {
			// 各検索結果をMappingに基づいてBeanに変換
			result_list.add(getBeanByResultSet(result));
		}

		result.close();
		return result_list;
	}

	/**
	 * selectOne<BR>
	 * 単一オブジェクトを取得するものとして検索実行
	 * @return 検索結果。Beanクラスインスタンス
	 * @throws SQLException SQL例外
	 * @throws POQLException PowerQL例外
	 * @throws POQLResultEmptyException 検索結果なしの場合の例外
	 * @throws POQLResultMultiException 検索結果複数の場合の例外
	 */
	public Object selectOne() throws SQLException, 
	POQLException, POQLResultEmptyException, POQLResultMultiException {

		Collection col = select();
		if (col.isEmpty()) {
			throw new POQLResultEmptyException("検索結果 0件");
		} else if (col.size() > 1) {
			throw new POQLResultMultiException("検索結果 " + col.size() + "件");
		}
		
		Object o = null;
		for(Iterator it = col.iterator(); it.hasNext();) {
			o = it.next(); 
		}
		return o;
	}

	/* (non-Javadoc)
	 * @see jp.co.powerbeans.powerql.POQLStatementIF#close()
	 */
	public void close() throws POQLException {
		if (preparedStatement != null) {
			try {
				preparedStatement.close();
			} catch (SQLException e) {
				throw new POQLException(e);
			}
		}
	}

	/**
	 * StatementがNULL以外の場合はclose, NULLの場合は何もしない
	 * @param st Statement
	 */
	public static void closeSafe(POQLViewPreparedStatement st) {
		try {
			if (st != null) {
				st.close();
			}
		} catch (POQLException e) {
			e.printStackTrace();
		}
	}
  
  /**
   * setArray<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setArray(int i, Array x) throws SQLException {
    preparedStatement.setArray(i, x);
  }

  /**
   * setAsciiStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setAsciiStream(parameterIndex, x, length);
  }

  /**
   * setBigDecimal<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBigDecimal(int parameterIndex, BigDecimal x) throws SQLException {
    preparedStatement.setBigDecimal(parameterIndex, x);
  }

  /**
   * setBinaryStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setBinaryStream(parameterIndex, x, length);
  }

  /**
   * setBlob<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBlob(int i, Blob x) throws SQLException {
    preparedStatement.setBlob(i, x);
  }

  /**
   * setBoolean<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBoolean(int parameterIndex, boolean x) throws SQLException {
    preparedStatement.setBoolean(parameterIndex, x);
  }

  /**
   * setByte<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setByte(int parameterIndex, byte x) throws SQLException {
    preparedStatement.setByte(parameterIndex, x);
  }

  /**
   * setBytes<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setBytes(int parameterIndex, byte[] x) throws SQLException {
    preparedStatement.setBytes(parameterIndex, x);
  }

  /**
   * setCharacterStream<BR>
   * @param parameterIndex
   * @param reader
   * @param length
   * @throws java.sql.SQLException
   */
  public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
    preparedStatement.setCharacterStream(parameterIndex, reader, length);
  }

  /**
   * setClob<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setClob(int i, Clob x) throws SQLException {
    preparedStatement.setClob(i, x);
  }

  /**
   * setCursorName<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setCursorName(String arg0) throws SQLException {
    preparedStatement.setCursorName(arg0);
  }

  /**
   * setDate<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setDate(int parameterIndex, Date x) throws SQLException {
    preparedStatement.setDate(parameterIndex, x);
  }

  /**
   * setDate<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setDate(int parameterIndex, Date x, Calendar cal) throws SQLException {
    preparedStatement.setDate(parameterIndex, x, cal);
  }

  /**
   * setDouble<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setDouble(int parameterIndex, double x) throws SQLException {
    preparedStatement.setDouble(parameterIndex, x);
  }

  /**
   * setEscapeProcessing<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setEscapeProcessing(boolean arg0) throws SQLException {
    preparedStatement.setEscapeProcessing(arg0);
  }

  /**
   * setFetchDirection<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setFetchDirection(int arg0) throws SQLException {
    preparedStatement.setFetchDirection(arg0);
  }

  /**
   * setFloat<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setFloat(int parameterIndex, float x) throws SQLException {
    preparedStatement.setFloat(parameterIndex, x);
  }

  /**
   * setInt<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setInt(int parameterIndex, int x) throws SQLException {
    preparedStatement.setInt(parameterIndex, x);
  }

  /**
   * setLong<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setLong(int parameterIndex, long x) throws SQLException {
    preparedStatement.setLong(parameterIndex, x);
  }

  /**
   * setMaxFieldSize<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setMaxFieldSize(int arg0) throws SQLException {
    preparedStatement.setMaxFieldSize(arg0);
  }

  /**
   * setMaxRows<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setMaxRows(int arg0) throws SQLException {
    preparedStatement.setMaxRows(arg0);
  }

  /**
   * setNull<BR>
   * @param parameterIndex
   * @param sqlType
   * @throws java.sql.SQLException
   */
  public void setNull(int parameterIndex, int sqlType) throws SQLException {
    preparedStatement.setNull(parameterIndex, sqlType);
  }

  /**
   * setNull<BR>
   * @param paramIndex
   * @param sqlType
   * @param typeName
   * @throws java.sql.SQLException
   */
  public void setNull(int paramIndex, int sqlType, String typeName) throws SQLException {
    preparedStatement.setNull(paramIndex, sqlType, typeName);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x) throws SQLException {
    preparedStatement.setObject(parameterIndex, x);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @param targetSqlType
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x, int targetSqlType) throws SQLException {
    preparedStatement.setObject(parameterIndex, x, targetSqlType);
  }

  /**
   * setObject<BR>
   * @param parameterIndex
   * @param x
   * @param targetSqlType
   * @param scale
   * @throws java.sql.SQLException
   */
  public void setObject(int parameterIndex, Object x, int targetSqlType, int scale) throws SQLException {
    preparedStatement.setObject(parameterIndex, x, targetSqlType, scale);
  }

  /**
   * setQueryTimeout<BR>
   * @param arg0
   * @throws java.sql.SQLException
   */
  public void setQueryTimeout(int arg0) throws SQLException {
    preparedStatement.setQueryTimeout(arg0);
  }

  /**
   * setRef<BR>
   * @param i
   * @param x
   * @throws java.sql.SQLException
   */
  public void setRef(int i, Ref x) throws SQLException {
    preparedStatement.setRef(i, x);
  }

  /**
   * setShort<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setShort(int parameterIndex, short x) throws SQLException {
    preparedStatement.setShort(parameterIndex, x);
  }

  /**
   * setString<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setString(int parameterIndex, String x) throws SQLException {
    preparedStatement.setString(parameterIndex, x);
  }

  /**
   * setTime<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setTime(int parameterIndex, Time x) throws SQLException {
    preparedStatement.setTime(parameterIndex, x);
  }

  /**
   * setTime<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setTime(int parameterIndex, Time x, Calendar cal) throws SQLException {
    preparedStatement.setTime(parameterIndex, x, cal);
  }

  /**
   * setTimestamp<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setTimestamp(int parameterIndex, Timestamp x) throws SQLException {
    preparedStatement.setTimestamp(parameterIndex, x);
  }

  /**
   * setTimestamp<BR>
   * @param parameterIndex
   * @param x
   * @param cal
   * @throws java.sql.SQLException
   */
  public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal) throws SQLException {
    preparedStatement.setTimestamp(parameterIndex, x, cal);
  }

  /**
   * setUnicodeStream<BR>
   * @param parameterIndex
   * @param x
   * @param length
   * @throws java.sql.SQLException
   */
  public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
    preparedStatement.setUnicodeStream(parameterIndex, x, length);
  }

  /**
   * setURL<BR>
   * @param parameterIndex
   * @param x
   * @throws java.sql.SQLException
   */
  public void setURL(int parameterIndex, URL x) throws SQLException {
    preparedStatement.setURL(parameterIndex, x);
  }

    /*
     * (非 Javadoc)
     * 
     * @see jp.co.powerbeans.powerql.POQLStatementIF#getLastSQL()
     */
    public String getLastSQL() {
        return lastSQL;
    }

    /*
     * (非 Javadoc)
     * 
     * @see jp.co.powerbeans.powerql.POQLStatementIF#getLastBindValues()
     */
    public Object[] getLastBindValues() {
        return lastBindValues;
    }


}
