package jp.co.powerbeans.powerql;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Set;

import jp.co.powerbeans.powerql.exceptions.POQLException;
import jp.co.powerbeans.powerql.exceptions.POQLTableNotFoundException;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatement;
import jp.co.powerbeans.powerql.vendor.DBDependQLStatementFactory;


/**
 * <p>タイトル: POQLTransaction</p>
 * <p>説明: 
 * このクラスは業務Bean等のロジックごとにPOQLManagerから取得する。<BR>
 * コンストラクタでコネクションを作成し、業務Beanが終了する時点で
 * close() を実行する。
 * 
 * 
 * </p>
 * <p>著作権: 株式会社パワービーンズ</p>
 * <p>会社名: 株式会社パワービーンズ</p>
 * <p>Created on 2003/10/07</p>
 * @author 門田明彦
 * @version $Revision: 1.4 $
 */
public class POQLTransaction {

	/** JDBC コネクション */
//	private jp.co.powerbeans.powerql.Connection connection;
	private Connection connection;
	
	/** 生成元 POQLManager */
	POQLManager mgr;

  /** rtrim フラグ */
	private boolean rtrim;

	private String lastSQL;

    /** 自動で現在日付を格納するタイムスタンプフィールド名 */
    private String[] autoTimestampColName;

	/**
	 * コンストラクタ
	 */
	POQLTransaction(Connection con, POQLManager mgr) throws POQLException {
		super();
		
		if (con == null) {
			throw new POQLException("Connection is null.");
		}
		setConnection(con);
		this.mgr = mgr;
	}

	/**
	 * setConnection<BR>
	 * @param con JDBC Connection
	 */
	void setConnection(Connection con) {
		this.connection = con;
	}

	/**
	 * setAutoCommit<BR>
	 * @param b
	 */
	public void setAutoCommit(boolean b) throws POQLException {
		try {
			connection.setAutoCommit(b);
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

	/**
	 * close<BR>
	 * 
	 */
	public void close() throws POQLException {
		try {
			if (connection != null && !connection.isClosed()) {
				connection.close();
				Log.println("closed.");
			}
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

	/**
	 * createStatement<BR>
	 * クラスオブジェクトを元に POQLStatement を作成し返す。
	 * @param bean_class Beanクラス名
	 * @return 生成したPOQLStatement
	 */
	public POQLStatement createStatement(Class bean_class) throws POQLException, POQLTableNotFoundException {
		
		try {
			return new POQLStatement(bean_class, connection.createStatement(), this);
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

	/**
	 * createStatement<BR>
	 * クラス名を元に POQLStatement を作成し返す。
	 * @param bean_class_name Beanクラス名
	 * @return 生成したPOQLStatement
	 */
	public POQLStatement createStatement(String bean_class_name) throws POQLException, POQLTableNotFoundException {
		
		Class bean_class;
		try {
			bean_class = Class.forName(bean_class_name);
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
			throw new POQLException("cannot create class :" + bean_class_name, e);
		}
		return createStatement(bean_class);
	}

	/**
	 * トランザクションをコミットする。
	 * 
	 */
	public void commit() throws POQLException {
		try {
		    if (!connection.getAutoCommit()) {
		        connection.commit();
		    }
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

	/**
	 * トランザクションをロールバックする。
	 * 
	 */
	public void rollback() throws POQLException {
		try {
			if (!connection.getAutoCommit()) {
				connection.rollback();
			}
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

	/**
	 * コネクションの参照を返す。
	 * @return コネクションの参照
	 */
	public Connection getConnection() {
		return connection;
	}

	/**
	 * createSimpleViewStatement<BR>
	 * @param bean_class 値格納用Bean
	 * @return 生成したPOQLSimpleViewStatement
	 */
	public POQLSimpleViewStatement createSimpleViewStatement(Class bean_class) throws POQLException {
		try {
			return new POQLSimpleViewStatement(bean_class, connection.createStatement(),this);
		} catch (SQLException e) {
			throw new POQLException(e);
		}
	}

  /**
   * createPreparedStatement<BR>
   * @param bean_class 値格納用Bean
   * @param sql パラメータ未設定SQL
   * @return 生成したPOQLPreparedStatement
   */
  public POQLPreparedStatement createPreparedStatement(Class bean_class, String sql) throws POQLTableNotFoundException, POQLException {
    this.lastSQL = sql;
    try {
      return new POQLPreparedStatement(bean_class, connection.prepareStatement(sql), this);
    } catch (SQLException e) {
      throw new POQLException(e);
    }
  }

  /**
   * createPreparedStatement<BR>
   * @param bean_class_name 値格納用Bean名
   * @param sql パラメータ未設定SQL
   * @return 生成したPOQLPreparedStatement
   */
  public POQLPreparedStatement createPreparedStatement(String bean_class_name, String sql) throws POQLTableNotFoundException, POQLException {
    Class bean_class;
    this.lastSQL = sql;
    try {
      bean_class = Class.forName(bean_class_name);
    } catch (ClassNotFoundException e) {
      e.printStackTrace();
      throw new POQLException("cannot create class :" + bean_class_name, e);
    }
    return createPreparedStatement(bean_class, sql);
  }

  /**
   * createViewPreparedStatement<BR>
   * @param bean_class 
   * @param sql パラメータ未設定SQL
   * @return 生成したPOQLPreparedStatement
   */
  public POQLViewPreparedStatement createViewPreparedStatement(Class bean_class,String sql) throws POQLException {
    this.lastSQL = sql;
    try {
        POQLViewPreparedStatement st = new POQLViewPreparedStatement(bean_class, connection.prepareStatement(sql), sql);
        st.setRtrim(this.rtrim);
        return st;
    } catch (SQLException e) {
      throw new POQLException(e);
    }
  }

  /**
   * createViewPreparedStatement<BR>
   * @param bean_class_name
   * @param sql パラメータ未設定SQL
   * @return 生成したPOQLViewPreparedStatement
   */
  public POQLViewPreparedStatement createViewPreparedStatement(String bean_class_name,String sql) throws POQLException {
    Class bean_class;
    this.lastSQL = sql;
    try {
      bean_class = Class.forName(bean_class_name);
    } catch (ClassNotFoundException e) {
      e.printStackTrace();
      throw new POQLException("cannot create class :" + bean_class_name, e);
    }
    return createViewPreparedStatement(bean_class, sql);
  }

    /**
     * Insert用のPOQLPreparedStatement を作成する。 INSERT SQL
     * を自作する以外は他のcreatePreparedStatementと同様
     * 
     * @param bean_class
     *            値格納用Bean
     * @return 生成したPOQLPreparedStatement
     * @throws POQLException
     * @throws POQLTableNotFoundException
     */
    public POQLPreparedStatement createPreparedStatement_forInsert(
            Class bean_class) throws POQLTableNotFoundException,
            POQLException {

        StringBuffer sql = new StringBuffer();
        ArrayList bpList;
        DBDependQLStatement dqs = DBDependQLStatementFactory.create(
                connection, mgr);
        try {
            bpList = POQLStatementSupport.getBeanPropertyList(bean_class,
                    this);
            // INSERT SQLを作成
            sql.append("INSERT INTO "
                    + dqs.escape(POQLUtil.className2TableName(bean_class)) + " ( ");

            // フィールド指定
            int c = 0;
            for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                // Property
                BeanProperty bp = (BeanProperty) bpit.next();

                // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
                if (bp.getColumnProp().isNotUseInsertSql()) {
                    continue;
                }
                // SQLに追加
                if (c++ > 0) {
                    sql.append(",");
                }
                sql.append(bp.getColumnPropEscapedName());
            }

            sql.append(") VALUES (");

            // フィールドと同数の ? を追加
            for (int i = 0; i < c; i++) {
                if (i > 0) {
                    sql.append(",");
                }
                sql.append("?");
            }
            
            sql.append(")");

            this.lastSQL = sql.toString();
            return createPreparedStatement(bean_class, sql.toString());
        } catch (SQLException e) {
            throw new POQLException(e);
        }
    }

    /**
     * Update用のPOQLPreparedStatement を作成する。 UPDATE SQL
     * を自作する以外は他のcreatePreparedStatementと同様
     * 
     * @param bean_class
     *            値格納用Bean
     * @return 生成したPOQLPreparedStatement
     * @throws POQLException
     */
    public POQLPreparedStatement createPreparedStatement_forUpdate(Class bean_class) throws POQLException {
        return createPreparedStatement_forUpdate(bean_class, null);
    }
    
    /**
     * Update用のPOQLPreparedStatement を作成する。 UPDATE SQL
     * を自作する以外は他のcreatePreparedStatementと同様
     * 
     * @param bean_class
     *            値格納用Bean
     * @param where WHERE 条件, null or "" の場合はプライマリキーをWHERE条件とする
     * @return 生成したPOQLPreparedStatement
     * @throws POQLException
     */
    public POQLPreparedStatement createPreparedStatement_forUpdate(Class bean_class, String where) throws POQLException {
        StringBuffer sql = new StringBuffer();
        ArrayList bpList;
        DBDependQLStatement dqs = DBDependQLStatementFactory.create(
                connection, mgr);
        try {
            bpList = POQLStatementSupport.getBeanPropertyList(bean_class,
                    this);
            // INSERT SQLを作成
            sql.append("UPDATE "
                    + dqs.escape(POQLUtil.className2TableName(bean_class)) + " SET ");

            // フィールド指定
            int c = 0;
            for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                // Property
                BeanProperty bp = (BeanProperty) bpit.next();

                // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
                if (bp.getColumnProp().isNotUseInsertSql()) {
                    continue;
                }
                // SQLに追加
                if (c++ > 0) {
                    sql.append(",");
                }
                sql.append(bp.getColumnPropEscapedName());// 予約語などをエスケープ
                sql.append("=?");
            }
            
            String where_sql = ""; 
            if (where != null && where.trim().length() > 0) {
                // 独自 WHERE 条件
                sql.append(" WHERE " + dqs.escape(where));
                where_sql = where;
                
            } else {
                // プライマリキーWHERE条件
                sql.append(" WHERE ");
                
                StringBuffer where_sqlw = new StringBuffer();
                
                c = 0;
                for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                    // Property
                    BeanProperty bp = (BeanProperty) bpit.next();
                    
                    if (bp.columnProp.isPrimaryKey()) {
	                    // SQLに追加
	                    if (c++ > 0) {
	                        where_sqlw.append(" AND ");
	                    }
	                    where_sqlw.append(dqs.escape(bp.columnProp.getName()) + "=?");
                    }
                }
                where_sql = where_sqlw.toString();
                sql.append(where_sql);
            }

            this.lastSQL = sql.toString();
            return createPreparedStatement(bean_class, sql.toString(), where_sql);
        } catch (SQLException e) {
            throw new POQLException(e);
        }
    }

        /**
         * POQLPreparedStatement を生成して返す。
         * @param bean_class 値格納用Bean
         * @param sql パラメータ未設定SQL
         * @param where_sql 検索条件SQL
         * @return 生成した POQLPreparedStatement
         * @throws POQLException
         */
        public POQLPreparedStatement createPreparedStatement(Class bean_class, String sql, String where_sql) throws POQLException {
            this.lastSQL = sql;
            try {
                return new POQLPreparedStatement(bean_class, connection.prepareStatement(sql), where_sql, this);
              } catch (SQLException e) {
                throw new POQLException(e);
              }
        }

		/**
		 * @param b
		 */
		public void setRtrim(boolean rtrim) {
            this.rtrim = rtrim;
		}
        
    public boolean isRtrim() {
     return rtrim;   
    }


    /**
     * @return lastSQL を戻します。
     */
    public String getLastSQL() {
        return lastSQL;
    }

//    /**
//     * 自動で現在日付を格納するタイムスタンプフィールド名を取得
//     * @return 自動で現在日付を格納するタイムスタンプフィールド名
//     */
//    public String getAutoTimestampColName() {
//        return autoTimestampColName;
//    }
//
    /**
     * 自動で現在日付を格納するタイムスタンプフィールド名を設定
     * @param auto_timestamp_colname 自動で現在日付を格納するタイムスタンプフィールド名
     */
    public void setAutoTimestampColName(String[] auto_timestamp_colname) {
        this.autoTimestampColName = auto_timestamp_colname;
    }

    /**
     * 自動で現在日付を格納するタイムスタンプフィールド名を取得
     * @return 自動で現在日付を格納するタイムスタンプフィールド名
     */
    public String[] getAutoTimestampColName() {
        return autoTimestampColName;
    }

    /**
     * Insert用のPOQLPreparedStatement を作成する。 INSERT SQL
     * を自作する以外は他のcreatePreparedStatementと同様
     * 
     * @param bean_class
     *            値格納用Bean
     * @param colset SQLで値を指定するカラム名Set
     * @return 生成したPOQLPreparedStatement
     * @throws POQLException
     * @throws POQLTableNotFoundException
     */
    public POQLPreparedStatement createPreparedStatement_forInsert(Class bean_class, Set colset) throws POQLException {

        StringBuffer sql = new StringBuffer();
        ArrayList bpList;
        try {
            bpList = POQLStatementSupport.getBeanPropertyList(bean_class,
                    this);
            // INSERT SQLを作成
            sql.append("INSERT INTO "
                    + POQLUtil.className2TableName(bean_class) + " ( ");

            // フィールド指定
            int c = 0;
            for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                // Property
                BeanProperty bp = (BeanProperty) bpit.next();
                
                // colsetに存在しない, または
                // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
                if (!colset.contains(bp.getColumnProp().getName().toUpperCase()) || bp.getColumnProp().isNotUseInsertSql()) {
                    continue;
                }
                // SQLに追加
                if (c++ > 0) {
                    sql.append(",");
                }
                sql.append(bp.getColumnProp().getName());
            }

            sql.append(") VALUES (");

            // フィールドと同数の ? を追加
            for (int i = 0; i < c; i++) {
                if (i > 0) {
                    sql.append(",");
                }
                sql.append("?");
            }
            
            sql.append(")");

            this.lastSQL = sql.toString();
            return createPreparedStatement(bean_class, sql.toString());
        } catch (SQLException e) {
            throw new POQLException(e);
        }
    }

    /**
     * @param beanClass
     * @param columns
     * @return
     * @throws POQLException
     */
    public POQLPreparedStatement createPreparedStatement_forUpdateWithColumns(Class bean_class, Set colset) throws POQLException {
        return createPreparedStatement_forUpdateWithColumns(bean_class, null, colset);
    }
    
    /**
     * @param beanClass
     * @param columns
     * @return
     * @throws POQLException
     */
    public POQLPreparedStatement createPreparedStatement_forUpdateWithColumns(Class bean_class, String where, Set colset) throws POQLException {
        StringBuffer sql = new StringBuffer();
        ArrayList bpList;
        try {
            bpList = POQLStatementSupport.getBeanPropertyList(bean_class,
                    this);
            // INSERT SQLを作成
            sql.append("UPDATE "
                    + POQLUtil.className2TableName(bean_class) + " SET ");

            // フィールド指定
            int c = 0;
            for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                // Property
                BeanProperty bp = (BeanProperty) bpit.next();

                // カラム名Setに存在しない または
                // DB2 'GENERATED ALWAYS' 等 自動INSERTしてはいけないカラムは無視
                if (!colset.contains(bp.getColumnProp().getName().toUpperCase()) || bp.getColumnProp().isNotUseInsertSql()) {
                    continue;
                }
                // SQLに追加
                if (c++ > 0) {
                    sql.append(",");
                }
                sql.append(bp.getColumnProp().getName());
                sql.append("=?");
            }
            
            String where_sql = ""; 
            if (where != null && where.trim().length() > 0) {
                // 独自 WHERE 条件
                sql.append(" WHERE " + where);
                where_sql = where;
                
            } else {
                // プライマリキーWHERE条件
                sql.append(" WHERE ");
                
                StringBuffer where_sqlw = new StringBuffer();
                
                c = 0;
                for (Iterator bpit = bpList.iterator(); bpit.hasNext();) {
                    // Property
                    BeanProperty bp = (BeanProperty) bpit.next();
                    
                    if (bp.columnProp.isPrimaryKey()) {
                        // SQLに追加
                        if (c++ > 0) {
                            where_sqlw.append(" AND ");
                        }
                        where_sqlw.append(bp.columnProp.getName() + "=?");
                    }
                }
                where_sql = where_sqlw.toString();
                sql.append(where_sql);
            }

            this.lastSQL = sql.toString();
            return createPreparedStatement(bean_class, sql.toString(), where_sql);
        } catch (SQLException e) {
            throw new POQLException(e);
        }
    }
}
