/***************************************************************************

	mame.h

	Controls execution of the core MAME system.

***************************************************************************/

#ifndef MACHINE_H
#define MACHINE_H

#include "osdepend.h"
#include "drawgfx.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "palette.h"

#ifdef MAME32JP
#include "ui_font.h"
#endif

#ifdef MESS
#include "mess.h"
#endif

extern char build_version[];

#define MAX_GFX_ELEMENTS 32
#define MAX_MEMORY_REGIONS 32



/***************************************************************************

	Core description of the currently-running machine

***************************************************************************/

struct RegionInfo
{
	UINT8 *		base;
	size_t		length;
	UINT32		type;
	UINT32		flags;
};

struct RunningMachine
{
	struct RegionInfo memory_region[MAX_MEMORY_REGIONS];
	struct GfxElement *gfx[MAX_GFX_ELEMENTS];	/* graphic sets (chars, sprites) */
	struct mame_bitmap *scrbitmap;	/* bitmap to draw into */
	struct rectangle visible_area;
	pen_t *pens;	/* remapped palette pen numbers. When you write */
					/* directly to a bitmap, never use absolute values, */
					/* use this array to get the pen number. For example, */
					/* if you want to use color #6 in the palette, use */
					/* pens[6] instead of just 6. */
	rgb_t *palette;
	UINT16 *game_colortable;	/* lookup table used to map gfx pen numbers */
								/* to color numbers */
	pen_t *remapped_colortable;	/* the above, already remapped through */
								/* Machine->pens */
	const struct GameDriver *gamedrv;	/* contains the definition of the game machine */
	const struct InternalMachineDriver *drv;	/* same as gamedrv->drv */
	int color_depth;	/* video color depth: 8, 16, 15 or 32 */
	int sample_rate;	/* the digital audio sample rate; 0 if sound is disabled. */
						/* This is set to a default value, or a value specified by */
						/* the user; osd_init() is allowed to change it to the actual */
						/* sample rate supported by the audio card. */
	struct GameSamples *samples;	/* samples loaded from disk */
	struct InputPort *input_ports;	/* the input ports definition from the driver */
								/* is copied here and modified (load settings from disk, */
								/* remove cheat commands, and so on) */
	struct InputPort *input_ports_default; /* original input_ports without modifications */
	int orientation;	/* see #defines in driver.h */
	struct GfxElement *uifont;	/* font used by the user interface */
	int uifontwidth,uifontheight;
#ifdef MAME32JP
	struct GfxElement *uifont2;	/* font used by the user interface */
	int uifontwidth2,uifontheight2;
#endif
	int uixmin,uiymin;
	int uiwidth,uiheight;
	int ui_orientation;
	struct rectangle absolute_visible_area;	/* as passed to osd_set_visible_area() */

	/* stuff for the debugger */
	struct mame_bitmap *debug_bitmap;
	pen_t *debug_pens;
	pen_t *debug_remapped_colortable;
	struct GfxElement *debugger_font;
};



/***************************************************************************

	Options passed from the frontend to the main core

***************************************************************************/

#ifdef MESS
#define MAX_IMAGES	32
/*
 * This is a filename and it's associated peripheral type
 * The types are defined in mess.h (IO_...)
 */
struct ImageFile
{
	const char *name;
	int type;
};
#endif

/* The host platform should fill these fields with the preferences specified in the GUI */
/* or on the commandline. */
struct GameOptions
{
	void *record;
	void *playback;
	void *language_file; /* LBO 042400 */

	int mame_debug;
	int cheat;
	int gui_host;

	int samplerate;
	int use_samples;
	int use_filter;
	int force_panpot;
	int limit_adjust;
	int format_volume;
	int auto_adjust;
	int presoundfix;

	float	brightness;		/* brightness of the display */
	float	pause_bright;		/* additional brightness when in pause */
	float	gamma;			/* gamma correction of the display */
	int color_depth;	/* 8 or 16, any other value means auto */
	int vector_width;	/* requested width for vector games; 0 means default (640) */
	int vector_height;	/* requested height for vector games; 0 means default (480) */
	int debug_width;	/* initial size of the debug_bitmap */
	int debug_height;
	int norotate;
	int ror;
	int rol;
	int flipx;
	int flipy;
	int beam;
	float vector_flicker;
	int translucency;
	int antialias;
	int use_artwork;
	char savegame;
	int		bios_type;/* dbg-bios */

#ifdef MAME32JP
	int autofiredelay;
	int game_index;
	int use_joystick;
	int use_djoystick;
	int hiscore_force;
	int hiscore_disable;
	int blackout; // BLACKOUT
	int autofire_message; // AUTOFIRE_MESSAGE
#endif
#ifdef ROMPATCH
	int patch_roms;
#endif

	#ifdef MESS
	int append_no_file_extension;

	struct ImageFile image_files[MAX_IMAGES];
	int image_count;
	#endif
};



/***************************************************************************

	Globals referencing the current machine and the global options

***************************************************************************/

extern struct GameOptions options;
extern struct RunningMachine *Machine;
extern int partial_update_count;



/***************************************************************************

	Function prototypes

***************************************************************************/

int run_game(int game);
int updatescreen(void);
void draw_screen(void);

/* force a partial update of the screen up to and including the
   requested scanline */
void force_partial_update(int scanline);

/* called by cpuexec.c to reset updates at the end of VBLANK */
void reset_partial_updates(void);

/* set the current visible area of the screen bitmap */
void set_visible_area(int min_x, int max_x, int min_y, int max_y);

/* next time vh_screenrefresh is called, full_refresh will be true,
   thus requesting a redraw of the entire screen */
void schedule_full_refresh(void);

void update_video_and_audio(void);
/* osd_fopen() must use this to know if high score files can be used */
int mame_highscore_enabled(void);
#ifdef MAME32JP
#define set_led_status(num, on)
#else
void set_led_status(int num,int on);
#endif

/* return current performance data */
const struct performance_info *mame_get_performance_info(void);

/* return the index of the given CPU, or -1 if not found */
int mame_find_cpu_index(const char *tag);

/* expansion of tag/data pairs */
struct InternalMachineDriver;
void expand_machine_driver(void (*constructor)(struct InternalMachineDriver *), struct InternalMachineDriver *output);

#endif
