/***************************************************************************

                    -= Seta, Sammy, Visco (SSV) System =-

                    driver by   Luca Elia (l.elia@tin.it)


CPU          :      NEC V60

Sound Chip   :      Ensoniq ES5506 (OTTOR2)

Custom Chips :      ST-0004     (Video DAC)
                    ST-0005     (Parallel I/O)
                    ST-0006     (Video controller)
                    ST-0007     (System controller)

Others       :      Battery + MB3790 + LH5168D-10L (NVRAM)
                    DX-102              (I/O)
                    M62X42B             (RTC)
                    ST010
                    TA8210              (Audio AMP)
                    uPD71051/7001C      (UART)

-----------------------------------------------------------------------------------
Main Board  ROM Board   Year + Game                                 By
-----------------------------------------------------------------------------------
STA-0001    STS-0001    93  Super Real Mahjong PIV                  Seta
STA-0001    STS-0001    93  Dramatic Adventure Quiz Keith & Lucy    Visco
STA-0001    SAM-5127    93  Survival Arts                           Sammy
STA-0001    SAM-5127    93  Dyna Gears                              Sammy
STA-0001B   VISCO-001B  94  Drift Out '94                           Visco
STA-0001B   GOLF ROM    94  Eagle Shot Golf                         Sammy
STA-0001B   ?           94  Twin Eagle II - The Rescue Mission      Seta
STA-0001B   VISCO       95  Mobil Suit Gundam Final Shooting (1)    Banpresto
STA-0001B   P1-102A     95  Mahjong Hyper Reaction                  Sammy
?           ?           95  Ultra X Weapons / Ultra Keibitai        Banpresto + Tsuburaya Prod.
STA-0001B   VISCO-JJ1   96  Lovely Pop Mahjong JangJang Shimasho    Visco
STA-0001B   VISCO-001B  96  Storm Blade                             Visco
STA-0001B   P1-105A     96? Meosis Magic                            Sammy
STA-0001B   ?           97  Joryuu Syougi Kyoushitsu (2)            Visco
STA-0001B   VISCO-JJ1   97  Koi Koi Shimasho 2                      Visco
STA-0001B   P1-112A     97  Mahjong Hyper Reaction 2                Sammy
STA-0001B   ?           97  Monster Slider                          Visco / Datt
STA-0001    ?           97  Super Real Mahjong P7                   Seta
STA-0001B   ?           98  Gourmet Battle Quiz Ryorioh CooKing     Visco
STA-0001B   P1-112C     98  Pachinko Sexy Reaction                  Sammy
STA-0001B   B1-001A     99  Pachinko Sexy Reaction 2                Sammy
STA-0001B   P1-112C     99  Change Air Blade                        Visco
STA-0001B   VISCO-JJ1   00  Lovely Pop Mahjong JangJang Shimasho 2  Visco
STA-0001B   SSV_SUB     00  Vasara                                  Visco
STA-0001B   SSV_SUB     01  Vasara 2                                Visco
-----------------------------------------------------------------------------------

(1) Uses ST-0009 & ST-0020 & Light Gun
(2) Uses NEC V810 CPU as sub cpu

SSV Hardware Overview
Sammy/Seta/Visco, 1993-2001

SSV was a joint venture between "S"ammy, "S"eta and "V"isco to produce a high quality 32-bit
RISC-based hardware platform.

The SSV main board contains all of the CPUs, RAM, custom ICs and sound chips. The lower PCB plugs
into 4x 64-pin connectors and contains the game software and any additional game-specific connectors
for controllers etc. The SSV games can be swapped by simply exchanging the lower (game) PCB onto any
SSV main board. There is an older main board STA-0001 which is not compatible with _some_ of the
newer games. The actual PCB will plug in, but the games will not boot up, instead displaying some
kind of debug screen full of numbers. This is known to occur on Vasara and Vasara 2 but may occur
with other games also. AFAIK, the actual PCBs look identical!

PCB Layouts
-----------

Main Board (common to all SSV-based games)

STA-0001B SYSTEM SSV
|---------------------------------------------------------|
| 2066   VOL                                              |
|TA8210                    |--------|      74ALS245 x6    |
|      TL072 |--------|    |        |                     |
|H           |ENSONIC |    | ST0007 |                 JP1 |
|      TL072 |ES5506  |    |        |   |--------|JP2     |
|            |OTTOR2  |    |--------|   |        |74ALS74 |
|S           |--------|                 |  V60   |        |
|      D6376                74ALS08     |        |74F32   |
|   74LS07                              |--------|        |
|          TC4050                                 JP3     |
|J TD62064                                                |
|A              74ALS245                                  |
|M      LM358      74ALS245                               |
|M        |------|     74ALS245                           |
|A        |ST0004|                 681000  681000         |
|         |      |                                        |
||------| |------| 43256  43256 43256       43256  43256  |
||ST0005|                         |--------|      JP4     |
||      |      48MHz              |        |       MB3771 |
||------|42.9545MHz   74AC245     | ST0006 | 74ALS273     |
|                         74F245  |        |       74F04  |
| DSW1(8)   DSW2(8)               |--------|              |
|---------------------------------------------------------|
(Note All IC's Shown)

Notes:
      H       - 3 pin connector for additional sound output
      S       - 5 pin connector for additional sound output
      D6376   - NEC D6376 2-channel 16-bit D/A converter (SOIC16)
      V60     - NEC D70615GD-16 V60 CPU, clock 16.000MHz [48/3] (QFP120)
      ES5506  - Ensonic ES5506 sound IC, clock 16.000MHz [48/3] (PLCC68)
      LM358   - National Semiconductor LM358 Low Power Dual Operational Amplifier (DIP8)
      2066    - New Japan Radio NJM2066 Dual Operational Amplifier (DIP16)
      TL072   - Texas Instruments TL072 Low Noise JFET-Input Operational Amplifier (DIP8)
      TA8210  - Toshiba TA8210 20W 2-channel Audio Power Amplifier
      43256   - Sharp LH52B256-70LL 32K x8 SRAM (DIP28)
                Note all sockets with 43256 can accept DIP32 128K x8 SRAM instead depending on
                game requirements
      681000  - Samsung KM681000ALP-7L 128K x8 SRAM (DIP32)
      JP1     - 3 pin jumper, hardwired on PCB to 2-3
      JP2     - 3 pin jumper, hardwired on PCB to 2-3
      JP3     - 3 pin jumper, hardwired on PCB to 1-3
      JP4     - 3 pin jumper, hardwired on PCB to 1-3

      SETA custom IC's -
                         ST-0004 Video DAC (QFP64)
                         ST-0005 Parallel I/O (QFP100)
                         ST-0006 Video controller (QFP208, covered with large heaksink)
                         ST-0007 System controller (QFP160)

To Do:

- all games :   CRT controller (resolution+visible area+flip screen?)

- dynagear  :   visible area may be incorrect

- hypreac2  :   communication with other units
                tilemap sprites use the yoffset specified in the sprites-list?
                (see the 8 pixel gap between the backgrounds and the black rows)

- keithlcy  :   Backgrounds are offset by $20 pixels, so they're kludged to work; there is a
                x&y offsets in the sprite list but in a same frame there are sprite with yoffset = 0
                and sprite with a yoffset = $7c but for all sprite the offset must be the same ($20)

- srmp4     :   Backgrounds are offset by $60 pixels, so they're kludged to work

- srmp7     :   Needs interrupts by the sound chip (not properly hooked up yet). Kludged to work.

- stmblade  :   There is a rogue "tilemap" sprite that pops up at level 2 and stays
                there till the end of the game (a piece of sky to the left of the screen).
                It seems that the x&y offsets in the sprite list should be apllied
                to it (-$200,-$200) to move it off screen. But currently those offsets
                are ignored for "tilemap" sprites. This may be related to the kludge for srmp4
                and to the kludge for keithlcy;

- ultrax    :   Backgrounds are offset by $10 pixels, so they're kludged to work


    code @ $e75cdc

     W:
            0x482000 - 0x482007 - values taken from obj table
            0x482040 - 0x482043 - write latch ?

     R:
            0x482022 - 0x482023 - result = direction, probably : 00 = down, 40 = left, 80 = up, c0 = right
            0x482042 - 0x482043 - protection status bits ?

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "machine/random.h"
#include "machine/eeprom.h"

#include "seta.h"

#include <math.h>

#define NO_DUMP 0
#define REGION_PLDS REGION_USER2

/***************************************************************************


                                Interrupts


***************************************************************************/

static UINT8 requested_int;
static UINT16 *ssv_irq_vectors;
static UINT16 irq_enable;
static UINT16 *ssv_mainram;

/* Update the IRQ state based on all possible causes */
static void update_irq_state(void)
{
	cpu_set_irq_line(0, 0, (requested_int & irq_enable)? ASSERT_LINE : CLEAR_LINE);
}

int ssv_irq_callback(int level)
{
	int i;
	for ( i = 0; i <= 7; i++ )
	{
		if (requested_int & (1 << i))
		{
			UINT16 vector = ssv_irq_vectors[i * (16/2)] & 7;
			return vector;
		}
	}
	return 0;
}

WRITE16_HANDLER( ssv_irq_ack_w )
{
	int level = ((offset * 2) & 0x70) >> 4;
	requested_int &= ~(1 << level);

	update_irq_state();
}

/*
    IRQ Enable Register:

    drifto94:   c at the start
    hypreact:   ff at the start
    hypreac2:   ff at the start
    janjans1:   0,6c,60
    keithlcy:   c at the start
    meosism:    ff at the start
    mslider:    c at the start
    ryorioh:    0,c at the start
    srmp4:      8 at the start
    srmp7:      8 at the start, 28, 40 (seems related to 21000e writes)
    survarts:   0,8 at the start
    sxyreact:   ff at the start
    ultrax:     40,00 at the start then 42,4a
    twineag2:   40,00 at the start then 42,4a
*/
WRITE16_HANDLER( ssv_irq_enable_w )
{
	COMBINE_DATA(&irq_enable);
}

static int interrupt_ultrax;

INTERRUPT_GEN( ssv_interrupt )
{
	if (cpu_getiloops())
	{
		if(interrupt_ultrax)
		{
			requested_int |= 1 << 1;	// needed by ultrax to coin up, breaks cairblad
			update_irq_state();
		}
	}
	else
	{
		requested_int |= 1 << 3;	// vblank
		update_irq_state();
	}
}

INTERRUPT_GEN( gdfs_interrupt )
{
	if (cpu_getiloops())
	{
		requested_int |= 1 << 6;	// reads lightgun (4 times for 4 axis)
		update_irq_state();
	}
	else
	{
		requested_int |= 1 << 3;	// vblank
		update_irq_state();
	}
}

/***************************************************************************


                            Coins Lockout / Counter


***************************************************************************/

/*
    drifto94:   c3
    janjans1:   c3
    keithlcy:   c3
    mslider:    c3, 83 in test mode
    ryorioh:    c3

    hypreac2:   80
    hypreact:   80
    meosism:    83
    srmp4:      83, c0 in test mode (where only tilemap sprites are used)
    srmp7:      80
    survarts:   83
    sxyreact:   80
*/
static WRITE16_HANDLER( ssv_lockout_w )
{
//	usrintf_showmessage("%02X",data & 0xff);
	if (ACCESSING_LSB)
	{
		coin_lockout_w(1,~data & 0x01);
		coin_lockout_w(0,~data & 0x02);
		coin_counter_w(1, data & 0x04);
		coin_counter_w(0, data & 0x08);
//                        data & 0x40?
		ssv_enable_video( data & 0x80);
	}
}

/* Same as above but with inverted lockout lines */
static WRITE16_HANDLER( ssv_lockout_inv_w )
{
//	usrintf_showmessage("%02X",data & 0xff);
	if (ACCESSING_LSB)
	{
		coin_lockout_w(1, data & 0x01);
		coin_lockout_w(0, data & 0x02);
		coin_counter_w(1, data & 0x04);
		coin_counter_w(0, data & 0x08);
//                        data & 0x40?
		ssv_enable_video( data & 0x80);
	}
}

MACHINE_INIT( ssv )
{
	requested_int = 0;
	cpu_set_irq_callback(0, ssv_irq_callback);
	cpu_setbank(1, memory_region(REGION_USER1));
}


/***************************************************************************


                            Non-Volatile RAM


***************************************************************************/

static UINT16 *ssv_nvram;
static size_t    ssv_nvram_size;

NVRAM_HANDLER( ssv )
{
	if (read_or_write)
		osd_fwrite(file, ssv_nvram, ssv_nvram_size);
	else
		if (file)
			osd_fread(file, ssv_nvram, ssv_nvram_size);
}

NVRAM_HANDLER( gdfs )
{
	if (read_or_write)
		EEPROM_save(file);
	else
	{
		EEPROM_init(&eeprom_interface_93C46);

		if (file) EEPROM_load(file);
		else
		{
			/* Set the EEPROM to Factory Defaults */
		}
	}
}

/***************************************************************************


                                DSP


***************************************************************************/


static UINT16 *dsp_ram;

static WRITE16_HANDLER( dsp_w )
{
	COMBINE_DATA(dsp_ram+offset);
	if(offset == 0x21 && dsp_ram[0x21]) {
		switch(dsp_ram[0x20]) {
		case 0x0001:
			dsp_ram[0x11] = (UINT8)(128*atan2(dsp_ram[0] - dsp_ram[1], dsp_ram[2] - dsp_ram[3])/M_PI) ^ 0x80;
			dsp_ram[0x21] = 0;
			break;
		default:
			dsp_ram[0x21] = 0;
			logerror("SSV DSP: unknown function %x (%x)\n", dsp_ram[0x20], activecpu_get_pc());
			break;
		}
	}
}



/***************************************************************************


                                Memory Maps


***************************************************************************/

//static READ16_HANDLER( fake_r )	{	return ssv_scroll[offset];	}

#define SSV_READMEM( _ROM  )										\
	{ 0x000000, 0x00ffff, MRA16_RAM				},	/*	RAM		*/	\
	{ 0x100000, 0x13ffff, MRA16_RAM				},	/*	Sprites	*/	\
	{ 0x140000, 0x15ffff, MRA16_RAM				},	/*	Palette	*/	\
	{ 0x160000, 0x17ffff, MRA16_RAM				},	/*			*/	\
	{ 0x1c0000, 0x1c0001, ssv_vblank_r			},	/*	Vblank?	*/	\
/**/{ 0x1c0002, 0x1c007f, MRA16_RAM				},	/*	Scroll	*/	\
	{ 0x210002, 0x210003, input_port_0_word_r	},	/*	DSW		*/	\
	{ 0x210004, 0x210005, input_port_1_word_r	},	/*	DSW		*/	\
	{ 0x210008, 0x210009, input_port_2_word_r	},	/*	P1		*/	\
	{ 0x21000a, 0x21000b, input_port_3_word_r	},	/*	P2		*/	\
	{ 0x21000c, 0x21000d, input_port_4_word_r	},	/*	Coins	*/	\
	{ 0x21000e, 0x21000f, MRA16_NOP				},	/*			*/	\
	{ 0x300000, 0x30007f, ES5506_data_0_word_r	},	/*	Sound	*/	\
	{ 0x482000, 0x482fff, MRA16_RAM				},	\
	{ _ROM,     0xffffff, MRA16_BANK1			},	/*	ROM		*/	\
	/*{ 0x990000, 0x99007f, fake_r	},*/

#define SSV_WRITEMEM														\
	{ 0x000000, 0x00ffff, MWA16_RAM						},	/*	RAM			*/	\
	{ 0x100000, 0x13ffff, MWA16_RAM, &spriteram16		},	/*	Sprites		*/	\
	{ 0x140000, 0x15ffff, paletteram16_xrgb_swap_word_w, &paletteram16	},		\
	{ 0x160000, 0x17ffff, MWA16_RAM						},	/*				*/	\
	{ 0x1c0000, 0x1c007f, ssv_scroll_w, &ssv_scroll		},	/*	Scroll		*/	\
	{ 0x21000e, 0x21000f, ssv_lockout_w					},	/*	Lockout		*/	\
	{ 0x210010, 0x210011, MWA16_NOP						},	/*				*/	\
	{ 0x230000, 0x230071, MWA16_RAM, &ssv_irq_vectors	},	/*	IRQ Vectors	*/	\
	{ 0x240000, 0x240071, ssv_irq_ack_w					},	/*	IRQ Ack.	*/	\
	{ 0x260000, 0x260001, ssv_irq_enable_w				},	/*	IRQ Enable	*/	\
	{ 0x300000, 0x30007f, ES5506_data_0_word_w			},	/*	Sound		*/\
	{ 0x482000, 0x482fff, dsp_w, &dsp_ram				},	\
	/*{ 0x990000, 0x99007f, ssv_scroll_w	},*/


static UINT16 *ssv_input_sel;

/***************************************************************************
                                Drift Out '94
***************************************************************************/

static READ16_HANDLER( drifto94_rand_r )
{
	return mame_rand() & 0xffff;
}

static MEMORY_READ16_START( drifto94_readmem )
	{ 0x480000, 0x480001, MRA16_NOP				},	// ?
	{ 0x510000, 0x510001, drifto94_rand_r		},	// ??
	{ 0x520000, 0x520001, drifto94_rand_r		},	// ??
	{ 0x580000, 0x5807ff, MRA16_RAM				},	// NVRAM
	SSV_READMEM( 0xc00000 )
MEMORY_END
static MEMORY_WRITE16_START( drifto94_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP				},	// ? 1 at the start
	{ 0x400000, 0x47ffff, MWA16_RAM				},	// ?
	{ 0x480000, 0x480001, MWA16_NOP				},	// ?
	{ 0x483000, 0x485fff, MWA16_NOP				},	// ?
	{ 0x500000, 0x500001, MWA16_NOP				},	// ??
	{ 0x580000, 0x5807ff, MWA16_RAM, &ssv_nvram, &ssv_nvram_size	},	// NVRAM
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                     Mobil Suit Gundam Final Shooting
***************************************************************************/

static int gdfs_gfxram_bank, gdfs_lightgun_select;
static UINT16 *gdfs_blitram;

READ16_HANDLER( gdfs_eeprom_r )
{
	return (((gdfs_lightgun_select & 1) ? 0 : 0xff) ^ readinputport(5 + gdfs_lightgun_select)) | (EEPROM_read_bit() << 8);
}

WRITE16_HANDLER( gdfs_eeprom_w )
{
	static UINT16 data_old;

	if (data & ~0x7b00)
		logerror("CPU #0 PC: %06X - Unknown EEPROM bit written %04X\n",activecpu_get_pc(),data);

	if ( ACCESSING_MSB )
	{
//      data & 0x8000 ? (near palette writes)
//      data & 0x0001 ?

		// latch the bit
		EEPROM_write_bit(data & 0x4000);

		// reset line asserted: reset.
		EEPROM_set_cs_line((data & 0x1000) ? CLEAR_LINE : ASSERT_LINE );

		// clock line asserted: write latch or select next bit to read
		EEPROM_set_clock_line((data & 0x2000) ? ASSERT_LINE : CLEAR_LINE );

		if (!(data_old & 0x0800) && (data & 0x0800))	// rising clock
			gdfs_lightgun_select = (data & 0x0300) >> 8;
	}

	COMBINE_DATA(&data_old);
}


static READ16_HANDLER( gdfs_gfxram_r )
{
	return eaglshot_gfxram[offset + gdfs_gfxram_bank * 0x100000/2];
}

static WRITE16_HANDLER( gdfs_gfxram_w )
{
	offset += gdfs_gfxram_bank * 0x100000/2;
	COMBINE_DATA(&eaglshot_gfxram[offset]);

	eaglshot_dirty = 1;
	eaglshot_dirty_tile[offset / (16*8/2)] = 1;
}

static READ16_HANDLER( gdfs_blitram_r )
{
	switch (offset)
	{
		case 0x00/2:
			// blitter status? (bit C, bit A)
			return 0;
	}

	logerror("CPU #0 PC: %06X - Blit reg read: %02X\n",activecpu_get_pc(),offset*2);
	return 0;
}

static WRITE16_HANDLER( gdfs_blitram_w )
{
	COMBINE_DATA(&gdfs_blitram[offset]);

	switch (offset)
	{
		case 0x8a/2:
		{
			if (data & ~0x43)
				logerror("CPU #0 PC: %06X - Unknown gdfs_gfxram_bank bit written %04X\n",activecpu_get_pc(),data);

			if (ACCESSING_LSB)
				gdfs_gfxram_bank = data & 3;
		}
		break;

		case 0xc0/2:
		case 0xc2/2:
		case 0xc4/2:
		case 0xc6/2:
		case 0xc8/2:
		break;

		case 0xca/2:
		{
			UINT32 src	=	(gdfs_blitram[0xc0/2] + (gdfs_blitram[0xc2/2] << 16)) << 1;
			UINT32 dst	=	(gdfs_blitram[0xc4/2] + (gdfs_blitram[0xc6/2] << 16)) << 4;
			UINT32 len	=	(gdfs_blitram[0xc8/2]) << 4;

			UINT8 *rom	=	memory_region(REGION_GFX2);
			size_t size	=	memory_region_length(REGION_GFX2);

			if ( (src+len <= size) && (dst+len <= 4 * 0x100000) )
			{
				eaglshot_dirty = 1;

				memcpy( &eaglshot_gfxram[dst/2], &rom[src], len );

				if (len % (16*8))	len = len / (16*8) + 1;
				else				len = len / (16*8);

				memset( &eaglshot_dirty_tile[dst / (16*8)], 1, len );
			}
			else
			{
				logerror("CPU #0 PC: %06X - Blit out of range: src %x, dst %x, len %x\n",activecpu_get_pc(),src,dst,len);
			}
		}
		break;

		default:
			logerror("CPU #0 PC: %06X - Blit reg written: %02X <- %04X\n",activecpu_get_pc(),offset*2,data);
	}
}

static MEMORY_READ16_START( gdfs_readmem )
	{ 0x540000, 0x540001, gdfs_eeprom_r			},
	{ 0x600000, 0x600fff, MRA16_RAM				},
	{ 0x400000, 0x43ffff, MRA16_RAM				},
	{ 0x440000, 0x44003f, MRA16_RAM				},
	{ 0x800000, 0x87ffff, MRA16_RAM				},
	{ 0x8c0000, 0x8c00ff, gdfs_blitram_r		},
	{ 0x900000, 0x9fffff, gdfs_gfxram_r			},
	SSV_READMEM( 0xc00000 )
MEMORY_END
static MEMORY_WRITE16_START( gdfs_writemem )
	{ 0x400000, 0x41ffff, gdfs_tmapram_w, &gdfs_tmapram			},
	{ 0x420000, 0x43ffff, MWA16_RAM								},
	{ 0x440000, 0x44003f, MWA16_RAM, &gdfs_tmapscroll			},
	{ 0x500000, 0x500001, gdfs_eeprom_w							},
	{ 0x600000, 0x600fff, MWA16_RAM								},
	{ 0x800000, 0x87ffff, MWA16_RAM, &spriteram16_2				},
	{ 0x8c0000, 0x8c00ff, gdfs_blitram_w, &gdfs_blitram			},
	{ 0x900000, 0x9fffff, gdfs_gfxram_w							},
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Hyper Reaction
***************************************************************************/

/*
    The game prints "backup ram ok" and there is code to test some ram
    at 0x580000-0x5bffff. The test is skipped and this ram isn't used
    though. I guess it's either a left-over or there are different
    version with some battery backed RAM (which would indeed be on the
    rom-board, AFAIK)
*/

static READ16_HANDLER( hypreact_input_r )
{
	UINT16 input_sel = *ssv_input_sel;
	if (input_sel & 0x0001)	return readinputport(5);
	if (input_sel & 0x0002)	return readinputport(6);
	if (input_sel & 0x0004)	return readinputport(7);
	if (input_sel & 0x0008)	return readinputport(8);
	logerror("CPU #0 PC %06X: unknown input read: %04X\n",activecpu_get_pc(),input_sel);
	return 0xffff;
}

static MEMORY_READ16_START( hypreact_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r		},	// Watchdog
//	{ 0x280000, 0x280001, MRA16_NOP					},	// ? read at the start, value not used
	{ 0xc00000, 0xc00001, hypreact_input_r			},	// Inputs
	{ 0xc00006, 0xc00007, MRA16_RAM					},	//
	{ 0xc00008, 0xc00009, MRA16_NOP					},	//
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( hypreact_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP					},	// ? 5 at the start
	{ 0x21000e, 0x21000f, ssv_lockout_inv_w			},	// Inverted lockout lines
	{ 0xc00006, 0xc00007, MWA16_RAM, &ssv_input_sel	},	// Inputs
	{ 0xc00008, 0xc00009, MWA16_NOP					},	//
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Hyper Reaction 2
***************************************************************************/

static MEMORY_READ16_START( hypreac2_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r		},	// Watchdog
//	{ 0x280000, 0x280001, MRA16_NOP					},	// ? read at the start, value not used
	{ 0x500000, 0x500001, hypreact_input_r			},	// Inputs
	{ 0x500002, 0x500003, hypreact_input_r			},	// (again?)
//	  0x540000, 0x540003  communication with another unit
	SSV_READMEM( 0xe00000 )
MEMORY_END
static MEMORY_WRITE16_START( hypreac2_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP					},	// ? 5 at the start
	{ 0x21000e, 0x21000f, ssv_lockout_inv_w			},	// Inverted lockout lines
	{ 0x520000, 0x520001, MWA16_RAM, &ssv_input_sel	},	// Inputs
//	  0x540000, 0x540003  communication with other units
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Jan Jan Simasyo
***************************************************************************/

static READ16_HANDLER( srmp4_input_r );

static MEMORY_READ16_START( janjans1_readmem )
	{ 0x210006, 0x210007, MRA16_NOP					},
	{ 0x800002, 0x800003, srmp4_input_r				},	// Inputs
	SSV_READMEM( 0xc00000 )
MEMORY_END
static MEMORY_WRITE16_START( janjans1_writemem )
	{ 0x210000, 0x210001, MWA16_NOP					},	// koikois2 but not janjans1
//	{ 0x210002, 0x210003, MWA16_NOP					},	// ? 1 at the start
	{ 0x800000, 0x800001, MWA16_RAM, &ssv_input_sel	},	// Inputs
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Keith & Lucy
***************************************************************************/

static MEMORY_READ16_START( keithlcy_readmem )
	{ 0x21000e, 0x21000f, MRA16_NOP			},	//
	SSV_READMEM( 0xe00000 )
MEMORY_END
static MEMORY_WRITE16_START( keithlcy_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP			},	// ? 1 at the start
	{ 0x210010, 0x210011, MWA16_NOP			},	//
	{ 0x400000, 0x47ffff, MWA16_RAM			},	// ?
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Meosis Magic
***************************************************************************/

static MEMORY_READ16_START( meosism_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog
//	{ 0x280000, 0x280001, MRA16_NOP				},	// ? read once, value not used
	{ 0x580000, 0x58ffff, MRA16_RAM				},	// NVRAM
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( meosism_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP				},	// ? 5 at the start
//	{ 0x500004, 0x500005, MWA16_NOP				},	// ? 0,58,18
	{ 0x580000, 0x58ffff, MWA16_RAM, &ssv_nvram, &ssv_nvram_size	},	// NVRAM
	SSV_WRITEMEM
MEMORY_END

/***************************************************************************
                                Monster Slider
***************************************************************************/

/* Monster Slider needs the RAM mirrored for the gameplay logic to work correctly */

static READ16_HANDLER( ssv_mainram_r )
{
	return ssv_mainram[offset];
}

static WRITE16_HANDLER( ssv_mainram_w )
{
	COMBINE_DATA(&ssv_mainram[offset]);
}


static MEMORY_READ16_START( mslider_readmem )
	{ 0x010000, 0x01ffff, ssv_mainram_r		}, // RAM Mirror
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( mslider_writemem )
	{ 0x010000, 0x01ffff, ssv_mainram_w		}, // RAM Mirror
//	{ 0x210002, 0x210003, MWA16_NOP			},	// ? 1 at the start
	{ 0x400000, 0x47ffff, MWA16_RAM			},	// ?
//	{ 0x500000, 0x500001, MWA16_NOP			},	// ? ff at the start
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                    Gourmet Battle Quiz Ryohrioh CooKing
***************************************************************************/

static MEMORY_READ16_START( ryorioh_readmem )
	SSV_READMEM( 0xc00000 )
MEMORY_END
static MEMORY_WRITE16_START( ryorioh_writemem )
	{ 0x210000, 0x210001, watchdog_reset16_w	},	// Watchdog
//	{ 0x210002, 0x210003, MWA16_NOP				},	// ? 1 at the start
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
							Super Real Mahjong PIV
***************************************************************************/

static READ16_HANDLER( srmp4_input_r )
{
	UINT16 input_sel = *ssv_input_sel;
	if (input_sel & 0x0002)	return readinputport(5);
	if (input_sel & 0x0004)	return readinputport(6);
	if (input_sel & 0x0008)	return readinputport(7);
	if (input_sel & 0x0010)	return readinputport(8);
	logerror("CPU #0 PC %06X: unknown input read: %04X\n",activecpu_get_pc(),input_sel);
	return 0xffff;
}

static MEMORY_READ16_START( srmp4_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r		},	// Watchdog
	{ 0xc0000a, 0xc0000b, srmp4_input_r				},	// Inputs
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( srmp4_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP					},	// ? 1,5 at the start
	{ 0xc0000e, 0xc0000f, MWA16_RAM, &ssv_input_sel	},	// Inputs
	{ 0xc00010, 0xc00011, MWA16_NOP					},	//
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                            Super Real Mahjong P7
***************************************************************************/

/*
    Interrupts aren't supported by the chip emulator yet
    (lev 5 in this case, I guess)
*/
static READ16_HANDLER( srmp7_irqv_r )
{
	return 0x0080;
}

static WRITE16_HANDLER( srmp7_sound_bank_w )
{
	if (ACCESSING_LSB)
	{
		int bank = 0x400000/2 * (data & 1);	// UINT16 address
		int voice;
		for (voice = 0; voice < 32; voice++)
			ES5506_voice_bank_0_w(voice, bank);
	}
//	usrintf_showmessage("%04X",data);
}

static READ16_HANDLER( srmp7_input_r )
{
	UINT16 input_sel = *ssv_input_sel;
	if (input_sel & 0x0002)	return readinputport(5);
	if (input_sel & 0x0004)	return readinputport(6);
	if (input_sel & 0x0008)	return readinputport(7);
	if (input_sel & 0x0010)	return readinputport(8);
	logerror("CPU #0 PC %06X: unknown input read: %04X\n",activecpu_get_pc(),input_sel);
	return 0xffff;
}

static MEMORY_READ16_START( srmp7_readmem )
	{ 0x010000, 0x050faf, MRA16_RAM				},	// More RAM
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog
	{ 0x300076, 0x300077, srmp7_irqv_r			},	// Sound
//	  0x540000, 0x540003, related to lev 5 irq?
	{ 0x600000, 0x600001, srmp7_input_r			},	// Inputs
	SSV_READMEM( 0xc00000 )
MEMORY_END
static MEMORY_WRITE16_START( srmp7_writemem )
	{ 0x010000, 0x050faf, MWA16_RAM					},	// More RAM
//	{ 0x210002, 0x210003, MWA16_NOP					},	// ? 0,4 at the start
	{ 0x21000e, 0x21000f, ssv_lockout_inv_w			},	// Coin Counters / Lockouts
//	  0x540000, 0x540003, related to lev 5 irq?
	{ 0x580000, 0x580001, srmp7_sound_bank_w		},	// Sound Bank
	{ 0x680000, 0x680001, MWA16_RAM, &ssv_input_sel	},	// Inputs
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Survival Arts
***************************************************************************/

static MEMORY_READ16_START( survarts_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog
//	{ 0x290000, 0x290001, MRA16_NOP				},	// ?
//	{ 0x2a0000, 0x2a0001, MRA16_NOP				},	// ?

	{ 0x400000, 0x43ffff, MRA16_RAM				},	// dyna

	{ 0x500008, 0x500009, input_port_5_word_r	},	// Extra Buttons
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( survarts_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP				},	// ? 0,4 at the start
	{ 0x400000, 0x43ffff, MWA16_RAM				},	// dyna

	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                            Pachinko Sexy Reaction
***************************************************************************/

static UINT16 serial;

static READ16_HANDLER( sxyreact_ballswitch_r )
{
	return readinputport(5);
}

static READ16_HANDLER( sxyreact_dial_r )
{
	return ((serial >> 1) & 0x80);
}

static WRITE16_HANDLER( sxyreact_dial_w )
{
	if (ACCESSING_LSB)
	{
		static int old;

		if (data & 0x20)
			serial = readinputport(6) & 0xff;

		if ( (old & 0x40) && !(data & 0x40) )	// $40 -> $00
			serial <<= 1;						// shift 1 bit

		old = data;
	}
}

static WRITE16_HANDLER( sxyreact_motor_w )
{
//	usrintf_showmessage("%04X",data);	// 8 = motor on; 0 = motor off
}

static MEMORY_READ16_START( sxyreact_readmem )
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog
	{ 0x500002, 0x500003, sxyreact_ballswitch_r	},	// ?
	{ 0x500004, 0x500005, sxyreact_dial_r		},	// Dial Value (serial)
	{ 0x580000, 0x58ffff, MRA16_RAM				},	// NVRAM
	SSV_READMEM( 0xe00000 )
MEMORY_END
static MEMORY_WRITE16_START( sxyreact_writemem )
//	{ 0x210002, 0x210003, MWA16_NOP				},	// ? 1 at the start
	{ 0x21000e, 0x21000f, ssv_lockout_inv_w		},	// Inverted lockout lines
	{ 0x520000, 0x520001, sxyreact_dial_w		},	// Dial Value (advance 1 bit)
	{ 0x520004, 0x520005, sxyreact_motor_w		},	// Dial Motor?
	{ 0x580000, 0x58ffff, MWA16_RAM, &ssv_nvram, &ssv_nvram_size	},	// NVRAM
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                Twin Eagle II
***************************************************************************/

static MEMORY_READ16_START( twineag2_readmem )
	{ 0x010000, 0x03ffff, MRA16_RAM				},	// More RAM
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog (also value is cmp.b with mem 8)
	SSV_READMEM( 0xe00000 )
MEMORY_END
static MEMORY_WRITE16_START( twineag2_writemem )
	{ 0x010000, 0x03ffff, MWA16_RAM				},	// More RAM
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************
                                    Ultra X
***************************************************************************/

static MEMORY_READ16_START( ultrax_readmem )
	{ 0x010000, 0x03ffff, MRA16_RAM				},	// More RAM
	{ 0x210000, 0x210001, watchdog_reset16_r	},	// Watchdog (also value is cmp.b with memory address 8)
	SSV_READMEM( 0xe00000 )
MEMORY_END
static MEMORY_WRITE16_START( ultrax_writemem )
	{ 0x010000, 0x03ffff, MWA16_RAM			},	// More RAM
//	{ 0x210002, 0x210003, MWA16_NOP			},	// ? 2,6 at the start
	SSV_WRITEMEM
MEMORY_END

/***************************************************************************
  Eagle Shot Golf
***************************************************************************/

static UINT8 trackball_select, gfxrom_select;

static READ16_HANDLER( eaglshot_gfxrom_r )
{
	UINT8 *rom	=	memory_region(REGION_GFX1);
	size_t size	=	memory_region_length(REGION_GFX1);

	offset = offset * 2 + gfxrom_select * 0x200000;

	if (offset > size)
		return 0xffff;

	return rom[offset] + (rom[offset+1]<<8);
}

static WRITE16_HANDLER( eaglshot_gfxrom_w )
{
	if (ACCESSING_LSB)
		gfxrom_select = data;
}

static READ16_HANDLER( eaglshot_trackball_r )
{
	switch(trackball_select)
	{
		case 0x60:	return (readinputport(5) >> 8) & 0xff;
		case 0x40:	return (readinputport(5) >> 0) & 0xff;

		case 0x70:	return (readinputport(6) >> 8) & 0xff;
		case 0x50:	return (readinputport(6) >> 0) & 0xff;
	}
	return 0;
}

static WRITE16_HANDLER( eaglshot_trackball_w )
{
	if (ACCESSING_LSB)
		trackball_select = data;
}



static READ16_HANDLER( eaglshot_gfxram_r )
{
	return eaglshot_gfxram[offset + (ssv_scroll[0x76/2] & 0xf) * 0x40000/2];
}

static WRITE16_HANDLER( eaglshot_gfxram_w )
{
	offset += (ssv_scroll[0x76/2] & 0xf) * 0x40000/2;
	COMBINE_DATA(&eaglshot_gfxram[offset]);

	eaglshot_dirty = 1;
	eaglshot_dirty_tile[offset / (16*8/2)] = 1;
}


static MEMORY_READ16_START( eaglshot_readmem )
	{ 0x180000, 0x1bffff, eaglshot_gfxram_r							},
	{ 0x210000, 0x210001, /*watchdog_reset16_r*/MRA16_NOP			},	// Watchdog
	{ 0xa00000, 0xbfffff, eaglshot_gfxrom_r							},
	{ 0xc00000, 0xc007ff, MRA16_RAM									},	// NVRAM
	{ 0xd00000, 0xd00001, eaglshot_trackball_r						},
	SSV_READMEM( 0xf00000 )
MEMORY_END
static MEMORY_WRITE16_START( eaglshot_writemem )
	{ 0x180000, 0x1bffff, eaglshot_gfxram_w							},
//  { 0x210002, 0x210003, MWA16_NOP                                 },  // ? 0,4 at the start
	{ 0x21000e, 0x21000f, ssv_lockout_inv_w							},	// Inverted lockout lines
	{ 0x800000, 0x800001, eaglshot_gfxrom_w							},
	{ 0x900000, 0x900001, eaglshot_trackball_w						},
	{ 0xc00000, 0xc007ff, MWA16_RAM, &ssv_nvram, &ssv_nvram_size	},	// NVRAM
	SSV_WRITEMEM
MEMORY_END


/***************************************************************************


                                Input Ports


***************************************************************************/


/***************************************************************************
                           Change Air Blade
***************************************************************************/

INPUT_PORTS_START( cairblad )
	PORT_START	// IN0
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0060, 0x0060, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(      0x0040, "Every 2 Mil" )
	PORT_DIPSETTING(      0x0060, "2 Mil/6 Mil" )
	PORT_DIPSETTING(      0x0020, "4 Million" )
	PORT_DIPSETTING(      0x0000, "None" )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW, IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                                Drift Out '94
***************************************************************************/

INPUT_PORTS_START( drifto94 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0002, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0004, 0x0004, "Sound Test" )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x00c0, 0x00c0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 1C_2C ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x000c, 0x000c, "Unknown 2-2&3*" )
	PORT_DIPSETTING(      0x000c, "11 (0)" )
	PORT_DIPSETTING(      0x0008, "10 (1)" )
	PORT_DIPSETTING(      0x0004, "01 (0)" )
	PORT_DIPSETTING(      0x0000, "00 (2)" )
	PORT_DIPNAME( 0x0010, 0x0010, "Music Volume" )
	PORT_DIPSETTING(      0x0000, "Quiet" )
	PORT_DIPSETTING(      0x0010, "Loud" )
	PORT_DIPNAME( 0x0020, 0x0020, "Sound Volume" )
	PORT_DIPSETTING(      0x0000, "Quiet" )
	PORT_DIPSETTING(      0x0020, "Loud" )
	PORT_DIPNAME( 0x0040, 0x0040, "Save Best Time" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Unknown 2-7" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                                Eagle Shot Golf
***************************************************************************/

INPUT_PORTS_START( eaglshot )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x000f, 0x0009, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0009, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x000d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x000b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x000a, DEF_STR( 1C_6C ) )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0005, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0004, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0003, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0002, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c
	PORT_DIPSETTING(      0x0001, "Multiple Coin Feature E" )
// 1c-1c, 2c-3c
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x0010, 0x0010, "Discount to Continue" )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) ) // 2 Coins to start, 1 to continue
	PORT_DIPNAME( 0x0020, 0x0020, "Controls" )
	PORT_DIPSETTING(      0x0020, "Trackball" )
	PORT_DIPSETTING(      0x0000, "Joystick" )
	PORT_DIPNAME( 0x0040, 0x0040, "Trackball Type" )
	PORT_DIPSETTING(      0x0040, "24 Counts (USA)" )
	PORT_DIPSETTING(      0x0000, "12 Counts (Japan)" )
	PORT_DIPNAME( 0x0080, 0x0080, "Unused/Unknown" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, "Number of Holes" )
	PORT_DIPSETTING(      0x0002, "2 Holes" )
	PORT_DIPSETTING(      0x0003, "3 Holes" )
	PORT_DIPSETTING(      0x0001, "4 Holes" )
	PORT_DIPSETTING(      0x0000, "5 Holes" )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) ) // No listed value for ON & ON
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Allow Continue" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END

/***************************************************************************
                     Mobil Suit Gundam Final Shooting
***************************************************************************/

INPUT_PORTS_START( gdfs )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0000, "Controls" )
	PORT_DIPSETTING(      0x0001, "Joystick" )
	PORT_DIPSETTING(      0x0000, "Light_Gun" )
	PORT_DIPNAME( 0x0002, 0x0002, "Light Gun Calibration" )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0004, 0x0004, "Level_Select" ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0018, 0x0018, DEF_STR( Coinage ) )
//  PORT_DIPSETTING(      0x0000, DEF_STR( 2C_1C ) ) /* 2 Coins to Start, 1 Coin to Continue??? */
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Save Scores" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )	// Clear NVRAM on boot
	PORT_DIPSETTING(      0x0020, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, "Invert X Axis" )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Language" )
	PORT_DIPSETTING(      0x0000, "English" )
	PORT_DIPSETTING(      0x0008, "Japanese" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Damage From Machine Gun" )	// F76E34
	PORT_DIPSETTING(      0x0020, "Light" )
	PORT_DIPSETTING(      0x0000, "Heavy" )
	PORT_DIPNAME( 0x0040, 0x0040, "Damage From Beam Cannon" )	// F77487
	PORT_DIPSETTING(      0x0040, "Light" )
	PORT_DIPSETTING(      0x0000, "Heavy" )
	PORT_DIPNAME( 0x0080, 0x0080, "Damage From Missle" )	// F77255
	PORT_DIPSETTING(      0x0080, "Light" )
	PORT_DIPSETTING(      0x0000, "Heavy" )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BIT( 0x00f0, IP_ACTIVE_LOW, IPT_UNKNOWN  )

	PORT_START	// IN5 - $540000(0)
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 35, 10, 0, 255 )

	PORT_START	// IN6 - $540000(1)
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 35, 10, 0, 255 )

	PORT_START	// IN7 - $540000(2)
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X | IPF_PLAYER2, 35, 10, 0, 255 )

	PORT_START	// IN8 - $540000(3)
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y | IPF_PLAYER2, 35, 10, 0, 255 )
INPUT_PORTS_END


/***************************************************************************
                                Hyper Reaction
***************************************************************************/

INPUT_PORTS_START( hypreact )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Half Coins To Continue" )
	PORT_DIPSETTING(      0x0040, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy"    )
	PORT_DIPSETTING(      0x000c, "Normal"  )
	PORT_DIPSETTING(      0x0004, "Hard"    )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0010, 0x0010, "Controls" )
	PORT_DIPSETTING(      0x0010, "Keyboard" )
	PORT_DIPSETTING(      0x0000, "Joystick" )
	PORT_DIPNAME( 0x0020, 0x0020, "Unknown 2-5" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Unknown 2-6" )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008 (used in joystick mode)
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX( 0x0002, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE,    IP_JOY_NONE )
	PORT_BITX( 0x0004, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT,     IP_JOY_NONE )
	PORT_BITX( 0x0008, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN       )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN       )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP   )

	PORT_START	// IN3 - $21000a (used in joystick mode)
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BITX( 0x0002, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX( 0x0004, IP_ACTIVE_LOW, 0, "Ron",   KEYCODE_Z,        IP_JOY_NONE )
	PORT_BITX( 0x0008, IP_ACTIVE_LOW, 0, "Tsumo", KEYCODE_RCONTROL, IP_JOY_NONE )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN        )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN        )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )	// service coin & bet
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $c00000(0)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START1                              )
	PORT_BIT( 0xffc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6 - $c00000(1)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )
	PORT_BIT( 0xffc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $c00000(2)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0xffe0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $c00000(3)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0xfff0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                                Hyper Reaction 2
***************************************************************************/

INPUT_PORTS_START( hypreac2 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Half Coins To Continue" )
	PORT_DIPSETTING(      0x0040, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0010, 0x0010, "Controls" )
	PORT_DIPSETTING(      0x0010, "Keyboard" )
	PORT_DIPSETTING(      0x0000, "Joystick" )
	PORT_DIPNAME( 0x0020, 0x0020, "Communication 1" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Communication 2" )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $500000(0)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START1                              )
	PORT_BIT( 0xffc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6 - $500000(1)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BIT( 0xffe0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $500000(2)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0xffe0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $500000(3)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0xfff0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                                JangJang Shimasho
***************************************************************************/

INPUT_PORTS_START( janjans1 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, "Unknown 1-0" )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Voice" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Unknown 1-7" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0004, 0x0004, "Nudity" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Mini Game" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, "Initial Score" )
	PORT_DIPSETTING(      0x0020, "1000" )
	PORT_DIPSETTING(      0x0030, "1500" )
	PORT_DIPSETTING(      0x0010, "2000" )
	PORT_DIPSETTING(      0x0000, "3000" )
	PORT_DIPNAME( 0x00c0, 0x00c0, "Communication" )
//  PORT_DIPSETTING(      0x0080, "unused" )
	PORT_DIPSETTING(      0x00c0, "None" )
	PORT_DIPSETTING(      0x0040, "Board 1 (Main)" )
	PORT_DIPSETTING(      0x0000, "Board 2 (Sub)" )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $800002(0)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6 - $800002(1)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $800002(2)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $800002(3)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                                Keith & Lucy
***************************************************************************/

INPUT_PORTS_START( keithlcy )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x00c0, 0x00c0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 1C_2C ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy"	)	// 15 sec
	PORT_DIPSETTING(      0x0003, "Normal"	)	// 12
	PORT_DIPSETTING(      0x0001, "Hard"	)	// 10
	PORT_DIPSETTING(      0x0000, "Hardest"	)	// 8
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Lives ) )
	PORT_DIPSETTING(      0x0008, "2" )
	PORT_DIPSETTING(      0x000c, "3" )
	PORT_DIPSETTING(      0x0004, "4" )
	PORT_DIPSETTING(      0x0000, "5" )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(      0x0030, "Every 100k" )		//100
	PORT_DIPSETTING(      0x0020, "Every 150k" )		//150
	PORT_DIPSETTING(      0x0010, "100k & Every 200K" )	//100
	PORT_DIPSETTING(      0x0000, "Every 200k" )		//200
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) ) /* Manual lists these dips as "Unused" */
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) ) /* Manual lists these dips as "Unused" */
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1  )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2  )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                            Koi Koi Simasho 2
***************************************************************************/

INPUT_PORTS_START( koikois2 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Voice" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Controls" )
	PORT_DIPSETTING(      0x0080, "Joystick" )
	PORT_DIPSETTING(      0x0000, "Keyboard" )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0004, 0x0004, "Nudity" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Unknown ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x00c0, 0x00c0, "Communication" )
//  PORT_DIPSETTING(      0x0080, "unused" )
	PORT_DIPSETTING(      0x00c0, "None" )
	PORT_DIPSETTING(      0x0040, "Board 1 (Main)" )
	PORT_DIPSETTING(      0x0000, "Board 2 (Sub)" )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $800002(0)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6 - $800002(1)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $800002(2)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $800002(3)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                                Meosis Magic
***************************************************************************/

INPUT_PORTS_START( meosism )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0003, "1 Medal/1 Credit" )
	PORT_DIPSETTING(      0x0001, "1 Medal/5 Credits" )
	PORT_DIPSETTING(      0x0002, "1 Medal/10 Credits" )
	PORT_DIPSETTING(      0x0000, "1 Medal/20 Credits" )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Attendant Pay" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0010, 0x0010, "Medals Payout" )
	PORT_DIPSETTING(      0x0010, "400" )
	PORT_DIPSETTING(      0x0000, "800" )
	PORT_DIPNAME( 0x0020, 0x0020, "Max Credits" )
	PORT_DIPSETTING(      0x0020, "5000" )
	PORT_DIPSETTING(      0x0000, "9999" )
	PORT_DIPNAME( 0x0040, 0x0040, "Hopper" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Reel Speed" )
	PORT_DIPSETTING(      0x0080, "Low" )
	PORT_DIPSETTING(      0x0000, "High" )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, "Game Rate" )
	PORT_DIPSETTING(      0x0000, "80%" )
	PORT_DIPSETTING(      0x0002, "85%" )
	PORT_DIPSETTING(      0x0003, "90%" )
	PORT_DIPSETTING(      0x0001, "95%" )
	PORT_DIPNAME( 0x0004, 0x0004, "Unknown 2-2" )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Unknown 2-3" )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, "Unknown 2-4" )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0000, "Controls" )
//  PORT_DIPSETTING(      0x0020, "Simple") )
	PORT_DIPSETTING(      0x0000, "Complex" )
	PORT_DIPNAME( 0x0040, 0x0000, "Coin Sensor" )
	PORT_DIPSETTING(      0x0040, "Active High" )
	PORT_DIPSETTING(      0x0000, "Active Low" )
	PORT_DIPNAME( 0x0080, 0x0080, "Hopper Sensor" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_BUTTON4        )	//bet
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        )	//stop/r
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        )	//stop/c
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        )	//stop/l
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT )	//no
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  )	//yes
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_START1         )	//start
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN        )	//-

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN  )	//-
	PORT_BITX( 0x0002, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )	//test
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN  )	//-
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_SERVICE3 )	//payout
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_UNKNOWN  )	//-
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_TILT     )	//reset
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN  )	//-
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN  )	//-

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )	//service coin
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_SERVICE2 )	//analyzer
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON5  )	//max bet
	PORT_BIT(  0x00e0, IP_ACTIVE_LOW, IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                                Monster Slider
***************************************************************************/

INPUT_PORTS_START( mslider )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_SERVICE( 0x0040, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) ) /* Manual lists this dip as "Unused" */
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0030, 0x0030, "Rounds (Vs Mode)" )
	PORT_DIPSETTING(      0x0000, "1" )
	PORT_DIPSETTING(      0x0030, "2" )
	PORT_DIPSETTING(      0x0020, "3" )
	PORT_DIPSETTING(      0x0010, "4" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) ) /* Manual lists these dips as "Unused" */
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                    Gourmet Battle Quiz Ryohrioh CooKing
***************************************************************************/

INPUT_PORTS_START( ryorioh )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Unused ) ) /* Manual states this dip is "Unused" */
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) ) /* Manual states this dip is "Unused" */
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Unknown 1-7*" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Unused ) ) /* Manual states dips 3-8 are "Unused" */
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1  )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2  )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                            Super Real Mahjong PIV
***************************************************************************/

INPUT_PORTS_START( srmp4 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_4C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_4C ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Unknown 1-7" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0006, "Easiest" )
	PORT_DIPSETTING(      0x0005, "Easier" )
	PORT_DIPSETTING(      0x0004, "Easy" )
	PORT_DIPSETTING(      0x0007, "Normal" )
	PORT_DIPSETTING(      0x0003, "Medium" )
	PORT_DIPSETTING(      0x0002, "Hard" )
	PORT_DIPSETTING(      0x0001, "Harder" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0040, 0x0040, "Allow Continue" )
	PORT_DIPSETTING(      0x0000, DEF_STR( No ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Unknown 2-7" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $c0000a(0)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6 - $c0000a(1)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $c0000a(2)
	PORT_BITX(0x0001, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $c0000a(3)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                            Super Real Mahjong P7
***************************************************************************/

INPUT_PORTS_START( srmp7 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_4C ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Unknown 1-3" )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, "Unknown 1-4" )
	PORT_DIPSETTING(      0x0010, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Unknown 1-5" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Re-cloth" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Nudity" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0006, "Easiest" )
	PORT_DIPSETTING(      0x0005, "Easier" )
	PORT_DIPSETTING(      0x0004, "Easy" )
	PORT_DIPSETTING(      0x0007, "Normal" )
	PORT_DIPSETTING(      0x0003, "Medium" )
	PORT_DIPSETTING(      0x0002, "Hard" )
	PORT_DIPSETTING(      0x0001, "Harder" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0008, 0x0008, "Kuitan" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, "Allow Continue" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW,  IPT_UNKNOWN  )	// tested
	PORT_BIT(  0x00e0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN6 - $600000(0)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7 - $600000(1)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN8 - $600000(2)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX(0x0002, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN5 - $600000(3)
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x0004, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BITX(0x0008, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x0010, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x0020, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                                Storm Blade
***************************************************************************/

INPUT_PORTS_START( stmblade )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Rapid Fire" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Lives ) )
	PORT_DIPSETTING(      0x0020, "1" )
	PORT_DIPSETTING(      0x0010, "2" )
	PORT_DIPSETTING(      0x0030, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(      0x0040, "600000" )
	PORT_DIPSETTING(      0x0000, "800000" )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW, IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************
                                Survival Arts
***************************************************************************/

INPUT_PORTS_START( survarts )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x000f, 0x0009, DEF_STR( Coin_A ) ) // Verified Default is 2 coins 1 Credit
	PORT_DIPSETTING(      0x0007, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0009, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x000d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x000b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x000a, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(      0x0000, "2 Credits Start, 1 to continue" )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0005, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0004, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0003, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0002, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c
	PORT_DIPNAME( 0x00f0, 0x0090, DEF_STR( Coin_B ) ) // Verified Defualt is 2 coins 1 Credit
	PORT_DIPSETTING(      0x0070, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0090, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00f0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0060, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x00e0, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x00d0, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x00b0, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x00a0, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(      0x0000, "2 Credits Start, 1 to continue" )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0050, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0040, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0030, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0020, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0000, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0004, 0x0004, "Timer Speed" )
	PORT_DIPSETTING(      0x0004, "Normal" )
	PORT_DIPSETTING(      0x0000, "Fast" )
	PORT_DIPNAME( 0x0008, 0x0008, "Damage Level" )
	PORT_DIPSETTING(      0x0008, "Normal" )
	PORT_DIPSETTING(      0x0000, "High" )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0020, "Easy" )
	PORT_DIPSETTING(      0x0030, "Normal" )
	PORT_DIPSETTING(      0x0010, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x00c0, 0x0000, "Fatal Damage" )
	PORT_DIPSETTING(      0x00c0, "Light" )
	PORT_DIPSETTING(      0x0040, "Normal" )
	PORT_DIPSETTING(      0x0080, "Heavy" )
	PORT_DIPSETTING(      0x0000, "Heaviest" )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )

	PORT_START	// IN5 - $500008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

/***************************************************************************
                Dyna Gears
***************************************************************************/

INPUT_PORTS_START( dynagear )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x000f, 0x000f, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0009, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x000d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x000b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x000a, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(      0x0000, "2 Credits Start, 1 to continue" )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0005, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0004, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0003, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0002, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c
	PORT_DIPNAME( 0x00f0, 0x00f0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0070, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0090, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00f0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0060, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x00e0, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x00d0, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x00b0, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x00a0, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(      0x0000, "2 Credits Start, 1 to continue" )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0050, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0040, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0030, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0020, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c

	PORT_START	// IN0 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0000, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0030, 0x0030, "Lives" )
	PORT_DIPSETTING(      0x0010, "1" )
	PORT_DIPSETTING(      0x0030, "2" )
	PORT_DIPSETTING(      0x0020, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Health" )
	PORT_DIPSETTING(      0x0000, "3 Hearts" )
	PORT_DIPSETTING(      0x0080, "4 Hearts" )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
INPUT_PORTS_END

/***************************************************************************
                            Pachinko Sexy Reaction
***************************************************************************/

INPUT_PORTS_START( sxyreact )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_BIT(     0x0038, IP_ACTIVE_LOW, IPT_UNUSED )
//  PORT_DIPNAME( 0x0038, 0x0038, DEF_STR( Coin_B ) )
//  PORT_DIPSETTING(      0x0028, DEF_STR( 3C_1C ) )
//  PORT_DIPSETTING(      0x0030, DEF_STR( 2C_1C ) )
//  PORT_DIPSETTING(      0x0038, DEF_STR( 1C_1C ) )
//  PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
//  PORT_DIPSETTING(      0x0018, DEF_STR( 1C_3C ) )
//  PORT_DIPSETTING(      0x0010, DEF_STR( 1C_4C ) )
//  PORT_DIPSETTING(      0x0008, DEF_STR( 1C_5C ) )
//  PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Credits To Play" )
	PORT_DIPSETTING(      0x0040, "1" )
	PORT_DIPSETTING(      0x0000, "2" )
	PORT_DIPNAME( 0x0080, 0x0080, "Buy Balls With Credits" )	// press start
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( On ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x000c, 0x000c, "Difficulty" )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x000c, "Normal" )
	PORT_DIPSETTING(      0x0004, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0010, 0x0010, "Controls" )
	PORT_DIPSETTING(      0x0010, "Dial" )
	PORT_DIPSETTING(      0x0000, "Joystick" )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0040, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Unused ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )	// -> ball sensor on
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x00ff, IP_ACTIVE_LOW, IPT_UNKNOWN )	// (player 2, only shown in test mode)

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNUSED   )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BITX( 0x0010, IP_ACTIVE_LOW, IPT_SERVICE, "Test Advance", KEYCODE_F1, IP_JOY_DEFAULT )
	PORT_BIT(  0x00e0, IP_ACTIVE_LOW, IPT_UNKNOWN  )

	PORT_START	// IN5 - $500002
	PORT_BIT(  0x0001, IP_ACTIVE_HIGH,  IPT_SERVICE2 )	// ball switch on -> handle motor off

	PORT_START	// IN6 - $500004
	PORT_ANALOGX( 0xff, 0x00, IPT_PADDLE, 15, 15, 0, 0xcf, KEYCODE_N, KEYCODE_M, 0, 0 )
INPUT_PORTS_END


/***************************************************************************
                                Twin Eagle II
***************************************************************************/

INPUT_PORTS_START( twineag2 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x000f, 0x000f, DEF_STR( Coin_A ) ) // No values listed for all "ON"
	PORT_DIPSETTING(      0x0007, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0009, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x000d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x000b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x000a, DEF_STR( 1C_6C ) )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0005, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0004, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0003, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0002, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c
	PORT_DIPSETTING(      0x0001, "Multiple Coin Feature E" )
// 1c-1c, 2c-3c
	PORT_DIPNAME( 0x00f0, 0x00f0, DEF_STR( Coin_B ) ) // No values listed for all "ON"
	PORT_DIPSETTING(      0x0070, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0090, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00f0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0060, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x00e0, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x00d0, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x00b0, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x00a0, DEF_STR( 1C_6C ) )
// "** ADDED MULTIPLE COIN FEATURE **"
	PORT_DIPSETTING(      0x0050, "Multiple Coin Feature A" )
// 2c-1c, 4c-2c, 5c-3c & 6c-4c
	PORT_DIPSETTING(      0x0040, "Multiple Coin Feature B" )
// 2c-1c, 4c-3c
	PORT_DIPSETTING(      0x0030, "Multiple Coin Feature C" )
// 1c-1c, 2c-2c, 3c-3c, 4c-4c, 5c-6c
	PORT_DIPSETTING(      0x0020, "Multiple Coin Feature D" )
// 1c-1c, 2c-2c, 3c-3c & 4c-5c
	PORT_DIPSETTING(      0x0010, "Multiple Coin Feature E" )
// 1c-1c, 2c-3c

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0006, "Easiest" )
	PORT_DIPSETTING(      0x0005, "Easier" )
	PORT_DIPSETTING(      0x0004, "Easy" )
	PORT_DIPSETTING(      0x0007, "Normal" )
	PORT_DIPSETTING(      0x0003, "Medium" )
	PORT_DIPSETTING(      0x0002, "Hard" )
	PORT_DIPSETTING(      0x0001, "Harder" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Lives ) )
	PORT_DIPSETTING(      0x0000, "2" )
	PORT_DIPSETTING(      0x0010, "3" )
	PORT_DIPNAME( 0x0020, 0x0020, "Freeze" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW,  IPT_SERVICE1 )

	PORT_BIT(  0x0008, IP_ACTIVE_LOW,  IPT_SERVICE2 )
	PORT_BITX( 0x0010, IP_ACTIVE_LOW, IPT_SERVICE, "Test Advance", KEYCODE_F1, IP_JOY_DEFAULT )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW,  IPT_SERVICE4 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW,  IPT_START3 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW,  IPT_START4 )
INPUT_PORTS_END


/***************************************************************************
                                Ultra X
***************************************************************************/

INPUT_PORTS_START( ultrax )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x000f, 0x000f, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0009, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x000e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x000d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x000b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x000a, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x00f0, 0x00f0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0090, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00f0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x00e0, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x00d0, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(      0x00b0, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x00a0, DEF_STR( 1C_6C ) )

	PORT_START	// IN1 - $210004
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy"	)	//$140
	PORT_DIPSETTING(      0x0003, "Normal"	)	//$190
	PORT_DIPSETTING(      0x0001, "Hard"	)	//$200
	PORT_DIPSETTING(      0x0000, "Hardest"	)	//$300
	PORT_DIPNAME( 0x0014, 0x0014, "Country" )
	PORT_DIPSETTING(      0x0000, "China" )
	PORT_DIPSETTING(      0x0014, "Japan" )
//	PORT_DIPSETTING(      0x0010, "Japan" )
	PORT_DIPSETTING(      0x0004, "World" )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	// country            0x0010
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BITX( 0x0010, IP_ACTIVE_LOW, IPT_SERVICE, "Test Advance", KEYCODE_F1, IP_JOY_DEFAULT )
	PORT_BIT(  0x00e0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


/***************************************************************************
                               Vasara
***************************************************************************/

INPUT_PORTS_START( vasara )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Free_Play )  )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x00c0, 0x00c0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 1C_2C ) )

	PORT_START	// IN1
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x000c, 0x000c, "Bomber Stock" )
	PORT_DIPSETTING(      0x0000, "0" )
	PORT_DIPSETTING(      0x0004, "1" )
	PORT_DIPSETTING(      0x000c, "2" )
	PORT_DIPSETTING(      0x0008, "3" )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Lives ))
	PORT_DIPSETTING(      0x0000, "1" )
	PORT_DIPSETTING(      0x0010, "2" )
	PORT_DIPSETTING(      0x0030, "3" )
	PORT_DIPSETTING(      0x0020, "5" )
	PORT_DIPNAME( 0x0040, 0x0040, "Game Voice" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0000, "English Subtitles" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW, IPT_UNKNOWN  )
INPUT_PORTS_END

/***************************************************************************
                               Vasara 2
***************************************************************************/

INPUT_PORTS_START( vasara2 )
	PORT_START	// IN0 - $210002
	PORT_DIPNAME( 0x0001, 0x0001, DEF_STR( Free_Play )  )
	PORT_DIPSETTING(      0x0001, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0002, 0x0002, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_SERVICE( 0x0004, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x0008, 0x0008, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0030, 0x0030, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0030, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x00c0, 0x00c0, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x00c0, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( 1C_2C ) )

	PORT_START	// IN1
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0002, "Easy" )
	PORT_DIPSETTING(      0x0003, "Normal" )
	PORT_DIPSETTING(      0x0001, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Lives ) )
	PORT_DIPSETTING(      0x0000, "1" )
	PORT_DIPSETTING(      0x0004, "2" )
	PORT_DIPSETTING(      0x000c, "3" )
	PORT_DIPSETTING(      0x0008, "5" )
	PORT_DIPNAME( 0x0010, 0x0010, "Game Voice" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Vasara Stock" )
	PORT_DIPSETTING(      0x0020, "2" )
	PORT_DIPSETTING(      0x0000, "3" )
	PORT_DIPNAME( 0x0040, 0x0040, "English Subtitles" )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, "Secret Character" )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )

	PORT_START	// IN2 - $210008
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )

	PORT_START	// IN3 - $21000a
	PORT_BIT(  0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )

	PORT_START	// IN4 - $21000c
	PORT_BIT_IMPULSE( 0x0001, IP_ACTIVE_LOW, IPT_COIN1, 10 )
	PORT_BIT_IMPULSE( 0x0002, IP_ACTIVE_LOW, IPT_COIN2, 10 )
	PORT_BIT(  0x0004, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x0008, IP_ACTIVE_LOW, IPT_TILT     )
	PORT_BIT(  0x00f0, IP_ACTIVE_LOW, IPT_UNKNOWN  )
INPUT_PORTS_END


/***************************************************************************


                            Graphics Layouts


***************************************************************************/

/*  16 x 8 tiles. Depth is 8 bits, but can be decreased to 6 (and maybe
    less) at runtime.   */

static struct GfxLayout layout_16x8x8 =
{
	16,8,
	RGN_FRAC(1,4),
	8,
	{	RGN_FRAC(3,4)+8, RGN_FRAC(3,4)+0,
		RGN_FRAC(2,4)+8, RGN_FRAC(2,4)+0,
		RGN_FRAC(1,4)+8, RGN_FRAC(1,4)+0,
		RGN_FRAC(0,4)+8, RGN_FRAC(0,4)+0	},
	{	STEP8(0,1), STEP8(16,1)	},
	{	STEP8(0,16*2)	},
	16*8*2
};

static struct GfxLayout layout_16x8x6 =
{
	16,8,
	RGN_FRAC(1,4),
	6,
	{
		RGN_FRAC(2,4)+8, RGN_FRAC(2,4)+0,
		RGN_FRAC(1,4)+8, RGN_FRAC(1,4)+0,
		RGN_FRAC(0,4)+8, RGN_FRAC(0,4)+0	},
	{	STEP8(0,1), STEP8(16,1)	},
	{	STEP8(0,16*2)	},
	16*8*2
};

static struct GfxDecodeInfo ssv_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &layout_16x8x8, 0, 0x8000/64 }, // [0] Sprites (256 colors)
	{ REGION_GFX1, 0, &layout_16x8x6, 0, 0x8000/64 }, // [1] Sprites (64 colors)
	{ -1 }
};

static struct GfxLayout layout_16x8x8_2 =
{
	16,8,
	RGN_FRAC(1,1),
	8,
	{	STEP8(0,1)		},
	{	STEP16(0,8)		},
	{	STEP8(0,16*8)	},
	16*8*8
};

static struct GfxLayout layout_16x8x6_2 =
{
	16,8,
	RGN_FRAC(1,1),
	6,
	{	2,3,4,5,6,7		},
	{	STEP16(0,8)		},
	{	STEP8(0,16*8)	},
	16*8*8
};

static struct GfxDecodeInfo eaglshot_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &layout_16x8x8_2, 0, 0x8000/64 }, // [0] Sprites (256 colors, decoded from ram)
	{ REGION_GFX1, 0, &layout_16x8x6_2, 0, 0x8000/64 }, // [1] Sprites (64 colors, decoded from ram)
	{ -1 }
};

static struct GfxLayout layout_16x16x8 =
{
	16,16,
	RGN_FRAC(1,1),
	8,
	{	STEP8(0,1)		},
	{	STEP16(0,8)		},
	{	STEP16(0,16*8)	},
	16*16*8
};

static struct GfxDecodeInfo gdfs_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &layout_16x8x8,   0, 0x8000/64  }, // [0] Sprites (256 colors)
	{ REGION_GFX1, 0, &layout_16x8x6,   0, 0x8000/64  }, // [1] Sprites (64 colors)
	{ REGION_GFX2, 0, &layout_16x8x8_2, 0, 0x8000/64  }, // [2] Zooming Sprites (256 colors, decoded from ram)
	{ REGION_GFX3, 0, &layout_16x16x8,  0, 0x8000/256 }, // [3] Tilemap
	{ -1 }
};

/***************************************************************************


                                Machine Drivers


***************************************************************************/

static struct ES5506interface es5506_interface =
{
	1,
	{ 16000000 },
	{ REGION_SOUND1 },
	{ REGION_SOUND2 },
	{ REGION_SOUND3 },
	{ REGION_SOUND4 },
	{ YM3012_VOL(100,MIXER_PAN_LEFT,100,MIXER_PAN_RIGHT) },
	{ 0 },
	{ 0 }
};

/* Average clock cycles per instruction (12?) */
#define AVERAGE_CPI		(12)

#define CLOCK_16MHz			(16000000 / AVERAGE_CPI)	// Known speed for system boards STA-0001 & STA-0001B
#define CLOCK_12MHz			(12000000 / AVERAGE_CPI)

/***************************************************************************

    Some games (e.g. hypreac2) oddly map the high bits of the tile code
    to the gfx roms: arranging the roms accordingly would waste tens of
    megabytes. So we use a look-up table.

    We also need to set up game specific offsets for sprites and layers
    (at least until the CRT controlled will be emulated).

***************************************************************************/

void init_ssv(void)
{
	int i;
	for (i = 0; i < 16; i++)
		ssv_tile_code[i]	=	( (i & 8) ? (1 << 16) : 0 ) +
								( (i & 4) ? (2 << 16) : 0 ) +
								( (i & 2) ? (4 << 16) : 0 ) +
								( (i & 1) ? (8 << 16) : 0 ) ;
	ssv_enable_video(1);
	ssv_special = 0;
	interrupt_ultrax = 0;
	eaglshot_dirty = 0;
}

void init_ssv_hypreac2(void)
{
	int i;

	init_ssv();

	for (i = 0; i < 16; i++)
		ssv_tile_code[i]	=	(i << 16);
}


DRIVER_INIT( drifto94 )		{	init_ssv();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( eaglshot )		{	init_ssv_hypreac2();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = 0;	ssv_tilemap_offsy = -0xef; }
DRIVER_INIT( gdfs )			{	init_ssv();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = 1;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = 0;	}
DRIVER_INIT( hypreact )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf7;	}
DRIVER_INIT( hypreac2 )		{	init_ssv_hypreac2();	// different
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf8;	}
DRIVER_INIT( janjans1 )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( keithlcy )		{	init_ssv();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0xf1;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( meosism )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xef;	}
DRIVER_INIT( mslider )		{	init_ssv();
								ssv_sprites_offsx =-16;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +8;	ssv_tilemap_offsy = -0xf1;	}
DRIVER_INIT( ryorioh )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( srmp4 )		{	init_ssv();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;
//  ((UINT16 *)memory_region(REGION_USER1))[0x2b38/2] = 0x037a;   /* patch to see gal test mode */
							}
DRIVER_INIT( srmp7 )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = -0xf;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( stmblade )		{	init_ssv();
								ssv_sprites_offsx = -8; ssv_sprites_offsy = +0xef;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}
DRIVER_INIT( survarts )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xef;	}
DRIVER_INIT( dynagear )		{	init_ssv();
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0xec;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xef;	}
DRIVER_INIT( sxyreact )		{	init_ssv_hypreac2();	// different
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xef;	}
DRIVER_INIT( sxyreac2 )		{	init_ssv_hypreac2();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xef;	}
DRIVER_INIT( twineag2 )		{	init_ssv();interrupt_ultrax=1;
								ssv_sprites_offsx = -6; ssv_sprites_offsy = +0x01;
								ssv_tilemap_offsx = -2; ssv_tilemap_offsy = +0x00;	}
DRIVER_INIT( ultrax )		{	init_ssv();interrupt_ultrax=1;
								ssv_sprites_offsx = -8;	ssv_sprites_offsy = +0x01;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = +0x00;	}
DRIVER_INIT( vasara )		{	init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xf0;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf8;	}
DRIVER_INIT( jsk )			{	//cpu_setbank(2, memory_region(REGION_USER2));
								init_ssv();
								ssv_sprites_offsx = +0;	ssv_sprites_offsy = +0xe8;
								ssv_tilemap_offsx = +0;	ssv_tilemap_offsy = -0xf0;	}


static MACHINE_DRIVER_START( ssv )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", V60, CLOCK_16MHz) /* Based on STA-0001 & STA-0001B System boards */
	MDRV_CPU_VBLANK_INT(ssv_interrupt,2)	/* Vblank */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)	/* we use cpu_getvblank */

	MDRV_MACHINE_INIT(ssv)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN)

	MDRV_SCREEN_SIZE(0x180, 0x100)
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
	MDRV_GFXDECODE(ssv_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(0x8000)
	MDRV_VIDEO_START(ssv)
	MDRV_VIDEO_UPDATE(ssv)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(ES5506, es5506_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( drifto94 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(drifto94_readmem, drifto94_writemem)

	MDRV_NVRAM_HANDLER(ssv)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 4, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( gdfs )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(gdfs_readmem, gdfs_writemem)
	MDRV_CPU_VBLANK_INT(gdfs_interrupt,1+4)

	MDRV_NVRAM_HANDLER(gdfs)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)

	MDRV_GFXDECODE(gdfs_gfxdecodeinfo)
	MDRV_VIDEO_START(gdfs)
	MDRV_VIDEO_UPDATE(gdfs)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( hypreact )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hypreact_readmem, hypreact_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(8, 0x148-1, 16, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( hypreac2 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hypreac2_readmem, hypreac2_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 8, 0xf8-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( janjans1 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(janjans1_readmem, janjans1_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( keithlcy )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(keithlcy_readmem, keithlcy_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 4, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( meosism )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(meosism_readmem, meosism_writemem)

	MDRV_NVRAM_HANDLER(ssv)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mslider )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mslider_readmem, mslider_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( ryorioh )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(ryorioh_readmem, ryorioh_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( srmp4 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(srmp4_readmem, srmp4_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 4, 0xf4-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( srmp7 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(srmp7_readmem, srmp7_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( stmblade )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(drifto94_readmem, drifto94_writemem)

	MDRV_NVRAM_HANDLER(ssv)
	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x158-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( survarts )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(survarts_readmem, survarts_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 4, 0xf4-1)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( dynagear )
	/* basic machine hardware */
	MDRV_IMPORT_FROM(survarts)
	/* video hardware */
	MDRV_VISIBLE_AREA(8, 0x158-16-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( eaglshot )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(eaglshot_readmem, eaglshot_writemem)

	MDRV_NVRAM_HANDLER(ssv)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x140-1, 8, 0xe8-1)

	MDRV_GFXDECODE(eaglshot_gfxdecodeinfo)
	MDRV_VIDEO_START(eaglshot)
	MDRV_VIDEO_UPDATE(eaglshot)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( sxyreact )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(sxyreact_readmem, sxyreact_writemem)

	MDRV_NVRAM_HANDLER(ssv)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( twineag2 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(twineag2_readmem, twineag2_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( ultrax )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(ultrax_readmem, ultrax_writemem)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( jsk )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ssv)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(janjans1_readmem, janjans1_writemem)

	//MDRV_CPU_ADD(V810,25000000)
	//MDRV_CPU_PROGRAM_MAP(jsk_mem, 0)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 0x150-1, 0, 0xf0-1)
MACHINE_DRIVER_END


/***************************************************************************


                                ROMs Loading


***************************************************************************/


/***************************************************************************

                        Change Air Blade (Japan)

Change Air Blade
Sammy, 1999

ROM board for use with System SSV Main Board
PCB No: P1-112C

Fairly sparsely populated board containing not much except....

RAM   : 6262 (x1)
OTHER : 3.6V Ni-Cd Battery
PALs  : (x1, labelled AC412G00)

ROMs  : (Filename  = ROM Label)
        (Extension = PCB Location)
------------------------------
AC1801M01.U6    32M Mask
AC1802M01.U9    32M Mask

AC1805M01.U8    32M Mask
AC1806M01.U11   32M Mask

AC1803M01.U7    32M Mask
AC1804M01.U10   32M Mask

AC1807M01.U41   32M Mask
AC1810E01.U32   27C160


***************************************************************************/

ROM_START( cairblad )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "ac1810e0.u32",  0x000000, 0x200000, 0x13a0b4c2 ) // AC1810E01.U32    27C160

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "ac1801m0.u6",  0x0000000, 0x400000, 0x1b2b6943 ) // AC1801M01.U6    32M Mask
	ROM_LOAD( "ac1802m0.u9",  0x0400000, 0x400000, 0xe053b087 ) // AC1802M01.U9    32M Mask

	ROM_LOAD( "ac1803m0.u7",  0x0800000, 0x400000, 0x45484866 ) // AC1803M01.U7    32M Mask
	ROM_LOAD( "ac1804m0.u10", 0x0c00000, 0x400000, 0x5e0b2285 ) // AC1804M01.U10   32M Mask

	ROM_LOAD( "ac1805m0.u8",  0x1000000, 0x400000, 0x19771f43 ) // AC1805M01.U8    32M Mask
	ROM_LOAD( "ac1806m0.u11", 0x1400000, 0x400000, 0x816b97dc ) // AC1806M01.U11   32M Mask

	ROM_FILL(                 0x1800000, 0x800000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "ac1410m0.u41", 0x000000, 0x400000, 0xecf1f255 ) // AC1807M01.U41   32M Mask
ROM_END


/***************************************************************************

                        Drift Out '94 - The hard order

----------------------
System SSV (STA-0001B)
----------------------
CPU  : NEC D70615GD-16-S (V60)
Sound: Ensoniq ES5506 (OTTOR2)
OSC  : 42.9545MHz(X2) 48.0000MHz(X3)

Custom chips:
ST-0004 (Video DAC?)
ST-0005 (Parallel I/O?)
ST-0006 (Video controller)
ST-0007 (System controller)

Program Work RAM  : 256Kbitx2 (expandable to 1Mx2)
Object Work RAM   : 1Mbitx2
Color Palette RAM : 256Kbitx3 (expandable to 1Mx3)

-------------------------
SSV Subboard (VISCO-001B)
-------------------------
ROMs:
visco-33.bin - Main programs (27c4000)
visco-37.bin /

vg003-19.u26 - Data? (mask, read as 27c160)

vg003-17.u22 - Samples (mask, read as 27c160)
vg003-18.u15 /

vg003-01.a0 - Graphics (mask, read as 27c160)
vg003-05.a1 |
vg003-09.a2 |
vg009-13.a3 |
vg009-02.b0 |
vg003-06.b1 |
vg003-10.b2 |
vg003-14.b3 |
vg003-03.c0 |
vg003-07.c1 |
vg003-11.c2 |
vg003-15.c3 |
vg003-04.d0 |
vg003-08.d1 |
vg003-12.d2 |
vg003-16.d3 /

GAL:
vg003-22.u29 (16V8)

Custom chip:
ST010 (maybe D78C10?)

Others:
Lithium battery + MB3790 + LH5168D-10L

***************************************************************************/

ROM_START( drifto94 )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "vg003-19.u26", 0x000000, 0x200000, 0x238e5e2b )	// "SoundDriverV1.1a"
	ROM_LOAD16_BYTE( "visco-37.bin", 0x200000, 0x080000, 0x78fa3ccb )
	ROM_RELOAD(                      0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "visco-33.bin", 0x200001, 0x080000, 0x88351146 )
	ROM_RELOAD(                      0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "vg003-01.a0", 0x0000000, 0x200000, 0x2812aa1a )
	ROM_LOAD( "vg003-05.a1", 0x0200000, 0x200000, 0x1a1dd910 )
	ROM_LOAD( "vg003-09.a2", 0x0400000, 0x200000, 0x198f1c06 )
	ROM_LOAD( "vg003-13.a3", 0x0600000, 0x200000, 0xb45b2267 )

	ROM_LOAD( "vg003-02.b0", 0x0800000, 0x200000, 0xd7402027 )
	ROM_LOAD( "vg003-06.b1", 0x0a00000, 0x200000, 0x518c509f )
	ROM_LOAD( "vg003-10.b2", 0x0c00000, 0x200000, 0xc1ee9d8b )
	ROM_LOAD( "vg003-14.b3", 0x0e00000, 0x200000, 0x645b672b )

	ROM_LOAD( "vg003-03.c0", 0x1000000, 0x200000, 0x1ca7163d )
	ROM_LOAD( "vg003-07.c1", 0x1200000, 0x200000, 0x2ff113bb )
	ROM_LOAD( "vg003-11.c2", 0x1400000, 0x200000, 0xf924b105 )
	ROM_LOAD( "vg003-15.c3", 0x1600000, 0x200000, 0x83623b01 )

	ROM_LOAD( "vg003-04.d0", 0x1800000, 0x200000, 0x6be9bc62 )
	ROM_LOAD( "vg003-08.d1", 0x1a00000, 0x200000, 0xa7113cdb )
	ROM_LOAD( "vg003-12.d2", 0x1c00000, 0x200000, 0xac0fd855 )
	ROM_LOAD( "vg003-16.d3", 0x1e00000, 0x200000, 0x1a5fd312 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "vg003-17.u22", 0x000000, 0x200000, 0x6f9294ce )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "vg003-18.u15", 0x000000, 0x200000, 0x511b3e93 )
ROM_END


/***************************************************************************

Eagle Shot Golf
Sammy, 1994

GOLF ROM
|----------------------------------------|
| 514400                    GAL SI3-11   |
|                           GAL SI3-12   |
| 514400                                 |
|          SI003-06.u31    SI003-09.u18  |
| 514400                                 |
|                                        |
| 514400   SI003-05.u30    SI003-10.u20  |
|                                        |
| 514400                                 |
|          SI003-04.u10                  |
| 514400                                 |
|                                        |
| 514400   SI003-03.u11                  |
|                                        |
| 514400                                 |
|          SI003-02.u12                  |
|                                        |
|                                        |
|          SI003-01.u13    SI003-07.U23  |
|                                        |
|                                        |
|                          SI003-08.U24  |
|                                        |
|                             uPD4701    |
|                                        |
|                             mc14584b   |
|                                        |
|                                 P1     |
|-------------------------------||||||---|

 mc14584b - Motorola HEX Schmitt Trigger
  uPD4701 - Mouse/Trackball interface by NEC
   514400 - Hatachi HM514400BS7 1M x 4 DRAM (4 Megs of RAM)
      GAL - GAL16V8B
       P1 - 6 Pin trackball connector (same pinout as Golden Tee 9x series)

Lower PCB
PCB Number: GOLF ROM PCB
RAM       : HM514400 (x8)
PALs      : GAL16V8 (x2) labelled SI3-11 & SI3-12
OTHER     : NEC D4701AC
            Controls probably trackball, has 6 pin connector hooked up to a
            mc14584b Logic IC. Joystick appears to be used also for selecting
            stance, club and direction.

ROMs      : U18 & U20 are used for main program.
            All rest are 16M Mask
            U23 & U24 are sound related, all others for GFX.

Loc  ROMs           Use & eprom type
-----------------------------------------
U18  si003-09.prl - V60 Program (27C4001)
U20  si003-10.prh /

U23  si003-07.s0 - Samples (16M-Mask)
U24  si003-08.s1 /

U13  si003-01.d0 - Graphics (16M-Mask)
U12  si003-02.d1 |
U11  si003-03.d2 |
U10  si003-04.d3 |
U30  si003-05.d4 |
U31  si003-06.d5 /

NOTE: The "s" and "d" designations above are silk-screened on the rom PCB.

Chips of note:  mc14584b - Motorola HEX Schmitt Trigger

This chip is used for the trackball trigger / reading / converting values

***************************************************************************/

ROM_START( eaglshot )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "si003-09.u18",  0x000000, 0x080000, 0x219c71ce )
	ROM_LOAD16_BYTE( "si003-10.u20",  0x000001, 0x080000, 0xc8872e48 )

	ROM_REGION( 0x0c00000, REGION_GFX1, /*ROMREGION_DISPOSE*/0 )	/* Sprites - Read by the CPU */
	ROM_LOAD( "si003-01.u13", 0x0000000, 0x200000, 0xd7df0d52 )
	ROM_LOAD( "si003-02.u12", 0x0200000, 0x200000, 0x92b4d50d )
	ROM_LOAD( "si003-03.u11", 0x0400000, 0x200000, 0x6ede4012 )
	ROM_LOAD( "si003-04.u10", 0x0600000, 0x200000, 0x4c65d1a1 )
	ROM_LOAD( "si003-05.u30", 0x0800000, 0x200000, 0xdaf52d56 )
	ROM_LOAD( "si003-06.u31", 0x0a00000, 0x200000, 0x449f9ae5 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "si003-07.u23", 0x000000, 0x200000, 0x81679fd6 )
	ROM_LOAD16_WORD_SWAP( "si003-08.u24", 0x200000, 0x200000, 0xd0122ba2 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, 0 )	/* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )
ROM_END


/***************************************************************************

                    (Mahjong) Hyper Reaction (Japan)

(c)1995 Sammy, SSV system

P1-102A (ROM board)

***************************************************************************/

ROM_START( hypreact )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "s14-1-02.u2", 0x000000, 0x080000, 0xd90a383c )
	ROM_LOAD16_BYTE( "s14-1-01.u1", 0x000001, 0x080000, 0x80481401 )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "s14-1-07.u7",  0x0000000, 0x200000, 0x6c429fd0 )
	ROM_LOAD( "s14-1-05.u13", 0x0200000, 0x200000, 0x2ff72f98 )
	ROM_LOAD( "s14-1-06.u10", 0x0400000, 0x200000, 0xf470ec42 )

	ROM_LOAD( "s14-1-10.u6",  0x0600000, 0x200000, 0xfdd706ba )
	ROM_LOAD( "s14-1-08.u12", 0x0800000, 0x200000, 0x5bb9bb0d )
	ROM_LOAD( "s14-1-09.u9",  0x0a00000, 0x200000, 0xd1dda65f )

	ROM_LOAD( "s14-1-13.u8",  0x0c00000, 0x200000, 0x971caf11 )
	ROM_LOAD( "s14-1-11.u14", 0x0e00000, 0x200000, 0x6d8e7bae )
	ROM_LOAD( "s14-1-12.u11", 0x1000000, 0x200000, 0x233a8e23 )

	ROM_FILL(                 0x1200000, 0x600000, 0          )

//  The chip seems to use REGION1 too, but produces no sound from there.

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "s14-1-04.u4", 0x000000, 0x200000, 0xa5955336 )
	ROM_LOAD16_WORD_SWAP( "s14-1-03.u5", 0x200000, 0x200000, 0x283a6ec2 )
ROM_END


/***************************************************************************

                    (Mahjong) Hyper Reaction 2 (Japan)

(c)1997 Sammy,SSV system

P1-112A (ROM board)

***************************************************************************/

ROM_START( hypreac2 )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "u2.bin",  0x000000, 0x080000, 0x05c93266 )
	ROM_LOAD16_BYTE( "u1.bin",  0x000001, 0x080000, 0x80cf9e59 )
	ROM_LOAD16_BYTE( "u47.bin", 0x100000, 0x080000, 0xa3e9bfee )
	ROM_LOAD16_BYTE( "u46.bin", 0x100001, 0x080000, 0x68c41235 )

	ROM_REGION( 0x2800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "s16-1-16.u6",  0x0000000, 0x400000, 0xb308ac34 )
	ROM_LOAD( "s16-1-15.u9",  0x0400000, 0x400000, 0x2c8e381e )
	ROM_LOAD( "s16-1-14.u12", 0x0800000, 0x200000, 0xafe9d187 )

	ROM_LOAD( "s16-1-10.u7",  0x0a00000, 0x400000, 0x86a10cbd )
	ROM_LOAD( "s16-1-09.u10", 0x0e00000, 0x400000, 0x6b8e4d92 )
	ROM_LOAD( "s16-1-08.u13", 0x1200000, 0x200000, 0xb355f45d )

	ROM_LOAD( "s16-1-13.u8",  0x1400000, 0x400000, 0x89869af2 )
	ROM_LOAD( "s16-1-12.u11", 0x1800000, 0x400000, 0x87d9c748 )
	ROM_LOAD( "s16-1-11.u14", 0x1c00000, 0x200000, 0x70b3c0a0 )

	ROM_FILL(                 0x1e00000,0x0a00000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "s16-1-06.u41", 0x000000, 0x400000, 0x626e8a81 )

	ROM_REGION16_BE( 0x600000, REGION_SOUND2, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "s16-1-07.u42", 0x200000, 0x400000, 0x42bcb41b )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "s16-1-07.u42", 0x000000, 0x400000, 0x42bcb41b )
ROM_END


/***************************************************************************

             Lovely Pop Mahjong Jangjang Shimasho (Japan)

(c)1996 Visco, SSV System

***************************************************************************/

ROM_START( janjans1 )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "jj1-data.bin", 0x000000, 0x200000, 0x6734537e )
	ROM_LOAD16_BYTE( "jj1-prol.bin", 0x200000, 0x080000, 0x4231d928 )
	ROM_RELOAD(                      0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "jj1-proh.bin", 0x200001, 0x080000, 0x651383c6 )
	ROM_RELOAD(                      0x300001, 0x080000             )

	ROM_REGION( 0x2800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "jj1-a0.bin", 0x0000000, 0x400000, 0x39bbbc46 )
	ROM_LOAD( "jj1-a1.bin", 0x0400000, 0x400000, 0x26020133 )
	ROM_LOAD( "jj1-a2.bin", 0x0800000, 0x200000, 0xe993251e )

	ROM_LOAD( "jj1-b0.bin", 0x0a00000, 0x400000, 0x8ee66b0a )
	ROM_LOAD( "jj1-b1.bin", 0x0e00000, 0x400000, 0x048719b3 )
	ROM_LOAD( "jj1-b2.bin", 0x1200000, 0x200000, 0x6e95af3f )

	ROM_LOAD( "jj1-c0.bin", 0x1400000, 0x400000, 0x9df28afc )
	ROM_LOAD( "jj1-c1.bin", 0x1800000, 0x400000, 0xeb470ed3 )
	ROM_LOAD( "jj1-c2.bin", 0x1c00000, 0x200000, 0xaaf72c2d )

	ROM_LOAD( "jj1-d0.bin", 0x1e00000, 0x400000, 0x2b3bd591 )
	ROM_LOAD( "jj1-d1.bin", 0x2200000, 0x400000, 0xf24c0d36 )
	ROM_LOAD( "jj1-d2.bin", 0x2600000, 0x200000, 0x481b3be8 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "jj1-snd0.bin", 0x000000, 0x200000, 0x4f7d620a )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "jj1-snd1.bin", 0x000000, 0x200000, 0x9b3a7ae5 )
ROM_END

/***************************************************************************

            Lovely Pop Mahjong JangJang Shimasho 2 (Japan)

(c)1996-2000 Visco

This game runs on SSV hardware. The game PCB is VISCO-JJ1
The game PCB contains mostly just ROMs and logic, plus....
NEC D71051 (DIP28)
8MHz OSC TI 75175 (DIP8)
2x PALs (near program ROMs)
2x 8-pin connectors for extra controls

***************************************************************************/

ROM_START( janjans2 )
    ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )        /* V60 Code */
    ROM_LOAD16_WORD( "jan2-dat.u28",  0x000000, 0x200000, 0x0c9c62bf )
    ROM_LOAD16_BYTE( "jan2-prol.u26", 0x200000, 0x080000, 0x758a7249 )
    ROM_RELOAD(                       0x300000, 0x080000             )
    ROM_LOAD16_BYTE( "jan2-proh.u27", 0x200001, 0x080000, 0xfcd5da62 )
    ROM_RELOAD(                       0x300001, 0x080000             )

    ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE )    /* Sprites */
    ROM_LOAD( "jan2-a0.u13", 0x0000000, 0x400000, 0x37869bea )
    ROM_LOAD( "jan2-a1.u14", 0x0400000, 0x400000, 0x8189e74f )

    ROM_LOAD( "jan2-b0.u16", 0x0800000, 0x400000, 0x19877c5c )
    ROM_LOAD( "jan2-b1.u17", 0x0c00000, 0x400000, 0x8d0f7190 )

    ROM_LOAD( "jan2-c0.u21", 0x1000000, 0x400000, 0x8bdff3d5 )
    ROM_LOAD( "jan2-c1.u22", 0x1400000, 0x400000, 0xf7ea5934 )

    ROM_LOAD( "jan2-d0.u34", 0x1800000, 0x400000, 0x479fdb54 )
    ROM_LOAD( "jan2-d1.u35", 0x1c00000, 0x400000, 0xc0148895 )

    ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )    /* Samples */
    ROM_LOAD16_BYTE( "jan2-snd0.u29", 0x000000, 0x200000, 0x22cc054e )

    ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )    /* Samples */
    ROM_LOAD16_BYTE( "jan2-snd1.u33", 0x000000, 0x200000, 0xcbcac4a6 )
ROM_END


/***************************************************************************

                            Joryuu Syougi Kyoushitsu

(c)1997 Visco, System SSV ROM board

CPU : NEC JAPAN D70732GD-25 (C)NEC1991 V810 9651MK007
OSC : 48.0000MHz

ROMs:
JSK-U4 .BIN [ec22fb41] - (27c1001)
JSK-U24.BIN [1fa6e156]  |
JSK-U38.BIN [8e5c0de3]  |
JSK-U52.BIN [19cc585f] /

JSK-U71.BIN [b529f331] - (27c1001)
JSK-U72.BIN [41ed8a9f] /

JSK-S0.U65  [8d1a9aeb] - (16M mask)

JSK-A0.BIN  [4bac3196] - (16M mask)
JSK-B0.BIN  [40664e5a]  |
JSK-C0.BIN  [2a230e64]  |
JSK-D0.BIN  [911e53a6] /

GAL (not dumped):
U53.BIN     [--------] - GAL16V8B
U70.BIN     [--------] /

***************************************************************************/

ROM_START( jsk )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )
	ROM_LOAD16_WORD( "jsk-main.bin", 0x000000, 0x100000, NO_DUMP ) // missing ?

	ROM_LOAD16_BYTE( "jsk-u72.bin", 0x80000, 0x20000, 0x41ed8a9f )
	ROM_LOAD16_BYTE( "jsk-u71.bin", 0x80001, 0x20000, 0xb529f331 )

	ROM_REGION( 0x1000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "jsk-a0.bin", 0x0000000, 0x200000, 0x4bac3196 )
	ROM_LOAD( "jsk-b0.bin", 0x0400000, 0x200000, 0x40664e5a )
	ROM_LOAD( "jsk-c0.bin", 0x0800000, 0x200000, 0x2a230e64 )
	ROM_LOAD( "jsk-d0.bin", 0x0c00000, 0x200000, 0x911e53a6 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "jsk-s0.u65", 0x000000, 0x200000, 0x8d1a9aeb )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, 0 )	/* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION32_LE( 0x20000*4, REGION_USER2, 0 )
	ROM_LOAD32_BYTE( "jsk-u4.bin",  0x00003, 0x20000, 0xec22fb41 )
	ROM_LOAD32_BYTE( "jsk-u24.bin", 0x00002, 0x20000, 0x1fa6e156 )
	ROM_LOAD32_BYTE( "jsk-u38.bin", 0x00001, 0x20000, 0x8e5c0de3 )
	ROM_LOAD32_BYTE( "jsk-u52.bin", 0x00000, 0x20000, 0x19cc585f )
ROM_END


/***************************************************************************

                Dramatic Adventure Quiz Keith & Lucy (Japan)

(c)1993 Visco, SSV system

STS-0001 (ROM board)

***************************************************************************/

ROM_START( keithlcy )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "vg002-07.u28", 0x000000, 0x100000, 0x57f80ff5 )	// "SETA SoundDriver"
	ROM_LOAD16_BYTE( "kl-p0l.u26",   0x100000, 0x080000, 0xd7b177fb )
	ROM_LOAD16_BYTE( "kl-p0h.u27",   0x100001, 0x080000, 0x9de7add4 )

	ROM_REGION( 0x800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "vg002-01.u13", 0x000000, 0x200000, 0xb44d85b2 )
	ROM_LOAD( "vg002-02.u16", 0x200000, 0x200000, 0xaa05fd14 )
	ROM_LOAD( "vg002-03.u21", 0x400000, 0x200000, 0x299a8a7d )
	ROM_LOAD( "vg002-04.u34", 0x600000, 0x200000, 0xd3633f9b )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "vg002-05.u29", 0x000000, 0x200000, 0x66aecd79 )
	ROM_LOAD16_WORD_SWAP( "vg002-06.u33", 0x200000, 0x200000, 0x75d8c8ea )
ROM_END


/***************************************************************************

                            Koi Koi Shimasho 2 (Japan)

(c)1997 Visco, SSV System

VISCO-JJ1 (same board as janjans1)

OSC  :8.00000MHz

KK2_A0.BIN   [0c89a9ae] GFX (32M mask)
KK2_A1.BIN   [3f85ff19]  |
KK2_B0.BIN   [4d028972]  |
KK2_B1.BIN   [16e085f7]  |
KK2_C0.BIN   [34b699d9]  |
KK2_C1.BIN   [896fbb6f]  |
KK2_D0.BIN   [0e3005a4]  |
KK2_D1.BIN   [0116a5fb] /

U26.BIN      [4be937a1] Programs (27c4001) (PROL)
U27.BIN      [25f39d93] /                  (PROH)

KK2_SND0.BIN [b27eaa94] Sound (16M mask)
KK2_SND1.BIN [e5a963e1] /

(socket for DATA ROM is empty)

***************************************************************************/

ROM_START( koikois2 )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
//  socket for DATA ROM is empty
	ROM_LOAD16_BYTE( "u26.bin", 0x200000, 0x080000, 0x4be937a1 )
	ROM_RELOAD(                 0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "u27.bin", 0x200001, 0x080000, 0x25f39d93 )
	ROM_RELOAD(                 0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
 	ROM_LOAD( "kk2-a0.bin", 0x0000000, 0x400000, 0xb94b76c2 )
 	ROM_LOAD( "kk2-a1.bin", 0x0400000, 0x200000, 0xa7c99f56 )

 	ROM_LOAD( "kk2-b0.bin", 0x0800000, 0x400000, 0x4d028972 )
 	ROM_LOAD( "kk2-b1.bin", 0x0c00000, 0x200000, 0x778ec9fb )

 	ROM_LOAD( "kk2-c0.bin", 0x1000000, 0x400000, 0x34b699d9 )
 	ROM_LOAD( "kk2-c1.bin", 0x1400000, 0x200000, 0xab451e88 )

 	ROM_LOAD( "kk2-d0.bin", 0x1800000, 0x400000, 0x0e3005a4 )
 	ROM_LOAD( "kk2-d1.bin", 0x1c00000, 0x200000, 0x17a02252 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "kk2_snd0.bin", 0x000000, 0x200000, 0xb27eaa94 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "kk2_snd1.bin", 0x000000, 0x200000, 0xe5a963e1 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND2, 0x000000, 0x000000, 0x400000 )
ROM_END


/***************************************************************************

                        Meosis Magic (Japan, BET?)

(c)1996 Sammy, SSV System

P1-105A

Custom:     DX-102 (I/O)
Others:     M62X42B (RTC?)
            64k SRAM (Back up)
            Ni-Cd Battery

***************************************************************************/

ROM_START( meosism )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "s15-2-2.u47", 0x000000, 0x080000, 0x2ab0373f )
	ROM_LOAD16_BYTE( "s15-2-1.u46", 0x000001, 0x080000, 0xa4bce148 )

	ROM_REGION( 0x800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "s15-1-7.u7", 0x000000, 0x200000, 0xec5023cb )
	ROM_LOAD( "s15-1-8.u6", 0x200000, 0x200000, 0xf04b0836 )
	ROM_LOAD( "s15-1-5.u9", 0x400000, 0x200000, 0xc0414b97 )
	ROM_LOAD( "s15-1-6.u8", 0x600000, 0x200000, 0xd721aeb6 )

//  The chip seems to use REGION1 too, but produces no sound from there.

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "s15-1-4.u45", 0x000000, 0x200000, 0x0c6738a7 )
	ROM_LOAD16_WORD_SWAP( "s15-1-3.u43", 0x200000, 0x200000, 0xd7e83178 )
ROM_END


/***************************************************************************

                            Monster Slider (Japan)

(c)1997 Visco/PATT, System SSV

ms-pl.bin - V60 main program (27c4000, low)
ms-ph.bin - V60 main program (27c4000, high)

ms-snd0.bin \
            |- sound data (read as 27c160)
ms-snd1.bin /

ms-a0.bin \
ms-b0.bin |- Graphics (read as 27c160)
ms-c0.bin /

ms-a1.bin \
ms-b1.bin |- Graphics (27c4100)
ms-c1.bin /

vg001-14 \
         |- (GAL16V8. not dumped)
vg001-15 /

Other parts:    uPD71051
                OSC 8.0000MHz

***************************************************************************/

ROM_START( mslider )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "ms-pl.bin", 0x000000, 0x080000, 0x70b2a05d )
	ROM_LOAD16_BYTE( "ms-ph.bin", 0x000001, 0x080000, 0x34a64e9f )

	ROM_REGION( 0xa00000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "ms-a0.bin", 0x000000, 0x200000, 0x7ed38ccc )
	ROM_LOAD( "ms-a1.bin", 0x200000, 0x080000, 0x83f5995f )

	ROM_LOAD( "ms-b0.bin", 0x280000, 0x200000, 0xfaa076e1 )
	ROM_LOAD( "ms-b1.bin", 0x480000, 0x080000, 0xef9748db )

	ROM_LOAD( "ms-c0.bin", 0x500000, 0x200000, 0xf9d3e052 )
	ROM_LOAD( "ms-c1.bin", 0x700000, 0x080000, 0x7f910c5a )

	ROM_FILL(              0x780000, 0x280000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "ms-snd0.bin", 0x000000, 0x200000, 0xcda6e3a5 )
	ROM_LOAD16_WORD_SWAP( "ms-snd1.bin", 0x200000, 0x200000, 0x8f484b35 )
ROM_END


/***************************************************************************

                    Gourmet Battle Quiz Ryohrioh CooKing (Japan)

(c)1998 Visco, SSV System

***************************************************************************/

ROM_START( ryorioh )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD( "ryorioh.dat",      0x000000, 0x200000, 0xd1335a6a )
	ROM_LOAD16_BYTE( "ryorioh.l", 0x200000, 0x080000, 0x9ad60e7d )
	ROM_RELOAD(                   0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "ryorioh.h", 0x200001, 0x080000, 0x0655fcff )
	ROM_RELOAD(                   0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "ryorioh.a0", 0x0000000, 0x400000, 0xf76ee003 )
	ROM_LOAD( "ryorioh.a1", 0x0400000, 0x400000, 0xca44d66d )

	ROM_LOAD( "ryorioh.b0", 0x0800000, 0x400000, 0xdaa134f4 )
	ROM_LOAD( "ryorioh.b1", 0x0c00000, 0x400000, 0x7611697c )

	ROM_LOAD( "ryorioh.c0", 0x1000000, 0x400000, 0x20eb49cf )
	ROM_LOAD( "ryorioh.c1", 0x1400000, 0x400000, 0x1370c75e )

	ROM_LOAD( "ryorioh.d0", 0x1800000, 0x400000, 0xffa14ef1 )
	ROM_LOAD( "ryorioh.d1", 0x1c00000, 0x400000, 0xae6055e8 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "ryorioh.snd", 0x000000, 0x200000, 0x7bd38b76 )
ROM_END


/***************************************************************************

                            Super Real Mahjong PIV

(c)SETA 1993, System SSV

CPU        : V60 (12MHz)
Sound      : Ensoniq OTTO
Work RAM   : 256Kbit (expandable to 1Mbitx2. SRMP7 requires this)
Object RAM : 1Mbitx2
Palette RAM: 256Kbitx3 (expandable to 1Mbitx3)

sx001-01.a0 \
sx001-02.b0 |
sx001-03.c0 |
sx001-04.a1 |
sx001-05.b1 |- Graphics (16M Mask)
sx001-06.c1 |
sx001-07.a2 |
sx001-08.b2 |
sx001-09.c2 /

sx001-10.sd0 - Sound - 16M Mask

sx001-11.prl - Main program (low)  - 27c040
sx001-12.prh - Main program (high) - 27c040

Custom chips
ST-0004 (Video DAC)
ST-0005 (Parallel I/O)
ST-0006 (Video controller - 32768 palettes from 24bit color)
ST-0007 (System controller)

***************************************************************************/

ROM_START( srmp4 )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "sx001-14.prl", 0x000000, 0x080000, 0x19aaf46e )
	ROM_LOAD16_BYTE( "sx001-15.prh", 0x000001, 0x080000, 0xdbd31399 )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "sx001-01.a0", 0x0000000, 0x200000, 0x94ee9203 )
	ROM_LOAD( "sx001-04.a1", 0x0200000, 0x200000, 0x38c9c49a )
	ROM_LOAD( "sx001-07.a2", 0x0400000, 0x200000, 0xee66021e )

	ROM_LOAD( "sx001-02.b0", 0x0600000, 0x200000, 0xadffb598 )
	ROM_LOAD( "sx001-05.b1", 0x0800000, 0x200000, 0x4c400a38 )
	ROM_LOAD( "sx001-08.b2", 0x0a00000, 0x200000, 0x36efd52c )

	ROM_LOAD( "sx001-03.c0", 0x0c00000, 0x200000, 0x4336b037 )
	ROM_LOAD( "sx001-06.c1", 0x0e00000, 0x200000, 0x6fe7229e )
	ROM_LOAD( "sx001-09.c2", 0x1000000, 0x200000, 0x91dd8218 )

	ROM_FILL(                0x1200000, 0x600000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "sx001-10.sd0", 0x000000, 0x200000, 0x45409ef1 )
	ROM_RELOAD(                           0x200000, 0x200000             )
ROM_END

ROM_START( srmp4o )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "sx001-11.prl", 0x000000, 0x080000, 0xdede3e64 )
	ROM_LOAD16_BYTE( "sx001-12.prh", 0x000001, 0x080000, 0x739c53c3 )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "sx001-01.a0", 0x0000000, 0x200000, 0x94ee9203 )
	ROM_LOAD( "sx001-04.a1", 0x0200000, 0x200000, 0x38c9c49a )
	ROM_LOAD( "sx001-07.a2", 0x0400000, 0x200000, 0xee66021e )

	ROM_LOAD( "sx001-02.b0", 0x0600000, 0x200000, 0xadffb598 )
	ROM_LOAD( "sx001-05.b1", 0x0800000, 0x200000, 0x4c400a38 )
	ROM_LOAD( "sx001-08.b2", 0x0a00000, 0x200000, 0x36efd52c )

	ROM_LOAD( "sx001-03.c0", 0x0c00000, 0x200000, 0x4336b037 )
	ROM_LOAD( "sx001-06.c1", 0x0e00000, 0x200000, 0x6fe7229e )
	ROM_LOAD( "sx001-09.c2", 0x1000000, 0x200000, 0x91dd8218 )

	ROM_FILL(                0x1200000, 0x600000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "sx001-10.sd0", 0x000000, 0x200000, 0x45409ef1 )
	ROM_RELOAD(                           0x200000, 0x200000             )
ROM_END


/***************************************************************************

                            Super Real Mahjong P7 (Japan)

(c)1997 Seta, SSV system

***************************************************************************/

ROM_START( srmp7 )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "sx015-10.dat", 0x000000, 0x200000, 0xfad3ac6a )
	ROM_LOAD16_BYTE( "sx015-07.pr0", 0x200000, 0x080000, 0x08d7f841 )
	ROM_RELOAD(                      0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "sx015-08.pr1", 0x200001, 0x080000, 0x90307825 )
	ROM_RELOAD(                      0x300001, 0x080000             )

	ROM_REGION( 0x4000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "sx015-26.a0", 0x0000000, 0x400000, 0xa997be9d )
	ROM_LOAD( "sx015-25.a1", 0x0400000, 0x400000, 0x29ac4211 )
	ROM_LOAD( "sx015-24.a2", 0x0800000, 0x400000, 0xb8fea3da )
	ROM_LOAD( "sx015-23.a3", 0x0c00000, 0x400000, 0x9ec0b81e )

	ROM_LOAD( "sx015-22.b0", 0x1000000, 0x400000, 0x62c3df07 )
	ROM_LOAD( "sx015-21.b1", 0x1400000, 0x400000, 0x55b8a431 )
	ROM_LOAD( "sx015-20.b2", 0x1800000, 0x400000, 0xe84a64d7 )
	ROM_LOAD( "sx015-19.b3", 0x1c00000, 0x400000, 0x994b5063 )

	ROM_LOAD( "sx015-18.c0", 0x2000000, 0x400000, 0x72d43fd4 )
	ROM_LOAD( "sx015-17.c1", 0x2400000, 0x400000, 0xfdfd82f1 )
	ROM_LOAD( "sx015-16.c2", 0x2800000, 0x400000, 0x86aa314b )
	ROM_LOAD( "sx015-15.c3", 0x2c00000, 0x400000, 0x11f50e16 )

	ROM_LOAD( "sx015-14.d0", 0x3000000, 0x400000, 0x186f83fa )
	ROM_LOAD( "sx015-13.d1", 0x3400000, 0x400000, 0xea6e5329 )
	ROM_LOAD( "sx015-12.d2", 0x3800000, 0x400000, 0x80336523 )
	ROM_LOAD( "sx015-11.d3", 0x3c00000, 0x400000, 0x134c8e28 )

	ROM_REGION16_BE( 0x800000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "sx015-06.s0", 0x000000, 0x200000, 0x0d5a206c )
	ROM_RELOAD(                     0x400000, 0x200000             )

	ROM_REGION16_BE( 0x800000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "sx015-05.s1", 0x000000, 0x200000, 0xbb8cebe2 )
	ROM_RELOAD(                     0x400000, 0x200000             )

	ROM_REGION16_BE( 0x800000, REGION_SOUND3, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "sx015-04.s2", 0x000000, 0x200000, 0xf6e933df )
	ROM_LOAD16_BYTE( "sx015-02.s4", 0x400000, 0x200000, 0x6567bc3e )

	ROM_REGION16_BE( 0x800000, REGION_SOUND4, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "sx015-03.s3", 0x000000, 0x200000, 0x5b51ab21 )
	ROM_LOAD16_BYTE( "sx015-01.s5", 0x400000, 0x200000, 0x481b00ed )
ROM_END


/***************************************************************************

         Survival Arts

Sammy USA, 1994

This game runs on SSV hardware.

Game PCB Layout
---------------

SAM-5127
|----------------------------------------|
| SI001-13.U6               4P     3P    |
|                                        |
| SI001-12.U7                            |
|                                        |
| SI001-11.U8                            |
|                                        |
| SI001-10.U9                            |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                           SI001-08.U21 |
|                                        |
|                           SI001-05.U22 |
|                                        |
|                           SI001-02.U23 |
|                                        |
|                                        |
|USA-PR-H.U3                             |
|                                        |
|USA-PR-L.U4  SI001-09.U15  SI001-07.U25 |
|                                        |
|             SI001-06.U16  SI001-04.U26 |
|PAL                                     |
|(SI001-14)   SI001-03.U17  SI001-01.U27 |
|                                        |
|----------------------------------------|

 Manufacturer: Sammy USA
 System Type: System SSV

 ----------------------
 System SSV (STA-0001)
 ----------------------
 CPU  : NEC D70615GD-16 (V60)
 Sound: Ensoniq ES5506 (OTTOR2)
 OSC  : 42.9545MHz(X2) 48.0000MHz(X3)

 Custom chips:
 ST-0004 (Video DAC)
 ST-0005 (Parallel I/O)
 ST-0006 (Video controller)
 ST-0007 (System controller)

 Program Work RAM  : 256Kbitx2 (expandable to 1Mx2)
 Object Work RAM   : 1Mbitx2
 Color Palette RAM : 256Kbitx3 (expandable to 1Mx3)

 -------------------------
 SSV Subboard (SAM-5127)
 -------------------------
 ROMs:
 USA-PR-H.u3 - V60 Program (27C4001)
 USA-PR-L.u4 /

 si001-10.s0 - Samples (16M-Mask)
 si001-12.s2 /

 si001-11.s1 - Samples (8M-Mask)
 si001-13.s3 /

 si001-01.a0 - Graphics (16M-Mask)
 si001-04.a1 |
 si001-05.a2 |
 si001-02.b0 |
 si001-05.b1 |
 si001-07.b2 |
 si001-03.c0 |
 si001-06.c1 |
 si001-09.c2 /

 Empty Sockets:
 DATA --- 16M-Mask
 A3     |
 B3     |
 C3     |
 D0-D3 /

 GAL:
 si003-14.u5 (16V8B)

 MISC:
 P3 10 Pin Header for Buttons 4-6 (Kick) for each player
 P4 10 Pin Header (unused)

***************************************************************************/

ROM_START( survarts )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "prl-r6.u4", 0x000000, 0x080000, 0xef5f6e17 )
	ROM_LOAD16_BYTE( "prh-r5.u3", 0x000001, 0x080000, 0xd446f010 )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "si001-01.u27", 0x0000000, 0x200000, 0x8b38fbab ) /* A0 */
	ROM_LOAD( "si001-04.u26", 0x0200000, 0x200000, 0x34248b54 ) /* A1 */
	ROM_LOAD( "si001-07.u25", 0x0400000, 0x200000, 0x497d6151 ) /* A2 */

	ROM_LOAD( "si001-02.u23", 0x0600000, 0x200000, 0xcb4a2dbd ) /* B0 */
	ROM_LOAD( "si001-05.u22", 0x0800000, 0x200000, 0x8f092381 ) /* B1 */
	ROM_LOAD( "si001-08.u21", 0x0a00000, 0x200000, 0x182b88c4 ) /* B2 */

	ROM_LOAD( "si001-03.u17", 0x0c00000, 0x200000, 0x92fdf652 ) /* C0 */
	ROM_LOAD( "si001-06.u16", 0x0e00000, 0x200000, 0x9a62f532 ) /* C1 */
	ROM_LOAD( "si001-09.u15", 0x1000000, 0x200000, 0x0955e393 ) /* C2 */

	ROM_FILL(                0x1200000, 0x600000, 0          )

//  The chip seems to use REGION1 too, but produces no sound from there.

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "si001-10.u9", 0x000000, 0x100000, 0x5642b333 ) /* S0 */
	ROM_LOAD16_WORD_SWAP( "si001-11.u8", 0x100000, 0x100000, 0xa81e6ea6 ) /* S1 */
	ROM_LOAD16_WORD_SWAP( "si001-12.u7", 0x200000, 0x100000, 0xe9b2b45b ) /* S2 */
	ROM_LOAD16_WORD_SWAP( "si001-13.u6", 0x300000, 0x100000, 0xd66a7e26 ) /* S3 */

	ROM_REGION( 0x0200, REGION_PLDS, 0 )
	ROM_LOAD( "gal16v8b.u5",   0x0000, 0x0117, 0x378ce368 )
ROM_END

ROM_START( survartu )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "usa-pr-l.u4", 0x000000, 0x080000, 0xfa328673 )
	ROM_LOAD16_BYTE( "usa-pr-h.u3", 0x000001, 0x080000, 0x6bee2635 )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "si001-01.u27", 0x0000000, 0x200000, 0x8b38fbab ) /* A0 */
	ROM_LOAD( "si001-04.u26", 0x0200000, 0x200000, 0x34248b54 ) /* A1 */
	ROM_LOAD( "si001-07.u25", 0x0400000, 0x200000, 0x497d6151 ) /* A2 */

	ROM_LOAD( "si001-02.u23", 0x0600000, 0x200000, 0xcb4a2dbd ) /* B0 */
	ROM_LOAD( "si001-05.u22", 0x0800000, 0x200000, 0x8f092381 ) /* B1 */
	ROM_LOAD( "si001-08.u21", 0x0a00000, 0x200000, 0x182b88c4 ) /* B2 */

	ROM_LOAD( "si001-03.u17", 0x0c00000, 0x200000, 0x92fdf652 ) /* C0 */
	ROM_LOAD( "si001-06.u16", 0x0e00000, 0x200000, 0x9a62f532 ) /* C1 */
	ROM_LOAD( "si001-09.u15", 0x1000000, 0x200000, 0x0955e393 ) /* C2 */

	ROM_FILL(                0x1200000, 0x600000, 0          )

//  The chip seems to use REGION1 too, but produces no sound from there.

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "si001-10.u9", 0x000000, 0x100000, 0x5642b333 ) /* S0 */
	ROM_LOAD16_WORD_SWAP( "si001-11.u8", 0x100000, 0x100000, 0xa81e6ea6 ) /* S1 */
	ROM_LOAD16_WORD_SWAP( "si001-12.u7", 0x200000, 0x100000, 0xe9b2b45b ) /* S2 */
	ROM_LOAD16_WORD_SWAP( "si001-13.u6", 0x300000, 0x100000, 0xd66a7e26 ) /* S3 */
ROM_END

/***************************************************************************

                        Dynagears
Sammy, 1993

This game runs on SSV hardware.

Game PCB Layout
---------------

SAM-5127
|----------------------------------------|
| SI002-10.U6                            |
|                                        |
| SI002-09.U7                            |
|                                        |
| SI002-08.U8                            |
|                                        |
| SI002-07.U9                            |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                                        |
|                           SI002-05.U22 |
|                                        |
|                           SI002-02.U23 |
|                                        |
|                                        |
|SI002-PRH.U3                            |
|                                        |
|SI002-PRL.U4                            |
|                                        |
|             SI002-06.U16  SI002-04.U26 |
|PAL                                     |
|(SI002-14)   SI002-03.U17  SI002-01.U27 |
|                                        |
|----------------------------------------|

***************************************************************************/

ROM_START( dynagear )
	ROM_REGION16_LE( 0x100000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "si002-prl.u4", 0x000000, 0x080000, 0x71ba29c6 )
	ROM_LOAD16_BYTE( "si002-prh.u3", 0x000001, 0x080000, 0xd0947a12 )

	ROM_REGION( 0x1000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "si002-01.u27", 0x0000000, 0x200000, 0x0060a521 )
	ROM_LOAD( "si002-04.u26", 0x0200000, 0x200000, 0x6140f47d )

	ROM_LOAD( "si002-02.u23", 0x0400000, 0x200000, 0xc22f2a41 )
	ROM_LOAD( "si002-05.u22", 0x0600000, 0x200000, 0x482412fd )

	ROM_LOAD( "si002-03.u17", 0x0800000, 0x200000, 0x4261a6b8 )
	ROM_LOAD( "si002-06.u16", 0x0a00000, 0x200000, 0x0e1f23f6 )

	ROM_FILL(                0xc00000, 0x400000, 0          )

//  The chip seems to use REGION1 too, but produces no sound from there.

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "si002-07.u9", 0x000000, 0x100000, 0x30d2bf11 )
	ROM_LOAD16_WORD_SWAP( "si002-08.u8", 0x100000, 0x100000, 0x253704ee )
	ROM_LOAD16_WORD_SWAP( "si002-09.u7", 0x200000, 0x100000, 0x1ea86db7 )
	ROM_LOAD16_WORD_SWAP( "si002-10.u6", 0x300000, 0x100000, 0xe369c177 )
ROM_END


/***************************************************************************

                        Pachinko Sexy Reaction (Japan)

(c)1998 Sammy, SSV system

P1-112C (ROM board)

Chips:  DX-102 x2
        uPD7001C (ADC?)
        64k NVRAM

***************************************************************************/

ROM_START( sxyreact )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_BYTE( "ac414e00.u2",  0x000000, 0x080000, 0xd5dd7593 )
	ROM_LOAD16_BYTE( "ac413e00.u1",  0x000001, 0x080000, 0xf46aee4a )
	ROM_LOAD16_BYTE( "ac416e00.u47", 0x100000, 0x080000, 0xe0f7bba9 )
	ROM_LOAD16_BYTE( "ac415e00.u46", 0x100001, 0x080000, 0x92de1b5f )

	ROM_REGION( 0x2800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "ac1401m0.u6",  0x0000000, 0x400000, 0x0b7b693c )
	ROM_LOAD( "ac1402m0.u9",  0x0400000, 0x400000, 0x9d593303 )
	ROM_LOAD( "ac1403m0.u12", 0x0800000, 0x200000, 0xaf433eca )

	ROM_LOAD( "ac1404m0.u7",  0x0a00000, 0x400000, 0xcdda2ccb )
	ROM_LOAD( "ac1405m0.u10", 0x0e00000, 0x400000, 0xe5e7a5df )
	ROM_LOAD( "ac1406m0.u13", 0x1200000, 0x200000, 0xc7053409 )

	ROM_LOAD( "ac1407m0.u8",  0x1400000, 0x400000, 0x28c83d5e )
	ROM_LOAD( "ac1408m0.u11", 0x1800000, 0x400000, 0xc45bab47 )
	ROM_LOAD( "ac1409m0.u14", 0x1c00000, 0x200000, 0xbe1c66c2 )

	ROM_FILL(                 0x1e00000, 0xa00000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "ac1410m0.u41", 0x000000, 0x400000, 0x2a880afc )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD16_WORD_SWAP( "ac1411m0.u42", 0x200000, 0x200000, 0x2ba4ca43 )
	ROM_CONTINUE( 0x000000, 0x200000 )	// this will go in region 3

	// a few sparse samples are played from here
	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_COPY( REGION_SOUND2, 0x000000,    0x200000, 0x200000 )
ROM_END

/***************************************************************************

                        Pachinko Sexy Reaction 2 (Japan)

Pachinko Sexy Reaction 2
Sammy, 1999

SAMMY B1-001A
|---------------------------------------------------------|
|                                                         |
|                                                         |
|                                                         |
|                                                         |
|               U12   AC1702T00.U9                        |
|                         AC1701T00.U6                    |
|                                                         |
|                                                         |
|                                                         |
|CN8                  AC1704T00.U10       AC1706T00.U11   |
|               U13      AC1703T00.U7   U14   AC1705T00.U8|
|                                                         |
|                                                         |
|                                                         |
|                                                         |
|         AC1708T00.U42                                   |
|CN3             AC1707T00.U41                            |
|                                                  BATT   |
|                      AC1714E00.U32            U55       |
|        CN5                   U47  U46  U2 U1  U35       |
|     D7001C                                8464  GAL16V8 |
|                                                         |
|---------------------------------------------------------|
Notes:
      6264     - Fujitsu MB8464A-10LL-SK 8K x8 SRAM (DIP28)
      D7001C   - NEC D7001C UART (DIP16)
      BATT     - 3.6V NICD Battery
      CN5      - 3 pin connector
      CN3      - 10 pin connector
      CN8      - 9 pin connector
      U12,U13, \
      U14      / Locations for SOP44 surface mounted ROMs, unpopulated
      U47,U46, \
      U2,U1    / Locations for DIP32 ROMs, unpopulated
      U35      - Location for NEC D71051GU, unpopulated
      U55      - Location for MSM62X42B Real Time Clock, unpopulated
      GAL16V8  - Lattice GAL16V8D stamped 'AC1709G00' (DIP20)
      U41,U42, \
      U7,U8,   \
      U10,U11  / OKI M27C3252CZ 32MBit SOP44 surface mounted maskROMs

      U32      - ST Microelectronics M27C160 16MBit EPROM (DIP42)

***************************************************************************/

ROM_START( sxyreac2 )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "ac1714e00.u32",  0x000000, 0x200000, 0x78075d70 )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "ac1701t00.u6",  0x0000000, 0x400000, 0xe14611c2 )
	ROM_LOAD( "ac1702t00.u9",  0x0400000, 0x400000, 0x2c8b07f8 )

	ROM_LOAD( "ac1703t00.u7",  0x0800000, 0x400000, 0xd6c7e861 )
	ROM_LOAD( "ac1704t00.u10", 0x0c00000, 0x400000, 0x5fa7ccf0 )

	ROM_LOAD( "ac1705t00.u8",  0x1000000, 0x400000, 0x2dff0652 )
	ROM_LOAD( "ac1706t00.u11", 0x1400000, 0x400000, 0xe7a168e0 )

	ROM_FILL(                 0x1800000, 0x800000, 0         )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_WORD_SWAP( "ac1707t00.u41", 0x000000, 0x400000, 0x28999bc4 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, 0 ) /* Samples */
	ROM_LOAD16_WORD_SWAP( "ac1708t00.u42", 0x200000, 0x200000, 0x7001eec0 )
	ROM_CONTINUE( 0x000000, 0x200000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 )	/* Samples */
	ROM_COPY( REGION_SOUND2, 0x000000,    0x200000, 0x200000 )
ROM_END


/***************************************************************************

                                Storm Blade
CPU  : NEC D70615GD-16-S (V60)
Sound: Ensoniq ES5506 (OTTOR2)


Rom board  001B
SSV mother board

U37, U33 = 27c040
U22, U41, U35, U25, U21, U11, U7  = 16 MEG MASK ROMS
U32, U18, U4 = 4 MEG MASK ROMS
U26 = 8 MEG MASK ROM

There is a battery on the rom board @ BT1 (battery # CR2032 - 3 volts)

***************************************************************************/

ROM_START( stmblade )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "sb-pd0.u26",  0x000000, 0x100000, 0x91c4fbf7 )
	ROM_LOAD16_BYTE( "s-blade.u37", 0x200000, 0x080000, 0xa6a42cc7 )
	ROM_RELOAD(                     0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "s-blade.u33", 0x200001, 0x080000, 0x16104ca6 )
	ROM_RELOAD(                     0x300001, 0x080000             )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
	ROM_LOAD( "sb-a0.u41", 0x0000000, 0x200000, 0x2a327b51 )
	ROM_LOAD( "sb-a1.u35", 0x0200000, 0x200000, 0x246f6f28 )
	ROM_LOAD( "sb-a2.u32", 0x0400000, 0x080000, 0x2049acf3 )
	ROM_LOAD( "sb-b0.u25", 0x0600000, 0x200000, 0xb3aa3e68 )
	ROM_LOAD( "sb-b1.u21", 0x0800000, 0x200000, 0xe95b38e7 )
	ROM_LOAD( "sb-b2.u18", 0x0a00000, 0x080000, 0xd080e620 )
	ROM_LOAD( "sb-c0.u11", 0x0c00000, 0x200000, 0x825dd8f1 )
	ROM_LOAD( "sb-c1.u7",  0x0e00000, 0x200000, 0x744afcd7 )
	ROM_LOAD( "sb-c2.u4",  0x1000000, 0x080000, 0xfd1d2a92 )
	ROM_FILL(              0x1200000, 0x600000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "sb-snd0.u22", 0x000000, 0x200000, 0x4efd605b )
ROM_END


/***************************************************************************

                                Twin Eagle II

***************************************************************************/

ROM_START( twineag2 )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "sx002_12", 0x000000, 0x200000, 0x846044dc )

	ROM_REGION( 0x1800000, REGION_GFX1, ROMREGION_DISPOSE )	/* Sprites */
	ROM_LOAD( "sx002_01", 0x0000000, 0x200000, 0x6d6896b5 )
	ROM_LOAD( "sx002_02", 0x0200000, 0x200000, 0x3f47e97a )
	ROM_LOAD( "sx002_03", 0x0400000, 0x200000, 0x544f18bf )

	ROM_LOAD( "sx002_04", 0x0600000, 0x200000, 0x58c270e2 )
	ROM_LOAD( "sx002_05", 0x0800000, 0x200000, 0x3c310229 )
	ROM_LOAD( "sx002_06", 0x0a00000, 0x200000, 0x46d5b1f3 )

	ROM_LOAD( "sx002_07", 0x0c00000, 0x200000, 0xc30fa397 )
	ROM_LOAD( "sx002_08", 0x0e00000, 0x200000, 0x64edcefa )
	ROM_LOAD( "sx002_09", 0x1000000, 0x200000, 0x51527c56 )

	ROM_FILL(             0x1200000, 0x600000, 0          )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 ) /* Samples */
	ROM_LOAD16_BYTE( "sx002_10", 0x000000, 0x200000, 0xb0669dfa )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 ) /* Samples */
	ROM_LOAD16_BYTE( "sx002_11", 0x000000, 0x200000, 0xb8dd621a )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND2, 0x000000, 0x000000, 0x400000 )
ROM_END


/***************************************************************************

                Ultra Keibitai / Ultra X Weapon

(c)1995 Banpresto (developed by Seta)
Hardware is almost identical to SSV system

****************************************************************************/

ROM_START( ultrax )
	ROM_REGION16_LE( 0x200000, REGION_USER1, 0 )  /* V60 Code */
	ROM_LOAD16_BYTE( "71047-11.u64", 0x000000, 0x080000, 0x593b2678 )
	ROM_LOAD16_BYTE( "71047-09.u65", 0x000001, 0x080000, 0x08ea8d91 )
	ROM_LOAD16_BYTE( "71047-12.u62", 0x100000, 0x080000, 0x76a77ab2 )
	ROM_LOAD16_BYTE( "71047-10.u63", 0x100001, 0x080000, 0x7c79faf9 )

	ROM_REGION( 0xc00000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
	ROM_LOAD( "71047-01.u73", 0x0000000, 0x200000, 0x66662b08 )
	ROM_LOAD( "71047-02.u74", 0x0200000, 0x100000, 0x6b00dc0c )
	ROM_LOAD( "71047-03.u76", 0x0300000, 0x200000, 0x00fcd6c2 )
	ROM_LOAD( "71047-04.u77", 0x0500000, 0x100000, 0xd9e710d1 )
	ROM_LOAD( "71047-05.u75", 0x0600000, 0x200000, 0x10848193 )
	ROM_LOAD( "71047-06.u88", 0x0800000, 0x100000, 0xb8ac2942 )
	ROM_FILL(                 0x0900000, 0x300000, 0 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 ) /* Samples */
	ROM_LOAD16_BYTE( "71047-07.u59", 0x000000, 0x200000, 0xd9828b62 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 ) /* Samples */
	ROM_LOAD16_BYTE( "71047-08.u60", 0x000000, 0x200000, 0x30ebff6d )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND2, 0x000000, 0x000000, 0x400000 )
ROM_END


/***************************************************************************

                            Vasara / Vasara 2

Visco Games, 2000 / 2001
(info from the vasara 2 board but vasara should be the same)

This is a sub board that fits any standard SSV main board.

There's very little on the ROM board. Just 2x 27C040 EPROMs,
4x 64Mbit SOP44 MASK ROMs, 3x 16Mbit SOP44 MASK ROMs,
some logic and 2 PALs near the PROG & DATA ROMs.

The actual ROM PCB has the capability to accept SOP44 and
TSOP48 type 1 SMT ROMs though many of the positions are unpopulated.
It's likely this same ROM board could be used for other Visco games.

ROM PCB (PCB Number: SSV_SUB)

SSV-SUB
|-------------------------------------------|
|                                           |
|            logic chips (x14)              |
|                                           |
|                                           |
|                                           |
|  U4   U8   U3   U7   U2   U6   U1   U5    |
|                                           |
|                                           |
|                                           |
|                                           |
|   U12    U11   logic (x3)   U10     U9    |
|                                           |
|                                           |
|                                           |
|                                           |
|                                           |
|                                           |
|             U38             U34           |
| U37  U36         U30  U31        PAL PAL  |
|             U39             U35           |
|                                           |
|                                           |
|-------------------------------------------|
Notes:
LOC      Vasara                  Vasara 2           Device
----------------------------------------------------------------------
U4       VASARA-1-D0             VASARA-2-D0        64M SOP44 MaskROM
U3       VASARA-1-C0             VASARA-2-C0        64M SOP44 MaskROM
U2       VASARA-1-B0             VASARA-2-B0        64M SOP44 MaskROM
U1       VASARA-1-A0             VASARA-2-A0        64M SOP44 MaskROM
U37      VASARA-1 SND1           VASARA-1 SND1      Macronix MX29F1610 SOP44 MaskROM
U36      VASARA-1 SND0           VASARA-1 SND0      Macronix MX29F1610 SOP44 MaskROM
U30      PRG-L U30               PRG-L U30          TMS 27C040 EPROM
U31      PRG-H U31               PRG-H U31          TMS 27C040 EPROM
U34      VASARA-1 DAT            VASARA-1 DAT       Macronix MX29F1610 MaskROM

Locations U5, U6, U7 &U8 wired for Macronix MX29F1610 SOP44 but not populated.
Locations U9, U10, U11, U12, U35, U38 & U39 wired for Macronix MX29F1610 TSOP48 but not populated.


File Name     Labeled as        Loc. Printed*      ROM Type
-----------------------------------------------------------------------
prg-h.u31     PRG-H U31         U31  PRG H       | 27C040
prg-l.u30     PRG-L U20         U30  PRG L      /
s1.u37        C DAT VASARA-1    U37  S1         \
s0.u36        B DAT VASARA-1    U36  S0          | Surface Mounted 16Mbit SOP44 MASK ROMs
data.u34      A SND 1 VASARA-1  U34  DATA ROM   /
d0.u4         VASARA-2-D0       U4   D0.D1      \
c0.u3         VASARA-2-C0       U3   C0.C1       | Surface Mounted 64Mbit SOP44 MASK ROMs
b0.u2         VASARA-2-B0       U2   B0.B1       |
a0.u1         VASARA-2-A0       U1   A0.A1      /

2x GAL16V8D

Printed = Info silk-screened/printed on the actual PCB

Vasara 2 has a secret character code like the Raizing games:

    Insert Coin, UP UP UP DOWN DOWN DOWN UP UP UP UP UP UP UP  (Up x3, Down x3, Up x7)

    Using this code will add a fifth selectable character called Saiga Magoichi
    To enable this character full time, set dip 8 of switch 2

****************************************************************************/

ROM_START( vasara )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "data.u34",  0x000000, 0x200000, 0x7704cc7e )
	ROM_LOAD16_BYTE( "prg-l.u30", 0x200000, 0x080000, 0xf0547886 )
	ROM_RELOAD(                   0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "prg-h.u31", 0x200001, 0x080000, 0x6a39bba9 )
	ROM_RELOAD(                   0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
	ROM_LOAD( "a0.u1", 0x0000000, 0x800000, 0x673230a6 )
	ROM_LOAD( "b0.u2", 0x0800000, 0x800000, 0x31a2da7f )
	ROM_LOAD( "c0.u3", 0x1000000, 0x800000, 0xd110dacf )
	ROM_LOAD( "d0.u4", 0x1800000, 0x800000, 0x82d0ca55 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s0.u36", 0x000000, 0x200000, 0x754fca02 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s1.u37", 0x000000, 0x200000, 0x5f303698 )
ROM_END

ROM_START( vasara2 )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "data.u34",  0x000000, 0x200000, 0x493d0103 )
	ROM_LOAD16_BYTE( "prg-l.u30", 0x200000, 0x080000, 0x40e6f5f6 )
	ROM_RELOAD(                   0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "prg-h.u31", 0x200001, 0x080000, 0xc958e146 )
	ROM_RELOAD(                   0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
	ROM_LOAD( "a0.u1", 0x0000000, 0x800000, 0xa6306c75 )
	ROM_LOAD( "b0.u2", 0x0800000, 0x800000, 0x227cbd9f )
	ROM_LOAD( "c0.u3", 0x1000000, 0x800000, 0x54ede017 )
	ROM_LOAD( "d0.u4", 0x1800000, 0x800000, 0x4be8479d )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s0.u36", 0x000000, 0x200000, 0x2b381b33 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s1.u37", 0x000000, 0x200000, 0x11cd7098 )
ROM_END

ROM_START( vasara2a )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "data.u34",     0x000000, 0x200000, 0x493d0103 )
	ROM_LOAD16_BYTE( "basara-l.u30", 0x200000, 0x080000, 0xfd88b068 )
	ROM_RELOAD(                      0x300000, 0x080000             )
	ROM_LOAD16_BYTE( "basara-h.u31", 0x200001, 0x080000, 0x91d641e6 )
	ROM_RELOAD(                      0x300001, 0x080000             )

	ROM_REGION( 0x2000000, REGION_GFX1, ROMREGION_DISPOSE ) /* Sprites */
	ROM_LOAD( "a0.u1", 0x0000000, 0x800000, 0xa6306c75 )
	ROM_LOAD( "b0.u2", 0x0800000, 0x800000, 0x227cbd9f )
	ROM_LOAD( "c0.u3", 0x1000000, 0x800000, 0x54ede017 )
	ROM_LOAD( "d0.u4", 0x1800000, 0x800000, 0x4be8479d )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s0.u36", 0x000000, 0x200000, 0x2b381b33 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, ROMREGION_ERASE | 0 )	/* Samples */
	ROM_LOAD16_BYTE( "s1.u37", 0x000000, 0x200000, 0x11cd7098 )
ROM_END

/***************************************************************************

Mobile Suit Gundam Final Shooting
Banpresto, 1995

Uses main board STA-0001B SYSTEM SSV

Game Board Layout
-----------------

VISCO (no other PCB numbers)
|---------------------------------------------------------|
|                 GAL16V8(2)                              |
|   VG004-01.U33            VG004-07.U39                  |
|      VG004-02.U34           VG004-08.U40                |
|        VG004-03.U35           VG004-10.U45      TC514400|
|          VG004-04.U36           VG004-09.U43            |
|            VG004-05.U37           VG004-11.U48  TC514400|
|D1            VG004-06.U38                               |
|                                       100MHz    TC514400|
|                                                         |
|                                |--------------| TC514400|
|      TC551001                  |              |         |
|P           |-------| GAL20V8   |              | TC514400|
| S2914      | SETA  | GAL20V8   |     SETA     |         |
|            |ST-0009| GAL20V8   |              | TC514400|
|            |       |           |   ST-0020    |         |
|            |-------|           |              | TC514400|
|      TC551001                  |              |         |
|           SSVV7.U16            |--------------| TC514400|
|               VG004-12.U4                               |
|C1                 VG004-13.U5  TC514260      VG004-14.U3|
| ADC0809             TC55257    GAL16V8(1) SSV2SET0.U1   |
|                     TC55257           SSVSET1.U2        |
|---------------------------------------------------------|
Notes:
      D1         - 4 pin connector for power
      P          - 10 pin connector
      C1         - 10 pin connector for analog controls
      ADC0809    - National Semiconductor ADC0809 8-Bit Microprocessor Compatible A/D Converter with 8-Channel Multiplexer (DIP28)
      S2914      - Seiko S2914 EEPROM (DIP8)
      TC551001   - Toshiba TC551001BFL-70L 128K x8 SRAM (SOP32)
      TC55257    - Toshiba TC55257N-70L 32K x8 SRAM (SOP28)
      TC514260   - Toshiba TC514260BJ-70 256K x16 DRAM (SOJ40)
      GAL16V8(1) - Lattice GAL16V8B stamped 'VG004-18' (DIP20)
      GAL16V8(2) - Lattice GAL16V8B stamped 'VG004-19' (DIP20)
      GAL20V8    - Lattice GAL20V8B all 3 stamped 'VG004-20' and have identical contents (DIP24)
      U16,U2,U1  - 4M MaskROM (DIP32)
      All other ROMs are 8M/16M MaskROM (DIP42)
      Custom Seta ICs -
                       ST-0009 (QFP176)
                       ST-0020 (QFP304, heatsinked)

****************************************************************************/

ROM_START( gdfs )
	ROM_REGION16_LE( 0x400000, REGION_USER1, 0 )		/* V60 Code */
	ROM_LOAD16_WORD( "vg004-14.u3",   0x000000, 0x100000, 0xd88254df )
	ROM_RELOAD(0x100000,0x100000)
	ROM_LOAD16_BYTE( "ssv2set0.u1",   0x200000, 0x080000, 0xc23b9e2c )
	ROM_RELOAD(0x300000,0x80000)
	ROM_LOAD16_BYTE( "ssv2set1.u2",   0x200001, 0x080000, 0xd7d52570 )
	ROM_RELOAD(0x300001,0x80000)

	ROM_REGION( 0x800000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "vg004-09.u43", 0x000000, 0x200000, 0xb7382cfa )
	ROM_LOAD( "vg004-10.u45", 0x200000, 0x200000, 0xb3c6b1cb )
	ROM_LOAD( "vg004-11.u48", 0x400000, 0x200000, 0x1491def1 )

	ROM_REGION( 0x1000000, REGION_GFX2, /*ROMREGION_DISPOSE*/0 )	// Zooming Sprites, read by a blitter
	ROM_LOAD( "vg004-01.u33", 0x0000000, 0x200000, 0xaa9a81c2 )
	ROM_LOAD( "vg004-02.u34", 0x0200000, 0x200000, 0xfa40ecb4 )
	ROM_LOAD( "vg004-03.u35", 0x0400000, 0x200000, 0x90004023 )
	ROM_LOAD( "vg004-04.u36", 0x0600000, 0x200000, 0xfdafd289 )
	ROM_LOAD( "vg004-06.u38", 0x0a00000, 0x200000, 0x3402325f )
	ROM_LOAD( "vg004-05.u37", 0x0800000, 0x200000, 0x9ae488b0 )
	ROM_LOAD( "vg004-07.u39", 0x0c00000, 0x200000, 0x5e89fcf9 )
	ROM_LOAD( "vg004-08.u40", 0x0e00000, 0x200000, 0x6b1746dc )

	ROM_REGION( 0x80000, REGION_GFX3, ROMREGION_DISPOSE )	// Tilemap
	ROM_LOAD( "ssvv7.u16",    0x0000000, 0x080000, 0xf1c3ab6f )

	ROM_REGION16_BE( 0x400000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD16_BYTE( "vg004-12.u4", 0x000000, 0x200000, 0xeb41a4ef )
	ROM_LOAD16_BYTE( "vg004-13.u5", 0x000001, 0x200000, 0xa4ed3977 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND2, 0 )	/* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND3, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )

	ROM_REGION16_BE( 0x400000, REGION_SOUND4, 0 ) /* Samples */
	ROM_COPY( REGION_SOUND1, 0x000000, 0x000000, 0x400000 )
ROM_END

/***************************************************************************


                                Game Drivers


***************************************************************************/

//     year   rom       clone     machine   inputs    init      monitor manufacturer          title                                               flags

GAMEX(1993,  dynagear, 0,        dynagear, dynagear, dynagear, ROT0,   "Sammy",              "Dyna Gear",                                        GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(1993,  keithlcy, 0,        keithlcy, keithlcy, keithlcy, ROT0,   "Visco",              "Dramatic Adventure Quiz Keith & Lucy (Japan)",     GAME_NO_COCKTAIL )
GAMEX(1993,  srmp4,    0,        srmp4,    srmp4,    srmp4,    ROT0,   "Seta",               "Super Real Mahjong PIV (Japan)",                   GAME_NO_COCKTAIL )
GAMEX(1993,  srmp4o,   srmp4,    srmp4,    srmp4,    srmp4,    ROT0,   "Seta",               "Super Real Mahjong PIV (Japan, older set)",        GAME_NO_COCKTAIL ) // by the numbering of the program roms this should be older
GAMEX(1993,  survarts, 0,        survarts, survarts, survarts, ROT0,   "Sammy",              "Survival Arts (World)",                            GAME_NO_COCKTAIL )
GAMEX(1993,  survartu, survarts, survarts, survarts, survarts, ROT0,   "American Sammy",     "Survival Arts (USA)",                              GAME_NO_COCKTAIL )
GAMEX(1994,  drifto94, 0,        drifto94, drifto94, drifto94, ROT0,   "Visco",              "Drift Out '94 - The Hard Order (Japan)",           GAME_NO_COCKTAIL )
GAMEX(1994,  eaglshot, 0,        eaglshot, eaglshot, eaglshot, ROT0,   "Sammy",              "Eagle Shot Golf",                                  GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(1995,  hypreact, 0,        hypreact, hypreact, hypreact, ROT0,   "Sammy",              "Mahjong Hyper Reaction (Japan)",                   GAME_NO_COCKTAIL )
GAMEX(1994,  twineag2, 0,        twineag2, twineag2, twineag2, ROT270, "Seta",               "Twin Eagle II - The Rescue Mission",               GAME_NO_COCKTAIL )
GAMEX(1995,  gdfs,     0,        gdfs,     gdfs,     gdfs,     ROT0,   "Banpresto",          "Mobil Suit Gundam Final Shooting (Japan)",         GAME_NO_COCKTAIL )
GAMEX(1995,  ultrax,   0,        ultrax,   ultrax,   ultrax,   ROT270, "Banpresto + Tsuburaya Prod.", "Ultra X Weapons / Ultra Keibitai",        GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(1996,  janjans1, 0,        janjans1, janjans1, janjans1, ROT0,   "Visco",              "Lovely Pop Mahjong JangJang Shimasho (Japan)",     GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(1996?, meosism,  0,        meosism,  meosism,  meosism,  ROT0,   "Sammy",              "Meosis Magic (Japan)",                             GAME_NO_COCKTAIL )
GAMEX(1996,  stmblade, 0,        stmblade, stmblade, stmblade, ROT270, "Visco",              "Storm Blade (US)",                                 GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(1997,  hypreac2, 0,        hypreac2, hypreac2, hypreac2, ROT0,   "Sammy",              "Mahjong Hyper Reaction 2 (Japan)",                 GAME_NO_COCKTAIL )
GAMEX(1997,  koikois2, 0,        janjans1, koikois2, janjans1, ROT0,   "Visco",              "Koi Koi Shimasho 2 - Super Real Hanafuda (Japan)", GAME_NO_COCKTAIL )
GAMEX(1997,  mslider,  0,        mslider,  mslider,  mslider,  ROT0,   "Visco / Datt Japan", "Monster Slider (Japan)",                           GAME_NO_COCKTAIL )
GAMEX(1997,  srmp7,    0,        srmp7,    srmp7,    srmp7,    ROT0,   "Seta",               "Super Real Mahjong P7 (Japan)",                    GAME_NO_COCKTAIL | GAME_IMPERFECT_SOUND )
GAMEX(1998,  ryorioh,  0,        ryorioh,  ryorioh,  ryorioh,  ROT0,   "Visco",              "Gourmet Battle Quiz Ryohrioh CooKing (Japan)",     GAME_NO_COCKTAIL )
GAMEX(1998,  sxyreact, 0,        sxyreact, sxyreact, sxyreact, ROT0,   "Sammy",              "Pachinko Sexy Reaction (Japan)",                   GAME_NO_COCKTAIL )
GAMEX(1999,  sxyreac2, 0,        sxyreact, sxyreact, sxyreac2, ROT0,   "Sammy",              "Pachinko Sexy Reaction 2 (Japan)",                 GAME_NO_COCKTAIL )
GAMEX(1999,  cairblad, 0,        sxyreact, cairblad, sxyreact, ROT270, "Sammy",              "Change Air Blade (Japan)",                         GAME_NO_COCKTAIL )
GAMEX(2000,  janjans2, 0,        janjans1, janjans1, janjans1, ROT0,   "Visco",              "Lovely Pop Mahjong JangJang Shimasho 2 (Japan)",   GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )
GAMEX(2000,  vasara,   0,        ryorioh,  vasara,   vasara,   ROT270, "Visco",              "Vasara",                                           GAME_NO_COCKTAIL )
GAMEX(2001,  vasara2,  0,        ryorioh,  vasara2,  vasara,   ROT270, "Visco",              "Vasara 2 (set 1)",                                 GAME_NO_COCKTAIL )
GAMEX(2001,  vasara2a, vasara2,  ryorioh,  vasara2,  vasara,   ROT270, "Visco",              "Vasara 2 (set 2)",                                 GAME_NO_COCKTAIL )


//  Games not working properly:


//  Games not working at all:

GAMEX(1997,  jsk,      0,        jsk,      janjans1, jsk,      ROT0,   "Visco",              "Joryuu Syougi Kyoushitsu (Japan)",                 GAME_NO_COCKTAIL | GAME_NOT_WORKING )
