/***************************************************************************

					  -= Fantasy Land / Galaxy Gunners =-

					driver by	Luca Elia (l.elia@tin.it)


CPUs 	:	2 x 8086
Sound	:	YM2151 + DAC
Video	:	?

---------------------------------------------------------------------------
Year + Game				Boards
---------------------------------------------------------------------------
>=1987	Fantasy Land	?
1989	Galaxy Gunners	?
---------------------------------------------------------------------------

Notes:

- Clocks are unknown and the cpu might be an 8088 or a later x86.

[fantland]

- The year of creation isn't displayed, but it's no sooner than 1987 since
  embedded in the roms is: "MS Run-Time Library - Copyright (c) 1987, Microsoft Corp"
- Slowdowns on the ice level's boss


***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"

VIDEO_UPDATE( fantland );

/***************************************************************************

							Memory Maps - Main CPU

***************************************************************************/

static data8_t fantland_nmi_enable;

static WRITE_HANDLER( fantland_nmi_enable_w )
{
	fantland_nmi_enable = data;
//	if ((fantland_nmi_enable != 0) && (fantland_nmi_enable != 8))
//		logerror("CPU #0 : nmi_enable = %02x - PC = %04X\n", data, activecpu_get_pc());
}

static WRITE_HANDLER( fantland_soundlatch_w )
{
	soundlatch_w(0,data);
	cpu_set_nmi_line(1,PULSE_LINE);
}

/***************************************************************************
								Fantasy Land
***************************************************************************/

static MEMORY_READ_START( fantland_readmem )
	{ 0x00000, 0x07fff, MRA_RAM			},
	{ 0x08000, 0x7ffff, MRA_ROM			},
	{ 0xa2000, 0xa21ff, MRA_RAM			},	// not actually read
	{ 0xa3000, 0xa3000, input_port_0_r	},
	{ 0xa3001, 0xa3001, input_port_1_r	},
	{ 0xa3002, 0xa3002, input_port_2_r	},
	{ 0xa3003, 0xa3003, input_port_3_r	},
	{ 0xa4000, 0xa67ff, MRA_RAM			},	// not actually read
	{ 0xc0000, 0xc03ff, MRA_RAM			},	// ""
	{ 0xe0000, 0xfffff, MRA_ROM			},
MEMORY_END

static MEMORY_WRITE_START( fantland_writemem )
	{ 0x00000, 0x07fff, MWA_RAM					},
	{ 0x08000, 0x7ffff, MWA_ROM					},
	{ 0xa2000, 0xa21ff, paletteram_xRRRRRGGGGGBBBBB_w, &paletteram	},
	{ 0xa3000, 0xa3000, fantland_nmi_enable_w	},
	{ 0xa3002, 0xa3002, fantland_soundlatch_w	},
	{ 0xa4000, 0xa67ff, MWA_RAM, &spriteram		},
	{ 0xc0000, 0xc03ff, MWA_RAM, &spriteram_2	},
	{ 0xe0000, 0xfffff, MWA_ROM					},
MEMORY_END


/***************************************************************************
								Galaxy Gunners
***************************************************************************/

static MEMORY_READ_START( galaxygn_readmem )
	{ 0x00000, 0x07fff, MRA_RAM			},
	{ 0x10000, 0x2ffff, MRA_ROM			},
	{ 0x52000, 0x521ff, MRA_RAM			},	// not actually read
	{ 0x53000, 0x53000, input_port_0_r	},
	{ 0x53001, 0x53001, input_port_1_r	},
	{ 0x53002, 0x53002, input_port_2_r	},
	{ 0x53003, 0x53003, input_port_3_r	},
	{ 0x54000, 0x567ff, MRA_RAM			},	// not actually read
	{ 0x60000, 0x603ff, MRA_RAM			},	// ""
	{ 0x60000, 0x7ffff, MRA_ROM			},
	{ 0xf0000, 0xfffff, MRA_ROM			},
MEMORY_END

static MEMORY_WRITE_START( galaxygn_writemem )
	{ 0x00000, 0x07fff, MWA_RAM					},
	{ 0x10000, 0x2ffff, MWA_ROM					},
	{ 0x52000, 0x521ff, paletteram_xRRRRRGGGGGBBBBB_w, &paletteram	},
	{ 0x53000, 0x53000, fantland_nmi_enable_w	},
	{ 0x53002, 0x53002, fantland_soundlatch_w	},
	{ 0x54000, 0x567ff, MWA_RAM, &spriteram		},
	{ 0x60000, 0x603ff, MWA_RAM, &spriteram_2	},
	{ 0x60000, 0x7ffff, MWA_ROM					},
	{ 0xf0000, 0xfffff, MWA_ROM					},
MEMORY_END


/***************************************************************************

							Memory Maps - Sound CPU

***************************************************************************/

static MEMORY_READ_START( fantland_sound_readmem )
	{ 0x00000, 0x01fff, MRA_RAM },
	{ 0x80000, 0x9ffff, MRA_ROM	},
	{ 0xc0000, 0xfffff, MRA_ROM	},
MEMORY_END

static MEMORY_WRITE_START( fantland_sound_writemem )
	{ 0x00000, 0x01fff, MWA_RAM	},
	{ 0x80000, 0x9ffff, MWA_ROM	},
	{ 0xc0000, 0xfffff, MWA_ROM	},
MEMORY_END

static PORT_READ_START( fantland_sound_readport )
	{ 0x0080, 0x0080, soundlatch_r				},
	{ 0x0101, 0x0101, YM2151_status_port_0_r	},
PORT_END

static PORT_WRITE_START( fantland_sound_writeport )
	{ 0x0100, 0x0100, YM2151_register_port_0_w	},
	{ 0x0101, 0x0101, YM2151_data_port_0_w		},
	{ 0x0180, 0x0180, DAC_0_data_w				},
PORT_END


/***************************************************************************

								Input Ports

***************************************************************************/

/***************************************************************************
								Fantasy Land
***************************************************************************/

INPUT_PORTS_START( fantland )
	PORT_START	/* IN0 - a3000 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1			)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START1			)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP		)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN	)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT	)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT	)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1			)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2			)

	PORT_START	/* IN1 - a3001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN2			)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2			)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP		| IPF_PLAYER2 )	// used in test mode only
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN	| IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT	| IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT	| IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1			| IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2			| IPF_PLAYER2 )

	PORT_START	/* IN2 - a3002 */
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x00, "Invulnerability" )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Allow Continue" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Normal" )
	PORT_DIPSETTING(    0x40, "Hard" )
	PORT_DIPSETTING(    0x20, "Harder" )
	PORT_DIPSETTING(    0x00, "Hardest" )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START      /* IN3 - a3003 */
	PORT_DIPNAME( 0x01, 0x01, "Test Sound" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0e, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x0e, "1" )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x0a, "3" )
	PORT_DIPSETTING(    0x08, "4" )
	PORT_DIPSETTING(    0x06, "5" )
	PORT_DIPSETTING(    0x04, "6" )
	PORT_DIPSETTING(    0x02, "7" )
	PORT_DIPSETTING(    0x00, "8" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "800k" )
	PORT_DIPSETTING(    0x20, "1600k" )
	PORT_DIPSETTING(    0x10, "2400k" )
	PORT_DIPSETTING(    0x00, "3200k" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )	//unused?
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )	//unused?
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

/***************************************************************************
								Galaxy Gunners
***************************************************************************/

INPUT_PORTS_START( galaxygn )
	PORT_START	/* IN0 - 53000 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1			)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START1			)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP		)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN	)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT	)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT	)
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1			)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2			)

	PORT_START	/* IN1 - 53001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN2			)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2			)
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP		| IPF_PLAYER2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN	| IPF_PLAYER2 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT	| IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT	| IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1			| IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2			| IPF_PLAYER2 )

	PORT_START	/* IN2 - 53002 */
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x00, "Invulnerability" )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) ) 	// Demo Sounds? doesn't work
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )	// Allow Continue? doesn't work
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Normal" )
	PORT_DIPSETTING(    0x40, "Hard" )
	PORT_DIPSETTING(    0x20, "Harder" )
	PORT_DIPSETTING(    0x00, "Hardest" )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START      /* IN3 - 53003 */
	PORT_DIPNAME( 0x01, 0x01, "Test Sound" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0e, 0x08, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x0e, "1" )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x0a, "3" )
	PORT_DIPSETTING(    0x08, "4" )
	PORT_DIPSETTING(    0x06, "5" )
	PORT_DIPSETTING(    0x04, "6" )
	PORT_DIPSETTING(    0x02, "7" )
	PORT_DIPSETTING(    0x00, "8" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x30, "10k" )
	PORT_DIPSETTING(    0x20, "20k" )
	PORT_DIPSETTING(    0x10, "30k" )
	PORT_DIPSETTING(    0x00, "40k" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )	//unused?
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )	//unused?
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


/***************************************************************************

							Graphics Layouts

***************************************************************************/

static struct GfxLayout layout16x16x6 =
{
	16,16,
	RGN_FRAC(1,1),
	6,
	{ 0,1,2,3,4,5 },
	{ STEP4(3*6,-6),STEP4(7*6,-6),STEP4(11*6,-6),STEP4(15*6,-6) },
	{ STEP16(0,16*6) },
	16*16*6
};

static struct GfxDecodeInfo fantland_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &layout16x16x6, 0, 4 }, // [0] Sprites
	{ -1 }
};

/***************************************************************************

								Machine Drivers

***************************************************************************/

static MACHINE_INIT( fantland )
{
	fantland_nmi_enable = 0;
}

static INTERRUPT_GEN( fantland_irq )
{
	if (fantland_nmi_enable & 8)
		cpu_set_nmi_line(0, PULSE_LINE);
}

static INTERRUPT_GEN( fantland_sound_irq )
{
	cpu_set_irq_line_and_vector(1, 0, HOLD_LINE, 0x80/4);
}

static struct YM2151interface fantland_ym2151_interface =
{
	1,
	3000000,	// ?
	{ YM3012_VOL(35,MIXER_PAN_CENTER,25,MIXER_PAN_CENTER) },
	{ 0 }
};

static struct DACinterface fantland_dac_interface =
{
	1,
	{ 80 }
};

static MACHINE_DRIVER_START( fantland )
	/* basic machine hardware */
	MDRV_CPU_ADD(I86, 8000000)        // ?
	MDRV_CPU_MEMORY(fantland_readmem, fantland_writemem)
	MDRV_CPU_VBLANK_INT(fantland_irq,1)

	MDRV_CPU_ADD(I86, 8000000)        // ?
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(fantland_sound_readmem, fantland_sound_writemem)
	MDRV_CPU_PORTS(fantland_sound_readport,fantland_sound_writeport)
	MDRV_CPU_PERIODIC_INT(fantland_sound_irq,8000)
	// NMI when soundlatch is written

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(fantland)

	MDRV_INTERLEAVE(8000/60)	// sound irq must feed the DAC at 8kHz

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(352,256)
	MDRV_VISIBLE_AREA(0, 352-1, 0, 256-1)
	MDRV_GFXDECODE(fantland_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(256)

	MDRV_VIDEO_UPDATE(fantland)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2151,	fantland_ym2151_interface	)
	MDRV_SOUND_ADD(DAC,		fantland_dac_interface		)
MACHINE_DRIVER_END


static void galaxygn_sound_irq(int line)
{
	cpu_set_irq_line_and_vector(1, 0, line ? ASSERT_LINE : CLEAR_LINE, 0x80/4);
}

static struct YM2151interface galaxygn_ym2151_interface =
{
	1,
	3000000,	// ?
	{ YM3012_VOL(100,MIXER_PAN_CENTER,100,MIXER_PAN_CENTER) },
	{ galaxygn_sound_irq }
};

static MACHINE_DRIVER_START( galaxygn )
	/* basic machine hardware */
	MDRV_CPU_ADD(I86, 8000000)        // ?
	MDRV_CPU_MEMORY(galaxygn_readmem, galaxygn_writemem)
	MDRV_CPU_VBLANK_INT(fantland_irq,1)

	MDRV_CPU_ADD(I86, 8000000)        // ?
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(fantland_sound_readmem, fantland_sound_writemem)
	MDRV_CPU_PORTS(fantland_sound_readport,fantland_sound_writeport)
	// IRQ by YM2151, NMI when soundlatch is written

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(fantland)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(352,256)
	MDRV_VISIBLE_AREA(0, 352-1, 0, 256-1)
	MDRV_GFXDECODE(fantland_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(256)

	MDRV_VIDEO_UPDATE(fantland)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2151,	galaxygn_ym2151_interface	)
MACHINE_DRIVER_END

/***************************************************************************

								ROMs Loading

***************************************************************************/

/***************************************************************************

Fantasy Land by Electronic Devices of Italy

Fantasyl.ev2  od2  d0  d1 s1  s2 :are 27c010 devices
Fantasyl.ev1 &  od1  are 27c512 devices

s1 & s2 are sound files.  (labeled 5 and 6 on devices)
d0 & d1 are next to the image banks  (labeled  7 and 8 on devices)
ev1 od1  od2  ev2  are believed to be program eproms
     (labeled 2, 1, 3, 4 on devices respectively)
     (also labeled under the eprom, location U3, U8, U7, U2 respectively)


3) - 23c4000 mask roms  "05, 06, 07"               !!!! [ 32 pin devices!!! ]
5) - 23c4100 mask roms  "00, 01, 02, 03, 04"   !!!! [ 40 pin devices!!! ]

Fantasyl.01, 00, 02, 03, 04  was read as if it was a 27c400

Fantasy.05, 06, 07 was read as if it was a 27c040

***************************************************************************/

ROM_START( fantland )
	ROM_REGION( 0x100000, REGION_CPU1, 0 )					// Main CPU
	ROMX_LOAD( "fantasyl.ev2", 0x00000, 0x20000, 0xf5bdca0e , ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.od2", 0x00001, 0x20000, 0x9db35023 , ROM_SKIP(1) )
	ROM_COPY( REGION_CPU1,     0x00000, 0x40000, 0x40000 )
	ROMX_LOAD( "fantasyl.ev1", 0xe0000, 0x10000, 0x70e0ee30 , ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.od1", 0xe0001, 0x10000, 0x577b4bd7 , ROM_SKIP(1) )

	ROM_REGION( 0x100000, REGION_CPU2, 0 )					// Sound CPU
	ROM_LOAD16_WORD( "fantasyl.s2", 0x80000, 0x20000, 0xf23837d8 )	// samples (8 bit unsigned)
	ROM_LOAD16_WORD( "fantasyl.s1", 0xc0000, 0x20000, 0x1a324a69 )
	ROM_COPY( REGION_CPU2,          0xc0000, 0xe0000, 0x20000 )

	ROM_REGION( 0x480000, REGION_GFX1, ROMREGION_DISPOSE )	// Sprites
	ROMX_LOAD( "fantasyl.m00", 0x000000, 0x80000, 0x82d819ff , ROM_GROUPWORD | ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.m01", 0x0c0000, 0x80000, 0x70a94139 , ROM_GROUPWORD | ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.05",  0x000002, 0x80000, 0x62b9a00b , ROM_SKIP(2) )

	ROMX_LOAD( "fantasyl.m02", 0x180000, 0x80000, 0xae52bf37 , ROM_GROUPWORD | ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.m03", 0x240000, 0x80000, 0xf3f534a1 , ROM_GROUPWORD | ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.06",  0x180002, 0x80000, 0x867fa549 , ROM_SKIP(2) )

	ROMX_LOAD( "fantasyl.m04", 0x300000, 0x80000, 0xe7b1918c , ROM_GROUPWORD | ROM_SKIP(1) )
	ROMX_LOAD( "fantasyl.d0",  0x3c0001, 0x20000, 0x0f907f19 , ROM_SKIP(2) )
	ROMX_LOAD( "fantasyl.d1",  0x3c0000, 0x20000, 0x10d10389 , ROM_SKIP(2) )
	ROMX_LOAD( "fantasyl.07",  0x300002, 0x80000, 0x162ad422 , ROM_SKIP(2) )
ROM_END

/***************************************************************************
								Galaxy Gunners
***************************************************************************/

ROM_START( galaxygn )
	ROM_REGION( 0x100000, REGION_CPU1, 0 )					// Main CPU
	ROM_LOAD( "gg03.bin", 0x10000, 0x10000, 0x9e469189 )
	ROM_LOAD( "gg02.bin", 0x20000, 0x10000, 0xb87a438f )
	ROM_LOAD( "gg01.bin", 0x60000, 0x10000, 0xad0e5b29 )
	ROM_COPY( REGION_CPU1, 0x60000, 0x70000, 0x10000 )
	ROM_COPY( REGION_CPU1, 0x60000, 0xf0000, 0x10000 )

	ROM_REGION( 0x100000, REGION_CPU2, 0 )					// Sound CPU
	ROM_LOAD( "gg20.bin", 0xc0000, 0x10000, 0xf5c65a85 )
	ROM_COPY( REGION_CPU2, 0xc0000, 0xd0000, 0x10000 )
	ROM_COPY( REGION_CPU2, 0xc0000, 0xe0000, 0x10000 )
	ROM_COPY( REGION_CPU2, 0xc0000, 0xf0000, 0x10000 )

	ROM_REGION( 0x1b0000, REGION_GFX1, ROMREGION_DISPOSE )	// Sprites
	ROMX_LOAD( "gg54.bin", 0x000000, 0x10000, 0xb3621119 , ROM_SKIP(2) )
	ROMX_LOAD( "gg38.bin", 0x000001, 0x10000, 0x52b70f3e , ROM_SKIP(2) )
	ROMX_LOAD( "gg22.bin", 0x000002, 0x10000, 0xea49fee4 , ROM_SKIP(2) )
	ROMX_LOAD( "gg55.bin", 0x030000, 0x10000, 0xbffe278f , ROM_SKIP(2) )
	ROMX_LOAD( "gg39.bin", 0x030001, 0x10000, 0x3f7df1e6 , ROM_SKIP(2) )
	ROMX_LOAD( "gg23.bin", 0x030002, 0x10000, 0x4dcbbc99 , ROM_SKIP(2) )
	ROMX_LOAD( "gg56.bin", 0x060000, 0x10000, 0x0306069e , ROM_SKIP(2) )
	ROMX_LOAD( "gg40.bin", 0x060001, 0x10000, 0xf635aa7e , ROM_SKIP(2) )
	ROMX_LOAD( "gg24.bin", 0x060002, 0x10000, 0x733f5dcf , ROM_SKIP(2) )
	ROMX_LOAD( "gg57.bin", 0x090000, 0x10000, 0xc3919bef , ROM_SKIP(2) )
	ROMX_LOAD( "gg41.bin", 0x090001, 0x10000, 0x1f2757de , ROM_SKIP(2) )
	ROMX_LOAD( "gg25.bin", 0x090002, 0x10000, 0x1d094f95 , ROM_SKIP(2) )
	ROMX_LOAD( "gg58.bin", 0x0c0000, 0x10000, 0x4a459cb8 , ROM_SKIP(2) )
	ROMX_LOAD( "gg42.bin", 0x0c0001, 0x10000, 0xae7a8e1e , ROM_SKIP(2) )
	ROMX_LOAD( "gg26.bin", 0x0c0002, 0x10000, 0xc2f310b4 , ROM_SKIP(2) )
	ROMX_LOAD( "gg59.bin", 0x0f0000, 0x10000, 0xc8d4fbc2 , ROM_SKIP(2) )
	ROMX_LOAD( "gg43.bin", 0x0f0001, 0x10000, 0x74d3a0df , ROM_SKIP(2) )
	ROMX_LOAD( "gg27.bin", 0x0f0002, 0x10000, 0xc2cfd3f9 , ROM_SKIP(2) )
	ROMX_LOAD( "gg60.bin", 0x120000, 0x10000, 0x6e32b549 , ROM_SKIP(2) )
	ROMX_LOAD( "gg44.bin", 0x120001, 0x10000, 0xfcda6efa , ROM_SKIP(2) )
	ROMX_LOAD( "gg28.bin", 0x120002, 0x10000, 0x4d4fc01c , ROM_SKIP(2) )
	ROMX_LOAD( "gg61.bin", 0x150000, 0x10000, 0x177a767a , ROM_SKIP(2) )
	ROMX_LOAD( "gg45.bin", 0x150001, 0x10000, 0x2ba49d47 , ROM_SKIP(2) )
	ROMX_LOAD( "gg29.bin", 0x150002, 0x10000, 0xc1c68148 , ROM_SKIP(2) )
	ROMX_LOAD( "gg62.bin", 0x180000, 0x10000, 0x0fb2d41a , ROM_SKIP(2) )
	ROMX_LOAD( "gg46.bin", 0x180001, 0x10000, 0x5f1bf8ad , ROM_SKIP(2) )
	ROMX_LOAD( "gg30.bin", 0x180002, 0x10000, 0xded7cacf , ROM_SKIP(2) )
ROM_END

GAME( 19??, fantland, 0, fantland, fantland, 0, ROT0,  "Electronic Devices Italy", "Fantasy Land"   )
GAME( 1989, galaxygn, 0, galaxygn, galaxygn, 0, ROT90, "Electronic Devices Italy", "Galaxy Gunners" )

