package ams.com.ams.anyclone;

import java.beans.BeanInfo;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;

/**
 * <p>プロジェクト名: EnyClone</p>
 * <p>タイトル: AnyClone</p>
 * <p>説明: あらゆるクラスをDeepCopyするクラス。</p>
 * <p>Created on 2004/02/17</p>
 * @author 門田明彦
 * @version $Revision: 1.2 $
 */
public final class AnyClone {

  /**
   * cloneObject<BR>
   * 引数のObjectをできる限りDeepCopyして返す<BR>
   * 対応Object: 配列, Collection, プリミティブ型, Bean,
   * その他cloneメソッドがpublicなクラス<BR>
   *
   * <b>このメソッドを clone メソッド内から実行すると無限ループとなる
   * 可能性があるため、clone メソッド内から実行してはいけない。</b>
   * @param obj コピー元オブジェクト
   * @return cloned Object　コピー先オブジェクト
   */
  public static Object cloneObject(Object obj) {
    long start = System.currentTimeMillis();
    Object result = _cloneObject(obj);
    if (obj != null) {
      System.out.println("CloneUtil2.cloneObject " + (System.currentTimeMillis() - start) + "ms " + obj.getClass().getName());
    }
    return result;
  }

  /**
   * _cloneObject
   * DeepCopyメソッド（内部用）
   * @param obj コピー元オブジェクト
   * @return cloned Object コピー先オブジェクト
   */
  private static Object _cloneObject(Object obj) {
    Object result = null;

    if (obj == null) {
      return result;
    }

    Class obj_class = obj.getClass();

    if (obj_class.isArray()) {
      // 配列の場合
      result = arraycopy(obj);

    } else if (obj instanceof Collection) {
      // Collectionの場合
      try {
        Collection obj_list = (Collection)obj;
        Collection result_list = (Collection)obj_class.newInstance();

        Iterator it = obj_list.iterator();

        while(it.hasNext()) {
          result_list.add(_cloneObject(it.next()));
        }
        result = result_list;
      } catch (InstantiationException e) {
        result = obj;
      } catch (IllegalAccessException e) {
        result = obj;
      }

    } else {
      // 単一オブジェクト
      result = invokeClone(obj);
// for debug
//      System.out.print(" clone obj " + obj + " > " + result);
//      System.out.println("  deep ? " + (boolean)(obj != result));
    }
    return result;
  }

  /**
   * arraycopy<BR>
   * 配列オブジェクトをディープコピーして取得<BR>
   * プリミティブ型の配列とオブジェクト型の配列を区別する
   * @param obj コピ^-元オブジェクト
   * @return コピーした新しいオブジェクト
   */
  private static Object arraycopy(Object obj) {

    Object result;
    Class obj_class = obj.getClass();

    if (obj_class == int[].class) {
      // int 配列
      int[] objs = (int[]) obj;
      result = new int[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == byte[].class) {
      // byte 配列
      byte[] objs = (byte[]) obj;
      result = new byte[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == char[].class) {
      // char 配列
      char[] objs = (char[]) obj;
      result = new char[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == short[].class) {
      // short 配列
      short[] objs = (short[]) obj;
      result = new short[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == long[].class) {
      // long 配列
      long[] objs = (long[]) obj;
      result = new long[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == float[].class) {
      // float 配列
      float[] objs = (float[]) obj;
      result = new float[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == double[].class) {
      // double 配列
      double[] objs = (double[]) obj;
      result = new double[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else if (obj_class == boolean[].class) {
      // boolean 配列
      boolean[] objs = (boolean[]) obj;
      result = new boolean[objs.length];
      System.arraycopy(objs, 0, result, 0, objs.length);

    } else {
      // オブジェクトの配列
      // 具象クラスのの配列でnewする
      Object[] objs = (Object[]) obj;

      Object dst_obj = Array.newInstance(obj.getClass().getComponentType(), objs.length);
      Object[] dst_objs = (Object[])dst_obj;

      for (int i = 0; i < objs.length; i++) {
        dst_objs[i] = _cloneObject(objs[i]);
      }
      result = dst_objs;
    }
    return result;
  }

  /**
   * invokeClone
   * clone メソッドの実行を試みる。
   * @param o 非配列のコピー元オブジェクト
   * @return closeメソッドの実行に成功した場合はcloneの結果。失敗した場合は引数の参照
   */
  private static Object invokeClone(Object o) {
    Class c = o.getClass();

    if (o instanceof String) {
//      return new String((String) o);
      // final クラスなのでそのまま返す
      return (String)o;

    } else if (o instanceof Byte) {
      return new Byte(((Byte) o).byteValue());

    } else if (o instanceof Character) {
      return new Character(((Character) o).charValue());

    } else if (o instanceof Double) {
      return new Double(((Double) o).doubleValue());

    } else if (o instanceof Float) {
      return new Float(((Float) o).floatValue());

    } else if (o instanceof Integer) {
      return new Integer(((Integer) o).intValue());

    } else if (o instanceof Long) {
      return new Long(((Long) o).longValue());

    } else if (o instanceof Short) {
      return new Short(((Short) o).shortValue());

    } else if (o instanceof Boolean) {
      return new Boolean(((Boolean) o).booleanValue());

    }
// ここにその他の特別なクラス（引数無しのコンストラクタが存在しないクラス)
// の生成処理を記述する (java.sql.Timestamp..)

    // Cloneableクラス
    try {
      Method method = c.getDeclaredMethod("clone", null);
      return method.invoke(o, null);
    } catch (Exception e) {
      // public clone() になっていないか、clone() 中に例外発生。
    }

    // clone を利用できない場合はBeanとして扱う
    try {
      // Beanクラス
      BeanInfo info = Introspector.getBeanInfo(o.getClass());
      PropertyDescriptor[] properties = info.getPropertyDescriptors();
      Object obj_dst = Class.forName(o.getClass().getName()).newInstance();
//      BeanInfo info_dst = Introspector.getBeanInfo(obj_dst.getClass());
      //PropertyDescriptor[] properties_dst = info_dst.getPropertyDescriptors();

      for (int i = 0; i < properties.length; i++) {
//        Class type = properties[i].getPropertyType();
//        String name = properties[i].getName();
        Method m_get = properties[i].getReadMethod();
        Method m_set = properties[i].getWriteMethod();
//        Method m_set = properties_dst[i].getWriteMethod();
        if (m_set != null && m_get != null) {
          Object value = m_get.invoke(o, null);
          Object[] param = {_cloneObject(value)};
       //   System.out.println(" invoke[method:" + name + " type:" + type.getName() + " value:" + param[0].getClass().getName());
          m_set.invoke(obj_dst, param);
        }
      }

      return obj_dst;
    } catch (InvocationTargetException e) {
      e.getTargetException().printStackTrace();
    } catch (Exception e) {
      // 大いにExceptionがthrowされる可能性有り
      e.printStackTrace();
//    } catch (IllegalArgumentException e1) {
//    } catch (ClassNotFoundException e1) {
//    } catch (IntrospectionException e1) {
//    } catch (IllegalAccessException e1) {
    }

    // deepcopyに失敗した場合は引数をそのまま返してshallowcopyさせる
    return o;
  }


}
