// Menu.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _MENU_H_
#define _MENU_H_

#pragma warning(disable : 4297)

#include "../Manah/Object.h"
#include <commctrl.h>
#include <set>
#include <stdexcept>

namespace Manah {
namespace Windows {
namespace Controls {


// CMenu class definition
/////////////////////////////////////////////////////////////////////////////

class CMenu : public CSelfAssertable {
	// RXgN^
public:
	explicit CMenu(bool bPopup = true);
	explicit CMenu(HMENU hMenu);
	virtual ~CMenu();

	// \bh
public:
	bool			AppendMenuItem(UINT nID, UINT nType, const TCHAR* lpszCaption = 0);
	bool			AppendMenuItem(UINT nID, UINT nType, UINT nState, const TCHAR* lpszCaption = 0);
	DWORD			CheckMenuItem(UINT nID, bool bCheck = true);
	bool			CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID);
	bool			DeleteMenuItem(UINT iItem, bool bByCommand = true);
	bool			EnableMenuItem(UINT nID, bool bEnable = true);
	LRESULT			ExecuteMenuChar(TCHAR chCode, UINT nFlag);
	DWORD			GetContextHelpId() const;
	UINT			GetDefaultMenuItem(UINT nFlags) const;
	int				GetItemCount() const;
	bool			GetMenuItemCaption(UINT iItem, TCHAR* lpsz, int cchMax, bool bByCommand = true) const;
	int				GetMenuItemCaptionLength(UINT iItem, bool bByCommand = true) const;
	UINT			GetMenuItemID(int iItem) const;
	bool			GetMenuItemInfo(UINT iItem, MENUITEMINFO& mii, bool bByCommand = true) const;
	UINT			GetMenuItemState(UINT iItem, bool bByCommand = true) const;
	HMENU			GetSafeHmenu() const;
	CMenu			GetSubMenu(UINT iItem) const;
	bool			HasSubMenu(UINT iItem) const;
	bool			InsertMenuItem(UINT nID, UINT nPrevID, UINT nType, UINT nState, const TCHAR* lpszCaption);
	bool			IsMenu() const;
	static CMenu*	Load(HINSTANCE hInstance, const TCHAR* lpszName);
	static CMenu*	Load(const MENUTEMPLATE* pTemplate);
	int				MenuItemFromPoint(HWND hWnd, const POINT& pt) const;
	bool			ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt);
	bool			RemoveMenuItem(UINT iItem, bool bByCommand = true);
	bool			SetChildPopup(const CMenu& popup, UINT iItem, bool bDelegateOwnership = true, bool bByCommand = false);
	bool			SetContextHelpId(DWORD id);
	bool			SetDefaultMenuItem(UINT iItem, bool bByCommand = true);
	bool			SetMenuItemBitmaps(UINT iItem, HBITMAP hUncheckedBitmap, HBITMAP hCheckedBitmap, bool bByCommand = true);
	bool			TrackPopupMenu(UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect = 0);
	bool			TrackPopupMenuEx(UINT nFlags, int x, int y, HWND hWnd, const TPMPARAMS* lptpm = 0);
#if(WINVER >= 0x0500)
	bool			GetMenuInfo(MENUINFO& mi) const;
	bool			SetMenuInfo(const MENUINFO& mi);
#endif /* WINVER >= 0x0500 */

protected:
	virtual void	AssertValidAsMenu() const {
#ifdef _DEBUG
		AssertValid();
		assert(IsMenu());
#endif
	}

	// f[^o
private:
	HMENU					m_hMenu;				// handle for popup menu
	std::set<const CMenu*>	m_children;				// for deletion
	const bool				m_bManage;				// if contained menu was created/will be destroyed by this object
	bool					m_bCreatedByGetSubMenu;	// if object created by GetSubMenu method
};


// CMenu class implementation
/////////////////////////////////////////////////////////////////////////////

inline CMenu::CMenu(bool bPopup /* = true */)
		: m_hMenu(bPopup ? ::CreatePopupMenu() : ::CreateMenu()), m_bManage(true), m_bCreatedByGetSubMenu(false) {
}

inline CMenu::CMenu(HMENU hMenu) : m_hMenu(hMenu), m_bManage(false), m_bCreatedByGetSubMenu(false) {
	if(!IsMenu())
		throw std::invalid_argument("specified menu handle is not valid.");
}

inline CMenu::~CMenu() {
	if(!m_bCreatedByGetSubMenu) {
		while(m_bManage) {
			const UINT	c = GetItemCount();
			if(c == -1 || c == 0)
				break;
			RemoveMenuItem(0, false);
		}
		for(std::set<const CMenu*>::iterator it = m_children.begin(); it != m_children.end(); ++it)
			delete *it;
		if(m_bManage)
			::DestroyMenu(m_hMenu);
	}
}

inline bool CMenu::AppendMenuItem(UINT nID, UINT nType, const TCHAR* lpszCaption /* = 0 */) {
	AssertValidAsMenu();
	return AppendMenuItem(nID, nType, MFS_ENABLED, lpszCaption);
}

inline bool CMenu::AppendMenuItem(UINT nID, UINT nType, UINT nState, const TCHAR* lpszCaption /* = 0 */) {
	AssertValidAsMenu();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_FTYPE | MIIM_ID | MIIM_STATE;
	mii.fState = nState;
	mii.fType = nType;
	mii.wID = nID;
	if(toBoolean(nType & MFT_OWNERDRAW) && lpszCaption != 0) {
		mii.fMask |= MIIM_DATA;
		mii.dwItemData = reinterpret_cast<DWORD>(lpszCaption);
	}
	if(lpszCaption != 0) {
		mii.fMask |= MIIM_STRING;
		mii.dwTypeData = const_cast<TCHAR*>(lpszCaption);
	}
	return toBoolean(::InsertMenuItem(m_hMenu, ::GetMenuItemCount(m_hMenu), true, &mii));
}

inline DWORD CMenu::CheckMenuItem(UINT nID, bool bCheck /* = true */) {
	AssertValidAsMenu();
	return ::CheckMenuItem(m_hMenu, nID, bCheck ? MFS_CHECKED : MFS_UNCHECKED);
}

inline bool CMenu::CheckMenuRadioItem(UINT nFirstID, UINT nLastID, UINT nItemID) {
	AssertValidAsMenu();
	return toBoolean(::CheckMenuRadioItem(m_hMenu, nFirstID, nLastID, nItemID, MF_BYCOMMAND));
}

inline bool CMenu::DeleteMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValidAsMenu();
	return toBoolean(::DeleteMenu(m_hMenu, iItem, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline bool CMenu::EnableMenuItem(UINT nID, bool bEnable /* = true */) {
	AssertValidAsMenu();
	return toBoolean(::EnableMenuItem(m_hMenu, nID, MF_BYCOMMAND | (bEnable ? MFS_ENABLED : MFS_GRAYED)));
}

inline LRESULT CMenu::ExecuteMenuChar(TCHAR chCode, UINT nFlag) {
	AssertValidAsMenu();

	const UINT	cMenuItems = GetItemCount();

	if(chCode >= _T('a') && chCode <= _T('z'))	// make upper
		chCode -= 0x20;
	for(UINT iItem = 0; iItem < cMenuItems; ++iItem) {
		TCHAR			szCaption[100];
		const TCHAR*	pszAccel = 0;

		GetMenuItemCaption(iItem, szCaption, 100, false);
		pszAccel = std::_tcschr(szCaption, chCode);
		if(pszAccel != 0) {
			if(pszAccel[1] == chCode)
				return (iItem | 0x00020000);
		}
	}
	return MNC_IGNORE;
}

inline DWORD CMenu::GetContextHelpId() const {
	AssertValidAsMenu();
	return ::GetMenuContextHelpId(m_hMenu);
}

inline UINT CMenu::GetDefaultMenuItem(UINT nFlags) const {
	AssertValidAsMenu();
	return ::GetMenuDefaultItem(m_hMenu, false, nFlags);
}

inline bool CMenu::GetMenuItemCaption(UINT iItem, TCHAR* lpsz, int cchMax, bool bByCommand /* = true */) const {
	AssertValidAsMenu();
	return toBoolean(::GetMenuString(m_hMenu, iItem, lpsz, cchMax, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline int CMenu::GetMenuItemCaptionLength(UINT iItem, bool bByCommand /* = true */) const {
	AssertValidAsMenu();
	return ::GetMenuString(m_hMenu, iItem, 0, 0, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION);
}

inline int CMenu::GetItemCount() const {
	AssertValidAsMenu();
	return ::GetMenuItemCount(m_hMenu);
}

inline UINT CMenu::GetMenuItemID(int iItem) const {
	AssertValidAsMenu();
	return ::GetMenuItemID(m_hMenu, iItem);
}

inline bool CMenu::GetMenuItemInfo(UINT iItem, MENUITEMINFO& mii, bool bByCommand /* = true */) const {
	AssertValidAsMenu();
	return toBoolean(::GetMenuItemInfo(m_hMenu, iItem, !bByCommand, &mii));
}

inline UINT CMenu::GetMenuItemState(UINT iItem, bool bByCommand /* = true */) const {
	AssertValidAsMenu();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_STATE;
	::GetMenuItemInfo(m_hMenu, iItem, !bByCommand, &mii);
	return mii.fState;
}

inline HMENU CMenu::GetSafeHmenu() const {
	return (this != 0) ? m_hMenu : 0;
}

inline CMenu CMenu::GetSubMenu(UINT iItem) const {
	AssertValidAsMenu();
	HMENU	hMenu = ::GetSubMenu(m_hMenu, iItem);
	if(hMenu == 0)
		throw std::invalid_argument("Specified index is out of range or invalid.");
	CMenu	subMenu = CMenu(hMenu);
	subMenu.m_bCreatedByGetSubMenu = true;
	return subMenu;
}

inline bool CMenu::HasSubMenu(UINT iItem) const {
	AssertValidAsMenu();
	return toBoolean(::IsMenu(::GetSubMenu(m_hMenu, iItem)));
}

inline bool CMenu::InsertMenuItem(UINT nID, UINT nPrevID,
		UINT nType, UINT nState, const TCHAR* lpszCaption) {
	AssertValidAsMenu();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_FTYPE | MIIM_ID | MIIM_STATE;
	mii.fType = nType;
	mii.fState = nState;
	mii.wID = nID;
	if(toBoolean(nType & MFT_OWNERDRAW) && lpszCaption != 0) {
		mii.fMask |= MIIM_DATA;
		mii.dwItemData = reinterpret_cast<DWORD>(lpszCaption);
	}
	if(lpszCaption != 0) {
		mii.fMask |= MIIM_STRING;
		mii.dwTypeData = const_cast<TCHAR*>(lpszCaption);
	}
	return toBoolean(::InsertMenuItem(m_hMenu, nPrevID, false, &mii));
}

inline bool CMenu::IsMenu() const {
	// for diagnotics
	AssertValid();
	return toBoolean(::IsMenu(m_hMenu));
}

inline CMenu* CMenu::Load(HINSTANCE hInstance, const TCHAR* lpszName) {
	if(HMENU hMenu = ::LoadMenu(hInstance, lpszName)) {
		CMenu*	pMenu = new CMenu(hMenu);
		const_cast<bool&>(pMenu->m_bManage) = true;
		return pMenu;
	} else
		return 0;
}

inline CMenu* CMenu::Load(const MENUTEMPLATE* pTemplate) {
	assert(pTemplate != 0);
	if(HMENU hMenu = ::LoadMenuIndirect(pTemplate)) {
		CMenu*	pMenu = new CMenu(hMenu);
		const_cast<bool&>(pMenu->m_bManage) = true;
		return pMenu;
	} else
		return 0;
}

inline int CMenu::MenuItemFromPoint(HWND hWnd, const POINT& pt) const {
	AssertValidAsMenu();
	return ::MenuItemFromPoint(hWnd, m_hMenu, pt);
}

inline bool CMenu::ModifyMenuItem(UINT nID, UINT nFlags, const TCHAR* lpszPrompt) {
	AssertValidAsMenu();
	assert(!(nFlags & MF_POPUP));
	return toBoolean(::ModifyMenu(m_hMenu, nID, MF_BYCOMMAND | nFlags, nID, lpszPrompt));
}

inline bool CMenu::RemoveMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValidAsMenu();
	return toBoolean(::RemoveMenu(m_hMenu, iItem, bByCommand ? MF_BYCOMMAND : MF_BYPOSITION));
}

inline bool CMenu::SetChildPopup(const CMenu& popup, UINT iItem, bool bDelegateOwnership /* = true */, bool bByCommand /* = false */) {
	AssertValidAsMenu();

	MENUITEMINFO	mii;

	ZeroMemory(&mii, sizeof(MENUITEMINFO));
	mii.cbSize = sizeof(MENUITEMINFO);
	mii.fMask = MIIM_SUBMENU;
	mii.hSubMenu = popup.m_hMenu;

	if(toBoolean(::SetMenuItemInfo(m_hMenu, iItem, !bByCommand, &mii))) {
		if(bDelegateOwnership)
			m_children.insert(&popup);
		return true;
	} else
		return false;
}

inline bool CMenu::SetDefaultMenuItem(UINT iItem, bool bByCommand /* = true */) {
	AssertValidAsMenu();
	return toBoolean(::SetMenuDefaultItem(m_hMenu, iItem, !bByCommand));
}

inline bool CMenu::SetContextHelpId(DWORD id) {
	AssertValidAsMenu();
	return toBoolean(::SetMenuContextHelpId(m_hMenu, id));
}

inline bool CMenu::SetMenuItemBitmaps(UINT iItem, HBITMAP hUncheckedBitmap, HBITMAP hCheckedBitmap, bool bByCommand /* = true */) {
	AssertValidAsMenu();
	return toBoolean(::SetMenuItemBitmaps(m_hMenu, iItem,
		bByCommand ? MF_BYCOMMAND : MF_BYPOSITION, hUncheckedBitmap, hCheckedBitmap));
}

inline bool CMenu::TrackPopupMenu(UINT nFlags, int x, int y, HWND hWnd, const RECT* prcRect /* = 0 */) {
	AssertValidAsMenu();
	return toBoolean(::TrackPopupMenu(m_hMenu, nFlags, x, y, 0, hWnd, prcRect));
}

inline bool CMenu::TrackPopupMenuEx(UINT nFlags, int x, int y, HWND hWnd, const TPMPARAMS* lptpm /* = 0 */) {
	AssertValidAsMenu();
	return toBoolean(::TrackPopupMenuEx(m_hMenu, nFlags, x, y, hWnd, const_cast<TPMPARAMS*>(lptpm)));
}

#if(WINVER >= 0x0500)
inline bool CMenu::GetMenuInfo(MENUINFO& mi) const {
	AssertValidAsMenu();
	return toBoolean(::GetMenuInfo(m_hMenu, &mi));
}

inline bool CMenu::SetMenuInfo(const MENUINFO& mi) {
	AssertValidAsMenu();
	return toBoolean(::SetMenuInfo(m_hMenu, &mi));
}
#endif /* WINVER >= 0x0500 */

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#pragma warning(default : 4297)

#endif /* _MENU_OPERATOR_H_ */

/* [EOF] */