// ImageList.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _IMAGE_LIST_H_
#define _IMAGE_LIST_H_

#include "Object.h"

namespace Manah {
namespace Windows {

// CImageList class implementation
/////////////////////////////////////////////////////////////////////////////

class CImageList : public CSelfAssertable {
	// RXgN^
public:
	CImageList();
	virtual ~CImageList();

	// \bh
public:
	bool	Create(int cx, int cy, UINT nFlags, int nInitial, int nGrow);
	bool	Create(HINSTANCE hInstance, UINT nBitmapID, int cx, int nGrow, COLORREF clrMask);
	bool	Create(HINSTANCE hInstance, const TCHAR* lpszBitmapID, int cx, int nGrow, COLORREF clrMask);
	bool	Create(CImageList& refImageList1, int nImage1, CImageList& refImageList2, int nImage2, int dx, int dy);
	bool	Create(CImageList* pImageList);

	HIMAGELIST			GetSafeHandle() const;
	operator			HIMAGELIST() const;
	static CImageList*	FromHandle(HIMAGELIST hImageList);
	static CImageList*	FromHandlePermanent(HIMAGELIST hImageList);	// client MUST DELETE returned pointer, finally
	int					GetImageCount() const;
	COLORREF			GetBkColor() const;
	COLORREF			SetBkColor(COLORREF clr);
	bool				GetImageInfo(int iImage, IMAGEINFO& imageInfo) const;

	bool				Attach(HIMAGELIST hImageList);
	HIMAGELIST			Detach();
	bool				DeleteImageList();
	bool				SetImageCount(UINT nNewCount);
	int					Add(HBITMAP hImage, HBITMAP hMask);
	int					Add(HBITMAP hImage, COLORREF clrMask);
	int					Add(HICON hIcon);
	bool				Remove(int iImage);
	bool				Replace(int iImage, HBITMAP hBitmap, HBITMAP hMask);
	int					Replace(int iImage, HICON hIcon);
	HICON				ExtractIcon(HINSTANCE hInstance, int iImage);
	bool				Draw(HDC hDC, int iImage, const POINT& pt, UINT nStyle);
	bool				SetOverlayImage(int iImage, int iOverlay);
	bool				Copy(int iDest, int iSrc, UINT nFlags = ILCF_MOVE);
	bool				Copy(int iDest, const CImageList& imageList, int iSrc, UINT nFlags = ILCF_MOVE);
	bool				DrawIndirect(const IMAGELISTDRAWPARAMS& imldp);
	bool				DrawIndirect(HDC hDC, int iImage,
							const POINT& pt, const SIZE& size, const POINT& ptOrigin,
							UINT fStyle = ILD_NORMAL, DWORD dwRop = SRCCOPY,
							COLORREF rgbBack = CLR_DEFAULT, COLORREF rgbFore = CLR_DEFAULT);
	bool				SetDragCursorImage(int iDrag, const POINT& ptHotSpot);
	static CImageList*	GetDragImage(POINT& pt, POINT& ptHotSpot);
	bool				Read(IStream* lpStream);
	bool				Write(IStream* lpStream);
	bool				BeginDrag(int iImage, const POINT& ptHotSpot);
	static bool			DragEnter(HWND hwndLock, const POINT& pt);
	static void			EndDrag();
	static bool			DragLeave(HWND hwndLock);
	static bool			DragMove(const POINT& pt);
	static bool			DragShowNolock(bool bShow);

	// f[^o
private:
	HIMAGELIST	m_hImageList;
};


// CImageList class implementation
/////////////////////////////////////////////////////////////////////////////

inline CImageList::CImageList() : m_hImageList(0) {
}

inline CImageList::~CImageList() {
	if(m_hImageList != 0)
		DeleteImageList();
}

inline CImageList::operator HIMAGELIST() const {
	AssertValid();
	return m_hImageList;
}

inline int CImageList::Add(HBITMAP hImage, HBITMAP hMask) {
	AssertValid();
	return ::ImageList_Add(m_hImageList, hImage, hMask);
}

inline int CImageList::Add(HBITMAP hImage, COLORREF clrMask) {
	AssertValid();
	return ::ImageList_AddMasked(m_hImageList, hImage, clrMask);
}

inline int CImageList::Add(HICON hIcon) {
	AssertValid();
	return ::ImageList_AddIcon(m_hImageList, hIcon);
}

inline bool CImageList::Attach(HIMAGELIST hImageList) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = hImageList;
	return (m_hImageList != 0);
}

inline bool CImageList::BeginDrag(int iImage, const POINT& ptHotSpot) {
	AssertValid();
	return toBoolean(::ImageList_BeginDrag(m_hImageList, iImage, ptHotSpot.x, ptHotSpot.y));
}

inline bool CImageList::Copy(int iDest, int iSrc, UINT nFlags /* = ILCF_MOVE */) {
	AssertValid();
	return toBoolean(::ImageList_Copy(m_hImageList, iDest, m_hImageList, iSrc, nFlags));
}

inline bool CImageList::Copy(int iDest, const CImageList& imageList, int iSrc, UINT nFlags /* = ILCF_MOVE */) {
	AssertValid();
	return toBoolean(::ImageList_Copy(m_hImageList, iDest, imageList.GetSafeHandle(), iSrc, nFlags));
}

inline bool CImageList::Create(int cx, int cy, UINT nFlags, int nInitial, int nGrow) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = ::ImageList_Create(cx, cy, nFlags, nInitial, nGrow);
	return (m_hImageList != 0);
}

inline bool CImageList::Create(HINSTANCE hInstance, UINT nBitmapID, int cx, int nGrow, COLORREF clrMask) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = ::ImageList_LoadBitmap(hInstance, MAKEINTRESOURCE(nBitmapID), cx, nGrow, clrMask);
	return (m_hImageList != 0);
}

inline bool CImageList::Create(HINSTANCE hInstance, const TCHAR* lpszBitmapID, int cx, int nGrow, COLORREF clrMask) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = ::ImageList_LoadBitmap(hInstance, lpszBitmapID, cx, nGrow, clrMask);
	return (m_hImageList != 0);
}

inline bool CImageList::Create(
	CImageList& refImageList1, int nImage1, CImageList& refImageList2, int nImage2, int dx, int dy) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = ::ImageList_Merge(
		refImageList1.GetSafeHandle(), nImage1, refImageList2.GetSafeHandle(), nImage2, dx, dy);
	return (m_hImageList != 0);
}

inline bool CImageList::Create(CImageList* pImageList) {
	AssertValid();

	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = ::ImageList_Duplicate(pImageList->GetSafeHandle());
	return (m_hImageList != 0);
}

inline bool CImageList::DeleteImageList() {
	AssertValid();
	return toBoolean(::ImageList_Destroy(m_hImageList));
}

inline HIMAGELIST CImageList::Detach() {
	AssertValid();

	HIMAGELIST	hImageList = 0;

	hImageList = m_hImageList;
	m_hImageList = 0;
	return hImageList;
}

inline bool CImageList::DragEnter(HWND hwndLock, const POINT& pt) {
	return toBoolean(::ImageList_DragEnter(hwndLock, pt.x, pt.y));
}

inline bool CImageList::DragLeave(HWND hwndLock) {
	return toBoolean(::ImageList_DragLeave(hwndLock));
}

inline bool CImageList::DragMove(const POINT& pt) {
	return toBoolean(::ImageList_DragMove(pt.x, pt.y));
}

inline bool CImageList::DragShowNolock(bool bShow) {
	return toBoolean(::ImageList_DragShowNolock(bShow));
}

inline bool CImageList::Draw(HDC hDC, int iImage, const POINT& pt, UINT nStyle) {
	AssertValid();
	return toBoolean(::ImageList_Draw(m_hImageList, iImage, hDC, pt.x, pt.y, nStyle));
}

inline bool CImageList::DrawIndirect(const IMAGELISTDRAWPARAMS& imldp) {
	AssertValid();
	return toBoolean(::ImageList_DrawIndirect(const_cast<IMAGELISTDRAWPARAMS*>(&imldp)));
}

inline bool CImageList::DrawIndirect(HDC hDC, int iImage,
		const POINT& pt, const SIZE& size, const POINT& ptOrigin, UINT fStyle /* = ILD_NORMAL */,
		DWORD dwRop /* = SRCCOPY */, COLORREF rgbBack /* = CLR_DEFAULT */, COLORREF rgbFore /* = CLR_DEFAULT */) {
	AssertValid();

	IMAGELISTDRAWPARAMS	imldp = {
		sizeof(IMAGELISTDRAWPARAMS), m_hImageList, iImage, hDC,
		pt.x, pt.y, size.cx, size.cy, ptOrigin.x, ptOrigin.y, rgbBack, rgbFore, dwRop
	};
	return toBoolean(::ImageList_DrawIndirect(&imldp));
}

inline void CImageList::EndDrag() {
	::ImageList_EndDrag();
}

inline HICON CImageList::ExtractIcon(HINSTANCE hInstance, int iImage) {
	AssertValid();
	return ::ImageList_ExtractIcon(hInstance, m_hImageList, iImage);
}

inline CImageList* CImageList::FromHandle(HIMAGELIST hImageList) {
	static CImageList	objTemp;

	if(hImageList == 0)
		return 0;
	objTemp.m_hImageList = hImageList;
	return &objTemp;
}

inline CImageList* CImageList::FromHandlePermanent(HIMAGELIST hImageList) {
	CImageList*	pNewObject = 0;

	if(hImageList = 0)
		return 0;
	pNewObject = new CImageList();
	pNewObject->m_hImageList = hImageList;
	return pNewObject;
}

inline COLORREF CImageList::GetBkColor() const {
	AssertValid();
	return ::ImageList_GetBkColor(m_hImageList);
}

inline CImageList* CImageList::GetDragImage(POINT& pt, POINT& ptHotSpot) {
	static CImageList	temp;
	if(HIMAGELIST hImageList = ::ImageList_GetDragImage(&pt, &ptHotSpot)) {
		temp.m_hImageList = hImageList;
		return &temp;
	}
	return 0;
}

inline int CImageList::GetImageCount() const {
	AssertValid();
	return ::ImageList_GetImageCount(m_hImageList);
}

inline bool CImageList::GetImageInfo(int iImage, IMAGEINFO& imageInfo) const {
	AssertValid();
	return toBoolean(::ImageList_GetImageInfo(m_hImageList, iImage, &imageInfo));
}

inline HIMAGELIST CImageList::GetSafeHandle() const {
	AssertValid();
	return m_hImageList;
}

inline bool CImageList::Read(LPSTREAM lpStream) {
	AssertValid();

	HIMAGELIST	hImageList;

	hImageList = ::ImageList_Read(lpStream);
	if(hImageList == 0)
		return false;
	if(m_hImageList != 0)
		DeleteImageList();
	m_hImageList = hImageList;
	return true;
}

inline bool CImageList::Remove(int iImage) {
	AssertValid();
	return toBoolean(::ImageList_Remove(m_hImageList, iImage));
}

inline bool CImageList::Replace(int iImage, HBITMAP hBitmap, HBITMAP hMask) {
	AssertValid();
	return toBoolean(::ImageList_Replace(m_hImageList, iImage, hBitmap, hMask));
}

inline int CImageList::Replace(int iImage, HICON hIcon) {
	AssertValid();
	return ::ImageList_ReplaceIcon(m_hImageList, iImage, hIcon);
}

inline COLORREF CImageList::SetBkColor(COLORREF clr) {
	AssertValid();
	return ::ImageList_SetBkColor(m_hImageList, clr);
}

inline bool CImageList::SetDragCursorImage(int iDrag, const POINT& ptHotSpot) {
	AssertValid();
	return toBoolean(::ImageList_SetDragCursorImage(m_hImageList, iDrag, ptHotSpot.x, ptHotSpot.y));
}

inline bool CImageList::SetImageCount(UINT nNewCount) {
	AssertValid();
	return toBoolean(::ImageList_SetImageCount(m_hImageList, nNewCount));
}

inline bool CImageList::SetOverlayImage(int iImage, int iOverlay) {
	AssertValid();
	return toBoolean(::ImageList_SetOverlayImage(m_hImageList, iImage, iOverlay));
}

inline bool CImageList::Write(LPSTREAM lpStream) {
	AssertValid();
	return toBoolean(::ImageList_Write(m_hImageList, lpStream));
}

} /* namespace Windows */
} /* namespace Manah */

#endif /* _IMAGE_LIST_H_ */

/* [EOF] */