// DocumentView.h
// (c) 2003-2005 exeal

#ifndef _DOCUMENT_VIEW_H_
#define _DOCUMENT_VIEW_H_
#include "Window.h"
#include <vector>
#include <algorithm>	// std::find


namespace Manah {
namespace Windows {

template<typename Update> class CDocument;

template<typename DocumentUpdate>
class _CViewBase {
protected:
	virtual ~_CViewBase() {}
protected:
	virtual void OnDocumentSetup() = 0;	// called by CDocument::SetupAllView
	virtual void OnInitialUpdate() = 0;	// called by CDocument::AddView
	virtual void OnUpdate(const DocumentUpdate& update) = 0;	// called by CDocument::UpdateAllViews
	friend class Manah::Windows::CDocument<DocumentUpdate>;
};

template<class Document /* implements CDocument */,
	typename DocumentUpdate, class BaseWindow = Manah::Windows::Controls::CWindow>
class CView : public _CViewBase<DocumentUpdate>, public BaseWindow {
public:
	typedef Document	Document;
public:
	CView(Document& document) : m_document(document) {}
	CView(const CView<Document, DocumentUpdate, BaseWindow>& rhs)
		: /*View(rhs),*/ BaseWindow(rhs), m_document(rhs.m_document) {}
public:
	virtual bool	Create(HWND hwndParent, const RECT& rect, DWORD dwStyle, DWORD dwExStyle) {return false;}
	Document&		GetDocument() const {AssertValid(); return m_document;}
protected:
	virtual			~CView() {}
	virtual void	OnDocumentSetup() {assert(false);}
	virtual void	OnInitialUpdate() {assert(false);}
	virtual void	OnUpdate(const DocumentUpdate& update) {assert(false);}
	virtual void	OnSetFocus(HWND hwndOld) {
		static_cast<CDocument<DocumentUpdate>&>(m_document).SetActiveView(*this); BaseWindow::OnSetFocus(hwndOld);}
private:
	Document&	m_document;
};

template<typename Update>
class CDocument : public CSelfAssertable, public CNoncopyable {
public:
	typedef _CViewBase<Update>	View;
	CDocument() : m_pszPathName(0), m_bModified(false), m_pActiveView(0) {}
	virtual ~CDocument();
public:
	bool		AddView(View& view);
	View&		GetActiveView() const throw(std::logic_error);
	std::size_t	GetCount() const {AssertValid(); return m_views.size();}
	View&		GetView(std::size_t iView) const throw(std::out_of_range) {AssertValid(); return *m_views.at(iView);}
	void		RemoveAndReleaseView(View& view) throw(std::invalid_argument);
	void		RemoveAndReleaseView(std::size_t iView) throw(std::out_of_range) {RemoveAndReleaseView(*m_views.at(iView));}
	void		SetActiveView(View& view) throw(std::invalid_argument);
	void		SetActiveView(std::size_t iView) throw(std::out_of_range) {AssertValid(); m_pActiveView = m_views.at(iView);}
	void		SetupAllViews();
	void		UpdateAllViews(const Update& update);

	const TCHAR*	GetExtensionName() const;
	const TCHAR*	GetFileName() const;
	const TCHAR*	GetPathName() const {AssertValid(); return m_pszPathName;}
	bool			IsModified() const {AssertValid(); return m_bModified;}
	void			SetModified(bool bModified = true) {AssertValid(); m_bModified = bModified;}
	void			SetPathName(const TCHAR* pszPathName);
protected:
	virtual void	OnChangedViewList() {if(m_views.empty()) OnCloseDocument();}
	virtual void	OnCloseDocument() {}
private:
	TCHAR*				m_pszPathName;
	bool				m_bModified;
	std::vector<View*>	m_views;
	View*				m_pActiveView;
};


// CDocument class implementation
/////////////////////////////////////////////////////////////////////////////

template<typename Update> inline CDocument<Update>::~CDocument() {
	delete[] m_pszPathName;
	for(std::vector<View*>::iterator it = m_views.begin(); it != m_views.end(); ++it)
		delete *it;
	m_views.erase(m_views.begin(), m_views.end());
}

template<typename Update> inline bool CDocument<Update>::AddView(View& view) {
	AssertValid();
	if(std::find(m_views.begin(), m_views.end(), &view) != m_views.end())
		return false;
	if(m_views.empty())
		m_pActiveView = &view;
	m_views.push_back(&view);
	view.OnInitialUpdate();
	return true;
}

template<typename Update> inline _CViewBase<Update>& CDocument<Update>::GetActiveView() const {
	AssertValid();
	if(m_views.empty())
		throw std::logic_error("There are no views.");
	assert(m_pActiveView != 0);
	return *m_pActiveView;
}

template<typename Update> inline const TCHAR* CDocument<Update>::GetExtensionName() const {
	AssertValid();
	if(m_pszPathName == 0)											return 0;
	else if(const TCHAR* p = std::_tcsrchr(m_pszPathName, _T('.')))	return p + 1;
	else															return 0;
}

template<typename Update> inline const TCHAR* CDocument<Update>::GetFileName() const {
	AssertValid();
	if(m_pszPathName == 0)												return 0;
	else if(const TCHAR* p = std::_tcsrchr(m_pszPathName, _T('\\')))	return p + 1;
	else																return 0;
}

template<typename Update> inline void CDocument<Update>::RemoveAndReleaseView(View& view) throw(std::invalid_argument) {
	AssertValid();
	std::vector<View*>::iterator	it = std::find(m_views.begin(), m_views.end(), &view);
	if(it == m_views.end())
		throw std::invalid_argument("Specified view does not belong this document.");
	m_views.erase(it);
	if(&view == m_pActiveView)
		m_pActiveView = (m_views.empty()) ? 0 : m_views.front();
	delete &view;
}

template<typename Update> inline void CDocument<Update>::SetActiveView(View& view) throw(std::invalid_argument) {
	AssertValid();
	if(std::find(m_views.begin(), m_views.end(), &view) == m_views.end())
		throw std::invalid_argument("Specified view does not belong to this document.");
	m_pActiveView = &view;
}

template<typename Update> inline void CDocument<Update>::SetPathName(const TCHAR* pszPathName) {
	AssertValid();
	delete[] m_pszPathName;
	if(pszPathName != 0) {
		m_pszPathName = new TCHAR[std::_tcslen(pszPathName) + 1];
		std::_tcscpy(m_pszPathName, pszPathName);
	} else
		m_pszPathName = 0;
}

template<typename Update> inline void CDocument<Update>::SetupAllViews() {
	AssertValid();
	std::vector<View*>::iterator	it = m_views.begin();
	while(it != m_views.end()) {
		(*it)->OnDocumentSetup();
		++it;
	}
}

template<typename Update> inline void CDocument<Update>::UpdateAllViews(const Update& update) {
	AssertValid();
	std::vector<View*>::iterator	it = m_views.begin();
	while(it != m_views.end()) {
		(*it)->OnUpdate(update);
		++it;
	}
}

} // namespace Windows
} // namespace Manah

#endif /* _DOCUMENT_VIEW_H_ */

/* [EOF] */