// Dialog.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _DIALOG_H_
#define _DIALOG_H_

#include "Window.h"
#include "ToolTipCtrl.h"
#include <map>


// CDialog class definition
/////////////////////////////////////////////////////////////////////////////

namespace Manah {
namespace Windows {
namespace Controls {

class CDialog : public CWindow {
	// RXgN^
public:
	CDialog();
	virtual ~CDialog();
private:
	CDialog(const CDialog& rhs);
	
	// \bh
public:
	bool	AddToolTip(HWND hwndCtrl, const TCHAR* lpszText);
	bool	AddToolTip(UINT idCtrl, const TCHAR* lpszText);
	bool	CheckDlg2StateButton(int nIDButton, bool bCheck = true);
	bool	CheckDlgButton(int nIDButton, UINT uCheck);
	bool	CheckRadioButton(int nIDFirstButton, int nIDLastButton, int nIDCheckButton);
	bool	Create(HINSTANCE hInstance, int nID, HWND hwndParent);
	int		DoModal();
	bool	DoModeless(bool bShow = true);
	void	EnableToolTips(bool bEnable = true);
	void	EndDialog(int nResult);
	int		GetCheckedRadioButton(int nIDFirstButton, int nIDLastButton) const;
	DWORD	GetDefID() const;
	HWND	GetDlgItem(int nIDDlgItem) const;
	int		GetDlgItemInt(int nIDDlgItem, bool* lpTranslated = 0, bool bSigned = true) const;
	int		GetDlgItemText(int nIDDlgItem, TCHAR* lpszText, int nMaxCount) const;
	HWND	GetNextDlgGroupItem(HWND hCtrl, bool bPrevious = false) const;
	HWND	GetNextDlgGroupItem(int nIDDlgItem, bool bPrevious = false) const;
	HWND	GetNextDlgTabItem(HWND hCtrl, bool bPrevious = false) const;
	HWND	GetNextDlgTabItem(int nIDDlgItem, bool bPrevious = false) const;
	bool	IsDialogMessage(const MSG& msg);
	UINT	IsDlgButtonChecked(int nIDButton) const;
	void	NextDlgCtrl() const;
	void	PrevDlgCtrl() const;
	LRESULT	SendDlgItemMessage(int nIDDlgItem, UINT message, WPARAM wParam, LPARAM lParam);
	void	SetDefID(UINT nID);
	void	SetDlgItemInt(int nIDDlgItem, UINT nValue, bool bSigned = true);
	void	SetDlgItemText(int nIDDlgItem, const TCHAR* lpszText);
protected:
	virtual LRESULT	DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam);

	// bZ[Wnh
protected:
	/* ? need to any implement ? */
	virtual void	OnClose();													// WM_CLOSE
	virtual bool	OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl);		// WM_COMMAND
	virtual bool	OnInitDialog(HWND hwndFocus, LPARAM lInitParam);			// WM_INITDIALOG
	virtual void	OnOK();														// IDOK
	virtual void	OnCancel();													// IDCANCEL
	/* ? noneed to implement ? */
	virtual void	OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized);// WM_ACTIVATE

	// f[^o
protected:
	static INT_PTR CALLBACK	DlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
	static LRESULT CALLBACK	DlgProcDummy(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
	HINSTANCE		m_hInstance;
	CToolTipCtrl*	m_pToolTips;
private:
	int		m_nID;
	HWND	m_hwndParent;
	bool	m_bModeless;
};


// fixed dialog template id version
template<int id> class CFixedIdDialog : public CDialog {
public:
	bool Create(HINSTANCE hInstance, HWND hwndParent) {
		AssertValid();
		return CDialog::Create(hInstance, id, hwndParent);
	}
};


// CDialog class implementation
/////////////////////////////////////////////////////////////////////////////

inline CDialog::CDialog() : m_pToolTips(0), m_bModeless(false) {
//	m_OldProc = CDialog::DlgProcDummy;	// prevent to call DefWindowProc in CWindow::DispatchEvent
}

inline CDialog::~CDialog() {
	if(IsWindow())
		DestroyWindow();
	m_hWnd = 0;

	if(m_pToolTips != 0) {
		delete m_pToolTips;
		m_pToolTips = 0;
	}
}

inline bool CDialog::AddToolTip(HWND hwndCtrl, const TCHAR* lpszText) {
	AssertValidAsWindow();

	if(m_pToolTips == 0 || m_hWnd != ::GetParent(hwndCtrl))
		return false;
	return m_pToolTips->AddTool(hwndCtrl, TTF_SUBCLASS, lpszText);
}

inline bool CDialog::AddToolTip(UINT idCtrl, const TCHAR* lpszText) {
	AssertValidAsWindow();
	return AddToolTip(GetDlgItem(idCtrl), lpszText);
}

inline bool CDialog::CheckDlg2StateButton(int nIDButton, bool bCheck /* = true */) {
	AssertValidAsWindow();
	return toBoolean(::CheckDlgButton(m_hWnd, nIDButton, bCheck ? BST_CHECKED : BST_UNCHECKED));
}

inline bool CDialog::CheckDlgButton(int nIDButton, UINT uCheck) {
	AssertValidAsWindow();
	return toBoolean(::CheckDlgButton(m_hWnd, nIDButton, uCheck));
}

inline bool CDialog::CheckRadioButton(int nIDFirstButton, int nIDLastButton, int nIDCheckButton) {
	AssertValidAsWindow();
	return toBoolean(::CheckRadioButton(m_hWnd, nIDFirstButton, nIDLastButton, nIDCheckButton));
}

inline bool CDialog::Create(HINSTANCE hInstance, int nID, HWND hwndParent) {
	m_hInstance = hInstance;
	m_nID = nID;
	m_hwndParent = hwndParent;

	if(m_pToolTips != 0) {
		m_pToolTips->Create(m_hWnd);
		m_pToolTips->Activate(true);
	}

	return true;
}

inline LRESULT CDialog::DispatchEvent(UINT message, WPARAM wParam, LPARAM lParam) {
	if(PreTranslateMessage(message, wParam, lParam))
		return true;
	POINT pt;

	switch(message) {
	case WM_ACTIVATE:
		OnActivate(LOWORD(wParam), reinterpret_cast<HWND>(lParam), toBoolean(HIWORD(wParam)));
		break;
	case WM_CLOSE:
		OnClose();
		return true;
	case WM_COMMAND:
		return OnCommand(LOWORD(wParam), HIWORD(wParam), reinterpret_cast<HWND>(lParam));
	case WM_CONTEXTMENU:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		if(OnContextMenu(reinterpret_cast<HWND>(wParam), pt))
			return true;
		break;
	case WM_DESTROY:
		OnDestroy();
		return true;
	case WM_DRAWITEM:
		OnDrawItem(static_cast<UINT>(wParam), reinterpret_cast<LPDRAWITEMSTRUCT>(lParam));
		break;
	case WM_INITDIALOG:
		return OnInitDialog(reinterpret_cast<HWND>(wParam), lParam);
	case WM_KILLFOCUS:
		OnKillFocus(reinterpret_cast<HWND>(wParam));
		return true;
	case WM_LBUTTONDOWN:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDown(wParam, pt);
		break;
	case WM_LBUTTONDBLCLK:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonDblClk(wParam, pt);
		break;
	case WM_LBUTTONUP:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnLButtonUp(wParam, pt);
		break;
	case WM_MEASUREITEM:
		OnMeasureItem(static_cast<UINT>(wParam), reinterpret_cast<LPMEASUREITEMSTRUCT>(lParam));
		break;
	case WM_MOUSEMOVE:
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		OnMouseMove(wParam, pt);
		break;
	case WM_NOTIFY:
		return OnNotify(wParam, reinterpret_cast<LPNMHDR>(lParam));
	case WM_SETCURSOR:
		if(OnSetCursor(reinterpret_cast<HWND>(wParam),
			static_cast<UINT>(LOWORD(lParam)), static_cast<UINT>(HIWORD(lParam))))
			return false;
		break;
	case WM_SETFOCUS:
		OnSetFocus(reinterpret_cast<HWND>(wParam));
		return true;
	case WM_SHOWWINDOW:
		OnShowWindow(toBoolean(wParam), lParam);
		break;
	case WM_SIZE:
		OnSize(wParam, LOWORD(lParam), HIWORD(lParam));
		break;
	case WM_SYSCOLORCHANGE:
		OnSysColorChange();
		break;
	case WM_TIMER:
		OnTimer(wParam);
		break;
	}

	return false;
}

inline INT_PTR CALLBACK CDialog::DlgProc(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam) {
	CDialog*	pDialog;

	if(message == WM_INITDIALOG) {
		pDialog = reinterpret_cast<CDialog*>(lParam);
		pDialog->m_hWnd = hDlg;
		::SetWindowLong(pDialog->m_hWnd, DWL_USER, reinterpret_cast<LPARAM>(pDialog));
		return pDialog->DispatchEvent(message, wParam, lParam);
	} else {
		pDialog = reinterpret_cast<CDialog*>(::GetWindowLong(hDlg, DWL_USER));
		return (pDialog != 0) ? pDialog->DispatchEvent(message, wParam, lParam) : false;
	}
}

inline LRESULT CALLBACK CDialog::DlgProcDummy(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam) {
	return false;
}

inline int CDialog::DoModal() {
	AssertValid();

	m_bModeless = false;
	return ::DialogBoxParam(m_hInstance, MAKEINTRESOURCE(m_nID),
		m_hwndParent, CDialog::DlgProc, reinterpret_cast<LPARAM>(this));
}

inline bool CDialog::DoModeless(bool bShow /* = true */) {
	AssertValid();

	m_hWnd = ::CreateDialogParam(m_hInstance, MAKEINTRESOURCE(m_nID),
		m_hwndParent, CDialog::DlgProc, reinterpret_cast<LPARAM>(this));

	if(m_hWnd == 0)
		return false;
	m_bModeless = true;

	if(bShow)
		ShowWindow(SW_SHOW);

	return true;
}

inline void CDialog::EnableToolTips(bool bEnable /* = true */) {
	if(bEnable) {
		if(m_pToolTips != 0)
			return;
		m_pToolTips = new CToolTipCtrl();
		m_pToolTips->Create(m_hwndParent);
	} else {
		if(m_pToolTips == 0)
			return;
		m_pToolTips->Activate(false);
	}
}

inline void CDialog::EndDialog(int nResult) {
	AssertValidAsWindow();

	::EndDialog(m_hWnd, nResult);
	if(m_bModeless)
		DestroyWindow();
}

inline int CDialog::GetCheckedRadioButton(int nIDFirstButton, int nIDLastButton) const {
	AssertValidAsWindow();
	for(int nID = nIDFirstButton; nID <= nIDLastButton; ++nID) {
		if(::IsDlgButtonChecked(m_hWnd, nID) == BST_CHECKED)
			return nID;
	}
	return 0;
}

inline DWORD CDialog::GetDefID() const {
	AssertValidAsWindow();
	return ::SendMessage(m_hWnd, DM_GETDEFID, 0, 0L);
}

inline HWND CDialog::GetDlgItem(int nIDDlgItem) const {
	AssertValidAsWindow();
	return ::GetDlgItem(m_hWnd, nIDDlgItem);
}

inline int CDialog::GetDlgItemInt(int nIDDlgItem, bool* lpTranslated /* = 0 */, bool bSigned /* = true */) const {
	AssertValidAsWindow();
	return ::GetDlgItemInt(m_hWnd, nIDDlgItem, reinterpret_cast<int*>(lpTranslated), bSigned);
}

inline int CDialog::GetDlgItemText(int nIDDlgItem, TCHAR* lpszText, int nMaxCount) const {
	AssertValidAsWindow();
	return ::GetDlgItemText(m_hWnd, nIDDlgItem, lpszText, nMaxCount);
}

inline HWND CDialog::GetNextDlgGroupItem(HWND hCtrl, bool bPrevious /* = false */) const {
	AssertValidAsWindow();
	return ::GetNextDlgGroupItem(m_hWnd, hCtrl, bPrevious);
}

inline HWND CDialog::GetNextDlgGroupItem(int nIDDlgItem, bool bPrevious /* = false */) const {
	AssertValid();
	return ::GetNextDlgGroupItem(m_hWnd, ::GetDlgItem(m_hWnd, nIDDlgItem), bPrevious);
}

inline HWND CDialog::GetNextDlgTabItem(HWND hCtrl, bool bPrevious /* = false */) const {
	AssertValidAsWindow();
	return ::GetNextDlgTabItem(m_hWnd, hCtrl, bPrevious);
}

inline HWND CDialog::GetNextDlgTabItem(int nIDDlgItem, bool bPrevious /* = false */) const {
	AssertValidAsWindow();
	return ::GetNextDlgTabItem(m_hWnd, ::GetDlgItem(m_hWnd, nIDDlgItem), bPrevious);
}

inline bool CDialog::IsDialogMessage(const MSG& msg) {
	AssertValidAsWindow();
	return toBoolean(::IsDialogMessage(m_hWnd, const_cast<MSG*>(&msg)));
}

inline UINT CDialog::IsDlgButtonChecked(int nIDButton) const {
	AssertValidAsWindow();
	return ::IsDlgButtonChecked(m_hWnd, nIDButton);
}

inline void CDialog::NextDlgCtrl() const {
	AssertValidAsWindow();
	::SetFocus(GetNextDlgTabItem(GetFocus(), false));
}

inline void CDialog::PrevDlgCtrl() const {
	AssertValidAsWindow();
	::SetFocus(GetNextDlgTabItem(GetFocus(), true));
}

inline LRESULT CDialog::SendDlgItemMessage(int nIDDlgItem, UINT message, WPARAM wParam, LPARAM lParam) {
	AssertValidAsWindow();
	return ::SendDlgItemMessage(m_hWnd, nIDDlgItem, message, wParam, lParam);
}

inline void CDialog::SetDefID(UINT nID) {
	AssertValidAsWindow();
	::SendMessage(m_hWnd, DM_SETDEFID, nID, 0L);
}

inline void CDialog::SetDlgItemInt(int nIDDlgItem, UINT nValue, bool bSigned /* = true */) {
	AssertValidAsWindow();
	::SetDlgItemInt(m_hWnd, nIDDlgItem, nValue, bSigned);
}

inline void CDialog::SetDlgItemText(int nIDDlgItem, const TCHAR* lpszText) {
	AssertValidAsWindow();
	::SetDlgItemText(m_hWnd, nIDDlgItem, lpszText);
}

inline void CDialog::OnActivate(UINT nState, HWND hwndPrevious, bool bMinimized) {
}

inline void CDialog::OnClose() {
	EndDialog(IDCLOSE);
}

inline bool CDialog::OnCommand(WORD wID, WORD wNotifyCode, HWND hwndCtrl) {
	switch(wID) {
	case IDOK:
		OnOK();
		return true;
	case IDCANCEL:
		OnCancel();
		return true;
	}

	return false;
}

inline bool CDialog::OnInitDialog(HWND hwndFocus, LPARAM lInitParam) {
	return true;
}

inline void CDialog::OnOK() {
	AssertValidAsWindow();
	EndDialog(IDOK);
}

inline void CDialog::OnCancel() {
	AssertValidAsWindow();
	EndDialog(IDCANCEL);
}

} /* namespace Controls */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _DIALOG_H_ */

/* [EOF] */