// DC.h
/////////////////////////////////////////////////////////////////////////////

#ifndef _DC_H_
#define _DC_H_
#include "Object.h"


namespace {
	struct _WindowRelatedDCHolder {
		HWND	hWnd;
		HDC		hDC;
		ulong	c;
		_WindowRelatedDCHolder(HWND hWnd_, HDC hDC_) : hWnd(hWnd_), hDC(hDC_), c(1) {assert(hDC != 0);}
		~_WindowRelatedDCHolder() {assert(c == 0); ::ReleaseDC(hWnd, hDC);}
	};
}

namespace Manah {
namespace Windows {
	namespace Controls {
		class CWindow;
	}
namespace GDI {

class CDC : public CSelfAssertable {
	// RXgN^
public:
	CDC() : m_hDC(0) {}
	virtual ~CDC() {}
protected:
	CDC(const CDC& rhs) : m_hDC(rhs.m_hDC) {}

	// Zq
public:
	operator HDC() const {return m_hDC;}
private:
	CDC&	operator =(const CDC& rhs);

	// \bh
public:
	/* 쐬 */
	HBITMAP		GetCurrentBitmap() const;
	HBRUSH		GetCurrentBrush() const;
	HFONT		GetCurrentFont() const;
	HPALETTE	GetCurrentPalette() const;
	HPEN		GetCurrentPen() const;
	HWND		GetWindow() const;

	/* foCXReLXg */
	int				GetDeviceCaps(int iIndex) const;
	HDC				GetSafeHdc() const;
	virtual bool	RestoreDC(int nSavedDC);
	virtual int		SaveDC();

	/* `c[ */
	int		EnumObjects(int nObjectType, GOBJENUMPROC lpObjectFunc, LPARAM lpData);
	POINT	GetBrushOrg() const;
	POINT	SetBrushOrg(int x, int y);
	POINT	SetBrushOrg(const POINT& pt);

	/* I */
	HBITMAP			SelectObject(HBITMAP hBitmap);
	HBRUSH			SelectObject(HBRUSH hBrush);
	virtual HFONT	SelectObject(HFONT hFont);
	HPEN			SelectObject(HPEN hPen);
	HGDIOBJ			SelectStockObject(int nObject);

	/* Fƃpbg */
	COLORREF	GetNearestColor(COLORREF clr) const;
	UINT		RealizePalette();
	HPALETTE	SelectPalette(HPALETTE hPalette, bool bForceBackground);
	void		UpdateColors();

	/*  */
	COLORREF	GetBkColor() const;
	int			GetBkMode() const;
	bool		GetColorAdjustment(COLORADJUSTMENT& colorAdjust) const;
	int			GetPolyFillMode() const;
	int			GetROP2() const;
	int			GetStretchBltMode() const;
	COLORREF	GetTextColor() const;
	COLORREF	SetBkColor(COLORREF clr);
	int			SetBkMode(int nBkMode);
	bool		SetColorAdjustment(const COLORADJUSTMENT& colorAdjust);
	int			SetPolyFillMode(int nPolyFillMode);
	int			SetROP2(int nDrawMode);
	int			SetStretchBltMode(int nStretchMode);
	COLORREF	SetTextColor(COLORREF clr);

	/* ̈ */
	bool	FillRgn(HRGN hRgn, HBRUSH hBrush);
	bool	FrameRgn(HRGN hRgn, HBRUSH hBrush, int nWidth, int nHeight);
	bool	InvertRgn(HRGN hRgn);
	bool	PaintRgn(HRGN hRgn);

	/* NbsO */
	int		ExcludeClipRect(int x1, int y1, int x2, int y2);
	int		ExcludeClipRect(const RECT& rect);
	int		ExcludeUpdateRgn(HWND hWnd);
	UINT	GetBoundsRect(RECT& rcBounds, UINT nFlags);
	int		GetClipBox(RECT& rect) const;
	int		IntersectClipRect(int x1, int y1, int x2, int y2);
	int		IntersectClipRect(const RECT& rect);
	int		OffsetClipRgn(int x, int y);
	int		OffsetClipRgn(const SIZE& size);
	bool	PtVisible(int x, int y) const;
	bool	PtVisible(const POINT& pt) const;
	bool	RectVisible(const RECT& rect) const;
	int		SelectClipRgn(HRGN hRgn);
	UINT	SetBoundsRect(const RECT& rcBounds, UINT nFlags);

	/* o */
	bool	AngleArc(int x, int y, int nRadius, float fStartAngle, float fSweepAngle);
	bool	Arc(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Arc(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	bool	ArcTo(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	ArcTo(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	int		GetArcDirection() const;
	POINT	GetCurrentPosition() const;
	bool	LineTo(int x, int y);
	bool	LineTo(const POINT& pt);
	POINT	MoveTo(int x, int y);
	POINT	MoveTo(const POINT& pt);
	bool	PolyBezier(const POINT* lpPoints, int nCount);
	bool	PolyBezierTo(const POINT* lpPoints, int nCount);
	bool	PolyDraw(const POINT* lpPoints, const BYTE* lpTypes, int nCount);
	bool	Polyline(LPPOINT lpPoints, int nCount);
	bool	PolylineTo(const POINT* lpPoints, int nCount);
	bool	PolyPolyline(const POINT* lpPoints, const DWORD* lpPolyPoints, int nCount);
	int		SetArcDirection(int nArcDirection);

	/* P` */
//	void	Draw3dRect(const RECT* lpRect, COLORREF clrTopLeft, COLORREF clrRightBottom);
//	void	Draw3dRect(int x, int y, int cx, int cy, COLORREF clrTopLeft, COLORREF clrRightBottom);
//	void	DrawDragIcon(const RECT* lpRect, SIZE size,
//				const RECT* lpRectLast, SIZE sizeLast, HBRUSH hBrush, HBRUSH hLastBrush);
	bool	DrawEdge(const RECT& rect, UINT nEdge, UINT grfFlags);
	bool	DrawFrameControl(const RECT& rect, UINT nType, UINT nState);
	bool	DrawIcon(int x, int y, HICON hIcon);
	bool	DrawIcon(const POINT& pt, HICON hIcon);
	void	FillRect(const RECT& rect, HBRUSH hBrush);
	void	FillSolidRect(const RECT& rect, COLORREF clr);
	void	FillSolidRect(int x, int y, int cx, int cy, COLORREF clr);
	void	FrameRect(const RECT& rect, HBRUSH hBrush);
	void	InvertRect(const RECT& rect);

	/* ȉ~Ƒp` */
	bool	Chord(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Chord(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	void	DrawFocusRect(const RECT& rect);
	bool	Ellipse(int x1, int y1, int x2, int y2);
	bool	Ellipse(const RECT& rect);
	bool	Pie(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4);
	bool	Pie(const RECT& rect, const POINT& ptStart, const POINT& ptEnd);
	bool	Polygon(const POINT* lpPoints, int nCount);
	bool	PolyPolygon(const POINT* lpPoints, const int* lpPolyCounts, int nCount);
	bool	Rectangle(int x1, int y1, int x2, int y2);
	bool	Rectangle(const RECT& rect);
	bool	RoundRect(int x1, int y1, int x2, int y2, int x3, int y3);
	bool	RoundRect(const RECT& rect, const POINT& pt);

	/* rbg}bv */
	bool		BitBlt(int x, int y, int nWidth, int nHeight, HDC hSrcDC, int xSrc, int ySrc, DWORD dwRop);
	bool		ExtFloodFill(int x, int y, COLORREF clr, UINT nFillType);
	bool		FloodFill(int x, int y, COLORREF clr);
	COLORREF	GetPixel(int x, int y) const;
	COLORREF	GetPixel(const POINT& pt) const;
	bool		MaskBlt(int x, int y, int nWidth, int nHeight,
					HDC hDC, int xSrc, int ySrc, HBITMAP hBitmap, int xMask, int yMask, DWORD dwRop);
	bool		PatBlt(int x, int y, int nWidth, int nHeight, DWORD dwRop);
	bool		PatBlt(const RECT& rect, DWORD dwRop);
	bool		PlgBlt(const POINT* lpPoint, HDC hDC,
					int xSrc, int ySrc, int nWidth, int nHeight, HBITMAP hBitmap, int xMask, int yMask);
	COLORREF	SetPixel(int x, int y, COLORREF clr);
	COLORREF	SetPixel(const POINT& pt, COLORREF clr);
	bool		SetPixelV(int x, int y, COLORREF clr);
	bool		SetPixelV(const POINT& pt, COLORREF clr);
	bool		StretchBlt(int x, int y, int nWidth, int nHeight,
					HDC hSrcDC, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight, DWORD dwRop);

	/* eLXg */
	virtual int		DrawText(const TCHAR* lpszText, int cchText, const RECT& rect, UINT nFormat);
	virtual bool	ExtTextOut(int x, int y, UINT nOptions, const RECT* lpRect,
						const TCHAR* lpszText, UINT cchText, const int* lpDxWidths);
	SIZE			GetCharacterPlacement(const TCHAR* lpszText, int nCount,
						int nMaxExtent, GCP_RESULTS& results, DWORD dwFlags) const;
	SIZE			GetTabbedTextExtent(const TCHAR* lpszText,
						int cchText, int nTabPositions, int* lpnTabStopPositions) const;
	UINT			GetTextAlign() const;
	int				GetTextCharacterExtra() const;
	SIZE			GetTextExtent(const TCHAR* lpszText, int cchText) const;
	bool			GetTextExtentExPoint(const TCHAR* pgiln, int cgi,
						int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const;
	bool			GetTextExtentExPointI(LPWORD pgiln, int cgi,
						int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const;
	bool			GetTextExtentPointI(LPWORD pgiln, int cgi, LPSIZE lpSize) const;
	int				GetTextFace(int cchFaceName, TCHAR* lpszFaceName) const;
	bool			GetTextMetrics(TEXTMETRIC& metrics) const;
	virtual bool	GrayString(HBRUSH hBrush, GRAYSTRINGPROC lpfnOutput,
						LPARAM lpData, int cchText, int x, int y, int nWidth, int nHeight);
	bool			PolyTextOut(const POLYTEXT* pTexts, int cTexts);
	UINT			SetTextAlign(UINT nFlags);
	int				SetTextCharacterExtra(int nCharExtra);
	int				SetTextJustification(int nBreakExtra, int nBreakCount);
	virtual SIZE	TabbedTextOut(int x, int y, const TCHAR* lpszText,
						int cchText, int nTabPositions, int* lpnTabStopPositions, int nTabOrigin);
	virtual bool	TextOut(int x, int y, const TCHAR* lpszText, int cchText);

	/* tHg */
	int		EnumFontFamilies(const TCHAR* pszName, FONTENUMPROC lpEnumFontFamProc, LPARAM lParam = 0UL) const;
	int		EnumFontFamilies(const LOGFONT& condition, FONTENUMPROC lpEnumFontFamProc, LPARAM lParam = 0UL) const;
	bool	GetAspectRatioFilterEx(SIZE& size) const;
	bool	GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABC& abc) const;
	bool	GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABCFLOAT& abcf) const;
	bool	GetCharWidth(UINT nFirstChar, UINT nLastChar, int* lpBuffer) const;
	bool	GetCharWidth(UINT nFirstChar, UINT nLastChar, float* lpBuffer) const;
	DWORD	GetFontData(DWORD dwTable, DWORD dwOffset, LPVOID lpData, DWORD cbData) const;
	DWORD	GetFontLanguageInfo() const;
#if(_WIN32_WINNT >= 0x0500)
	DWORD	GetFontUnicodeRanges(GLYPHSET& glyphset) const;
	DWORD	GetGlyphIndices(const TCHAR* psz, int cch, WORD* pIndices, DWORD dwFlags) const;
#endif
	DWORD	GetGlyphOutline(UINT nChar, UINT nFormat,
				LPGLYPHMETRICS lpgm, DWORD cbBuffer, LPVOID lpData, const MAT2* lpmat2) const;
	DWORD	GetKerningPairs(DWORD dwPairs, LPKERNINGPAIR lpkrnpair) const;
	UINT	GetOutlineTextMetrics(UINT cbData, LPOUTLINETEXTMETRIC lpotm) const;
	bool	GetRasterizerCaps(RASTERIZER_STATUS& status, UINT cb) const;
	DWORD	SetMapperFlags(DWORD dwFlag);

	/* XN[ */
	bool	ScrollDC(int dx, int dy,
				const RECT& scrollRect, const RECT& clipRect, HRGN hUpdateRgn, RECT* lpUpdateRect);

	/* \ */
	void	AssertValidAsDC() const;

	// f[^o
protected:
	HDC		m_hDC;
};


class CAutoDC : public CDC {
public:
	CAutoDC(HDC hDC = 0) : m_bAttached(false) {if(hDC != 0) Attach(hDC);}
	CAutoDC(const CAutoDC& rhs) : m_bAttached(false) {if(rhs.m_hDC != 0) Attach(rhs.m_hDC);}
	virtual ~CAutoDC() {
		if(m_hDC != 0 && !m_bAttached)
			::DeleteDC(Detach());
	}
	bool Attach(HDC hDC) {
		AssertValid();
		assert(m_hDC == 0);

		if(hDC == 0)
			return false;
		m_hDC = hDC;
		m_bAttached = true;
		return true;
	}
	bool CreateCompatibleDC(HDC hDC) {
		AssertValid();
		assert(m_hDC == 0);

		m_hDC = ::CreateCompatibleDC(hDC);
		return m_hDC != 0;
	}
	bool DeleteDC() {
		AssertValid();

		if(m_hDC == 0)
			return false;
		return toBoolean(::DeleteDC(Detach()));
	}
	HDC Detach() {
		AssertValid();

		HDC	hDC = m_hDC;

		m_hDC = 0;
		m_bAttached = false;
		return hDC;
	}
private:
	bool	m_bAttached;
};


class CPaintDC : public CDC {
public:
	CPaintDC(HWND hWnd) : m_hWnd(hWnd), m_bCreatedByCWindow(false) {
		assert(::IsWindow(m_hWnd));
		m_hDC = ::BeginPaint(hWnd, &m_paint);
		assert(m_hDC != 0);
	}
	CPaintDC(const CPaintDC& rhs) :
		CDC(rhs), m_hWnd(rhs.m_hWnd), m_bCreatedByCWindow(rhs.m_bCreatedByCWindow) {const_cast<CPaintDC&>(rhs).m_hWnd = 0;}
private:
	CPaintDC(HWND hWnd, PAINTSTRUCT& paint) : m_hWnd(hWnd), m_paint(paint), m_bCreatedByCWindow(true) {
		m_hDC = m_paint.hdc;
		assert(::IsWindow(m_hWnd));
		assert(m_hDC != 0);
	}
public:
	~CPaintDC() {
		if(!m_bCreatedByCWindow)
			::EndPaint(m_hWnd, &m_paint);
	}
private:
	operator HDC() const;
public:
	const PAINTSTRUCT& GetPaintStruct() const {return m_paint;}
private:
	HWND		m_hWnd;
	PAINTSTRUCT	m_paint;
	bool		m_bCreatedByCWindow;
	friend class Controls::CWindow;	// for CWindow::BeginPaint
};


class CClientDC : public CDC {
protected:
	CClientDC(HWND hWnd) : m_pHolder(new _WindowRelatedDCHolder(hWnd, ::GetDC(hWnd))) {m_hDC = m_pHolder->hDC;}
	CClientDC(HWND hWnd, HRGN hrgnClip, DWORD dwFlags) :
		m_pHolder(new _WindowRelatedDCHolder(hWnd, ::GetDCEx(hWnd, hrgnClip, dwFlags))) {m_hDC = m_pHolder->hDC;}
public:
	CClientDC(const CClientDC& rhs) : CDC(rhs), m_pHolder(rhs.m_pHolder) {++m_pHolder->c; m_hDC = rhs.m_hDC;}
	virtual ~CClientDC() {if(--m_pHolder->c == 0) delete m_pHolder;}
private:
	operator HDC() const;
private:
	_WindowRelatedDCHolder*	m_pHolder;
	friend class Controls::CWindow;	// for CWindow::GetDC
};


class CWindowDC : public CDC {
protected:
	CWindowDC(HWND hWnd) : m_pHolder(new _WindowRelatedDCHolder(hWnd, ::GetWindowDC(hWnd))) {m_hDC = m_pHolder->hDC;}
public:
	CWindowDC(const CWindowDC& rhs) : CDC(rhs), m_pHolder(rhs.m_pHolder) {++m_pHolder->c; m_hDC = rhs.m_hDC;}
	virtual ~CWindowDC() {if(--m_pHolder->c == 0) delete m_pHolder;}
private:
	operator HDC() const;
private:
	_WindowRelatedDCHolder*	m_pHolder;
	friend class Controls::CWindow;	// for CWindow::GetWindowDC
};


class CScreenDC : public CClientDC {
public:
	CScreenDC() : CClientDC(0) {}
private:
	operator HDC() const;
};


// CDC class implementation
/////////////////////////////////////////////////////////////////////////////

inline bool CDC::AngleArc(int x, int y, int nRadius, float fStartAngle, float fSweepAngle) {
	AssertValidAsDC();
	return toBoolean(::AngleArc(m_hDC, x, y, nRadius, fStartAngle, fSweepAngle));
}

inline bool CDC::Arc(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Arc(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

inline bool CDC::Arc(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Arc(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

inline bool CDC::ArcTo(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::ArcTo(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

inline bool CDC::ArcTo(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::ArcTo(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

inline void CDC::AssertValidAsDC() const {
	AssertValid();
	assert(m_hDC != 0);
}

inline bool CDC::BitBlt(int x, int y, int nWidth, int nHeight, HDC hSrcDC, int xSrc, int ySrc, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::BitBlt(m_hDC, x, y, nWidth, nHeight, hSrcDC, xSrc, ySrc, dwRop));
}

inline bool CDC::Chord(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Chord(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

inline bool CDC::Chord(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Chord(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

inline bool CDC::DrawEdge(const RECT& rect, UINT nEdge, UINT grfFlags) {
	AssertValidAsDC();
	return toBoolean(::DrawEdge(m_hDC, const_cast<RECT*>(&rect), nEdge, grfFlags));
}

inline void CDC::DrawFocusRect(const RECT& rect) {
	AssertValidAsDC();
	::DrawFocusRect(m_hDC, &rect);
}

inline bool CDC::DrawFrameControl(const RECT& rect, UINT nType, UINT nState) {
	AssertValidAsDC();
	return toBoolean(::DrawFrameControl(m_hDC, const_cast<RECT*>(&rect), nType, nState));
}

inline bool CDC::DrawIcon(int x, int y, HICON hIcon) {
	AssertValidAsDC();
	return toBoolean(::DrawIcon(m_hDC, x, y, hIcon));
}

inline bool CDC::DrawIcon(const POINT& pt, HICON hIcon) {
	AssertValidAsDC();
	return toBoolean(::DrawIcon(m_hDC, pt.x, pt.y, hIcon));
}

inline int CDC::DrawText(const TCHAR* lpszText, int cchText, const RECT& rect, UINT nFormat) {
	AssertValidAsDC();
	return ::DrawText(m_hDC, lpszText, cchText, const_cast<RECT*>(&rect), nFormat);
}

inline bool CDC::Ellipse(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return toBoolean(::Ellipse(m_hDC, x1, y1, x2, y2));
}

inline bool CDC::Ellipse(const RECT& rect) {
	AssertValidAsDC();
	return toBoolean(::Ellipse(m_hDC, rect.left, rect.top, rect.right, rect.bottom));
}

inline int CDC::EnumFontFamilies(const TCHAR* pszName, FONTENUMPROC lpEnumFontFamProc, LPARAM lParam /* = 0UL */) const {
	AssertValidAsDC();
	return ::EnumFontFamilies(m_hDC, pszName, lpEnumFontFamProc, lParam);
}

inline int CDC::EnumFontFamilies(const LOGFONT& condition, FONTENUMPROC lpEnumFontFamProc, LPARAM lParam /* = 0UL */) const {
	AssertValidAsDC();
	return ::EnumFontFamiliesEx(m_hDC, const_cast<LOGFONT*>(&condition), lpEnumFontFamProc, lParam, 0);
}

inline int CDC::EnumObjects(int nObjectType, GOBJENUMPROC lpObjectFunc, LPARAM lpData) {
	AssertValidAsDC();
	return ::EnumObjects(m_hDC, nObjectType, lpObjectFunc, lpData);
}

inline int CDC::ExcludeClipRect(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return ::ExcludeClipRect(m_hDC, x1, y1, x2, y2);
}

inline int CDC::ExcludeClipRect(const RECT& rect) {
	AssertValidAsDC();
	return ::ExcludeClipRect(m_hDC, rect.left, rect.top, rect.right, rect.bottom);
}

inline int CDC::ExcludeUpdateRgn(HWND hWnd) {
	AssertValidAsDC();
	return ::ExcludeUpdateRgn(m_hDC, hWnd);
}

inline bool CDC::ExtFloodFill(int x, int y, COLORREF clr, UINT nFillType) {
	AssertValidAsDC();
	return toBoolean(::ExtFloodFill(m_hDC, x, y, clr, nFillType));
}

inline bool CDC::ExtTextOut(int x, int y, UINT nOptions,
		const RECT* lpRect, const TCHAR* lpszText, UINT cchText, const int* lpDxWidths) {
	AssertValidAsDC();
	return toBoolean(::ExtTextOut(m_hDC, x, y, nOptions, lpRect, lpszText, cchText, lpDxWidths));
}

inline bool CDC::FillRgn(HRGN hRgn, HBRUSH hBrush) {
	AssertValidAsDC();
	return toBoolean(::FillRgn(m_hDC, hRgn, hBrush));
}

inline void CDC::FillSolidRect(const RECT& rect, COLORREF clr) {
	AssertValidAsDC();

	const COLORREF	clrOrg = GetBkColor();

	SetBkColor(clr);
	ExtTextOut(0, 0, ETO_OPAQUE, &rect, _T(""), 0, 0);
	SetBkColor(clrOrg);
}

inline void CDC::FillSolidRect(int x, int y, int cx, int cy, COLORREF clr) {
	AssertValidAsDC();
	
	COLORREF	clrOrg = GetBkColor();
	RECT		rect;

	::SetRect(&rect, x, y, x + cx, y + cy);
	SetBkColor(clr);
	ExtTextOut(0, 0, ETO_OPAQUE, &rect, _T(""), 0, 0);
	SetBkColor(clrOrg);
}

inline bool CDC::FloodFill(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::FloodFill(m_hDC, x, y, clr));
}

inline bool CDC::FrameRgn(HRGN hRgn, HBRUSH hBrush, int nWidth, int nHeight) {
	AssertValidAsDC();
	return toBoolean(::FrameRgn(m_hDC, hRgn, hBrush, nWidth, nHeight));
}

inline int CDC::GetArcDirection() const {
	AssertValidAsDC();
	return ::GetArcDirection(m_hDC);
}

inline void CDC::FillRect(const RECT& rect, HBRUSH hBrush) {
	AssertValidAsDC();
	::FillRect(m_hDC, &rect, hBrush);
}

inline void CDC::FrameRect(const RECT& rect, HBRUSH hBrush) {
	AssertValidAsDC();
	::FrameRect(m_hDC, &rect, hBrush);
}

inline bool CDC::GetAspectRatioFilterEx(SIZE& size) const {
	AssertValidAsDC();
	return toBoolean(::GetAspectRatioFilterEx(m_hDC, &size));
}

inline COLORREF CDC::GetBkColor() const {
	AssertValidAsDC();
	return ::GetBkColor(m_hDC);
}

inline int CDC::GetBkMode() const {
	AssertValidAsDC();
	return ::GetBkMode(m_hDC);
}

inline UINT CDC::GetBoundsRect(RECT& rcBounds, UINT nFlags) {
	AssertValidAsDC();
	return ::GetBoundsRect(m_hDC, &rcBounds, nFlags);
}

inline POINT CDC::GetBrushOrg() const {
	AssertValidAsDC();

	POINT	pt;
	::GetBrushOrgEx(m_hDC, &pt);
	return pt;
}

inline SIZE CDC::GetCharacterPlacement(const TCHAR* lpszText,
		int nCount, int nMaxExtent, GCP_RESULTS& results, DWORD dwFlags) const {
	AssertValidAsDC();
	DWORD	dw = ::GetCharacterPlacement(m_hDC, lpszText, nCount, nMaxExtent, &results, dwFlags);
	SIZE	size = {LOWORD(dw), HIWORD(dw)};
	return size;
}

inline bool CDC::GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABC& abc) const {
	AssertValidAsDC();
	return toBoolean(::GetCharABCWidths(m_hDC, nFirstChar, nLastChar, &abc));
}

inline bool CDC::GetCharABCWidths(UINT nFirstChar, UINT nLastChar, ABCFLOAT& abcf) const {
	AssertValidAsDC();
	return toBoolean(::GetCharABCWidthsFloat(m_hDC, nFirstChar, nLastChar, &abcf));
}

inline bool CDC::GetCharWidth(UINT nFirstChar, UINT nLastChar, int* lpBuffer) const {
	AssertValidAsDC();
	return toBoolean(::GetCharWidth32(m_hDC, nFirstChar, nLastChar, lpBuffer));
}

inline bool CDC::GetCharWidth(UINT nFirstChar, UINT nLastChar, float* lpBuffer) const {
	AssertValidAsDC();
	return toBoolean(::GetCharWidthFloat(m_hDC, nFirstChar, nLastChar, lpBuffer));
}

inline int CDC::GetClipBox(RECT& rect) const {
	AssertValidAsDC();
	return ::GetClipBox(m_hDC, &rect);
}

inline bool CDC::GetColorAdjustment(COLORADJUSTMENT& colorAdjust) const {
	AssertValidAsDC();
	return toBoolean(::GetColorAdjustment(m_hDC, &colorAdjust));
}

inline HBITMAP CDC::GetCurrentBitmap() const {
	AssertValidAsDC();
	return reinterpret_cast<HBITMAP>(::GetCurrentObject(m_hDC, OBJ_BITMAP));
}

inline HBRUSH CDC::GetCurrentBrush() const {
	AssertValidAsDC();
	return reinterpret_cast<HBRUSH>(::GetCurrentObject(m_hDC, OBJ_BRUSH));
}

inline HFONT CDC::GetCurrentFont() const {
	AssertValidAsDC();
	return reinterpret_cast<HFONT>(::GetCurrentObject(m_hDC, OBJ_FONT));
}

inline HPALETTE CDC::GetCurrentPalette() const {
	AssertValidAsDC();
	return reinterpret_cast<HPALETTE>(::GetCurrentObject(m_hDC, OBJ_PAL));
}

inline HPEN CDC::GetCurrentPen() const {
	AssertValidAsDC();
	return reinterpret_cast<HPEN>(::GetCurrentObject(m_hDC, OBJ_PEN));
}

inline POINT CDC::GetCurrentPosition() const {
	AssertValidAsDC();

	POINT	pt;
	::GetCurrentPositionEx(m_hDC, &pt);
	return pt;
}

inline int CDC::GetDeviceCaps(int iIndex) const {
	AssertValidAsDC();
	return ::GetDeviceCaps(m_hDC, iIndex);
}

inline DWORD CDC::GetFontData(DWORD dwTable, DWORD dwOffset, LPVOID lpData, DWORD cbData) const {
	AssertValidAsDC();
	return ::GetFontData(m_hDC, dwTable, dwOffset, lpData, cbData);
}

inline DWORD CDC::GetFontLanguageInfo() const {
	AssertValidAsDC();
	return ::GetFontLanguageInfo(m_hDC);
}

#if(_WIN32_WINNT >= 0x0500)
inline DWORD CDC::GetFontUnicodeRanges(GLYPHSET& glyphset) const {
	AssertValidAsDC();
	return ::GetFontUnicodeRanges(m_hDC, &glyphset);
}

inline DWORD CDC::GetGlyphIndices(const TCHAR* psz, int cch, WORD* pIndices, DWORD dwFlags) const {
	AssertValidAsDC();
	return ::GetGlyphIndices(m_hDC, psz, cch, pIndices, dwFlags);
}
#endif

inline DWORD CDC::GetGlyphOutline(UINT nChar, UINT nFormat,
		LPGLYPHMETRICS lpgm, DWORD cbBuffer, LPVOID lpData, const MAT2* lpmat2) const {
	AssertValidAsDC();
	return ::GetGlyphOutline(m_hDC, nChar, nFormat, lpgm, cbBuffer, lpData, lpmat2);
}

inline DWORD CDC::GetKerningPairs(DWORD dwPairs, LPKERNINGPAIR lpkrnpair) const {
	AssertValidAsDC();
	return ::GetKerningPairs(m_hDC, dwPairs, lpkrnpair);
}

inline COLORREF CDC::GetNearestColor(COLORREF clr) const {
	AssertValidAsDC();
	return ::GetNearestColor(m_hDC, clr);
}

inline UINT CDC::GetOutlineTextMetrics(UINT cbData, LPOUTLINETEXTMETRIC lpotm) const {
	AssertValidAsDC();
	return ::GetOutlineTextMetrics(m_hDC, cbData, lpotm);
}

inline COLORREF CDC::GetPixel(int x, int y) const {
	AssertValidAsDC();
	return ::GetPixel(m_hDC, x, y);
}

inline COLORREF CDC::GetPixel(const POINT& pt) const {
	AssertValidAsDC();
	return ::GetPixel(m_hDC, pt.x, pt.y);
}

inline int CDC::GetPolyFillMode() const {
	AssertValidAsDC();
	return ::GetPolyFillMode(m_hDC);
}

inline bool CDC::GetRasterizerCaps(RASTERIZER_STATUS& status, UINT cb) const {
	AssertValidAsDC();
	return toBoolean(::GetRasterizerCaps(&status, cb));
}

inline int CDC::GetROP2() const {
	AssertValidAsDC();
	return ::GetROP2(m_hDC);
}

inline HDC CDC::GetSafeHdc() const {
	return (this == 0) ? 0 : m_hDC;
}

inline int CDC::GetStretchBltMode() const {
	AssertValidAsDC();
	return ::GetStretchBltMode(m_hDC);
}

inline SIZE CDC::GetTabbedTextExtent(const TCHAR* lpszText,
		int cchText, int nTabPositions, int* lpnTabStopPositions) const {
	AssertValidAsDC();

	SIZE	size;
	DWORD	dw = ::GetTabbedTextExtent(m_hDC, lpszText, cchText, nTabPositions, lpnTabStopPositions);

	size.cx = LOWORD(dw);
	size.cy = HIWORD(dw);
	return size;
}

inline UINT CDC::GetTextAlign() const {
	AssertValidAsDC();
	return ::GetTextAlign(m_hDC);
}

inline int CDC::GetTextCharacterExtra() const {
	AssertValidAsDC();
	return ::GetTextCharacterExtra(m_hDC);
}

inline COLORREF CDC::GetTextColor() const {
	AssertValidAsDC();
	return ::GetTextColor(m_hDC);
}

inline SIZE CDC::GetTextExtent(const TCHAR* lpszText, int cchText) const {
	AssertValidAsDC();

	SIZE	size;

	::GetTextExtentPoint32(m_hDC, lpszText, cchText, &size);
	return size;
}

inline bool CDC::GetTextExtentExPoint(const TCHAR* pgiln,
		int cgi, int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentExPoint(m_hDC, pgiln, cgi, nMaxExtent, lpnFit, alpDx, lpSize));
}

#if(_WIN32_WINNT >= 0x0500)
inline bool CDC::GetTextExtentExPointI(LPWORD pgiln,
		int cgi, int nMaxExtent, LPINT lpnFit, LPINT alpDx, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentExPointI(m_hDC, pgiln, cgi, nMaxExtent, lpnFit, alpDx, lpSize));
}
#endif

#if(_WIN32_WINNT >= 0x0500)
inline bool CDC::GetTextExtentPointI(LPWORD pgiln, int cgi, LPSIZE lpSize) const {
	AssertValidAsDC();
	return toBoolean(::GetTextExtentPointI(m_hDC, pgiln, cgi, lpSize));
}
#endif

inline int CDC::GetTextFace(int cchFaceName, TCHAR* lpszFaceName) const {
	AssertValidAsDC();
	return ::GetTextFace(m_hDC, cchFaceName, lpszFaceName);
}

inline bool CDC::GetTextMetrics(TEXTMETRIC& metrics) const {
	AssertValidAsDC();
	return toBoolean(::GetTextMetrics(m_hDC, &metrics));
}

inline HWND CDC::GetWindow() const {
	AssertValidAsDC();
	return ::WindowFromDC(m_hDC);
}

inline bool CDC::GrayString(HBRUSH hBrush,
		GRAYSTRINGPROC lpfnOutput, LPARAM lpData, int cchText, int x, int y, int nWidth, int nHeight) {
	AssertValidAsDC();
	return toBoolean(::GrayString(m_hDC, hBrush, lpfnOutput, lpData, cchText, x, y, nWidth, nHeight));
}

inline int CDC::IntersectClipRect(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return ::IntersectClipRect(m_hDC, x1, y1, x2, y2);
}

inline int CDC::IntersectClipRect(const RECT& rect) {
	AssertValidAsDC();
	return ::IntersectClipRect(m_hDC, rect.left, rect.top, rect.right, rect.bottom);
}

inline void CDC::InvertRect(const RECT& rect) {
	AssertValidAsDC();
	::InvertRect(m_hDC, &rect);
}

inline bool CDC::InvertRgn(HRGN hRgn) {
	AssertValidAsDC();
	return toBoolean(::InvertRgn(m_hDC, hRgn));
}

inline bool CDC::LineTo(int x, int y) {
	AssertValidAsDC();
	return toBoolean(::LineTo(m_hDC, x, y));
}

inline bool CDC::LineTo(const POINT& pt) {
	AssertValidAsDC();
	return toBoolean(::LineTo(m_hDC, pt.x, pt.y));
}

inline bool CDC::MaskBlt(int x, int y, int nWidth, int nHeight,
		HDC hDC, int xSrc, int ySrc, HBITMAP hBitmap, int xMask, int yMask, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::MaskBlt(m_hDC, x, y, nWidth, nHeight, hDC, xSrc, ySrc, hBitmap, xMask, yMask, dwRop));
}

inline POINT CDC::MoveTo(int x, int y) {
	AssertValidAsDC();

	POINT	pt;
	::MoveToEx(m_hDC, x, y, &pt);
	return pt;
}

inline POINT CDC::MoveTo(const POINT& pt) {
	AssertValidAsDC();

	POINT	pt_;
	::MoveToEx(m_hDC, pt.x, pt.y, &pt_);
	return pt_;
}

inline int CDC::OffsetClipRgn(int x, int y) {
	AssertValidAsDC();
	return ::OffsetClipRgn(m_hDC, x, y);
}

inline int CDC::OffsetClipRgn(const SIZE& size) {
	AssertValidAsDC();
	return ::OffsetClipRgn(m_hDC, size.cx, size.cy);
}

inline bool CDC::PaintRgn(HRGN hRgn) {
	AssertValidAsDC();
	return toBoolean(::PaintRgn(m_hDC, hRgn));
}

inline bool CDC::PatBlt(int x, int y, int nWidth, int nHeight, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::PatBlt(m_hDC, x, y, nWidth, nHeight, dwRop));
}

inline bool CDC::PatBlt(const RECT& rect, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::PatBlt(m_hDC, rect.left, rect.top, rect.right - rect.left, rect.bottom - rect.top, dwRop));
}

inline bool CDC::Pie(int x1, int y1, int x2, int y2, int x3, int y3, int x4, int y4) {
	AssertValidAsDC();
	return toBoolean(::Pie(m_hDC, x1, y1, x2, y2, x3, y3, x4, y4));
}

inline bool CDC::Pie(const RECT& rect, const POINT& ptStart, const POINT& ptEnd) {
	AssertValidAsDC();
	return toBoolean(::Pie(m_hDC, rect.left, rect.top,
		rect.right, rect.bottom, ptStart.x, ptStart.y, ptEnd.x, ptEnd.y));
}

inline bool CDC::PlgBlt(const POINT* lpPoint, HDC hDC,
		int xSrc, int ySrc, int nWidth, int nHeight, HBITMAP hBitmap, int xMask, int yMask) {
	AssertValidAsDC();
	return toBoolean(::PlgBlt(m_hDC, lpPoint, hDC, xSrc, ySrc, nWidth, nHeight, hBitmap, xMask, yMask));
}

inline bool CDC::PolyBezier(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyBezier(m_hDC, lpPoints, nCount));
}

inline bool CDC::PolyBezierTo(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyBezierTo(m_hDC, lpPoints, nCount));
}

inline bool CDC::PolyDraw(const POINT* lpPoints, const BYTE* lpTypes, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyDraw(m_hDC, lpPoints, lpTypes, nCount));
}

inline bool CDC::Polygon(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::Polygon(m_hDC, lpPoints, nCount));
}

inline bool CDC::Polyline(LPPOINT lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::Polyline(m_hDC, lpPoints, nCount));
}

inline bool CDC::PolylineTo(const POINT* lpPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolylineTo(m_hDC, lpPoints, nCount));
}

inline bool CDC::PolyPolygon(const POINT* lpPoints, const int* lpPolyCounts, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyPolygon(m_hDC, lpPoints, lpPolyCounts, nCount));
}

inline bool CDC::PolyPolyline(const POINT* lpPoints, const DWORD *lpPolyPoints, int nCount) {
	AssertValidAsDC();
	return toBoolean(::PolyPolyline(m_hDC, lpPoints, lpPolyPoints, nCount));
}

inline bool CDC::PolyTextOut(const POLYTEXT* pTexts, int cTexts) {
	AssertValidAsDC();
	return toBoolean(::PolyTextOut(m_hDC, pTexts, cTexts));
}

inline bool CDC::PtVisible(int x, int y) const {
	AssertValidAsDC();
	return toBoolean(::PtVisible(m_hDC, x, y));
}

inline bool CDC::PtVisible(const POINT& pt) const {
	AssertValidAsDC();
	return toBoolean(::PtVisible(m_hDC, pt.x, pt.y));
}

inline UINT CDC::RealizePalette() {
	AssertValidAsDC();
	return ::RealizePalette(m_hDC);
}

inline bool CDC::Rectangle(int x1, int y1, int x2, int y2) {
	AssertValidAsDC();
	return toBoolean(::Rectangle(m_hDC, x1, y1, x2, y2));
}

inline bool CDC::Rectangle(const RECT& rect) {
	AssertValidAsDC();
	return toBoolean(::Rectangle(m_hDC, rect.left, rect.top, rect.right, rect.bottom));
}

inline bool CDC::RectVisible(const RECT& rect) const {
	AssertValidAsDC();
	return toBoolean(::RectVisible(m_hDC, &rect));
}

inline bool CDC::RestoreDC(int nSavedDC) {
	AssertValidAsDC();
	return toBoolean(::RestoreDC(m_hDC, nSavedDC));
}

inline bool CDC::RoundRect(int x1, int y1, int x2, int y2, int x3, int y3) {
	AssertValidAsDC();
	return toBoolean(::RoundRect(m_hDC, x1, y1, x2, y2, x3, y3));
}

inline bool CDC::RoundRect(const RECT& rect, const POINT& pt) {
	AssertValidAsDC();
	return toBoolean(::RoundRect(m_hDC, rect.left, rect.top, rect.right, rect.bottom, pt.x, pt.y));
}

inline int CDC::SaveDC() {
	AssertValidAsDC();
	return ::SaveDC(m_hDC);
}

inline bool CDC::ScrollDC(int dx, int dy,
		const RECT& scrollRect, const RECT& clipRect, HRGN hUpdateRgn, RECT* lpUpdateRect) {
	AssertValidAsDC();
	return toBoolean(::ScrollDC(m_hDC, dx, dy, &scrollRect, &clipRect, hUpdateRgn, lpUpdateRect));
}

inline int CDC::SelectClipRgn(HRGN hRgn) {
	AssertValidAsDC();
	return ::SelectClipRgn(m_hDC, hRgn);
}

inline HBITMAP CDC::SelectObject(HBITMAP hBitmap) {
	AssertValidAsDC();
	return reinterpret_cast<HBITMAP>(::SelectObject(m_hDC, hBitmap));
}

inline HBRUSH CDC::SelectObject(HBRUSH hBrush) {
	AssertValidAsDC();
	return reinterpret_cast<HBRUSH>(::SelectObject(m_hDC, hBrush));
}

inline HFONT CDC::SelectObject(HFONT hFont) {
	AssertValidAsDC();
	return reinterpret_cast<HFONT>(::SelectObject(m_hDC, hFont));
}

inline HPEN CDC::SelectObject(HPEN hPen) {
	AssertValidAsDC();
	return reinterpret_cast<HPEN>(::SelectObject(m_hDC, hPen));
}

inline HPALETTE CDC::SelectPalette(HPALETTE hPalette, bool bForceBackground) {
	AssertValidAsDC();
	return ::SelectPalette(m_hDC, hPalette, bForceBackground);
}

inline HGDIOBJ CDC::SelectStockObject(int nObject) {
	AssertValidAsDC();
	return ::SelectObject(m_hDC, ::GetStockObject(nObject));
}

inline int CDC::SetArcDirection(int nArcDirection) {
	AssertValidAsDC();
	return ::SetArcDirection(m_hDC, nArcDirection);
}

inline COLORREF CDC::SetBkColor(COLORREF clr) {
	AssertValidAsDC();
	return ::SetBkColor(m_hDC, clr);
}

inline int CDC::SetBkMode(int nBkMode) {
	AssertValidAsDC();
	return ::SetBkMode(m_hDC, nBkMode);
}

inline UINT CDC::SetBoundsRect(const RECT& rcBounds, UINT nFlags) {
	AssertValidAsDC();
	return ::SetBoundsRect(m_hDC, &rcBounds, nFlags);
}

inline POINT CDC::SetBrushOrg(int x, int y) {
	AssertValidAsDC();

	POINT	pt;
	::SetBrushOrgEx(m_hDC, x, y, &pt);
	return pt;
}

inline POINT CDC::SetBrushOrg(const POINT& pt) {
	AssertValidAsDC();

	POINT	pt_;
	::SetBrushOrgEx(m_hDC, pt.x, pt.y, &pt_);
	return pt_;
}

inline bool CDC::SetColorAdjustment(const COLORADJUSTMENT& colorAdjust) {
	AssertValidAsDC();
	return toBoolean(::SetColorAdjustment(m_hDC, &colorAdjust));
}

inline DWORD CDC::SetMapperFlags(DWORD dwFlag) {
	AssertValidAsDC();
	return ::SetMapperFlags(m_hDC, dwFlag);
}

inline COLORREF CDC::SetPixel(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return ::SetPixel(m_hDC, x, y, clr);
}

inline COLORREF CDC::SetPixel(const POINT& pt, COLORREF clr) {
	AssertValidAsDC();
	return ::SetPixel(m_hDC, pt.x, pt.y, clr);
}

inline bool CDC::SetPixelV(int x, int y, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::SetPixelV(m_hDC, x, y, clr));
}

inline bool CDC::SetPixelV(const POINT& pt, COLORREF clr) {
	AssertValidAsDC();
	return toBoolean(::SetPixelV(m_hDC, pt.x, pt.y, clr));
}

inline int CDC::SetPolyFillMode(int nPolyFillMode) {
	AssertValidAsDC();
	return ::SetPolyFillMode(m_hDC, nPolyFillMode);
}

inline int CDC::SetROP2(int nDrawMode) {
	AssertValidAsDC();
	return ::SetROP2(m_hDC, nDrawMode);
}

inline int CDC::SetStretchBltMode(int nStretchMode) {
	AssertValidAsDC();
	return ::SetStretchBltMode(m_hDC, nStretchMode);
}

inline UINT CDC::SetTextAlign(UINT nFlags) {
	AssertValidAsDC();
	return ::SetTextAlign(m_hDC, nFlags);
}

inline int CDC::SetTextCharacterExtra(int nCharExtra) {
	AssertValidAsDC();
	return ::SetTextCharacterExtra(m_hDC, nCharExtra);
}

inline COLORREF CDC::SetTextColor(COLORREF clr) {
	AssertValidAsDC();
	return ::SetTextColor(m_hDC, clr);
}

inline int CDC::SetTextJustification(int nBreakExtra, int nBreakCount) {
	AssertValidAsDC();
	return ::SetTextJustification(m_hDC, nBreakExtra, nBreakExtra);
}

inline bool CDC::StretchBlt(int x, int y, int nWidth, int nHeight,
		HDC hSrcDC, int xSrc, int ySrc, int nSrcWidth, int nSrcHeight, DWORD dwRop) {
	AssertValidAsDC();
	return toBoolean(::StretchBlt(m_hDC,
		x, y, nWidth, nHeight, hSrcDC, xSrc, ySrc, nSrcWidth, nSrcHeight, dwRop));
}

inline SIZE CDC::TabbedTextOut(int x, int y,
		const TCHAR* lpszText, int cchText, int nTabPositions, int* lpnTabStopPositions, int nTabOrigin) {
	AssertValidAsDC();
	SIZE	size;
	long	n = ::TabbedTextOut(m_hDC, x, y,
		lpszText, cchText, nTabPositions, lpnTabStopPositions, nTabOrigin);

	size.cx = LOWORD(n);
	size.cy = HIWORD(n);
	return size;
}

inline bool CDC::TextOut(int x, int y, const TCHAR* lpszText, int cchText) {
	AssertValidAsDC();
	return toBoolean(::TextOut(m_hDC, x, y, lpszText, cchText));
}

inline void CDC::UpdateColors() {
	AssertValidAsDC();
	::UpdateColors(m_hDC);
}

} /* namespace GDI */
} /* namespace Windows */
} /* namespace Manah */

#endif /* _DC_H_ */

/* [EOF] */