// Japanese.cpp
// (c) 2004-2005 exeal

#include "StdAfx.h"
#include "Encoder.h"
#include <memory>		// std::auto_ptr
#include <map>
#include <algorithm>	// std::binary_search

using namespace Ascension;
using namespace Ascension::Encodings;
using namespace std;


BEGIN_ENCODER_DEFINITION()
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_SHIFTJIS, Japanese_ShiftJis, 2, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_SHIFTJIS2004, Japanese_ShiftJis2004, 2, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_EUC, Japanese_EucJp, 3, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_EUCJIS2004, Japanese_EucJis2004, 3, 1)
	DEFINE_ENCODER_CLASS_(51932, Japanese_EucJpWindows)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP, Japanese_Iso2022Jp, 8, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP1, Japanese_Iso2022Jp1, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP2, Japanese_Iso2022Jp2, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP2004, Japanese_Iso2022Jp2004, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP2004_STRICT, Japanese_Iso2022Jp2004_Strict, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP2004_COMPATIBLE, Japanese_Iso2022Jp2004_Compatible, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP3, Japanese_Iso2022Jp3, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP3_STRICT, Japanese_Iso2022Jp3_Strict, 9, 1)
	DEFINE_ENCODER_CLASS(CPEX_JAPANESE_ISO2022JP3_COMPATIBLE, Japanese_Iso2022Jp3_Compatible, 9, 1)
//	DEFINE_ENCODER_CLASS(50221, Japanese_Iso2022JpWindows)
END_ENCODER_DEFINITION()
DEFINE_DETECTOR(CPEX_JAPANESE_AUTODETECT, Japanese);

/**
 *	@file Japanese.cpp	{GR[_̎
 *
 *	<h3>镶WƃGR[h</h3>
 *
 *	̃t@Cł͈ȉ̃GR[h:
 *	<ul>
 *		<li>JIS X0208 -- Vtg JIS AISO-2022-JP</li>
 *		<li>JIS X0208  JIS X0212 -- EUC AISO-2022-JP-1 AISO-2022-JP-2</li>
 *		<li>JIS X0213 -- Shift_JIS-2004 AEUC-JIS-2004 AISO-2022-JP-3 (3)AISO-2022-JP-2004 (3)</li>
 *		<li>CP932 -- Vtg JIS AEUC AISO-2022-JP</li>
 *	</ul>
 *	CP932 x[X̃GR[h Windows ̕ϊ\ɊÂ̂ŁA
 *	EUC ł JIS X0212 W܂łȂB܂VXe
 *	CP932 CXg[ĂȂΖ
 *
 *	eGR[h̊ȒPȉ Ascension ł̎舵ɂăR[hɃRg
 *
 *	ISO-2022-JP*  UCS ւ̕ϊɂāAsȃGXP[vV[PXƂ̌㑱f[^1oCgȃR[hl
 *	UCS ɕϊB̓[Uϊ𔭌ՂȂʂ邪Â܂ܓGR[hŔ
 *	UCS ɕϊẴf[^ɂ͖߂Ȃ̂Œ
 *
 *	<h3>JIS X0208  UCS ̑Ήŕ̉߂镶</h3>
 *
 *	KUBOTA ̒ (http://www.debian.or.jp/~kubota/unicode-symbols-map2.html) ɂ
 *	JIS X0208 12̕ UCS Ƃ̑Ήɂĕ̕ϊ\ŉ߂ɈႢB
 *	Ascension ł JISX0213 InfoCenter ̕\ JIS X0208 AJIS X0213 ̃e[u쐬ɗpĂA
 *	̕\L̔rΏۂɊ܂܂ĂB͂̕\̒ŉ߂ɗĥ12
 *	libiconv EUC-JP ̂̂ɕύXÃe[u쐬 (Jis fBNg)
 *
 *	<h3>ISO-2022-JP-2004 3̃GR[h</h3>
 *
 *	Emacs  ISO-2022-JP Ƃ̌݊̂߂ ISO-2022-JP-3 Ƃ̕ώ킹3ĂB
 *	 JIS X0208  JIS X0213 Ŋ̕ۊقȂ邽߂łB
 *	ڍׂeGR[hɂĂ͈ȉ̃y[WQ:
 *	<ul>
 *		<li>JIS X 0213̓ƁAEmacsł̎
 *		(http://www.m17n.org/m17n2000_all_but_registration/proceedings/kawabata/jisx0213.html)</li>
 *		<li>Becky! JIS X 0213 vOC
 *		(http://members.at.infoseek.co.jp/jisx0213/bk0213.html)</li>
 *	</ul>
 *
 *	<h3></h3>
 *
 *	JIS X0213 ɂ͍\ȔL܂܂ĂAUCS ̕ϊ
 *	JIS ɖςݕɂĂ͊bƔLɕ邱Ƃŗ_͕\łB
 *	 Ascension ł͂̕sȂA_ł͕ϊ͕s\łB
 *	JIS X0213 Ɍ鍇ς݉ɂĂ͑ΉĂ
 *
 *	<h3>L̍</h3>
 *
 *	JIS X0213 2̐LA㏸ (1-11-69) Ɖ~ (1-11-70) ɒڑΉ UCS ͖A
 *	2̃R[h|Cg̍ΉƍlBȂ킿 JIS  UCS ւ̕ϊɂāA㏸
 *	U+02E9+02E5A~ U+02E5+02E9 ƂȂB̂悤ȒPȕϊs JIS  UCS
 *	̃R[hĂ܂悤 (http://wakaba-web.hp.infoseek.co.jp/table/jis-note.ja.html)B
 *	Ascension ł ZWNJ gčƈӐ}R[h|Cg̑gݍ킹ƂłȂ̂ʂB
 *	܂AJIS Œ (1-11-64) ƒ (1-11-68) łꍇ́Aꂼ̕ UCS
 *	ɕϊĊԂ ZWNJ ށBt UCS  U+02E5  U+02E9 łꍇ JIS
 *	̑Ή1̐LɕϊAԂ ZWNJ ꍇ2̐Lɕϊ
 */


#define IS_ISO2022JP3(cp)	\
	(cp >= CPEX_JAPANESE_ISO2022JP3 && cp <= CPEX_JAPANESE_ISO2022JP3_COMPATIBLE)

#define IS_ISO2022JP2004(cp)	\
	(cp >= CPEX_JAPANESE_ISO2022JP2004 && cp <= CPEX_JAPANESE_ISO2022JP2004_COMPATIBLE)

#define JK(ku, ten)	((ku << 8) | ten) + 0x2020


namespace {
	// JIS <-> UCS ϊe[u (make_table.pl 쐬t@C؂荏񂾂)B
	// JIS X0213 ɂĂ͗Ƃ32rbglŃe[u쐬B
	// UCS  0x0000-0xFFFF ͂̂܂ UCS-2A0x10000-0x10FFFF  UCS-4A0xFFFFFFF ȏ UCS-2 2\
	// JIS ł

	const uchar		ESC = '\x1B';
	const wchar_t	ZWNJ = L'\x200C';	// Zero Width Non-Joiner (U+200C)
	const wchar_t	JISX0208toUCS_2121[] = {	// 0x2121-0x2840
		#include "Jis\JISX0208toUCS_2121"
	};
	const wchar_t	JISX0208toUCS_3021[] = {	// 0x3021-0x4F53
		#include "Jis\JISX0208toUCS_3021"
	};
	const wchar_t	JISX0208toUCS_5021[] = {	// 0x5021-0x7426
		#include "Jis\JISX0208toUCS_5021"
	};
	const ushort	UCStoJISX0208_00A2[] = {	// U+00A2-U+00F7
		#include "Jis\UCStoJISX0208_00A2"
	};
	const ushort	UCStoJISX0208_0391[] = {	// U+0391-U+0451
		#include "Jis\UCStoJISX0208_0391"
	};
	const ushort	UCStoJISX0208_2010[] = {	// U+2010-U+2312
		#include "Jis\UCStoJISX0208_2010"
	};
	const ushort	UCStoJISX0208_2500[] = {	// U+2500-U+266F
		#include "Jis\UCStoJISX0208_2500"
	};
	const ushort	UCStoJISX0208_3000[] = {	// U+3000-U+30FE
		#include "Jis\UCStoJISX0208_3000"
	};
	const ushort	UCStoJISX0208_4E00[] = {	// U+4E00-U+9FA0
		#include "Jis\UCStoJISX0208_4E00"
	};
	const ushort	UCStoJISX0208_FF01[] = {	// U+FF01-U+FFE5
		#include "Jis\UCStoJISX0208_FF01"
	};
	const wchar_t	JISX0212toUCS_222F[] = {	// 0x222F-0x2271
		#include "Jis\JISX0212toUCS_222F"
	};
	const wchar_t	JISX0212toUCS_2661[] = {	// 0x2661-0x2B77
		#include "Jis\JISX0212toUCS_2661"
	};
	const wchar_t	JISX0212toUCS_3021[] = {	// 0x3021-0x6D63
		#include "Jis\JISX0212toUCS_3021"
	};
	const ushort	UCStoJISX0212_007E[] = {	// U+007E-U+045F
		#include "Jis\UCStoJISX0212_007E"
	};
	const ushort	UCStoJISX0212_2116[] = {	// U+2116-U+2122
		#include "Jis\UCStoJISX0212_2116"
	};
	const ushort	UCStoJISX0212_4E02[] = {	// U+4E02-U+9FA5
		#include "Jis\UCStoJISX0212_4E02"
	};
	const ulong		JISX0213P1toUCS_2121[] = {	// 0x2121-0x2F7E
		#include "Jis\JISX0213P1toUCS_2121"
	};
	const ulong		JISX0213P1toUCS_4F54[] = {	// 0x4F54-0x4F7E
		#include "Jis\JISX0213P1toUCS_4F54"
	};
	const ulong		JISX0213P1toUCS_7427[] = {	// 0x7427-0x7E7E
		#include "Jis\JISX0213P1toUCS_7427"
	};
	const ulong		JISX0213P2toUCS_2121[] = {	// 0x2121-0x217E
		#include "Jis\JISX0213P2toUCS_2121"
	};
	const ulong		JISX0213P2toUCS_2321[] = {	// 0x2321-0x257E
		#include "Jis\JISX0213P2toUCS_2321"
	};
	const ulong		JISX0213P2toUCS_2821[] = {	// 0x2821-0x287E
		#include "Jis\JISX0213P2toUCS_2821"
	};
	const ulong		JISX0213P2toUCS_2C21[] = {	// 0x2C21-0x2F7E
		#include "Jis\JISX0213P2toUCS_2C21"
	};
	const ulong		JISX0213P2toUCS_6E21[] = {	// 0x6E21-0x7E76
		#include "Jis\JISX0213P2toUCS_6E21"
	};
	const ulong		UCStoJISX0213_00A0[] = {	// U+00A0-U+0451
		#include "Jis\UCStoJISX0213_00A0"
	};
	const ulong		UCStoJISX0213_1E3E[] = {	// U+1E3E-U+29FB
		#include "Jis\UCStoJISX0213_1E3E"
	};
	const ulong		UCStoJISX0213_3000[] = {	// U+3000-U+6568
		#include "Jis\UCStoJISX0213_3000"
	};
	const ulong		UCStoJISX0213_F91D[] = {	// U+F91D-U+FA6A
		#include "Jis\UCStoJISX0213_F91D"
	};
	const ulong		UCStoJISX0213_FE45[] = {	// U+FE45-U+FFE5
		#include "Jis\UCStoJISX0213_FE45"
	};

	// ISO-2022-JP-3 ̋֎~ (JIS X0213:2000 2\1)
	const ushort	ProhibitedIdeographs_2000[] = {	// sA
		JK( 3,26),JK( 3,27),JK( 3,28),JK( 3,29),JK( 3,30),JK( 3,31),	// <L>
		JK( 3,32),
		JK( 3,59),JK( 3,60),JK( 3,61),JK( 3,62),JK( 3,63),JK( 3,64),	// <L>
		JK( 3,91),JK( 3,92),JK( 3,93),JK( 3,94),
		JK( 4,84),JK( 4,85),JK( 4,86),JK( 8,87),JK( 4,88),JK( 4,89),	// <>
		JK( 4,90),JK( 4,91),
		JK( 5,87),JK( 5,88),JK( 5,89),JK( 5,90),JK( 5,91),JK( 5,92),	// <Љ>
		JK( 5,93),JK( 5,94),
		JK( 6,25),JK( 6,26),JK( 6,27),JK( 6,28),JK( 6,29),JK( 6,30),	// <gv>
		JK( 6,31),JK( 6,32),
												JK(13,83),JK(13,88),	// @@@@
		JK(13,89),JK(13,93),JK(13,94),									// EE
														  JK(16, 2),	// 
		JK(16,19),JK(16,79),JK(17,58),JK(17,75),JK(17,79),JK(18, 3),	// y
		JK(18, 9),JK(18,10),JK(18,11),JK(18,25),JK(18,50),JK(18,89),	// Љ
		JK(19, 4),JK(19,20),JK(19,21),JK(19,34),JK(19,41),JK(19,69),	// CSTah
		JK(19,73),JK(19,76),JK(19,86),JK(19,90),JK(20,18),JK(20,33),	// 
		JK(20,35),JK(20,50),JK(20,79),JK(20,91),JK(21, 7),JK(21,85),	// ЊF
		JK(22, 2),JK(22,31),JK(22,33),JK(22,38),JK(22,48),JK(22,64),	// ċ΋
		JK(22,77),JK(23,16),JK(23,39),JK(23,59),JK(23,66),JK(24, 6),	// Ofz
		JK(24,20),JK(25,60),JK(25,77),JK(25,82),JK(25,85),JK(27, 6),	// {E
		JK(27,67),JK(27,75),JK(28,40),JK(28,41),JK(28,49),JK(28,50),	// Ǝǎώ
		JK(28,52),JK(29,11),JK(29,13),JK(29,43),JK(29,75),JK(29,77),	// ҏJLj
		JK(29,79),JK(29,80),JK(29,84),JK(30,36),JK(30,45),JK(30,53),	// ˏ
		JK(30,63),JK(30,85),JK(31,32),JK(31,57),JK(32, 5),JK(32,65),	// ݏ_x
		JK(32,70),JK(33, 8),JK(33,36),JK(33,46),JK(33,56),JK(33,63),	// Gcmw~
		JK(33,67),JK(33,93),JK(33,94),JK(34, 3),JK(34, 8),JK(34,45),	// 
		JK(34,86),JK(35,18),JK(35,29),JK(35,86),JK(35,88),JK(36, 7),	// Q\
		JK(36, 8),JK(36,45),JK(36,47),JK(36,59),JK(36,87),JK(37,22),	// ˒ْ͒U
		JK(37,31),JK(37,52),JK(37,55),JK(37,78),JK(37,83),JK(37,88),	// ^sv
		JK(38,33),JK(38,34),JK(38,45),JK(38,81),JK(38,86),JK(39,25),	// ˓X
		JK(39,63),JK(39,72),JK(40,14),JK(40,16),JK(40,43),JK(40,53),	// ~ɔ
		JK(40,60),JK(40,74),JK(41,16),JK(41,48),JK(41,49),JK(41,50),	// ڔOopq
		JK(41,51),JK(41,78),JK(42, 1),JK(42,27),JK(42,29),JK(42,57),	// r
		JK(42,66),JK(43,43),JK(43,47),JK(43,72),JK(43,74),JK(43,89),	// jn
		JK(44,40),JK(44,45),JK(44,65),JK(44,89),JK(45,20),JK(45,58),	// Ɩ˖ߖSy
		JK(45,73),JK(45,74),JK(45,83),JK(46,20),JK(46,26),JK(46,48),	// 
		JK(46,62),JK(46,64),JK(46,81),JK(46,82),JK(46,93),JK(47, 3),	// ܗޗB
		JK(47,13),JK(47,15),JK(47,22),JK(47,25),JK(47,26),JK(47,31),	// LNUXY^
							JK(48,54),JK(52,68),JK(57,88),JK(58,25),	// @@Ԛ❘
		JK(59,56),JK(59,77),JK(62,25),JK(62,85),JK(63,70),JK(64,86),	// w
		JK(66,72),JK(66,74),JK(67,62),JK(68,38),JK(73, 2),JK(73,14),	// }AM
		JK(73,58),JK(74, 4),JK(75,61),JK(76,45),JK(77,78),JK(80,55),	// y|
		JK(80,84),JK(82,45),JK(82,84),JK(84, 1),JK(84, 2),JK(84, 3),	// 
		JK(84, 4),JK(84, 5),JK(84, 6),									// 
	};
	const ushort	ProhibitedIdeographs_2004[] = {	// ISO-2022-JP-2004 ̋֎~ (JIS X0213:2004 2\2)
		JK(14, 1),JK(15,94),JK(17,19),JK(22,70),JK(23,50),JK(28,24),	// EERq
		JK(33,73),JK(38,61),JK(39,77),JK(47,52),JK(47,94),JK(53,11),	// ۔EEJ
		JK(54, 2),JK(54,58),JK(84, 7),JK(94,90),JK(94,91),JK(94,92),	// EEEE
		JK(94,93),JK(94,94)												// EE
	};
	const ushort	CjkExtBUcs[] = {
		0x000B,0x0089,0x00A2,0x00A4,0x01A2,0x0213,0x032B,0x0371,0x0381,0x03F9,0x044A,0x0509,0x05D6,0x0628,0x074F,0x0807,
		0x083A,0x08B9,0x097C,0x099D,0x0AD3,0x0B1D,0x0B9F,0x0D45,0x0DE1,0x0E64,0x0E6D,0x0E95,0x0F5F,0x1201,0x123D,0x1255,
		0x1274,0x127B,0x12D7,0x12E4,0x12FD,0x131B,0x1336,0x1344,0x13C4,0x146D,0x146E,0x15D7,0x1647,0x16B4,0x1706,0x1742,
		0x18BD,0x19C3,0x1C56,0x1D2D,0x1D45,0x1D62,0x1D78,0x1D92,0x1D9C,0x1DA1,0x1DB7,0x1DE0,0x1E33,0x1E34,0x1F1E,0x1F76,
		0x1FFA,0x217B,0x2218,0x231E,0x23AD,0x26F3,0x285B,0x28AB,0x298F,0x2AB8,0x2B46,0x2B4F,0x2B50,0x2BA6,0x2C1D,0x2C24,
		0x2DE1,0x31B6,0x31C3,0x31C4,0x31F5,0x3372,0x33D0,0x33D2,0x33D3,0x33D5,0x33DA,0x33DF,0x33E4,0x344A,0x344B,0x3451,
		0x3465,0x34E4,0x355A,0x3594,0x35C4,0x3638,0x3639,0x363A,0x3647,0x370C,0x371C,0x373F,0x3763,0x3764,0x37E7,0x37FF,
		0x3824,0x383D,0x3A98,0x3C7F,0x3CFE,0x3D00,0x3D0E,0x3D40,0x3DD3,0x3DF9,0x3DFA,0x3F7E,0x4096,0x4103,0x41C6,0x41FE,
		0x43BC,0x4629,0x46A5,0x47F1,0x4896,0x4A4D,0x4B56,0x4B6F,0x4C16,0x4D14,0x4E0E,0x4E37,0x4E6A,0x4E8B,0x504A,0x5055,
		0x5122,0x51A9,0x51CD,0x51E5,0x521E,0x524C,0x542E,0x548E,0x54D9,0x550E,0x55A7,0x5771,0x57A9,0x57B4,0x59C4,0x59D4,
		0x5AE3,0x5AE4,0x5AF1,0x5BB2,0x5C4B,0x5C64,0x5DA1,0x5E2E,0x5E56,0x5E62,0x5E65,0x5EC2,0x5ED8,0x5EE8,0x5F23,0x5F5C,
		0x5FD4,0x5FE0,0x5FFB,0x600C,0x6017,0x6060,0x60ED,0x6270,0x6286,0x634C,0x6402,0x667E,0x66B0,0x671D,0x68DD,0x68EA,
		0x6951,0x696F,0x69DD,0x6A1E,0x6A58,0x6A8C,0x6AB7,0x6AFF,0x6C29,0x6C73,0x6CDD,0x6E40,0x6E65,0x6F94,0x6FF6,0x6FF7,
		0x6FF8,0x70F4,0x710D,0x7139,0x73DA,0x73DB,0x73FE,0x7410,0x7449,0x7614,0x7615,0x7631,0x7684,0x7693,0x770E,0x7723,
		0x7752,0x7985,0x7A84,0x7BB3,0x7BBE,0x7BC7,0x7CB8,0x7DA0,0x7E10,0x7FB7,0x808A,0x80BB,0x8277,0x8282,0x82F3,0x83CD,
		0x840C,0x8455,0x856B,0x85C8,0x85C9,0x86D7,0x86FA,0x8946,0x8949,0x896B,0x8987,0x8988,0x89BA,0x89BB,0x8A1E,0x8A29,
		0x8A43,0x8A71,0x8A99,0x8ACD,0x8ADD,0x8AE4,0x8BC1,0x8BEF,0x8D10,0x8D71,0x8DFB,0x8E1F,0x8E36,0x8E89,0x8EEB,0x8F32,
		0x8FF8,0x92A0,0x92B1,0x9490,0x95CF,0x967F,0x96F0,0x9719,0x9750,0x98C6,0x9A72,0x9DDB,0x9E15,0x9E3D,0x9E49,0x9E8A,
		0x9EC4,0x9EDB,0x9EE9,0x9FCE,0xA01A,0xA02F,0xA082,0xA0F9,0xA190,0xA38C,0xA437,0xA5F1,0xA602,0xA61A,0xA6B2,
	};
	const ulong		CjkExtBJis[] = {
		0x2E22,0x12121,0x1212B,0x1212E,0x12136,0x12146,0x12170,0x12179,0x12177,0x12322,0x12325,0x12327,0x12331,0x12332,0x12338,0x1233F,
		0x12341,0x1234A,0x12352,0x12353,0x12359,0x1235C,0x4F54,0x12377,0x1242A,0x1243A,0x12432,0x12431,0x1243D,0x12459,0x2F42,0x1245C,
		0x12463,0x1245E,0x1246B,0x1246A,0x12472,0x2F4C,0x12474,0x12475,0x12525,0x12532,0x2F60,0x1253E,0x12547,0x4F63,0x12555,0x12556,
		0x2F7B,0x1257E,0x12830,0x12837,0x12838,0x1283B,0x1283A,0x12845,0x12840,0x1283F,0x12848,0x1284A,0x1284B,0x4F6E,0x1285B,0x12866,
		0x1286C,0x12C22,0x17E53,0x12C2B,0x12C30,0x12C50,0x12C65,0x12C6D,0x12C72,0x12D24,0x12D32,0x12D29,0x12D2A,0x12D35,0x12D34,0x12D39,
		0x12D56,0x12E24,0x12D7D,0x753A,0x12E23,0x12E3A,0x12E42,0x12E3D,0x12E3C,0x12E44,0x12E47,0x12E49,0x12E43,0x12E55,0x12E57,0x12E56,
		0x12E5B,0x12E77,0x12E78,0x12F2A,0x7572,0x12F42,0x12F3F,0x12F43,0x12F40,0x12F59,0x12F4E,0x7629,0x7632,0x12F61,0x12F6A,0x12F69,
		0x12F70,0x12F75,0x16E23,0x16E34,0x7660,0x16E49,0x17475,0x16E5C,0x16E60,0x16E5F,0x16E5E,0x16F32,0x16F47,0x16F4D,0x16F61,0x16F64,
		0x17022,0x17033,0x17039,0x776C,0x17053,0x1707B,0x1712E,0x17130,0x17135,0x17144,0x1715D,0x17161,0x17166,0x17169,0x17175,0x17177,
		0x1717A,0x17221,0x17224,0x17223,0x17228,0x1722C,0x1723D,0x787E,0x17248,0x7929,0x1725B,0x7947,0x17275,0x17276,0x7954,0x17332,
		0x1733E,0x1733D,0x17340,0x17352,0x1735D,0x1735E,0x796E,0x17373,0x17374,0x17377,0x17375,0x1737D,0x1737B,0x17422,0x17424,0x17427,
		0x1742F,0x1742E,0x17435,0x17434,0x1743D,0x17442,0x1744F,0x17469,0x1746B,0x17472,0x17479,0x17535,0x1753A,0x17546,0x17556,0x17558,
		0x1755A,0x1755D,0x1755F,0x17563,0x1756A,0x17570,0x17573,0x7A5D,0x12544,0x17644,0x1764E,0x7B33,0x1765D,0x17675,0x17721,0x17722,
		0x1767E,0x7B49,0x17733,0x17736,0x17765,0x17764,0x1776B,0x1776E,0x17773,0x1782A,0x17829,0x1782C,0x7B6C,0x17834,0x1783C,0x1783E,
		0x17842,0x17856,0x17863,0x17877,0x17879,0x1787A,0x17925,0x1792F,0x17932,0x17939,0x17942,0x17948,0x7C49,0x17959,0x1795E,0x7C51,
		0x17966,0x1796B,0x1797A,0x1797E,0x17A21,0x17A2C,0x17A2F,0x17A50,0x17A4F,0x17A57,0x17A65,0x17A66,0x17A71,0x17A72,0x17A7E,0x17B21,
		0x17B2D,0x17B2C,0x17B36,0x17B37,0x17B3E,0x17B3D,0x17B4E,0x17B4F,0x17B57,0x17B5A,0x17B5C,0x17B5D,0x17B61,0x17B65,0x17B67,0x17B69,
		0x17B71,0x17C22,0x17C23,0x17C38,0x17C42,0x17C4C,0x17C56,0x17C59,0x17C5D,0x17C76,0x17D2C,0x17D4B,0x17D59,0x17D4C,0x17D5D,0x17D5B,
		0x17D67,0x17D70,0x17D6D,0x17E25,0x17E2B,0x17E29,0x17E35,0x17E32,0x7E66,0x17E58,0x17E5A,0x17E6E,0x17E70,0x17E72,0x17E76,
	};
	static map<ushort, ulong>	cjkExtBtoJis;
	enum Iso2022JpCharset_G0 {
		ascii, jisx0201_roman, /*jisx0201_kana,*/ jisx0208, jisx0212, jisx0213p1, jisx0213p2, gb2312, ksc5601
	};
	enum Iso2022JpCharset_G2 {
		undesignated = ksc5601 + 1 , iso8859_1, iso8859_7
	};

	// ISO-2022-JP-3 ̋֎~
	inline bool IsIso2022Jp3ProhibitedIdeograph(ushort jis) {
		return (jis >= JK(6, 57) && jis <= JK(6, 94))
			|| (jis >= JK(7, 34) && jis <= JK(7, 48))
			|| (jis >= JK(7, 82) && jis <= JK(8, 62))
			|| (jis >= JK(8, 71) && jis <= JK(8, 92))
			|| (jis >= JK(9, 1) && jis <= JK(12, 83))
			|| (jis >= JK(12, 93) && jis <= JK(13, 55))
			|| (jis >= JK(13, 63) && jis <= JK(13, 79))
			|| (jis >= JK(14, 2) && jis <= JK(15, 93))
			|| (jis >= JK(47, 53) && jis <= JK(47, 93))
			|| (jis >= JK(84, 8) && jis <= JK(94, 89))
			|| binary_search(ProhibitedIdeographs_2000,
				ProhibitedIdeographs_2000 + _countof(ProhibitedIdeographs_2000), jis);
	}
	// JIS X0213:2004 Œǉꂽ֎~
	inline bool IsIso2022Jp2004ProhibitedIdeograph(ushort jis) {
		return binary_search(ProhibitedIdeographs_2004,
			ProhibitedIdeographs_2004 + _countof(ProhibitedIdeographs_2004), jis);
	}

	// JIS X0201 Roman -> UCS ϊ
	inline wchar_t JISX0201RomanToUCS(uchar ch) {
		if(ch == 0x5C)						return 0x00A5;					// Yen Sign
		else if(ch == 0x7E)					return 0x203E;					// Overline
		else if(ch >= 0x20 && ch <= 0x7D)	return ch;						// 7-bit
		else								return __REPLACEMENT_CHARACTER;	// invalid
	}

	// UCS -> JIS X0201 Roman ϊ
	inline uchar UCSToJISX0201Roman(wchar_t ch) {
		if(ch >= 0x0020 && ch <= 0x005B)		return static_cast<uchar>(ch);	// 7-bit
		else if(ch >= 0x005D && ch <= 0x007D)	return static_cast<uchar>(ch);	// 7-bit
		else if(ch == 0x00A5)					return 0x5C;					// Yen Sign
		else if(ch == 0x203E)					return 0x7E;					// Overline
		else									return 0x00;					// invalid
	}

	// JIS X0201 Kana -> UCS ϊ
	inline wchar_t JISX0201KanaToUCS(uchar ch) {
		return (ch >= 0xA1 && ch <= 0xDF) ? ch + 0xFEC0 : __REPLACEMENT_CHARACTER;
	}

	// UCS -> JIS X0201 Kana ϊ
	inline uchar UCSToJISX0201Kana(wchar_t ch) {
		return (ch >= 0xFF61 && ch <= 0xFF9F) ? static_cast<uchar>(ch - 0xFEC0) : 0x00;
	}

	// JIS X0208 -> UCS ϊ
	inline wchar_t JISX0208ToUCS(ushort jis) {
		if(jis >= 0x2121 && jis < 0x2121 + _countof(JISX0208toUCS_2121))
			return JISX0208toUCS_2121[jis - 0x2121];
		else if(jis >= 0x3021 && jis < 0x3021 + _countof(JISX0208toUCS_3021))
			return JISX0208toUCS_3021[jis - 0x3021];
		else if(jis >= 0x5021 && jis < 0x5021 + _countof(JISX0208toUCS_5021))
			return JISX0208toUCS_5021[jis - 0x5021];
		else
			return __REPLACEMENT_CHARACTER;
	}

	// UCS -> JIS X0208 ϊ
	inline ushort UCSToJISX0208(wchar_t wch) {
		if(wch >= 0x00A2 && wch < 0x00A2 + _countof(UCStoJISX0208_00A2))
			return UCStoJISX0208_00A2[wch - 0x00A2];
		else if(wch >= 0x0391 && wch < 0x0391 + _countof(UCStoJISX0208_0391))
			return UCStoJISX0208_0391[wch - 0x0391];
		else if(wch >= 0x2010 && wch < 0x2010 + _countof(UCStoJISX0208_2010))
			return UCStoJISX0208_2010[wch - 0x2010];
		else if(wch >= 0x2500 && wch < 0x2500 + _countof(UCStoJISX0208_2500))
			return UCStoJISX0208_2500[wch - 0x2500];
		else if(wch >= 0x3000 && wch < 0x3000 + _countof(UCStoJISX0208_3000))
			return UCStoJISX0208_3000[wch - 0x3000];
		else if(wch >= 0x4E00 && wch < 0x4E00 + _countof(UCStoJISX0208_4E00))
			return UCStoJISX0208_4E00[wch - 0x4E00];
		else if(wch >= 0xFF01 && wch < 0xFF01 + _countof(UCStoJISX0208_FF01))
			return UCStoJISX0208_FF01[wch - 0xFF01];
		else
			return 0x0000;
	}

	// JIS X0212 -> UCS ϊ
	inline wchar_t JISX0212ToUCS(ushort jis) {
		if(jis >= 0x222F && jis < 0x222F + _countof(JISX0212toUCS_222F))
			return JISX0212toUCS_222F[jis - 0x222F];
		else if(jis >= 0x2661 && jis < 0x2661 + _countof(JISX0212toUCS_2661))
			return JISX0212toUCS_2661[jis - 0x2661];
		else if(jis >= 0x3021 && jis < 0x3021 + _countof(JISX0212toUCS_3021))
			return JISX0212toUCS_3021[jis - 0x3021];
		else
			return __REPLACEMENT_CHARACTER;
	}

	// UCS -> JIS X0212 ϊ
	inline ushort UCSToJISX0212(wchar_t wch) {
		if(wch >= 0x007E && wch < 0x007E + _countof(UCStoJISX0212_007E))
			return UCStoJISX0212_007E[wch - 0x007E];
		else if(wch >= 0x2116 && wch < 0x2116 + _countof(UCStoJISX0212_2116))
			return UCStoJISX0212_2116[wch - 0x2116];
		else if(wch >= 0x4E02 && wch < 0x4E02 + _countof(UCStoJISX0212_4E02))
			return UCStoJISX0212_4E02[wch - 0x4E02];
		else
			return 0x0000;
	}

	// JIS X0213 1 -> UCS ϊ
	inline ulong JISX0213P1ToUCS(ushort jis) {
		if(jis >= 0x2121 && jis < 0x2121 + _countof(JISX0213P1toUCS_2121))
			return JISX0213P1toUCS_2121[jis - 0x2121];
		else if(jis >= 0x4F54 && jis < 0x4F54 + _countof(JISX0213P1toUCS_4F54))
			return JISX0213P1toUCS_4F54[jis - 0x4F54];
		else if(jis >= 0x7427 && jis < 0x7427 + _countof(JISX0213P1toUCS_7427))
			return JISX0213P1toUCS_7427[jis - 0x7427];
		else
			return JISX0208ToUCS(jis);
	}

	// JIS X0213 2 -> UCS ϊ
	inline ulong JISX0213P2ToUCS(ushort jis) {
		if(jis >= 0x2121 && jis < 0x2121 + _countof(JISX0213P2toUCS_2121))
			return JISX0213P2toUCS_2121[jis - 0x2121];
		else if(jis >= 0x2321 && jis < 0x2321 + _countof(JISX0213P2toUCS_2321))
			return JISX0213P2toUCS_2321[jis - 0x2321];
		else if(jis >= 0x2821 && jis < 0x2821 + _countof(JISX0213P2toUCS_2821))
			return JISX0213P2toUCS_2821[jis - 0x2821];
		else if(jis >= 0x2C21 && jis < 0x2C21 + _countof(JISX0213P2toUCS_2C21))
			return JISX0213P2toUCS_2C21[jis - 0x2C21];
		else if(jis >= 0x6E21 && jis < 0x6E21 + _countof(JISX0213P2toUCS_6E21))
			return JISX0213P2toUCS_6E21[jis - 0x6E21];
		else
			return 0x00000000;
	}

	// UCS -> JIS X0213 ϊ (cchEaten ͓͎ pwsz ̕Ao͎͕ϊɎgp)
	inline ushort UCSToJISX0213(const wchar_t* pwsz, ushort& cchEaten, bool& bPlane2) {
		assert(pwsz != 0 && cchEaten != 0);
		if(cjkExtBtoJis.empty()) {
			for(size_t i = 0; i < _countof(CjkExtBUcs); ++i)
				cjkExtBtoJis[CjkExtBUcs[i]] = CjkExtBJis[i];
		}

		ulong	jis = 0x0000;

		bPlane2 = false;
		if(cchEaten > 1) {
			if(pwsz[1] == 0x309A) {	// <kana> + Combining Katakana-Hiragana Semi-Voiced Sound Mark
				switch(pwsz[0]) {
				case 0x304B:	jis = 0x2477;	break;	// ka -> bidakuon nga
				case 0x304D:	jis = 0x2478;	break;	// ki -> bidakuon ngi
				case 0x304F:	jis = 0x2479;	break;	// ku -> bidakuon ngu
				case 0x3051:	jis = 0x247A;	break;	// ke -> bidakuon nge
				case 0x3053:	jis = 0x247B;	break;	// ko -> bidakuon ngo
				case 0x30AB:	jis = 0x2577;	break;	// ka -> bidakuon nga
				case 0x30AD:	jis = 0x2578;	break;	// ki -> bidakuon ngi
				case 0x30AF:	jis = 0x2579;	break;	// ku -> bidakuon ngu
				case 0x30B1:	jis = 0x257A;	break;	// ke -> bidakuon nge
				case 0x30B3:	jis = 0x257B;	break;	// ko -> bidakuon ngo
				case 0x30BB:	jis = 0x257C;	break;	// se -> ainu ce
				case 0x30C4:	jis = 0x257D;	break;	// tu -> ainu tu (tu)
				case 0x30C8:	jis = 0x257E;	break;	// to -> ainu to (tu)
				case 0x31F7:	jis = 0x2678;	break;	// small fu -> ainu p
				}
			} else if(pwsz[1] == 0x0300) {	// X + Combining Grave Accent
				switch(pwsz[0]) {
				case 0x00E6:	jis = 0x2B44;	break;	// ae
				case 0x0254:	jis = 0x2B48;	break;	// open o
				case 0x0259:	jis = 0x2B4C;	break;	// schwa
				case 0x025A:	jis = 0x2B4E;	break;	// schwa with hook
				case 0x028C:	jis = 0x2B4A;	break;	// turned v
				}
			} else if(pwsz[1] == 0x0301) {	// X + Combining Acute Accent
				switch(pwsz[0]) {
				case 0x0254:	jis = 0x2B49;	break;	// open o
				case 0x0259:	jis = 0x2B4D;	break;	// schwa
				case 0x025A:	jis = 0x2B4F;	break;	// schwa with hook
				case 0x028C:	jis = 0x2B4B;	break;	// turned v
				}
			} else if(pwsz[0] == 0x02E9) {
				if(pwsz[1] == 0x02E5)
					jis = 0x2B65;	// extra-low tone bar + extra-high tone bar -> rising symbol
				else if(pwsz[1] == ZWNJ && cchEaten > 2 && pwsz[2] == 0x02E5)
					jis = 0x2B64;	// just dependent extra-low tone bar
			} else if(pwsz[0] == 0x02E5) {
				if(pwsz[1] == 0x02E9)
					jis = 0x2B66;	// extra-high tone bar + extra-low tone bar -> falling symbol
				else if(pwsz[1] == ZWNJ && cchEaten > 2 && pwsz[2] == 0x02E9)
					jis = 0x2B60;	// just dependent extra-high tone bar
			}
			if(jis != 0x0000)
				cchEaten = 2;
		}
		if(jis == 0x0000) {
			if(pwsz[0] >= 0x00A0 && pwsz[0] < 0x00A0 + _countof(UCStoJISX0213_00A0)) {
				jis = UCStoJISX0213_00A0[pwsz[0] - 0x00A0]; cchEaten = 1;
			} else if(pwsz[0] >= 0x1E3E && pwsz[0] < 0x1E3E + _countof(UCStoJISX0213_1E3E)) {
				jis = UCStoJISX0213_1E3E[pwsz[0] - 0x1E3E]; cchEaten = 1;
			} else if(pwsz[0] >= 0x3000 && pwsz[0] < 0x3000 + _countof(UCStoJISX0213_3000)) {
				jis = UCStoJISX0213_3000[pwsz[0] - 0x3000]; cchEaten = 1;
			} else if(pwsz[0] >= 0xF91D && pwsz[0] < 0xF91D + _countof(UCStoJISX0213_F91D)) {
				jis = UCStoJISX0213_F91D[pwsz[0] - 0xF91D]; cchEaten = 1;
			} else if(pwsz[0] >= 0xF91D && pwsz[0] < 0xFE45 + _countof(UCStoJISX0213_FE45)) {
				jis = UCStoJISX0213_FE45[pwsz[0] - 0xFE45]; cchEaten = 1;
			} else if(cchEaten > 1 && IsUtf16HighSurrogate(pwsz[0]) && IsUtf16LowSurrogate(pwsz[1])) {
				const CodePoint	cp = DecodeUtf16SurrogatesToCodePoint(pwsz, cchEaten);

				cchEaten = 1;
				if(cp >= 0x020000) {
					const map<ushort, ulong>::const_iterator	it = cjkExtBtoJis.find(static_cast<ushort>(cp - 0x020000));
					if(it != cjkExtBtoJis.end()) {
						cchEaten = 2;
						jis = it->second;
					}
				}
			} else
				cchEaten = 1;
		}

		bPlane2 = jis > 0xFFFF;
		return static_cast<ushort>(jis);
	}

	// ISO-2022-JP-X -> UTF-16 ϊwp
	size_t _ConvertIso2022JpXToUtf16(CodePage cp, wchar_t* pwszDest, size_t cchDest,
			const uchar* pszSrc, size_t cchSrc, IUnconvertableCharCallback* pCallback) {
		assert(cp == CPEX_JAPANESE_ISO2022JP || cp == CPEX_JAPANESE_ISO2022JP1
			|| cp == CPEX_JAPANESE_ISO2022JP2 || IS_ISO2022JP3(cp) || IS_ISO2022JP2004(cp));

		// ܂߂邱Ƃ̂ł镶WƎwV[PX͎̒ʂBɋLq G0:
		//
		// ISO-2022-JP
		//	ASCII					ESC ( B
		//	JIS X0201:1976-Roman	ESC ( J
		//	JIS X0208:1978			ESC $ @
		//	JIS X0208:1983			ESC $ B
		//
		// ISO-2022-JP-1 (ISO-2022-JP Ɉȉǉ)
		//	JIS X0212:1990			ESC $ ( D
		//
		// ISO-2022-JP-2 (ISO-2022-JP-1 Ɉȉǉ)
		//	GB2312:1980				ESC $ A
		//	KSC5601:1987			ESC $ ( C
		//	ISO-8859-1				ESC . A		96Wɂ G2
		//	ISO-8859-7				ESC . F		96Wɂ G2
		//
		// ISO-2022-JP-3
		//	ASCII					ESC ( B
		//	JIS X0213:2000 1		ESC $ ( O
		//							ESC $ B		֎~
		//	JIS X0213:2000 2		ESC $ ( P
		//
		// ISO-2022-JP-2004
		//	ASCII					ESC ( B
		//	JIS X0213:2004 1		ESC $ ( Q
		//							ESC $ B		֎~
		//	JIS X0213:2004 2		ESC $ ( P
		//	JIS X0213:2000 1		ESC $ ( O	֎~
		//
		// JIS X0213 1ʂ̎wł JIS X0208 w邽߂ɎgpĂ ESC $ B p邱ƂF߂Ă邪A
		// ۊ̕ύXɂǍ݊V[PX JIS X0213 1ʂwꍇAJIS X0208
		// Ɋ܂܂镶̒Ŏgp֎~Ă镶
		//
		// JIS X0213:2004 (Ǖ1) ł͂̋֎~ǉꂽB̕ ISO-2022-JP-2004 ɂ
		// ESC $ ( O  JIS X0213 1ʂwꍇɂgpłȂ
		//
		// Ascension ̐A߂͎̒ʂ:
		//	- JIS X0201-Kana g邪AAscension ł͂̕WgȂB
		//	  x_gϊ\珜OĂAgȂ
		//	- JIS X0208 ͔NʂAS JIS X0208:1997 g
		//	- _ł ISO-2022-JP-2 ̒Ɗ؍ꕶW̕ϊɂ Windows ̕ϊ\g
		//	- JIS X0213 1ʂ UCS ւ̕ϊɂāAwV[PX͔NxʂȂ
		//	- ISO-2022-JP-*-Strict AISO-2022-JP-*-Compatible ł JIS X0208 ŕ\\ȕ
		//	  "ESC $ B" Ŏw
		//	- ISO-2022-JP-*-Compatible ł͋֎~ "ESC $ B" Ŏw
		//	- ISO-2022-JP-*-Strict  UCS ւ̕ϊł͋֎~lȂ
		//	- ISO-2022-JP-3 ñGR[fBOł JIS X0213 1ʂ̎w ESC $ ( O gB
		//	  ISO-2022-JP-2004 nGR[fBOƂ̍ق͂ꂾŁAISO-2022-JP-3
		//	  nGR[fBOłǕ1ŒǉꂽgƂł (UCS ւ̕ϊɂĂ͋ʂ)
		//  - ISO-2022-JP-2004-Compatible ̌݊ ISO-2022-JP ɑ΂̂łAISO-2022-JP-3 ɑ΂̂ł͂Ȃ

		size_t				iSrc = 0, iDest = 0;
		Iso2022JpCharset_G0	g0 = ascii;
		Iso2022JpCharset_G2	g2 = undesignated;

		CEncoderFactory&	encoders = CEncoderFactory::GetInstance();
		auto_ptr<CEncoder>	pIso88591Encoder, pIso88597Encoder;

		while(iSrc < cchSrc && iDest < cchDest) {
			if(pszSrc[iSrc] == ESC && cchSrc - iSrc >= 3) {	// expect esc. seq.
				if(memcmp(pszSrc + iSrc + 1, "(B", 2) == 0) {
					g0 = ascii; iSrc += 3; continue;
				} else if(memcmp(pszSrc + iSrc + 1, "(J", 2) == 0) {
					g0 = jisx0201_roman; iSrc += 3; continue;
//				} else if(memcmp(pszSrc + iSrc + 1, "(I", 2) == 0) {
//					g0 = jisx0201_kana; iSrc += 3; continue;
				} else if(cp != CPEX_JAPANESE_ISO2022JP2004 && cp != CPEX_JAPANESE_ISO2022JP3
						&& (memcmp(pszSrc + iSrc + 1, "$@", 2) == 0 || memcmp(pszSrc + iSrc + 1, "$B", 2) == 0)) {
					g0 = jisx0208; iSrc += 3; continue;
				} else if((cp == CPEX_JAPANESE_ISO2022JP1 || cp == CPEX_JAPANESE_ISO2022JP2)
						&& cchSrc - iSrc >= 4 && memcmp(pszSrc + iSrc + 1, "$(D", 3) == 0) {
					g0 = jisx0212; iSrc += 4; continue;
				} else if(cp == CPEX_JAPANESE_ISO2022JP2) {
					if(memcmp(pszSrc + iSrc + 1, "$A", 2) == 0 && encoders.IsValidCodePage(936)) {
						g0 = gb2312; iSrc += 3; continue;
					} else if(cchSrc - iSrc >= 4
							&& memcmp(pszSrc + iSrc + 1, "$(C", 3) == 0 && encoders.IsValidCodePage(949)) {
						g0 = ksc5601; iSrc += 4; continue;
					} else if(memcmp(pszSrc + iSrc + 1, ".A", 2) == 0) {
						g2 = iso8859_1; iSrc += 3; continue;
					} else if(memcmp(pszSrc + iSrc + 1, ".F", 2) == 0) {
						g2 = iso8859_7; iSrc += 3; continue;
					}
				} else if((IS_ISO2022JP3(cp) || IS_ISO2022JP2004(cp)) && cchSrc - iSrc >= 4) {
					if(memcmp(pszSrc + iSrc + 1, "$(O", 3) == 0 || memcmp(pszSrc + iSrc + 1, "$(Q", 3) == 0) {
						g0 = jisx0213p1; iSrc += 4; continue;
					} else if(memcmp(pszSrc + iSrc + 1, "$(P", 3) == 0) {
						g0 = jisx0213p2; iSrc += 4; continue;
					}
				}
			}

			if((pszSrc[iSrc] <= 0x20 && pszSrc[iSrc] != ESC) || (pszSrc[iSrc] >= 0x80 && pszSrc[iSrc] < 0xA0)) {	// C0 AC1
				if(pszSrc[iSrc] == 0x0A || pszSrc[iSrc] == 0x0D) {
					g0 = ascii;
					g2 = undesignated;
				}
				pwszDest[iDest++] = pszSrc[iSrc++];	// SI ASO A(1oCg) SS2 ASS3 ͖
			} else if(cchSrc - iSrc > 1 && cchDest - iDest > 1
					&& memcmp(pszSrc + iSrc, "\x1BN", 2) == 0) {	// SS2
				iSrc += 2;
				if(cchSrc > iSrc) {
					const uchar	chAnsi = pszSrc[iSrc] | 0x80;
					if(g2 == iso8859_1) {	// ISO-8859-1
						if(pIso88591Encoder.get() == 0)
							pIso88591Encoder.reset(encoders.CreateEncoder(CPEX_ISO8859_1));
						const size_t	cchConverted = pIso88591Encoder->ConvertToUnicode(
											pwszDest + iDest, cchDest - iDest,
											reinterpret_cast<const uchar*>(&chAnsi), 1, pCallback);
						if(cchConverted == 0)
							return 0;
						++iSrc;
						iDest += cchConverted;
					} else if(g2 == iso8859_7) {	// ISO-8859-7
						if(pIso88597Encoder.get() == 0)
							pIso88597Encoder.reset(encoders.CreateEncoder(CPEX_ISO8859_7));
						const size_t	cchConverted = pIso88597Encoder->ConvertToUnicode(
											pwszDest + iDest, cchDest - iDest,
											reinterpret_cast<const uchar*>(&chAnsi), 1, pCallback);
						if(cchConverted == 0)
							return 0;
						++iSrc;
						iDest += cchConverted;
					} else {
						wchar_t	ucs;
						CONFIRM_ILLEGAL_CHAR(ucs);
						pwszDest[iDest++] = ucs;
						++iSrc;
					}
				}
			} else if(g0 == jisx0201_roman) {	// JIS X0201-Roman
				wchar_t	ucs = JISX0201RomanToUCS(pszSrc[iSrc]);
				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				pwszDest[iDest++] = ucs;
				++iSrc;
/*			} else if(g0 == jisx0201_kana) {	// JIS X0201-Kana
				wchar_t	ucs;
				if(pszSrc[iSrc] >= 0x80) {
					CONFIRM_ILLEGAL_CHAR(ucs);
				} else {
					ucs = _JISX0201ToUCS(pszSrc[iSrc] + 0x80);
					if(ucs == 0)
						CONFIRM_ILLEGAL_CHAR(ucs);
				}
				pwszDest[iDest++] = ucs;
				++iSrc;
*/			} else if(g0 == ascii || cchSrc - iSrc == 1 || cchDest - iDest == 1) {	// ASCII or illegal char
				uchar	jis = pszSrc[iSrc];
				if(jis >= 0x80)
					CONFIRM_ILLEGAL_CHAR(jis);
				pwszDest[iDest++] = jis;
				++iSrc;
			} else if(g0 == jisx0208) {	// JIS X0208:1978 or :1983
				const ushort	jis = (pszSrc[iSrc] << 8) | pszSrc[iSrc + 1];
				wchar_t			ucs = JISX0208ToUCS(jis);

				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				pwszDest[iDest++] = ucs;
				iSrc += 2;
			} else if(g0 == jisx0212) {	// JIS X0212:1990
				const ushort	jis = (pszSrc[iSrc] << 8) | pszSrc[iSrc + 1];
				wchar_t			ucs = JISX0212ToUCS(jis);

				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				pwszDest[iDest++] = ucs;
				iSrc += 2;
			} else if(g0 == gb2312 || g0 == ksc5601) {	// GB2312:1980 or KSC5601:1987
				wchar_t		ucs;	// for error
				char		sz[2] = {pszSrc[iSrc] | 0x80, pszSrc[iSrc + 1] | 0x80};
				const int	cch = ::MultiByteToWideChar(
									(g0 == gb2312) ? 936 : 949, MB_PRECOMPOSED, sz, 2, pwszDest + iDest, 2);
				if(cch == 0) {
					CONFIRM_ILLEGAL_CHAR(ucs);
					pwszDest[iDest++] = ucs;
				} else
					iDest += cch;
				iSrc += 2;
			} else if(g0 == jisx0213p1 || g0 == jisx0213p2) {	// JIS X0213:2004 or 2000
				const ushort	jis = (pszSrc[iSrc] << 8) | pszSrc[iSrc + 1];
				ulong			ucs = (g0 == jisx0213p1) ? JISX0213P1ToUCS(jis) : JISX0213P2ToUCS(jis);

				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				if(ucs > 0xFFFF) {
					if(cchDest - iDest == 2)
						return iDest;
					if(ucs > 0x0010FFFF) {	// 2 UCS characters
						pwszDest[iDest++] = static_cast<wchar_t>(ucs >> 16);
						pwszDest[iDest++] = static_cast<wchar_t>(ucs);
					} else {
						EncodeCodePointToUtf16Surrogates(ucs, pwszDest + iDest);
						iDest += 2;
					}
				} else {
					if(iDest > 0) {
						if((pwszDest[iDest - 1] == L'\x02E9' && ucs == 0x02E5)
								|| (pwszDest[iDest - 1] == L'\x02E5' && ucs == 0x02E9)) {
							if(iDest + 1 >= cchDest)
								return iDest;
							pwszDest[iDest++] = ZWNJ;
						}
					}
					pwszDest[iDest++] = static_cast<wchar_t>(ucs);
				}
				iSrc += 2;
			}
		}
		return iDest;
	}

	// UTF-16 -> ISO-2022-JP-X ϊwp
	size_t _ConvertUtf16ToIso2022JpX(CodePage cp, uchar* pszDest, size_t cchDest,
			const wchar_t* pwszSrc, size_t cchSrc, IUnconvertableCharCallback* pCallback) {
		assert(cp == CPEX_JAPANESE_ISO2022JP || cp == CPEX_JAPANESE_ISO2022JP1
			|| cp == CPEX_JAPANESE_ISO2022JP2 || IS_ISO2022JP3(cp) || IS_ISO2022JP2004(cp));

		size_t				iSrc = 0, iDest = 0;
		int					charset = ascii;
		Iso2022JpCharset_G0	g0 = ascii;
		Iso2022JpCharset_G2	g2 = undesignated;

		CEncoderFactory&	encoders = CEncoderFactory::GetInstance();
		auto_ptr<CEncoder>	pIso88591Encoder((cp == CPEX_JAPANESE_ISO2022JP2) ? encoders.CreateEncoder(CPEX_ISO8859_1) : 0);
		auto_ptr<CEncoder>	pIso88597Encoder((cp == CPEX_JAPANESE_ISO2022JP2) ? encoders.CreateEncoder(CPEX_ISO8859_7) : 0);

		while(iSrc < cchSrc && iDest < cchDest) {
			const wchar_t	ucs = pwszSrc[iSrc];
			ushort			jis, cchEatenUtf16 = 1;
			uchar			szMultiBytes[2];

			if(ucs < 0x80) {
				jis = ucs;
				szMultiBytes[0] = static_cast<uchar>(ucs);
				szMultiBytes[1] = 0;
				charset = ascii;
			} else if(jis = UCSToJISX0201Roman(ucs) && jis < 0x80)
				charset = /*(jis < 0x80) ?*/ jisx0201_roman /*: jisx0201_kana*/;
			else if(IS_ISO2022JP3(cp) || IS_ISO2022JP2004(cp)) {
				bool	bIsPlane2;

				cchEatenUtf16 = cchSrc - iSrc;
				jis = UCSToJISX0213(pwszSrc + iSrc, cchEatenUtf16, bIsPlane2);

				if(jis != 0) {
					charset = undesignated;
					if(!bIsPlane2) {	// JIS X0208 ݊V[PXg
						if((cp == CPEX_JAPANESE_ISO2022JP3_COMPATIBLE
								|| cp == CPEX_JAPANESE_ISO2022JP2004_COMPATIBLE) && UCSToJISX0208(ucs) != 0)
							charset = jisx0208;
						else if((cp == CPEX_JAPANESE_ISO2022JP3_STRICT
								|| cp == CPEX_JAPANESE_ISO2022JP2004_STRICT)
								&& !IsIso2022Jp3ProhibitedIdeograph(jis)
								&& !IsIso2022Jp2004ProhibitedIdeograph(jis))
							charset = jisx0208;
					}
					if(charset == undesignated)
						charset = bIsPlane2 ? jisx0213p2 : jisx0213p1;
				} else {
					CONFIRM_ILLEGAL_CHAR(jis);
					charset = ascii;
				}
			} else if(jis = UCSToJISX0208(ucs))
				charset = jisx0208;
			else if((cp == CPEX_JAPANESE_ISO2022JP1 || cp == CPEX_JAPANESE_ISO2022JP2)
					&& toBoolean(jis = UCSToJISX0212(ucs)))
				charset = jisx0212;
			else if(cp == CPEX_JAPANESE_ISO2022JP2
					&& encoders.IsValidCodePage(936)
					&& ::WideCharToMultiByte(936, 0, pwszSrc + iSrc, 1, reinterpret_cast<char*>(szMultiBytes), 2, 0, 0) != 0)
				charset = gb2312;
			else if(cp == CPEX_JAPANESE_ISO2022JP2
					&& encoders.IsValidCodePage(949)
					&& ::WideCharToMultiByte(949, 0, pwszSrc + iSrc, 1, reinterpret_cast<char*>(szMultiBytes), 2, 0, 0) != 0)
				charset = ksc5601;
			else if(cp == CPEX_JAPANESE_ISO2022JP2
					&& pIso88591Encoder->ConvertFromUnicode(szMultiBytes, 2, pwszSrc + iSrc, 1, 0) != 0)
				charset = iso8859_1;
			else if(cp == CPEX_JAPANESE_ISO2022JP2
					&& pIso88597Encoder->ConvertFromUnicode(szMultiBytes, 2, pwszSrc + iSrc, 1, 0) != 0)
				charset = iso8859_7;
			else {
				CONFIRM_ILLEGAL_CHAR(jis);
				charset = ascii;
			}

#define DESIGNATE_TO_G0(esc_sequence, cch_esc)						\
	if(g0 != charset) {												\
		if(cchDest < cch_esc + 1 || iDest > cchDest - cch_esc - 1)	\
			break;													\
		memcpy(pszDest + iDest, esc_sequence, cch_esc);				\
		iDest += cch_esc;											\
		g0 = static_cast<Iso2022JpCharset_G0>(charset);				\
	}
#define DESIGNATE_TO_G2(esc_sequence, cch_esc)						\
	if(g2 != charset) {												\
		if(cchDest < cch_esc + 3 || iDest > cchDest - cch_esc - 3)	\
			break;													\
		memcpy(pszDest + iDest, esc_sequence, cch_esc);				\
		iDest += cch_esc;											\
		g2 = static_cast<Iso2022JpCharset_G2>(charset);				\
	}

			if(charset == ascii) {	// ASCII
				DESIGNATE_TO_G0("\x1B(B", 3);
				pszDest[iDest++] = static_cast<uchar>(jis);
			} else if(charset == jisx0201_roman) {	// JIS X0201-Roman
				DESIGNATE_TO_G0("\x1B(J", 3);
				pszDest[iDest++] = static_cast<uchar>(jis);
//			} else if(charset == jisx0201_kana) {	// JIS X0201-Kana
//				DESIGNATE_TO_G0("\x1B(I", 3);
//				pszDest[iDest++] = static_cast<uchar>(jis & 0x7F);
			} else if(charset == jisx0208) {	// JIS X0208:1997 (:1990)
				DESIGNATE_TO_G0("\x1B$B", 3);
				pszDest[iDest++] = static_cast<uchar>(jis >> 8);
				pszDest[iDest++] = static_cast<uchar>(jis);
			} else if(charset == jisx0212) {	// JIS X0212:1990
				DESIGNATE_TO_G0("\x1B$(D", 4);
				pszDest[iDest++] = static_cast<uchar>(jis >> 8);
				pszDest[iDest++] = static_cast<uchar>(jis);
			} else if(charset == jisx0213p1) {	// JIS X0213:2004 plane-1 or :2000 plane-1
				DESIGNATE_TO_G0(IS_ISO2022JP2004(cp) ? "\x1B$(Q" : "\x1B$(O", 4);
				pszDest[iDest++] = static_cast<uchar>(jis >> 8);
				pszDest[iDest++] = static_cast<uchar>(jis);
			} else if(charset == jisx0213p2) {	// JIS X0213:2004 (:2000) plane-2
				DESIGNATE_TO_G0("\x1B$(P", 4);
				pszDest[iDest++] = static_cast<uchar>(jis >> 8);
				pszDest[iDest++] = static_cast<uchar>(jis);
			} else if(charset == gb2312) {	// GB2312:1980
				DESIGNATE_TO_G0("\x1B$A", 3);
				pszDest[iDest++] = static_cast<uchar>(szMultiBytes[0] & 0x7F);
				if(szMultiBytes[1] != 0)
					pszDest[iDest++] = static_cast<uchar>(szMultiBytes[1] & 0x7F);
			} else if(charset == ksc5601) {	// KSC5601:1987
				DESIGNATE_TO_G0("\x1B$(C", 4);
				pszDest[iDest++] = static_cast<uchar>(szMultiBytes[0] & 0x7F);
				if(szMultiBytes[1] != 0)
					pszDest[iDest++] = static_cast<uchar>(szMultiBytes[1] & 0x7F);
			} else if(charset == iso8859_1) {	// ISO-8859-1
				DESIGNATE_TO_G2("\x1B.A", 3);
				if(iDest + 3 >= cchDest)
					break;
				pszDest[iDest++] = 0x1B;	// SS2
				pszDest[iDest++] = 'N';
				pszDest[iDest++] = static_cast<uchar>(szMultiBytes[0]);
			} else if(charset == iso8859_7) {	// ISO-8859-7
				DESIGNATE_TO_G2("\x1B.F", 3);
				if(iDest + 3 >= cchDest)
					break;
				pszDest[iDest++] = 0x1B;	// SS2
				pszDest[iDest++] = 'N';
				pszDest[iDest++] = static_cast<uchar>(szMultiBytes[0]);
			}
			iSrc += cchEatenUtf16;
		}

		// G0  ASCII ɖ߂ďI
		if(g0 != ascii && cchDest > 3 && iDest <= cchDest - 3) {
			memcpy(pszDest + iDest, "\x1B(B", 3);
			iDest += 3;
		}
		return iDest;

#undef DESIGNATE_TO_G0
#undef DESIGNATE_TO_G2
	}

	// JIS X0208 or JIS X0213 <-> Vtg JIS 2oCg̕ϊ
	inline void ConvertX0208ToShiftJisDbcs(ushort jis, uchar* pszShiftJis) {
		assert(pszShiftJis != 0);
		const uchar	jk = (jis - 0x2020) >> 8;		// 
		const uchar	jt = (jis - 0x2020) & 0x00FF;	// _

		assert(jk >= 1 && jk <= 94 && jt >= 1 && jt <= 94);
		pszShiftJis[0] = (jk - 1) / 2 + ((jk <= 62) ? 0x81 : 0xC1);
		if(jk % 2 == 0)	pszShiftJis[1] = jt + 0x9E;
		else			pszShiftJis[1] = jt + ((jt <= 63) ? 0x3F : 0x40);
	}
	inline ushort ConvertShiftJisDbcsToX0208(const uchar* pszShiftJis) {
		assert(pszShiftJis != 0);
		uchar	jk, jt;

		if(pszShiftJis[0] >= 0x81 && pszShiftJis[0] <= 0x9F)	// : 01-62
			jk = (pszShiftJis[0] - 0x81) * 2 + ((pszShiftJis[1] > 0x9E) ? 2 : 1);	// < leadbyte = (jk - 1) / 2 + 0x81
		else	// : 63-94
			jk = (pszShiftJis[0] - 0xC1) * 2 + ((pszShiftJis[1] > 0x9E) ? 2 : 1);	// < leadbyte = (jk - 1) / 2 + 0xC1
		if(jk % 2 == 0)
			jt = pszShiftJis[1] - 0x9E;	// < trailbyte = jt + 0x9E
		else if(pszShiftJis[1] <= 0x3F + 63)	// _: 01-63
			jt = pszShiftJis[1] - 0x3F;	// < trailbyte = jt + 0x3F
		else	// _: 64-94
			jt = pszShiftJis[1] - 0x40;	// < trailbyte = jt + 0x40
		return ((jk << 8) | jt) + 0x2020;
	}
	inline ushort ConvertShiftJisDbcsToX0213(const uchar* pszShiftJis, bool& bIsPlane2) {
		uchar		jk, jt;
		const bool	bKuIsEven = pszShiftJis[1] > 0x9E;

		bIsPlane2 = pszShiftJis[0] >= 0xF0;
		if(pszShiftJis[0] >= 0x81 && pszShiftJis[0] <= 0x9F)
			jk = pszShiftJis[0] * 2 - 0x101 + (bKuIsEven ? 1 : 0);
		else if(pszShiftJis[0] >= 0xE0 && pszShiftJis[0] <= 0xEF)
			jk = pszShiftJis[0] * 2 - 0x181 + (bKuIsEven ? 1 : 0);
		else if((pszShiftJis[0] == 0xF4 && bKuIsEven) || (pszShiftJis[0] >= 0xF5 && pszShiftJis[0] <= 0xFC))
			jk = pszShiftJis[0] * 2 - 0x19B + (bKuIsEven ? 1 : 0);
		else if((pszShiftJis[0] >= 0xF0 && pszShiftJis[0] <= 0xF3) || (pszShiftJis[0] == 0xF4 && !bKuIsEven)) {
			switch(pszShiftJis[0]) {
			case 0xF0:	jk = bKuIsEven ? 8 : 1; break;
			case 0xF1:	jk = bKuIsEven ? 4 : 3; break;
			case 0xF2:	jk = bKuIsEven ? 12 : 5; break;
			case 0xF3:	jk = bKuIsEven ? 14 : 13; break;
			case 0xF4:	jk = 15; break;
			}
		}
		if(jk % 2 == 0)
			jt = pszShiftJis[1] - 0x9E;	// < trailbyte = jt + 0x9E
		else if(pszShiftJis[1] <= 0x3F + 63)	// _: 01-63
			jt = pszShiftJis[1] - 0x3F;	// < trailbyte = jt + 0x3F
		else	// _: 64-94
			jt = pszShiftJis[1] - 0x40;	// < trailbyte = jt + 0x40
		return ((jk << 8) | jt) + 0x2020;
	}

	// ʃwp
	inline size_t IsShiftJis(const uchar* psz, size_t cch, CodePage& cp, bool& bFoundKana) {
		cp = 932;
		bFoundKana = false;
		for(size_t i = 0; i < cch; ++i) {
			const uchar	ch = psz[i];

			if(ch == 0x1B)	// ESC
				return i;
			else if(ch < 0x80)	// ASCII
				continue;
			else if(ch >= 0xA1 && ch <= 0xDF) {	// JIS X0201 
				bFoundKana = true;
				continue;
			} else if(i < cch - 1) {	// 2oCg?
				if(ch < 0x81 || ch > 0xFC || (ch > 0x9F && ch < 0xE0))
					return i;
				const uchar	chTrail = psz[i + 1];
				if(chTrail < 0x40 || chTrail > 0xFC || chTrail == 0x7F)
					return i;

				bool	bPlane2;
				if(cp == 932) {
					if(JISX0208ToUCS(ConvertShiftJisDbcsToX0208(psz + i)) == __REPLACEMENT_CHARACTER) {
						const ushort	jis = ConvertShiftJisDbcsToX0213(psz + i, bPlane2);
						if(!bPlane2 && JISX0213P1ToUCS(jis) == __REPLACEMENT_CHARACTER)
							return i;
						cp = CPEX_JAPANESE_SHIFTJIS2004;
					}
				} else {	// Shift_JIS-2004
					if(ConvertShiftJisDbcsToX0213(psz + i, bPlane2) == 0)
						return i;
				}
				++i;
				continue;
			}
			return i;
		}
		return cch;
	}
	inline size_t IsEucJp(const uchar* psz, size_t cch, CodePage& cp, bool& bFoundKana) {
		cp = 51932;
		bFoundKana = false;
		for(size_t i = 0; i < cch; ++i) {
			const uchar	ch = psz[i];

			if(ch == 0x1B)	// ESC
				return i;
			else if(ch < 0x80)	// ASCII
				continue;
			else if(ch == 0x8E) {	// SS2 -> JIS X0201 
				if(i + 1 >= cch || psz[i + 1] < 0xA0 || psz[i + 1] > 0xE0)
					return i;
				bFoundKana = true;
				++i;
			} else if(ch == 0x8F) {	// SS3 -> JIS X0212 or JIS X0213 plane2
				if(i + 2 >= cch)
					return i;
				ushort	jis = psz[i + 1] << 8 | psz[i + 2];
				if(jis < 0x8080)
					return i;
				jis -= 0x8080;
				if(JISX0212ToUCS(jis) != __REPLACEMENT_CHARACTER) {
					if(cp == CPEX_JAPANESE_EUCJIS2004)
						return i;
					cp = CPEX_JAPANESE_EUC;
				} else if(JISX0213P2ToUCS(jis) != __REPLACEMENT_CHARACTER) {
					if(cp == CPEX_JAPANESE_EUC)
						return i;
					cp = CPEX_JAPANESE_EUCJIS2004;
				} else
					return i;
				i += 2;
			} else if(i < cch - 1) {	// 2oCg?
				ushort	jis = ch << 8 | psz[i + 1];
				if(jis > 0x8080) {
					jis -= 0x8080;
					if(JISX0208ToUCS(jis) != __REPLACEMENT_CHARACTER)
						++i;
					else if(JISX0213P1ToUCS(jis) != __REPLACEMENT_CHARACTER) {
						if(cp == CPEX_JAPANESE_EUC)
							return i;
						cp = CPEX_JAPANESE_EUCJIS2004;
						++i;
					} else
						return i;
					continue;
				}
				return i;
			}
		}
		return cch;
	}
	inline size_t IsIso2022Jp(const uchar* psz, size_t cch, CodePage& cp, bool& bFoundKana) {
		// X0201, X0208, X0212, JP2, JP3-plane1, JP2004-plane1, X0213-plane2
		size_t	iFoundEsc[7] = {0, 0, 0, 0, 0, 0, 0};

		cp = 50221;
		bFoundKana = false;
		for(size_t i = 0; i < cch; ++i) {
			const uchar	ch = static_cast<uchar>(psz[i]);

			if(ch >= 0x80)	// 8bit
				return i;
			else if(ch == 0x1B) {
				if(i + 2 >= cch)
					return i;
				if(memcmp(psz + i + 1, "(J", 2) == 0 || memcmp(psz + i + 1, "(I", 2) == 0) {	// JIS X0201
					iFoundEsc[0] = i + 3; i += 2; bFoundKana = true;
				} else if(memcmp(psz + i + 1, "$@", 2) == 0 || memcmp(psz + i + 1, "$B", 2) == 0) {	// JIS X0208
					iFoundEsc[1] = i + 3; i += 2;
				} else if(memcmp(psz + i + 1, "$A", 2) == 0		// GB2312
						|| memcmp(psz + i + 1, ".A", 2) == 0	// ISO-8859-1
						|| memcmp(psz + i + 1, ".F", 2) == 0) {	// ISO-8859-7
					iFoundEsc[3] = i + 3; i += 2;
				} else if(i + 3 < cch) {
					if(memcmp(psz + i + 1, "$(D", 3) == 0) {	// JIS X0212
						iFoundEsc[2] = i + 4; i += 3;
					} else if(memcmp(psz + i + 1, "$(C", 3) == 0) {	// KSC5601
						iFoundEsc[3] = i + 4; i += 3;
					} else if(memcmp(psz + i + 1, "$(O", 3) == 0) {	// JIS X0213:2000 plane1
						iFoundEsc[4] = i + 4; i += 3;
					} else if(memcmp(psz + i + 1, "$(Q", 3) == 0) {	// JIS X0213:2004 plane1
						iFoundEsc[5] = i + 4; i += 3;
					} else if(memcmp(psz + i + 1, "$(P", 3) == 0) {	// JIS X0213 plane2
						iFoundEsc[6] = i + 4; i += 3;
					}
				} else
					return i;
			}
		}

		if(iFoundEsc[2] > 0 || iFoundEsc[3] > 0) {
//			if(iFoundEsc[4] > 0 || iFoundEsc[5] > 0 || iFoundEsc[6] > 0)
//				cp = CPEX_MULTILINGUAL_ISO2022_7BIT;
//			else
				cp = CPEX_JAPANESE_ISO2022JP2;
		} else if(iFoundEsc[5] > 0) {
//			if(iFoundEsc[2] > 0 || iFoundEsc[3] > 0)
//				cp = CPEX_MULTILINGUAL_ISO2022_7BIT;
//			else
			if(iFoundEsc[1] > 0)	cp = CPEX_JAPANESE_ISO2022JP2004_STRICT;
			else					cp = CPEX_JAPANESE_ISO2022JP2004;
		} else if(iFoundEsc[4] > 0) {
//			if(iFoundEsc[2] > 0 || iFoundEsc[3] > 0)
//				cp = CPEX_MULTILINGUAL_ISO2022_7BIT;
//			else
			if(iFoundEsc[1] > 0)	cp = CPEX_JAPANESE_ISO2022JP3_STRICT;
			else					cp = CPEX_JAPANESE_ISO2022JP3;
		} else if(iFoundEsc[6] > 0) {
//			if(iFoundEsc[2] > 0 || iFoundEsc[3] > 0)
//				cp = CPEX_MULTILINGUAL_ISO2022_7BIT;
//			else
				cp = CPEX_JAPANESE_ISO2022JP2004;
		} else
			cp = 50221;
		return cch;
	}

	void DetectCodePage_Japanese(const uchar* psz, size_t cch, CodePage& cpResult, size_t& cchConvertable) {
		// ܂ Unicode 𒲂ׂ
		if(CEncoderFactory::CodePageDetector unicodeDetector = CEncoderFactory::GetInstance().GetUnicodeDetector()) {
			unicodeDetector(psz, cch, cpResult, cchConvertable);
			if(cch == cchConvertable)
				return;
		} else
			cchConvertable = 0;

		size_t	cchConverted;
		bool	bFoundKana;

		CodePage	sjisCodePage;
		cchConverted = IsShiftJis(psz, cch, sjisCodePage, bFoundKana);
		if(cchConverted > cchConvertable) {
			cpResult = sjisCodePage;
			cchConvertable = cchConverted;
		}
		if(cchConverted == cch && !bFoundKana) {
			cchConvertable = cch;
			return;
		}

		CodePage	eucCodePage;
		cchConverted = IsEucJp(psz, cch, eucCodePage, bFoundKana);
		if(cchConverted >= cchConvertable && !bFoundKana) {
			cpResult = eucCodePage;
			cchConvertable = cchConverted;
		}
		if(cchConverted == cch && !bFoundKana)
			return;

		CodePage	cpIso2022Jp;
		cchConverted = IsIso2022Jp(psz, cch, cpIso2022Jp, bFoundKana);
		if(cchConverted >= cchConvertable && !bFoundKana) {
			cpResult = cpIso2022Jp;
			cchConvertable = cchConverted;
		}
	}
} // namespace `anonymous'


// { (Vtg JIS) ///////////////////////////////////////////////////////////////

size_t CEncoder_Japanese_ShiftJis::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	while(iSrc < cchSrc && iDest < cchDest) {
		if(pwszSrc[iSrc] < 0x80)
			pszDest[iDest++] = static_cast<uchar>(pwszSrc[iSrc]);
		else {
			ushort	jis = UCSToJISX0208(pwszSrc[iSrc]);
			if(jis == 0) {
				if(const uchar kana = UCSToJISX0201Kana(pwszSrc[iSrc]))
					pszDest[iDest++] = kana;
				else
					CONFIRM_ILLEGAL_CHAR(pszDest[iDest++]);
			} else if(iDest + 1 < cchDest) {
				ConvertX0208ToShiftJisDbcs(jis, reinterpret_cast<uchar*>(pszDest + iDest));
				iDest += 2;
			} else
				return iDest;
		}
		++iSrc;
	}
	return iDest;
}

size_t CEncoder_Japanese_ShiftJis::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	while(iSrc < cchSrc && iDest < cchDest) {
		if(pszSrc[iSrc] < 0x80)	// ascii
			pwszDest[iDest++] = pszSrc[iSrc++];
		else if(pszSrc[iSrc] >= 0xA1 && pszSrc[iSrc] <= 0xDF)	// kana
			pwszDest[iDest++] = JISX0201KanaToUCS(pszSrc[iSrc++]);
		else if(pszSrc[iSrc] == 0xA0) {	// illegal byte
			CONFIRM_ILLEGAL_CHAR(pwszDest[iDest]);
			++iDest;
		} else {	// dbcs lead byte
			const uchar	nTrailByte = pszSrc[iSrc + 1];
			if(iSrc < cchSrc - 1
					&& (nTrailByte >= 0x40 && nTrailByte <= 0xFC && nTrailByte != 0x7F)) {	// double-byte
				wchar_t	ucs = JISX0208ToUCS(ConvertShiftJisDbcsToX0208(pszSrc + iSrc));
				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				pwszDest[iDest++] = ucs;
				iSrc += 2;
			} else {	// illegal couple
				CONFIRM_ILLEGAL_CHAR(pwszDest[iDest]);
				++iDest;
			}
		}
	}
	return iDest;
}


// { (Shift_JIS-2004) /////////////////////////////////////////////////

size_t CEncoder_Japanese_ShiftJis2004::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	ushort	cchUtf16;
	bool	bIsPlane2;
	while(iSrc < cchSrc && iDest < cchDest) {
		cchUtf16 = (pwszSrc[iSrc] > 0x007F) ? static_cast<ushort>(cchSrc - iSrc) : 1;

		ushort	jis = (pwszSrc[iSrc] > 0x007F) ? UCSToJISX0213(pwszSrc + iSrc, cchUtf16, bIsPlane2) : pwszSrc[iSrc];
		if(jis == 0 && (jis = UCSToJISX0201Kana(pwszSrc[iSrc])) == 0)
			CONFIRM_ILLEGAL_CHAR(jis);
		if(jis < 0x0100)	// ascii or kana
			pszDest[iDest++] = static_cast<char>(jis);
		else if(iDest + 1 < cchDest) {
			const uchar	jk = (jis - 0x2020) >> 8;		// 
			const uchar	jt = (jis - 0x2020) & 0x00FF;	// _

			assert(jk >= 1 && jk <= 94 && jt >= 1 && jt <= 94);
			if(!bIsPlane2)	// 1
				pszDest[iDest++] = (jk + ((jk <= 62) ? 0x101 : 0x181)) / 2;
			else	// 2
				pszDest[iDest++] = (jk >= 78) ? ((jk + 0x19B) / 2) : ((jk + 0x1DF) / 2 - jk / 8 * 3);
			if(jk % 2 == 0)	pszDest[iDest++] = jt + 0x9E;
			else			pszDest[iDest++] = jt + ((jt <= 63) ? 0x3F : 0x40);
		}
		iSrc += cchUtf16;
	}
	return iDest;
}

size_t CEncoder_Japanese_ShiftJis2004::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	while(iSrc < cchSrc && iDest < cchDest) {
		if(pszSrc[iSrc] < 0x80)	// ascii
			pwszDest[iDest++] = pszSrc[iSrc++];
		else if(pszSrc[iSrc] >= 0xA1 && pszSrc[iSrc] <= 0xDF)	// kana
			pwszDest[iDest++] = JISX0201KanaToUCS(pszSrc[iSrc++]);
		else if(pszSrc[iSrc] == 0xA0) {	// illegal byte
			CONFIRM_ILLEGAL_CHAR(pwszDest[iDest]);
			++iDest;
		} else {
			const uchar	nTrailByte = pszSrc[iSrc + 1];
			if(iSrc < cchSrc - 1
					&& (nTrailByte >= 0x40 && nTrailByte <= 0xFC && nTrailByte != 0x7F)) {	// double-byte
				bool			bPlane2;
				const ushort	jis = ConvertShiftJisDbcsToX0213(pszSrc + iSrc, bPlane2);
				ulong			ucs = !bPlane2 ? JISX0213P1ToUCS(jis) : JISX0213P2ToUCS(jis);

				if(ucs == __REPLACEMENT_CHARACTER)
					CONFIRM_ILLEGAL_CHAR(ucs);
				if(ucs > 0x0010FFFF) {	// 2R[h|Cg
					if(iDest + 1 >= cchDest)
						return iDest;
					pwszDest[iDest++] = static_cast<wchar_t>(ucs >> 16);
					pwszDest[iDest++] = static_cast<wchar_t>(ucs);
				} else if(ucs >= 0x00010000) {	//  BMP
					if(iDest + 1 >= cchDest)
						return iDest;
					EncodeCodePointToUtf16Surrogates(ucs, pwszDest + iDest);
					iDest += 2;
				} else {
					if(iDest > 0) {
						if((pwszDest[iDest - 1] == L'\x02E9' && ucs == 0x02E5)
								|| (pwszDest[iDest - 1] == L'\x02E5' && ucs == 0x02E9)) {
							if(iDest + 1 >= cchDest)
								return iDest;
							pwszDest[iDest++] = ZWNJ;
						}
					}
					pwszDest[iDest++] = static_cast<wchar_t>(ucs);
				}
				iSrc += 2;
			} else {
				CONFIRM_ILLEGAL_CHAR(pwszDest[iDest]);
				++iDest;
			}
		}
	}
	return iDest;
}


// { (EUC) ////////////////////////////////////////////////////////////

size_t CEncoder_Japanese_EucJp::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;

	while(iSrc < cchSrc && iDest < cchDest) {
		if(pwszSrc[iSrc] < 0x0080) {	// ascii
			pszDest[iDest++] = static_cast<uchar>(pwszSrc[iSrc++]);
			continue;
		}

		bool	bX0212 = false;
		ushort	jis = UCSToJISX0208(pwszSrc[iSrc]);

		if(jis == 0) {
			jis = UCSToJISX0212(pwszSrc[iSrc]);
			if(jis != 0)
				bX0212 = true;
			else if(uchar kana = UCSToJISX0201Kana(pwszSrc[iSrc])) {
				// JIS X 0201 Kana
				if(cchDest - iDest < 2)
					return iDest;
				pszDest[iDest++] = '\x8E';	// SS2
				pszDest[iDest++] = kana;
				++iSrc;
				continue;
			} else {
				CONFIRM_ILLEGAL_CHAR(jis);
				pszDest[iDest++] = __NATIVE_DEFAULT_CHARACTER;
				++iSrc;
				continue;
			}
		} else if(cchDest - iDest < 2)
			return iDest;
		else {
			jis += 0x8080;	// jis -> euc-jp
			if(!bX0212) {	// JIS X0208
				pszDest[iDest++] = static_cast<char>(jis >> 8);
				pszDest[iDest++] = static_cast<char>(jis);
			} else if(cchDest - iDest < 3)
				return iDest;
			else {	// JIS X0212
				pszDest[iDest++] = '\x8F';	// SS3
				pszDest[iDest++] = static_cast<char>(jis >> 8);
				pszDest[iDest++] = static_cast<char>(jis);
			}
		}
		++iSrc;
	}
	return iDest;
}

size_t CEncoder_Japanese_EucJp::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;

	while(iSrc < cchSrc && iDest < cchDest) {
		const uchar	nFirstByte = pszSrc[iSrc];

		if(nFirstByte == 0x8E) {	// SS2 -> JIS X 0201 Kana
			if(iSrc + 2 > cchSrc)
				return iDest;
			wchar_t	ucs = JISX0201KanaToUCS(pszSrc[iSrc + 1]);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			pwszDest[iDest] = ucs;
			iSrc += 2;
		} else if(nFirstByte == 0x8F) {	// SS3 -> JIS X0212
			if(iSrc + 3 > cchSrc)
				return iDest;
			const ushort	jis = ((pszSrc[iSrc + 1] << 8) | pszSrc[iSrc + 2]) - 0x8080;
			wchar_t			ucs = JISX0212ToUCS(jis);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			pwszDest[iDest] = ucs;
			iSrc += 3;
		} else if(nFirstByte >= 0x80) {	// JIS X0208
			if(iSrc + 2 > cchSrc)
				return iDest;
			const ushort	jis = ((nFirstByte << 8) | pszSrc[iSrc + 1]) - 0x8080;
			wchar_t			ucs = JISX0208ToUCS(jis);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			pwszDest[iDest] = ucs;
			iSrc += 2;
		} else
			pwszDest[iDest] = pszSrc[iSrc++];
		++iDest;
	}
	return iDest;
}


// { (EUC-JIS-2004) /////////////////////////////////////////////////

size_t CEncoder_Japanese_EucJis2004::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	ushort	cchUtf16;
	bool	bIsPlane2;

	while(iSrc < cchSrc && iDest < cchDest) {
		cchUtf16 = (pwszSrc[iSrc] >= 0x80) ? cchSrc - iSrc : 1;

		ushort	jis = (pwszSrc[iSrc] >= 0x80) ? UCSToJISX0213(pwszSrc + iSrc, cchUtf16, bIsPlane2) : pwszSrc[iSrc];

		if(jis == 0) {
			if((jis = UCSToJISX0201Kana(pwszSrc[iSrc])) != 0) {
				if(cchDest - iDest < 2)
					return iDest;
				pszDest[iDest++] = '\x8E';
			} else
				CONFIRM_ILLEGAL_CHAR(jis);
		}
		if(jis < 0x100)
			pszDest[iDest++] = static_cast<char>(jis);
		else if(cchDest - iDest < 2)
			return iDest;
		else {
			jis += 0x8080;	// jis -> euc-jp
			if(!bIsPlane2) {	// 1
				pszDest[iDest++] = static_cast<char>(jis >> 8);
				pszDest[iDest++] = static_cast<char>(jis);
			} else if(cchDest - iDest < 3)
				return iDest;
			else {	// 2
				pszDest[iDest++] = '\x8F';	// SS3
				pszDest[iDest++] = static_cast<char>(jis >> 8);
				pszDest[iDest++] = static_cast<char>(jis);
			}
		}
		iSrc += cchUtf16;
	}
	return iDest;
}

size_t CEncoder_Japanese_EucJis2004::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;

	while(iSrc < cchSrc && iDest < cchDest) {
		const uchar	nFirstByte = pszSrc[iSrc];

		if(nFirstByte == 0x8E) {	// SS2 -> JIS X 0201 Kana
			if(iSrc + 2 > cchSrc)
				return iDest;
			wchar_t	ucs = JISX0201KanaToUCS(pszSrc[iSrc + 1]);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			pwszDest[iDest++] = ucs;
			iSrc += 2;
		} else if(nFirstByte == 0x8F) {	// SS3 -> plane-2
			if(iSrc + 3 > cchSrc)
				return iDest;
			const ushort	jis = ((pszSrc[iSrc + 1] << 8) | pszSrc[iSrc + 2]) - 0x8080;
			ulong			ucs = JISX0213P2ToUCS(jis);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			if(ucs > 0x0010FFFF) {	// 2R[h|Cg (Ǝv)
				if(iDest + 1 >= cchDest)
					return iDest;
				pwszDest[iDest++] = static_cast<wchar_t>(ucs >> 16);
				pwszDest[iDest++] = static_cast<wchar_t>(ucs);
			} else if(ucs >= 0x00010000) {	//  BMP
				if(iDest + 1 >= cchDest)
					return iDest;
				EncodeCodePointToUtf16Surrogates(ucs, pwszDest + iDest);
				iDest += 2;
			} else
				pwszDest[iDest++] = static_cast<wchar_t>(ucs);
			iSrc += 3;
		} else if(nFirstByte >= 0x80) {	// plane-1
			if(iSrc + 2 > cchSrc)
				return iDest;
			const ushort	jis = ((nFirstByte << 8) | pszSrc[iSrc + 1]) - 0x8080;
			ulong			ucs = JISX0213P1ToUCS(jis);

			if(ucs == __REPLACEMENT_CHARACTER)
				CONFIRM_ILLEGAL_CHAR(ucs);
			if(ucs > 0x0010FFFF) {	// 2R[h|Cg
				pwszDest[iDest++] = static_cast<wchar_t>(ucs >> 16);
				pwszDest[iDest++] = static_cast<wchar_t>(ucs);
			} else if(ucs >= 0x00010000) {	//  BMP
				EncodeCodePointToUtf16Surrogates(ucs, pwszDest + iDest);
				iDest += 2;
			} else {
				if(iDest > 0) {
					if((pwszDest[iDest - 1] == L'\x02E9' && ucs == 0x02E5)
							|| (pwszDest[iDest - 1] == L'\x02E5' && ucs == 0x02E9)) {
						if(iDest + 1 >= cchDest)
							return iDest;
						pwszDest[iDest++] = ZWNJ;
					}
				}
				pwszDest[iDest++] = static_cast<wchar_t>(ucs);
			}
			iSrc += 2;
		} else
			pwszDest[iDest++] = pszSrc[iSrc++];
	}
	return iDest;
}


// { (ISO-2022-JP) //////////////////////////////////////////////////////////////

// cchDest ȂƃGXP[vV[PX܂Ȃ\
size_t CEncoder_Japanese_Iso2022Jp::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(CPEX_JAPANESE_ISO2022JP, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(CPEX_JAPANESE_ISO2022JP, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-1) ///////////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp1::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(CPEX_JAPANESE_ISO2022JP1, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp1::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(CPEX_JAPANESE_ISO2022JP1, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-2) ///////////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp2::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(CPEX_JAPANESE_ISO2022JP2, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp2::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(CPEX_JAPANESE_ISO2022JP2, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-2004) ////////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp2004::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP2004, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp2004::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP2004, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-2004-Strict) ////////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp2004_Strict::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP2004_STRICT, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp2004_Strict::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP2004_STRICT, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-2004-Compatible) ////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp2004_Compatible::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP2004_COMPATIBLE, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp2004_Compatible::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP2004_COMPATIBLE, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-3) ////////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp3::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP3, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp3::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP3, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-3-Strict) ////////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp3_Strict::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP3_STRICT, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp3_Strict::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP3_STRICT, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (ISO-2022-JP-3-Compatible) ////////////////////////////////////////

size_t CEncoder_Japanese_Iso2022Jp3_Compatible::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();
	return _ConvertUtf16ToIso2022JpX(
		CPEX_JAPANESE_ISO2022JP3_COMPATIBLE, pszDest, cchDest, pwszSrc, cchSrc, pCallback);
}

size_t CEncoder_Japanese_Iso2022Jp3_Compatible::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();
	return _ConvertIso2022JpXToUtf16(
		CPEX_JAPANESE_ISO2022JP3_COMPATIBLE, pwszDest, cchDest, pszSrc, cchSrc, pCallback);
}


// { (EUC, windows-51932) //////////////////////////////////////////////

CEncoder_Japanese_EucJpWindows::CEncoder_Japanese_EucJpWindows() {
	if(!::IsValidCodePage(932))
		throw exception("This code page is not unsupported.");
}

size_t CEncoder_Japanese_EucJpWindows::ConvertFromUnicode(CFU_ARGLIST) {
	CFU_CHECKARGS();

	BOOL	bUsedDefaultChar;
	size_t	iSrc = 0, iDest = 0;
	uchar	szSJis[2];
	while(iSrc < cchSrc && iDest < cchDest) {
		// UTF-16 -> cp932
		const int	cConvertedBytes = ::WideCharToMultiByte(932, 0,
			pwszSrc + iSrc, 1, reinterpret_cast<char*>(szSJis), 2, 0, &bUsedDefaultChar);
		if(bUsedDefaultChar) {
			if(pCallback != 0 && pCallback->OnFoundUnconvertableCharacter())
				return 0;
			pCallback = 0;
		}

		// cp932 -> cp51932
		if(cConvertedBytes == 1 && szSJis[0] >= 0xA1 && szSJis[0] <= 0xDF) {	// pJi
			pszDest[iDest + 0] = '\x8E';
			pszDest[iDest + 1] = szSJis[0];
			iDest += 2;
		} else if(cConvertedBytes == 2
				&& ((szSJis[0] >= 0x81 && szSJis[0] <= 0x9F) || (szSJis[0] >= 0xE0 && szSJis[0] <= 0xFC))
				&& (szSJis[1] >= 0x40 && szSJis[1] <= 0xFC) && szSJis[1] != 0x7F) {	// 2oCg
			const ushort	jis = ConvertShiftJisDbcsToX0208(szSJis);
			pszDest[iDest + 0] = (jis | 0x8000) >> 8;
			pszDest[iDest + 1] = (jis | 0x0080) >> 0;
			iDest += 2;
		} else	// ̑
			pszDest[iDest++] = szSJis[0];
		++iSrc;
	}
	return iDest;
}

size_t CEncoder_Japanese_EucJpWindows::ConvertToUnicode(CTU_ARGLIST) {
	CTU_CHECKARGS();

	size_t	iSrc = 0, iDest = 0;
	uchar	szSJis[2];
	while(iSrc < cchSrc && iDest < cchDest) {
		// cp51932 -> cp932
		if(pszSrc[iSrc] == 0x8E) {	// pJi
			if(iSrc + 1 >= cchSrc)
				break;
			szSJis[0] = pszSrc[iSrc + 1];
			szSJis[1] = 0;
			iSrc += 2;
		} else if(pszSrc[iSrc] >= 0xA1 && pszSrc[iSrc] <= 0xFE
				&& pszSrc[iSrc + 1] >= 0xA1 && pszSrc[iSrc + 1] <= 0xFE) {	// 2oCg
			ConvertX0208ToShiftJisDbcs(((pszSrc[iSrc] << 8) | pszSrc[iSrc + 1]) - 0x8080, szSJis);
			iSrc += 2;
		} else {
			szSJis[0] = pszSrc[iSrc++];
			szSJis[1] = 0;
		}

		// cp932 -> UTF-16
		::MultiByteToWideChar(932, 0, reinterpret_cast<const char*>(szSJis), (szSJis[1] == 0) ? 1 : 2, pwszDest + iDest, 1);
		++iDest;
	}
	return iDest;
}

uchar CEncoder_Japanese_EucJpWindows::GetMaxNativeCharLength() const {
	return 2;
}

uchar CEncoder_Japanese_EucJpWindows::GetMaxUcsCharLength() const {
	return 1;
}


#undef IS_ISO2022JP3
#undef IS_ISO2022JP2004
#undef JK

/* [EOF] */