/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2008 Aimluck,Inc.
 * http://aipostyle.com/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package com.aimluck.eip.workflow.util;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.apache.cayenne.DataObjectUtils;
import org.apache.cayenne.DataRow;
import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.cayenne.query.SelectQuery;
import org.apache.jetspeed.om.security.UserIdPrincipal;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.jetspeed.services.security.JetspeedSecurityException;
import org.apache.turbine.services.InstantiationException;
import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowCategory;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowRequest;
import com.aimluck.eip.cayenne.om.portlet.EipTWorkflowRequestMap;
import com.aimluck.eip.cayenne.om.security.TurbineUser;
import com.aimluck.eip.common.ALBaseUser;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.orm.DatabaseOrmService;
import com.aimluck.eip.services.accessctl.ALAccessControlConstants;
import com.aimluck.eip.services.accessctl.ALAccessControlFactoryService;
import com.aimluck.eip.services.accessctl.ALAccessControlHandler;
import com.aimluck.eip.user.beans.UserLiteBean;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.whatsnew.util.WhatsNewUtils;
import com.aimluck.eip.workflow.WorkflowCategoryResultData;
import com.aimluck.eip.workflow.WorkflowDecisionRecordData;
import com.aimluck.eip.workflow.WorkflowDetailResultData;
import com.aimluck.eip.workflow.WorkflowOldRequestResultData;

/**
 * ワークフローのユーティリティクラスです。 <BR>
 * 
 */
public class WorkflowUtils {

  /** logger */
  private static final JetspeedLogger logger = JetspeedLogFactoryService
      .getLogger(WorkflowUtils.class.getName());

  /** 申請 */
  public static final String DB_STATUS_REQUEST = "R";

  /** 確認中 */
  public static final String DB_STATUS_CONFIRM = "C";

  /** 確認前 */
  public static final String DB_STATUS_WAIT = "W";

  /** 承認 */
  public static final String DB_STATUS_ACCEPT = "A";

  /** 否認 */
  public static final String DB_STATUS_DENIAL = "D";

  /** すべて承認 */
  public static final String DB_PROGRESS_ACCEPT = "A";

  /** 確認中 */
  public static final String DB_PROGRESS_WAIT = "W";

  /** 申請者に差し戻し */
  public static final String DB_PROGRESS_DENAIL = "D";

  /** 再申請済み */
  public static final String DB_PROGRESS_REAPPLY = "R";

  /**
   * Request オブジェクトモデルを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static EipTWorkflowRequest getEipTWorkflowRequest(RunData rundata,
      Context context, boolean mode_update) throws ALPageNotFoundException {
    int uid = ALEipUtils.getUserId(rundata);
    String requestid = ALEipUtils.getTemp(rundata, context,
        ALEipConstants.ENTITY_ID);
    try {
      if (requestid == null || Integer.valueOf(requestid) == null) {
        // Request IDが空の場合
        logger.debug("[WorkflowUtils] Empty ID...");
        return null;
      }

      // アクセス権の判定
      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query10 = new SelectQuery(EipTWorkflowRequestMap.class);
      Expression exp10 = ExpressionFactory.matchExp(
          EipTWorkflowRequestMap.REQUEST_ID_PROPERTY, Integer
              .valueOf(requestid));
      Expression exp11 = ExpressionFactory.matchExp(
          EipTWorkflowRequestMap.USER_ID_PROPERTY, uid);
      query10.setQualifier(exp10.andExp(exp11));
      List maps = dataContext.performQuery(query10);
      if (maps == null || maps.size() == 0) {
        // 指定したアカウントIDのレコードが見つからない場合
        logger.debug("[WorkFlow] Invalid user access...");
        throw new ALPageNotFoundException();
      }

      SelectQuery query = new SelectQuery(EipTWorkflowRequest.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          EipTWorkflowRequest.REQUEST_ID_PK_COLUMN, requestid);
      query.setQualifier(exp1);

      Expression exp2 = ExpressionFactory.matchExp(
          EipTWorkflowRequest.EIP_TWORKFLOW_REQUEST_MAP_PROPERTY + "."
              + EipTWorkflowRequestMap.USER_ID_PROPERTY, Integer
              .valueOf(ALEipUtils.getUserId(rundata)));
      query.andQualifier(exp2);

      if (mode_update) {
        Expression exp3 = ExpressionFactory.matchExp(
            EipTWorkflowRequest.EIP_TWORKFLOW_REQUEST_MAP_PROPERTY + "."
                + EipTWorkflowRequestMap.STATUS_PROPERTY, DB_STATUS_CONFIRM);
        query.andQualifier(exp3);

        Expression exp4 = ExpressionFactory.matchExp(
            EipTWorkflowRequest.PROGRESS_PROPERTY, DB_PROGRESS_WAIT);
        query.andQualifier(exp4);
      }

      List requests = dataContext.performQuery(query);

      if (requests == null || requests.size() == 0) {
        // 指定した Request IDのレコードが見つからない場合
        logger.debug("[WorkflowUtils] Not found ID...");
        return null;
      }

      EipTWorkflowRequest request = (EipTWorkflowRequest) requests.get(0);

      return (request);
    } catch (ALPageNotFoundException pageNotFound) {
      logger.error(pageNotFound);
      throw pageNotFound;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * Request オブジェクトモデルを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static EipTWorkflowRequest getEipTWorkflowRequestAll(RunData rundata,
      Context context) {
    String requestid = ALEipUtils.getTemp(rundata, context,
        ALEipConstants.ENTITY_ID);

    try {
      /**
       * 新着ポートレット既読処理
       */
      WhatsNewUtils.shiftWhatsNewReadFlag(
          WhatsNewUtils.WHATS_NEW_TYPE_WORKFLOW_REQUEST, Integer
              .parseInt(requestid), ALEipUtils.getUserId(rundata));
    } catch (Exception e) {
      // TODO: handle exception
      logger.error("Exception", e);
    }
    try {
      if (requestid == null || Integer.valueOf(requestid) == null) {
        // Request IDが空の場合
        logger.debug("[WorkflowUtils] Empty ID...");
        return null;
      }

      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowRequest.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          EipTWorkflowRequest.REQUEST_ID_PK_COLUMN, requestid);
      query.setQualifier(exp1);

      List requests = dataContext.performQuery(query);

      if (requests == null || requests.size() == 0) {
        // 指定した Request IDのレコードが見つからない場合
        logger.debug("[WorkflowUtils] Not found ID...");
        return null;
      }
      return ((EipTWorkflowRequest) requests.get(0));
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * Request オブジェクトモデルを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static EipTWorkflowRequest getEipTWorkflowRequestForOwner(
      RunData rundata, Context context) {
    String requestid = ALEipUtils.getTemp(rundata, context,
        ALEipConstants.ENTITY_ID);
    try {
      if (requestid == null || Integer.valueOf(requestid) == null) {
        // Request IDが空の場合
        logger.debug("[WorkflowUtils] Empty ID...");
        return null;
      }

      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowRequest.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          EipTWorkflowRequest.REQUEST_ID_PK_COLUMN, requestid);
      query.setQualifier(exp1);

      Expression exp2 = ExpressionFactory.matchExp(
          EipTWorkflowRequest.USER_ID_PROPERTY, Integer.valueOf(ALEipUtils
              .getUserId(rundata)));
      query.andQualifier(exp2);

      List requests = dataContext.performQuery(query);

      if (requests == null || requests.size() == 0) {
        // 指定した Request IDのレコードが見つからない場合
        logger.debug("[WorkflowUtils] Not found ID...");
        return null;
      }
      return ((EipTWorkflowRequest) requests.get(0));
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  public static List getEipTWorkflowRequestMap(EipTWorkflowRequest request) {
    try {
      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowRequestMap.class);
      Expression exp = ExpressionFactory.matchExp(
          EipTWorkflowRequestMap.REQUEST_ID_PROPERTY, request.getRequestId());
      query.setQualifier(exp);
      query.addOrdering(EipTWorkflowRequestMap.ORDER_INDEX_PROPERTY, true);

      List maps = dataContext.performQuery(query);

      if (maps == null || maps.size() == 0) {
        // 指定した Request IDのレコードが見つからない場合
        logger.debug("[WorkflowSelectData] Not found ID...");
        return null;
      }
      return maps;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * ワークフローカテゴリ オブジェクトモデルを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param mode_update
   * @return
   */
  public static EipTWorkflowCategory getEipTWorkflowCategory(RunData rundata,
      Context context) {
    String categoryid = ALEipUtils.getTemp(rundata, context,
        ALEipConstants.ENTITY_ID);
    try {
      if (categoryid == null || Integer.valueOf(categoryid) == null) {
        // Request IDが空の場合
        logger.debug("[WorkflowUtils] Empty ID...");
        return null;
      }

      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowCategory.class);
      Expression exp1 = ExpressionFactory.matchDbExp(
          EipTWorkflowCategory.CATEGORY_ID_PK_COLUMN, categoryid);
      query.setQualifier(exp1);

      List categories = dataContext.performQuery(query);

      if (categories == null || categories.size() == 0) {
        // 指定したカテゴリIDのレコードが見つからない場合
        logger.debug("[WorkflowUtils] Not found ID...");
        return null;
      }
      return ((EipTWorkflowCategory) categories.get(0));
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * ワークフローカテゴリ オブジェクトモデルを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  public static EipTWorkflowCategory getEipTWorkflowCategory(
      DataContext dataContext, Long category_id) {
    try {
      EipTWorkflowCategory category = (EipTWorkflowCategory) DataObjectUtils
          .objectForPK(dataContext, EipTWorkflowCategory.class, category_id);

      return category;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * カテゴリの一覧を取得する。
   * 
   * @param rundata
   * @param context
   */
  public static List loadCategoryList(RunData rundata, Context context) {
    try {
      // カテゴリ一覧
      List categoryList = new ArrayList();

      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowCategory.class);
      List aList = dataContext.performQuery(query);

      int size = aList.size();
      for (int i = 0; i < size; i++) {
        EipTWorkflowCategory record = (EipTWorkflowCategory) aList.get(i);
        WorkflowCategoryResultData rd = new WorkflowCategoryResultData();
        rd.initField();
        rd.setCategoryId(record.getCategoryId().longValue());
        rd.setCategoryName(record.getCategoryName());
        categoryList.add(rd);
      }

      return categoryList;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 依頼の詳細情報を取得する。
   * 
   * @param rundata
   * @param context
   */
  public static Object getResultDataDetail(Object obj) {
    try {
      EipTWorkflowRequest record = (EipTWorkflowRequest) obj;
      WorkflowDetailResultData rd = new WorkflowDetailResultData();
      rd.initField();
      rd.setUserId(record.getUserId().longValue());
      rd.setRequestName(record.getRequestName());
      rd.setRequestId(record.getRequestId().longValue());
      rd.setCategoryId(record.getEipTWorkflowCategory().getCategoryId()
          .longValue());
      rd.setCategoryName(record.getEipTWorkflowCategory().getCategoryName());
      rd.setPriorityString(WorkflowUtils.getPriorityString(record.getPriority()
          .intValue()));
      rd.setNote(record.getNote());
      rd.setPrice(record.getPrice().longValue());
      rd.setProgress(record.getProgress());

      List drList = new ArrayList();
      ALEipUser user = null;
      EipTWorkflowRequestMap map = null;
      WorkflowDecisionRecordData drd = null;
      List maps = WorkflowUtils.getEipTWorkflowRequestMap(record);
      int size = maps.size();
      for (int i = 0; i < size; i++) {
        map = (EipTWorkflowRequestMap) maps.get(i);
        drd = new WorkflowDecisionRecordData();
        drd.initField();

        user = ALEipUtils.getALEipUser(map.getUserId().intValue());
        drd.setUserId(map.getUserId().intValue());
        drd.setUserAliasName(user.getAliasName().getValue());
        drd.setStatus(map.getStatus());
        drd.setStatusString(WorkflowUtils.getStatusString(map.getStatus()));
        drd.setOrder(map.getOrderIndex().intValue());
        drd.setNote(map.getNote());
        drd.setUpdateDate(WorkflowUtils.translateDate(map.getUpdateDate(),
            "yyyy年M月d日H時m分"));
        drList.add(drd);
      }
      rd.setDecisionRecords(drList);

      // 過去の申請内容
      if (record.getParentId().intValue() != 0) {
        List oldReuqests = getOldRequests(record);
        if (oldReuqests != null && oldReuqests.size() > 0) {
          List oldList = new ArrayList();
          int osize = oldReuqests.size();
          for (int i = 0; i < osize; i++) {
            EipTWorkflowRequest request = (EipTWorkflowRequest) oldReuqests
                .get(i);
            WorkflowOldRequestResultData orrd = new WorkflowOldRequestResultData();
            orrd.initField();
            orrd.setRequestId(request.getRequestId().intValue());
            orrd.setRequestName(request.getRequestName());
            orrd.setCategoryName(request.getEipTWorkflowCategory()
                .getCategoryName());
            orrd.setUpdateDate(WorkflowUtils.translateDate(request
                .getUpdateDate(), "yyyy年M月d日H時m分"));
            oldList.add(orrd);
          }
          rd.setOldRequestLinks(oldList);
        }
      }

      rd.setCreateDate(WorkflowUtils.translateDate(record.getCreateDate(),
          "yyyy年M月d日H時m分"));
      rd.setUpdateDate(WorkflowUtils.translateDate(record.getUpdateDate(),
          "yyyy年M月d日H時m分"));
      return rd;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  private static List getOldRequests(EipTWorkflowRequest request) {
    try {
      DataContext dataContext = DatabaseOrmService.getInstance()
          .getDataContext();
      SelectQuery query = new SelectQuery(EipTWorkflowRequest.class);
      Expression exp11 = ExpressionFactory.matchDbExp(
          EipTWorkflowRequest.REQUEST_ID_PK_COLUMN, request.getParentId());
      Expression exp12 = ExpressionFactory.matchExp(
          EipTWorkflowRequest.PARENT_ID_PROPERTY, request.getParentId());
      query.setQualifier(exp11.orExp(exp12));
      Expression exp2 = ExpressionFactory.noMatchDbExp(
          EipTWorkflowRequest.REQUEST_ID_PK_COLUMN, request.getRequestId());
      query.andQualifier(exp2);

      query.addOrdering(EipTWorkflowRequest.UPDATE_DATE_PROPERTY, true);

      List requests = dataContext.performQuery(query);

      if (requests == null || requests.size() == 0) {
        // 指定した Request IDのレコードが見つからない場合
        logger.debug("[WorkflowSelectData] Not found ID...");
        return null;
      }
      return requests;
    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 重要度を表す画像名を取得します。 <BR>
   * 1 : 高い : priority_high.gif <BR>
   * 2 : やや高い : priority_middle_high.gif <BR>
   * 3 : 普通 : priority_middle.gif <BR>
   * 4 : やや低い : priority_middle_low.gif <BR>
   * 5 : 低い : priority_low.gif <BR>
   * 
   * @param i
   * @return
   */
  public static String getPriorityImage(int i) {
    String[] temp = { "priority_high.gif", "priority_middle_high.gif",
        "priority_middle.gif", "priority_middle_low.gif", "priority_low.gif" };
    String image = null;
    try {
      image = temp[i - 1];
    } catch (Exception ex) {
      // logger.error("Exeption", ex);
    }
    return image;
  }

  /**
   * 重要度を表す文字列を取得します。 <BR>
   * 1 : 高い : priority_high.gif <BR>
   * 2 : やや高い : priority_middle_high.gif <BR>
   * 3 : 普通 : priority_middle.gif <BR>
   * 4 : やや低い : priority_middle_low.gif <BR>
   * 5 : 低い : priority_low.gif <BR>
   * 
   * @param i
   * @return
   */
  public static String getPriorityString(int i) {
    String[] temp = { "高い", "やや高い", "普通", "やや低い", "低い" };
    String string = null;
    try {
      string = temp[i - 1];
    } catch (Exception ex) {
      // logger.error("Exeption", ex);
    }
    return string;
  }

  /**
   * 状態を表す画像名を取得します。 <BR>
   * 0 : 未着手 <BR>
   * 10 : 10% <BR>
   * 20 : 20% <BR>: :<BR>
   * 90 : 90% <BR>
   * 100 : 完了 <BR>
   * 
   * @param i
   * @return
   */
  public static String getStateImage(int i) {
    String[] temp = { "state_000.gif", "state_010.gif", "state_020.gif",
        "state_030.gif", "state_040.gif", "state_050.gif", "state_060.gif",
        "state_070.gif", "state_080.gif", "state_090.gif", "state_100.gif" };
    String image = null;
    try {
      image = temp[i / 10];
    } catch (Exception ex) {
      // logger.error("Exeption", ex);
    }
    return image;
  }

  /**
   * 状態を表す文字列を取得します。 <BR>
   * 0 : 未着手 <BR>
   * 10 : 10% <BR>
   * 20 : 20% <BR>: :<BR>
   * 90 : 90% <BR>
   * 100 : 完了 <BR>
   * 
   * @param i
   * @return
   */
  public static String getStateString(int i) {
    if (i == 0)
      return "未着手";
    else if (i == 100)
      return "完了";
    else {
      return new StringBuffer().append(i).append("%").toString();
    }
  }

  /**
   * 
   * @param status
   */
  public static String getStatusString(String status) {
    String res = "";
    if (DB_STATUS_REQUEST.equals(status)) {
      res = "申請";
    } else if (DB_STATUS_CONFIRM.equals(status)) {
      res = "確認中";
    } else if (DB_STATUS_WAIT.equals(status)) {
      res = "確認前";
    } else if (DB_STATUS_ACCEPT.equals(status)) {
      res = "承認";
    } else if (DB_STATUS_DENIAL.equals(status)) {
      res = "否認";
    }
    return res;
  }

  /**
   * Date のオブジェクトを指定した形式の文字列に変換する．
   * 
   * @param date
   * @param dateFormat
   * @return
   */
  public static String translateDate(Date date, String dateFormat) {
    if (date == null)
      return "Unknown";

    // 日付を表示形式に変換
    SimpleDateFormat sdf = new SimpleDateFormat(dateFormat);
    sdf.setTimeZone(TimeZone.getTimeZone("JST"));
    return sdf.format(date);
  }

  /**
   * 3 桁でカンマ区切りした文字列を取得する．
   * 
   * @param money
   * @return
   */
  public static String translateMoneyStr(String money) {
    if (money == null || money.length() == 0)
      return money;

    StringBuffer sb = new StringBuffer();
    int len = money.length();
    int count = len / 3;
    int del = len % 3;
    sb.append(money.substring(0, del));
    if (count > 0) {
      if (len > 3 && del != 0) {
        sb.append(",");
      }
      for (int i = 0; i < count; i++) {
        sb.append(money.substring(del + 3 * i, del + 3 * i + 3));
        if (i != count - 1) {
          sb.append(",");
        }
      }
    }
    return sb.toString();
  }

  /**
   * メール送信
   */
  public static boolean sendMail(RunData rundata, EipTWorkflowRequest request,
      ALEipUser destUser, ArrayList msgList) throws Exception {

    String org_id = DatabaseOrmService.getInstance().getOrgId(rundata);
    String subject = "[Aipo]ワークフロー";

    ArrayList memberList = new ArrayList();
    memberList.add(destUser);
    List destMemberList = ALMailUtils.getALEipUserAddrs(memberList, ALEipUtils
        .getUserId(rundata), false);

    return ALMailUtils.sendMailDelegate(org_id, (int) destUser.getUserId()
        .getValue(), destMemberList, subject, subject, WorkflowUtils
        .createMsgForPc(rundata, request), WorkflowUtils.createMsgForCellPhone(
        rundata, request, destUser), ALMailUtils
        .getSendDestType(ALMailUtils.KEY_MSGTYPE_WORKFLOW), msgList);
  }

  /**
   * パソコンへ送信するメールの内容を作成する．
   * 
   * @return
   */
  public static String createMsgForPc(RunData rundata,
      EipTWorkflowRequest request) {
    String CR = System.getProperty("line.separator");

    ALBaseUser user = null;
    ALEipUser user2 = null;

    try {
      user = (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(request
          .getUserId().toString()));
      user2 = ALEipUtils.getALEipUser(request.getUserId().intValue());
    } catch (Exception e) {
      return "";
    }
    StringBuffer body = new StringBuffer("");
    body.append(user2.getAliasName().toString());
    if (!user.getEmail().equals("")) {
      body.append("(").append(user.getEmail()).append(")");
    }

    if ("D".equals(request.getProgress())) {
      body.append("さんの申請は差し戻されました。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("差戻要確認").append(CR);
    } else if ("A".equals(request.getProgress())) {
      body.append("さんの申請は承認されました。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("承認済み").append(CR);
    } else {
      body.append("さんからの承認依頼です。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("決裁待ち").append(CR);
    }

    body.append("[表題]").append(CR);
    body.append(request.getEipTWorkflowCategory().getCategoryName()).append(CR);

    if (request.getRequestName() != null
        && (!"".equals(request.getRequestName()))) {
      body.append(request.getRequestName()).append(CR);
    }

    body.append("[申請日]").append(CR).append(
        WorkflowUtils.translateDate(request.getCreateDate(), "yyyy年M月d日H時m分"))
        .append(CR);

    body.append("[重要度]").append(CR).append(
        WorkflowUtils.getPriorityString(request.getPriority().intValue()))
        .append(CR);
    body.append("[申請内容]").append(CR).append(request.getNote()).append(CR);

    if (request.getPrice() != null && (request.getPrice().intValue() > 0)) {
      body.append("[金額]").append(CR).append(request.getPrice()).append(" 円")
          .append(CR);
    }

    body.append(CR);
    body.append("[Aipoへのアクセス]").append(CR);
    body.append("・社外").append(CR);
    body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
    body.append("・社内").append(CR);
    body.append("　").append(ALMailUtils.getLocalurl()).append(CR).append(CR);
    body.append("---------------------").append(CR);
    body.append("Aipo").append(CR);

    return body.toString();
  }

  /**
   * 携帯電話へ送信するメールの内容を作成する．
   * 
   * @return
   */
  public static String createMsgForCellPhone(RunData rundata,
      EipTWorkflowRequest request, ALEipUser login_user) {
    String CR = System.getProperty("line.separator");

    ALBaseUser user = null;
    ALEipUser user2 = null;

    try {
      user = (ALBaseUser) JetspeedSecurity.getUser(new UserIdPrincipal(request
          .getUserId().toString()));
      user2 = ALEipUtils.getALEipUser(request.getUserId().intValue());
    } catch (JetspeedSecurityException e) {
      return "";
    } catch (ALDBErrorException e) {
      return "";
    }

    StringBuffer body = new StringBuffer("");
    body.append(user2.getAliasName().toString());
    if (!user.getEmail().equals("")) {
      body.append("(").append(user.getEmail()).append(")");
    }

    if ("D".equals(request.getProgress())) {
      body.append("さんの申請は差し戻されました。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("差戻要確認").append(CR);
    } else if ("A".equals(request.getProgress())) {
      body.append("さんの申請は承認されました。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("承認済み").append(CR);
    } else {
      body.append("さんからの承認依頼です。").append(CR).append(CR);
      body.append("[決裁状況]").append(CR);
      body.append("決裁待ち").append(CR);
    }

    body.append("[表題]").append(CR);
    body.append(request.getEipTWorkflowCategory().getCategoryName()).append(CR);

    if (request.getRequestName() != null
        && (!"".equals(request.getRequestName()))) {
      body.append(request.getRequestName()).append(CR);
    }

    body.append("[申請日]").append(CR).append(
        WorkflowUtils.translateDate(request.getCreateDate(), "yyyy年M月d日H時m分"))
        .append(CR);

    body.append("[重要度]").append(CR).append(
        WorkflowUtils.getPriorityString(request.getPriority().intValue()))
        .append(CR);
    // body.append("[申請内容]").append(CR).append(request.getNote()).append(CR);

    if (request.getPrice() != null && (request.getPrice().intValue() > 0)) {
      body.append("[金額]").append(CR).append(request.getPrice()).append(" 円")
          .append(CR);
    }

    body.append(CR);
    body.append("[Aipoへのアクセス]").append(CR);
    body.append("　").append(ALMailUtils.getGlobalurl()).append(CR);
    body.append("---------------------").append(CR);
    body.append("Aipo").append(CR);

    return body.toString();
  }

  public static boolean hasWorkFlowAuthority(List memberList) {
    int size = memberList.size();
    List userIds = new ArrayList();
    for (int i = 0; i < size; i++) {
      userIds.add(((ALEipUser) memberList.get(i)).getUserId());
    }

    return false;
  }

  public static List getAuthorityUsers(RunData rundata, String groupname,
      boolean includeLoginuser) {

    try {
      // アクセス権限
      ALAccessControlFactoryService aclservice = (ALAccessControlFactoryService) ((TurbineServices) TurbineServices
          .getInstance())
          .getService(ALAccessControlFactoryService.SERVICE_NAME);
      ALAccessControlHandler aclhandler = aclservice.getAccessControlHandler();

      List ulist = aclhandler.getAuthorityUsersFromGroup(rundata,
          ALAccessControlConstants.POERTLET_FEATURE_WORKFLOW_REQUEST_SELF,
          groupname, includeLoginuser);

      int recNum = ulist.size();

      List list = new ArrayList();

      UserLiteBean user;
      DataRow dataRow;
      // ユーザデータを作成し、返却リストへ格納
      for (int j = 0; j < recNum; j++) {
        dataRow = (DataRow) ulist.get(j);
        user = new UserLiteBean();
        user.initField();
        user.setUserId(((Integer) ALEipUtils.getObjFromDataRow(dataRow,
            TurbineUser.USER_ID_PK_COLUMN)).intValue());
        user.setName((String) ALEipUtils.getObjFromDataRow(dataRow,
            TurbineUser.LOGIN_NAME_COLUMN));
        user.setAliasName((String) ALEipUtils.getObjFromDataRow(dataRow,
            TurbineUser.FIRST_NAME_COLUMN), (String) ALEipUtils
            .getObjFromDataRow(dataRow, TurbineUser.LAST_NAME_COLUMN));
        list.add(user);
      }
      return list;
    } catch (InstantiationException e) {
      // TODO Auto-generated catch block
      return null;
    }

  }

}
