////////////////////////////////////////////////////////////////////////////////
// マルチGadgetライブラリ
//   Copyright (C) 2008 - 2011 AAOh inc.
////////////////////////////////////////////////////////////////////////////////

/**
 * @fileOverview マルチGadgetライブラリ(アニメーション)
 * @author       $Author: fujishiro $
 */

/**
 * namespace。
 *   aaoh.kaze.animation.Animation
 *   aaoh.kaze.animation.FadeinAnimation
 *   aaoh.kaze.animation.FadeoutAnimation
 *   aaoh.kaze.animation.MoveAnimation
 *   aaoh.kaze.animation.ResizeAnimation
 *   aaoh.kaze.animation.ForegroundColorAnimation
 *   aaoh.kaze.animation.BackgroundColorAnimation
 *   aaoh.kaze.animation.BorderColorAnimation
 */

(function () {
	try {
		if (!aaoh.kaze.animation) {
			/**
			 * aaoh.kaze.animation。
			 *
			 * @namespace
			 */
			aaoh.kaze.animation = {};
		}
	} catch(ex) {
		aaoh.kaze.animation = {};
	}
})();

(function () {
/**
 * Animation。
 *
 * @constructor
 */
aaoh.kaze.animation.Animation = function() {
	this.callObj       = null;
	this.element       = null;
	this.speed         = null;
	this.startValue    = null;
	this.stopValue     = null;
	this.stepValue     = null;
	this.startCallable = null;
	this.stopCallable  = null;
	this.stepCallable  = null;

	this.flgInit      = false;
	this.currentValue = null;
	this.timer        = null;
}
aaoh.kaze.animation.Animation.prototype = {
	/*
	 * @param {Object}   callObj_       呼び出し元オブジェクト
	 * @param {Object}   element_       エレメント
	 * @param {Number}   speed_         スピード(msec)
	 * @param {Object}   startValue_    開始値
	 * @param {Object}   stopValue_     終了値
	 * @param {Object}   stepValue_     ステップ値
	 * @param {Function} startCallable_ 開始ファンクション
	 * @param {Function} stopCallable_  終了ファンクション
	 * @param {Function} stepCallable_  ステップファンクション
	 */
	init: function(
		callObj_, element_, speed_,
		startValue_, stopValue_, stepValue_,
		startCallable_, stopCallable_, stepCallable_
	) {
		this.callObj       = callObj_;
		this.element       = element_;
		this.speed         = speed_;
		this.startValue    = startValue_;
		this.stopValue     = stopValue_;
		this.stepValue     = stepValue_;
		this.startCallable = startCallable_;
		this.stopCallable  = stopCallable_;
		this.stepCallable  = stepCallable_;
		if (this.callObj == null) {
			this.callObj = this;
		}

		if (this.element == null) {
			throw new Error('element is null');

		} else if (isNaN(this.speed)) {
			throw new Error('speed is null, element is ' + this.element.id);

		} else if (this.startValue == null) {
			throw new Error(
				'startValue is null, '
				+ 'element is ' + this.element.id + ', '
				+ 'speed is '   + this.speed
			);

		} else if (this.stopValue == null) {
			throw new Error(
				'stopValue is null, '
				+ 'element is '    + this.element.id + ', '
				+ 'speed is '      + this.speed      + ', '
				+ 'startValue is ' + this.startValue
			);

		} else if (this.stepValue == null) {
			throw new Error(
				'stepValue is null, '
				+ 'element is '    + this.element.id + ', '
				+ 'speed is '      + this.speed      + ', '
				+ 'startValue is ' + this.startValue + ', '
				+ 'stopValue is '  + this.stopValue
			);
		}

		this.flgInit = true;
		this.timer   = new aaoh.kaze.Timer(this);
	},

	/**
	 * アニメーションの開始。
	 */
	start: function() {
		if (!this.flgInit) {
			this.init(arguments);
		}
		this.currentValue = this.startValue;

		if (this.startCallable != null) {
			this.startCallable.call(
				this.callObj, this.element, this.startValue
			);
		}

		this.timer.setInterval(this.step, this.speed);
	},

	/**
	 * アニメーションの停止。
	 */
	stop: function() {
		this.timer.clearInterval();

		if (this.stopCallable != null) {
			this.stopCallable.call(
				this.callObj, this.element, this.stopValue
			);
		}

		this.currentValue = null;
	},

	/**
	 * アニメーションのステップ実行。
	 */
	step: function() {
		var nextValue = this.nextValue();
		if (nextValue == null) {
			this.stop();
			return;
		}

		if (this.stepCallable != null) {
			this.stepCallable.call(
				this.callObj, this.element, nextValue
			);
		}
	},

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	nextValue: function() {
		if (isNaN(this.currentValue)) {
			return null;
		}

		var nextValue = this.currentValue + this.stepValue;
		if (nextValue >= this.stopValue) {
			return null;
		}

		this.currentValue = nextValue;
		return nextValue;
	}
}
})();

(function () {
/**
 * FadeinAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element_ エレメント
 * @param {Number} speed_   スピード(msec)
 */
aaoh.kaze.animation.FadeinAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element_, speed_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = 0;
	this.DEFAULT_STOP_VALUE  = 100;
	this.DEFAULT_STEP_VALUE  = 10;

	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}

	// 初期化
	this.init(
		this, element_, speed_,
		this.DEFAULT_START_VALUE, this.DEFAULT_STOP_VALUE,
		this.DEFAULT_STEP_VALUE,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setOpacity(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setOpacity(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setOpacity(element, nextValue);
	}
})();

(function () {
/**
 * FadeoutAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element_ エレメント
 * @param {Number} speed_   スピード(msec)
 */
aaoh.kaze.animation.FadeoutAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element_, speed_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = 100;
	this.DEFAULT_STOP_VALUE  = 0;
	this.DEFAULT_STEP_VALUE  = -10;

	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}

	// 初期化
	this.init(
		this, element_, speed_,
		this.DEFAULT_START_VALUE, this.DEFAULT_STOP_VALUE,
		this.DEFAULT_STEP_VALUE,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setOpacity(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setOpacity(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setOpacity(element, nextValue);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	nextValue = function() {
		if (isNaN(this.currentValue)) {
			return null;
		}

		var nextValue = this.currentValue + this.stepValue;
		if (nextValue <= this.stopValue) {
			return null;
		}

		this.currentValue = nextValue;
		return nextValue;
	}
})();

(function () {
/**
 * MoveAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element_    エレメント
 * @param {Number} speed_      スピード(msec)
 * @param {Object} startValue_ 開始値
 * @param {Object} stopValue_  終了値
 * @param {Object} stepValue_  ステップ値
 */
aaoh.kaze.animation.MoveAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element_, speed_,
		startValue_, stopValue_, stepValue_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = { x: 0,  y: 0  };
	this.DEFAULT_STOP_VALUE  = { x: 10, y: 10 };
	this.DEFAULT_STEP_VALUE  = { x: 1,  y: 1  };

	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}
	if (startValue_ == null) {
		startValue_ = this.DEFAULT_START_VALUE
	}
	if (stopValue_ == null) {
		stopValue_ = this.DEFAULT_STOP_VALUE
	}
	if (stepValue_ == null) {
		stepValue_ = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element_, speed_,
		startValue_, stopValue_, stepValue_,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setPosition(
			element, startValue.x, startValue.y
		);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setPosition(
			element, stopValue.x, stopValue.y
		);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setPosition(
			element, nextValue.x, nextValue.y
		);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	nextValue = function() {
		var nextValue = { x:0, y: 0 };
		nextValue.x = this.currentValue.x + this.stepValue.x;
		nextValue.y = this.currentValue.y + this.stepValue.y;
		if (nextValue.x >= this.stopValue.x) {
			nextValue.x = this.stopValue.x;
		}
		if (nextValue.y >= this.stopValue.y) {
			nextValue.y = this.stopValue.y;
		}

		if (
			nextValue.x >= this.stopValue.x
			&& nextValue.y >= this.stopValue.y
		) {
			return null;
		}

		this.currentValue = nextValue;
		return nextValue;
	}
})();

(function () {
/**
 * ResizeAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element_    エレメント
 * @param {Number} speed_      スピード(msec)
 * @param {Object} startValue_ 開始値
 * @param {Object} stopValue_  終了値
 * @param {Object} stepValue_  ステップ値
 */
aaoh.kaze.animation.ResizeAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element_, speed_,
		startValue_, stopValue_, stepValue_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = { width: 0,  height: 0 };
	this.DEFAULT_STOP_VALUE  = { width: 10, height: 10 };
	this.DEFAULT_STEP_VALUE  = { width: 1,  height: 1 };

	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}
	if (startValue_ == null) {
		startValue_ = this.DEFAULT_START_VALUE
	}
	if (stopValue_ == null) {
		stopValue_ = this.DEFAULT_STOP_VALUE
	}
	if (stepValue_ == null) {
		stepValue_ = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element_, speed_,
		startValue_, stopValue_, stepValue_,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setSize(
			element, startValue.width, startValue.height
		);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setSize(
			element, stopValue.width, stopValue.height
		);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setSize(
			element, nextValue.width, nextValue.height
		);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	nextValue = function() {
		var nextValue = { width:0, height: 0 };
		nextValue.width  = this.currentValue.width  + this.stepValue.width;
		nextValue.height = this.currentValue.height + this.stepValue.height;
		if (nextValue.width >= this.stopValue.width) {
			nextValue.width = this.stopValue.width;
		}
		if (nextValue.height >= this.stopValue.height) {
			nextValue.height = this.stopValue.height;
		}

		if (
			nextValue.width >= this.stopValue.width
			&& nextValue.height >= this.stopValue.height
		) {
			return null;
		}

		this.currentValue = nextValue;
		return nextValue;
	}
})();

(function () {
/**
 * BaseColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 */
aaoh.kaze.animation.BaseColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function() {
	// 特になし
});
	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.BaseColorAnimation.prototype.
	nextValue = function() {
		var nextValue = { r: 0, g: 0, b: 0 };

		nextValue.r = this.currentValue.r + this.stepValue.r;
		nextValue.g = this.currentValue.g + this.stepValue.g;
		nextValue.b = this.currentValue.b + this.stepValue.b;
		if (nextValue.r >= this.stopValue.r) {
			nextValue.r = this.stopValue.r;
		}
		if (nextValue.g >= this.stopValue.g) {
			nextValue.g = this.stopValue.g;
		}
		if (nextValue.b >= this.stopValue.b) {
			nextValue.b = this.stopValue.b;
		}

		if (
			nextValue.r >= this.stopValue.r
			&& nextValue.g >= this.stopValue.g
			&& nextValue.b >= this.stopValue.b
		) {
			return null;
		}

		this.currentValue = nextValue;
		return nextValue;
	}

	/**
	 * RGBへの変換。
	 *
	 * @return {String} 色値
	 * @return {Object} RGB値
	 */
	aaoh.kaze.animation.BaseColorAnimation.prototype.
	toRGB = function(color) {
		if (color == null) {
			return null;
		}

		if (!!color.r && !!color.g && !!color.b) {
			return color;

		} else if (
			color.length != 7
			|| color.indexOf('#') != 0
		) {
			return null;
		}

		var rColor = color.substring(1, 3);
		var gColor = color.substring(3, 5);
		var bColor = color.substring(5, 7);
		if (
			isNaN('0x' + rColor)
			|| isNaN('0x' + gColor)
			|| isNaN('0x' + bColor)
		) {
			return null;
		}

		var rgb = { r:0, g:0, b:0 };
		rgb.r = parseInt(rColor, 16);
		rgb.g = parseInt(gColor, 16);
		rgb.b = parseInt(bColor, 16);

		return rgb;
	}

	/**
	 * Colorへの変換。
	 *
	 * @return {Object} RGB値
	 * @return {String} 色値
	 */
	aaoh.kaze.animation.BaseColorAnimation.prototype.
	toColor = function(rgb) {
		if (rgb == null) {
			return null;
		}

		var rColor = rgb.r.toString(16);
		var gColor = rgb.g.toString(16);
		var bColor = rgb.b.toString(16);

		if (rColor.length <= 1) {
			rColor = '0' + rColor;
		}
		if (gColor.length <= 1) {
			gColor = '0' + gColor;
		}
		if (bColor.length <= 1) {
			bColor = '0' + bColor;
		}

		var color = '#' + rColor + gColor + bColor;
		return color;
	}
})();

(function () {
/**
 * ForegroundColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element_    エレメント
 * @param {Number} speed_      スピード(msec)
 * @param {Object} startValue_ 開始値
 * @param {Object} stopValue_  終了値
 * @param {Object} stepValue_  ステップ値
 */
aaoh.kaze.animation.ForegroundColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element_, speed_,
		startValue_, stopValue_, stepValue_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = this.toRGB('#000000');
	this.DEFAULT_STOP_VALUE  = this.toRGB('#ffffff');
	this.DEFAULT_STEP_VALUE  = this.toRGB('#202020');

	startValue_ = this.toRGB(startValue_);
	stopValue_  = this.toRGB(stopValue_);
	stepValue_  = this.toRGB(stepValue_);
	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}
	if (startValue_ == null) {
		startValue_ = this.DEFAULT_START_VALUE
	}
	if (stopValue_ == null) {
		stopValue_ = this.DEFAULT_STOP_VALUE
	}
	if (stepValue_ == null) {
		stepValue_ = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element_, speed_,
		startValue_, stopValue_, stepValue_,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	startCallable = function(element, startValue) {
		var color = this.toColor(startValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	stopCallable = function(element, stopValue) {
		var color = this.toColor(stopValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	stepCallable = function(element, nextValue) {
		var color = this.toColor(nextValue);

		aaoh.kaze.Element.setForeground(element, color);
	}
})();

(function () {
/**
 * BackgroundColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element_    エレメント
 * @param {Number} speed_      スピード(msec)
 * @param {Object} startValue_ 開始値
 * @param {Object} stopValue_  終了値
 * @param {Object} stepValue_  ステップ値
 */
aaoh.kaze.animation.BackgroundColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element_, speed_,
		startValue_, stopValue_, stepValue_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = this.toRGB('#ffffff');
	this.DEFAULT_STOP_VALUE  = this.toRGB('#000000');
	this.DEFAULT_STEP_VALUE  = this.toRGB('#202020');

	startValue_ = this.toRGB(startValue_);
	stopValue_  = this.toRGB(stopValue_);
	stepValue_  = this.toRGB(stepValue_);
	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}
	if (startValue_ == null) {
		startValue_ = this.DEFAULT_START_VALUE
	}
	if (stopValue_ == null) {
		stopValue_ = this.DEFAULT_STOP_VALUE
	}
	if (stepValue_ == null) {
		stepValue_ = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element_, speed_,
		startValue_, stopValue_, stepValue_,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	startCallable = function(element, startValue) {
		var color = this.toColor(startValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	stopCallable = function(element, stopValue) {
		var color = this.toColor(stopValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	stepCallable = function(element, nextValue) {
		var color = this.toColor(nextValue);

		aaoh.kaze.Element.setForeground(element, color);
	}
})();

(function () {
/**
 * BorderColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element_    エレメント
 * @param {Number} speed_      スピード(msec)
 * @param {Object} startValue_ 開始値
 * @param {Object} stopValue_  終了値
 * @param {Object} stepValue_  ステップ値
 */
aaoh.kaze.animation.BorderColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element_, speed_,
		startValue_, stopValue_, stepValue_
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = this.toRGB('#000000');
	this.DEFAULT_STOP_VALUE  = this.toRGB('#ffffff');
	this.DEFAULT_STEP_VALUE  = this.toRGB('#202020');

	startValue_ = this.toRGB(startValue_);
	stopValue_  = this.toRGB(stopValue_);
	stepValue_  = this.toRGB(stepValue_);
	if (speed_ == null) {
		speed_ = this.DEFAULT_SPEED;
	}
	if (startValue_ == null) {
		startValue_ = this.DEFAULT_START_VALUE
	}
	if (stopValue_ == null) {
		stopValue_ = this.DEFAULT_STOP_VALUE
	}
	if (stepValue_ == null) {
		stepValue_ = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element_, speed_,
		startValue_, stopValue_, stepValue_,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	startCallable = function(element, startValue) {
		var color = this.toColor(startValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	stopCallable = function(element, stopValue) {
		var color = this.toColor(stopValue);

		aaoh.kaze.Element.setForeground(element, color);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	stepCallable = function(element, nextValue) {
		var color = this.toColor(nextValue);

		aaoh.kaze.Element.setForeground(element, color);
	}
})();

